#include "unavailability.h"
#include "accesscontrol.h"
#include "cpersons.h"

Unavailability::Unavailability(QObject *parent)
    : Unavailability(-1, -1, QDate(), QDate(), parent)
{
}

Unavailability::Unavailability(const int id, const int personId, const QDate startDate, const QDate endDate, QObject *parent = nullptr)
    : QObject(parent), b_unavailabilityId(id), b_personId(personId), b_startDate(startDate), b_endDate(endDate)
{
    QObject::connect(this, &Unavailability::personIdChanged, [&]() { b_isDirty = true; });
    QObject::connect(this, &Unavailability::startDateChanged, [&]() { b_isDirty = true; });
    QObject::connect(this, &Unavailability::endDateChanged, [&]() { b_isDirty = true; });
}

bool Unavailability::save()
{
    if (!isDirty())
        return true;

    AccessControl *ac = &Singleton<AccessControl>::Instance();
    if (!ac->user() || !ac->user()->hasPermission(PermissionRule::CanEditAvailabilities))
        return false;

    sql_class *sql = &Singleton<sql_class>::Instance();
    sql_item queryitems;
    queryitems.insert(":id", unavailabilityId());
    int id = unavailabilityId() > 0 ? sql->selectScalar("SELECT id FROM unavailables WHERE id = :id", &queryitems, -1).toInt() : -1;

    sql_item insertItems;
    insertItems.insert("person_id", personId());
    insertItems.insert("start_date", startDate());
    insertItems.insert("end_date", endDate());

    bool ret = false;
    if (id > 0) {
        // update
        ret = sql->updateSql("unavailables", "id", QString::number(id), &insertItems);
    } else {
        // insert new row
        int newId = sql->insertSql("unavailables", &insertItems, "id");
        ret = newId > 0;
        if (newId > 0)
            setUnavailabilityId(newId);
    }
    setIsDirty(!ret);

    return ret;
}

UnavailabilityModel::UnavailabilityModel(QObject *parent)
    : QAbstractTableModel(parent)
{
}

UnavailabilityModel::~UnavailabilityModel()
{
    qDeleteAll(unavailabilities);
    unavailabilities.clear();
}

int UnavailabilityModel::rowCount(const QModelIndex &parent) const
{
    Q_UNUSED(parent)
    return unavailabilities.empty() ? 0 : unavailabilities.count();
}

int UnavailabilityModel::columnCount(const QModelIndex &parent) const
{
    Q_UNUSED(parent)
    return 6;
}

QHash<int, QByteArray> UnavailabilityModel::roleNames() const
{
    QHash<int, QByteArray> items;
    items[UnavailabilityIdRole] = "unavailabilityId";
    items[UnavailabilityRole] = "unavailability";
    items[PersonIdRole] = "personId";
    items[StartDateRole] = "startDate";
    items[EndDateRole] = "endDate";
    items[YearRole] = "year";
    return items;
}

QVariantMap UnavailabilityModel::get(int row) const
{
    QHash<int, QByteArray> names = roleNames();
    QHashIterator<int, QByteArray> i(names);
    QVariantMap res;
    QModelIndex idx = index(row, 0);
    while (i.hasNext()) {
        i.next();
        QVariant data = idx.data(i.key());
        res[i.value()] = data;
    }
    return res;
}

QVariant UnavailabilityModel::data(const QModelIndex &index, int role) const
{
    int row = index.row();
    if (row < 0 || row > unavailabilities.count())
        return QVariant();

    switch (role) {
    case UnavailabilityIdRole:
        return unavailabilities[index.row()]->unavailabilityId();
    case UnavailabilityRole:
        return QVariant::fromValue(unavailabilities[index.row()]);
    case PersonIdRole:
        return unavailabilities[index.row()]->personId();
    case StartDateRole:
        return unavailabilities[index.row()]->startDate();
    case EndDateRole:
        return unavailabilities[index.row()]->endDate();
    case YearRole:
        return unavailabilities[index.row()]->startDate().isValid()
                ? unavailabilities[index.row()]->startDate().year()
                : QDate::currentDate().year();
    default:
        return QVariant();
    }
}

bool UnavailabilityModel::setData(const QModelIndex &index, const QVariant &value, int role)
{
    if (index.row() < 0 || index.row() > unavailabilities.count())
        return false;
    Unavailability *currUnavailability = unavailabilities[index.row()];

    switch (role) {
    case Roles::PersonIdRole:
        currUnavailability->setPersonId(value.toInt());
        emit dataChanged(index, index, { role });
        break;
    case Roles::StartDateRole: {
        QDate newDate = value.toDate();
        if (newDate.addYears(99).year() <= QDate::currentDate().year())
            newDate = newDate.addYears(100);
        currUnavailability->setStartDate(newDate);
        emit dataChanged(this->index(index.row(), 3), this->index(index.row(), 3));
        emit dataChanged(index, index, { role });
        break;
    }
    case Roles::EndDateRole: {
        QDate newDate = value.toDate();
        if (newDate.addYears(99).year() <= QDate::currentDate().year())
            newDate = newDate.addYears(100);
        currUnavailability->setEndDate(newDate);
        emit dataChanged(this->index(index.row(), 4), this->index(index.row(), 4));
        emit dataChanged(index, index, { role });
        break;
    }
    default:
        break;
    }

    return true;
}

QModelIndex UnavailabilityModel::getUnavailabilityIndex(int unavailabilityId) const
{
    for (int row = 0; row < this->rowCount(); ++row) {
        QModelIndex rowIndex = this->index(row, 0);
        if (rowIndex.data(UnavailabilityIdRole) == unavailabilityId)
            return rowIndex;
    }
    return QModelIndex();
}

QModelIndex UnavailabilityModel::getUnavailabilityIndex(int personId, QDate startDate, QDate endDate) const
{
    for (int row = 0; row < this->rowCount(); ++row) {
        QModelIndex rowIndex = this->index(row, 0);
        if (rowIndex.data(PersonIdRole) == personId
            && rowIndex.data(StartDateRole) == startDate
            && rowIndex.data(EndDateRole) == endDate)
            return rowIndex;
    }
    return QModelIndex();
}

int UnavailabilityModel::addUnavailability(Unavailability *Unavailability)
{
    beginInsertRows(QModelIndex(), rowCount(), rowCount());
    unavailabilities << Unavailability;
    endInsertRows();
    return Unavailability->unavailabilityId();
}

QModelIndex UnavailabilityModel::addUnavailability(int personId)
{
    beginInsertRows(QModelIndex(), rowCount(), rowCount());
    Unavailability *newUnavailability = new Unavailability(this);
    newUnavailability->setPersonId(personId);
    unavailabilities << newUnavailability;
    endInsertRows();
    emit modelChanged();
    return index(rowCount() - 1, 0);
}

void UnavailabilityModel::removeUnavailability(int unavailabilityId)
{
    // remove unavailability
    QModelIndex unavailabilityIndex = getUnavailabilityIndex(unavailabilityId);
    if (unavailabilityIndex.isValid()) {
        int personId = this->data(unavailabilityIndex, PersonIdRole).toInt();
        QDate startDate = this->data(unavailabilityIndex, StartDateRole).toDate();
        QDate endDate = this->data(unavailabilityIndex, EndDateRole).toDate();
        removeUnavailability(personId, startDate, endDate);
    }
}

void UnavailabilityModel::removeUnavailability(int personId, QDate startDate, QDate endDate)
{
    // remove unavailability
    cpersons *cp = new cpersons;
    Person *p = cp->getPerson(personId);
    if (p == nullptr)
        return;
    if (p->removeUnavailability(startDate, endDate)) {
        QModelIndex unavailabilityIndex = getUnavailabilityIndex(personId, startDate, endDate);
        if (unavailabilityIndex.isValid()) {
            int row = unavailabilityIndex.row();
            beginRemoveRows(QModelIndex(), row, row);
            unavailabilities.erase(std::next(unavailabilities.begin(), row));
            endRemoveRows();
        }
    }
    emit modelChanged();
}

void UnavailabilityModel::loadUnavailabilities(int personId)
{
    beginResetModel();
    qDeleteAll(unavailabilities);
    unavailabilities.clear();

    sql_class *sql = &Singleton<sql_class>::Instance();
    sql_item s;
    s.insert("person_id", personId);
    sql_items UnavailabilityRows = sql->selectSql("SELECT * FROM unavailables WHERE person_id=:person_id AND active ORDER BY start_date DESC", &s);

    QDate firstDate = QDate(2100, 1, 1);
    QDate lastDate = QDate(1900, 1, 1);
    if (!UnavailabilityRows.empty()) {
        for (unsigned int i = 0; i < UnavailabilityRows.size(); i++) {
            sql_item s = UnavailabilityRows[i];
            QDate startDate = s.value("start_date").toDate();
            QDate endDate = s.value("end_date").toDate();
            firstDate = startDate < firstDate ? startDate : firstDate;
            lastDate = endDate > lastDate ? endDate : lastDate;
            addUnavailability(new Unavailability(s.value("id").toInt(),
                                                 s.value("person_id").toInt(),
                                                 s.value("start_date").toDate(),
                                                 s.value("end_date").toDate()));
        }
    }
    endResetModel();
    emit modelChanged();
}

bool UnavailabilityModel::isUnavailable(QDate date)
{
    for (int i = 0; i < unavailabilities.size(); i++) {
        if (unavailabilities[i]->startDate() <= date && date <= unavailabilities[i]->endDate())
            return true;
    }
    return false;
}

void UnavailabilityModel::save()
{
    sql_class *sql = &Singleton<sql_class>::Instance();
    sql->startTransaction();
    for (int i = 0; i < unavailabilities.count(); i++) {
        Unavailability *u = unavailabilities[i];
        if (u->isDirty()) {
            u->save();
        }
    }
    sql->commitTransaction();
}

UnavailabilitySortFilterProxyModel::UnavailabilitySortFilterProxyModel(QObject *parent)
    : QSortFilterProxyModel(parent)
{
}

QObject *UnavailabilitySortFilterProxyModel::source() const
{
    return sourceModel();
}

void UnavailabilitySortFilterProxyModel::setSource(QObject *source)
{
    setSourceModel(qobject_cast<QAbstractItemModel *>(source));
    emit sourceChanged();
}

QByteArray UnavailabilitySortFilterProxyModel::sortRole() const
{
    return roleNames().value(QSortFilterProxyModel::sortRole());
}

void UnavailabilitySortFilterProxyModel::setSortRole(const QByteArray &role)
{
    QSortFilterProxyModel::setSortRole(roleKey(role));
    emit sortChanged();
}

bool UnavailabilitySortFilterProxyModel::lessThan(const QModelIndex &left, const QModelIndex &right) const
{
    // sort by date within sections
    QVariant leftData = sourceModel()->data(left, UnavailabilityModel::Roles::StartDateRole);
    QVariant rightData = sourceModel()->data(right, UnavailabilityModel::Roles::EndDateRole);
    if (leftData.userType() == QMetaType::QDate) {
        return leftData.toDate() < rightData.toDate();
    } else
        return leftData.toString() < rightData.toString();
}

int UnavailabilitySortFilterProxyModel::roleKey(const QByteArray &role) const
{
    QHash<int, QByteArray> roles = roleNames();
    QHashIterator<int, QByteArray> it(roles);
    while (it.hasNext()) {
        it.next();
        if (it.value() == role)
            return it.key();
    }
    return -1;
}

UnavailabilityValidator::UnavailabilityValidator(QObject *parent)
    : QValidator(parent), m_model(nullptr), b_unavailabilityId(-1), m_role(UnavailabilityModel::Roles::None)
{
}

UnavailabilityValidator::~UnavailabilityValidator()
{
}

QValidator::State UnavailabilityValidator::validate(QString &input, int &pos) const
{
    Q_UNUSED(pos)

    if (input.isEmpty()) {
        emit errorChanged("");
        return Acceptable;
    }
    if (!model()) {
        emit errorChanged("");
        return Acceptable;
    }
    if (role() == UnavailabilityModel::Roles::None) {
        emit errorChanged("");
        return Acceptable;
    }
    QModelIndex modelIndex = model()->getUnavailabilityIndex(unavailabilityId());
    if (!modelIndex.isValid()) {
        emit errorChanged("");
        return Acceptable;
    }

    // check date
    QDate startDate = model()->data(modelIndex, UnavailabilityModel::Roles::StartDateRole).toDate();
    QDate endDate = model()->data(modelIndex, UnavailabilityModel::Roles::EndDateRole).toDate();
    if (role() == UnavailabilityModel::Roles::EndDateRole) {
        if (!startDate.isValid()) {
            emit errorChanged("");
            return Intermediate;
        }
        if (!endDate.isValid()) {
            emit errorChanged("");
            return Intermediate;
        }
        if (endDate < startDate) {
            emit errorChanged(tr("The end date is before the start. Please correct the dates.", "Unavailability date"));
            return Invalid;
        }
        if (endDate < QDate::currentDate()) {
            emit errorChanged(tr("The time period is in the past. Please correct the dates.", "Unavailability date"));
            return Invalid;
        }

        emit errorChanged("");
        return Acceptable;
    }

    emit errorChanged("");
    return Acceptable;
}

UnavailabilityModel *UnavailabilityValidator::model() const
{
    return m_model;
}

void UnavailabilityValidator::setModel(UnavailabilityModel *newModel)
{
    if (m_model == newModel)
        return;
    m_model = newModel;
    emit modelChanged();
}

UnavailabilityModel::Roles UnavailabilityValidator::role() const
{
    return m_role;
}

void UnavailabilityValidator::setRole(UnavailabilityModel::Roles newRole)
{
    if (m_role == newRole)
        return;
    m_role = newRole;
    emit roleChanged();
}
