/**
 * This file is part of TheocBase.
 *
 * Copyright (C) 2011-2019, TheocBase Development Team, see AUTHORS.
 *
 * TheocBase is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * TheocBase is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with TheocBase.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef TERRITORYSTREET_H
#define TERRITORYSTREET_H

#include <QObject>
#include <QAbstractTableModel>
#include <QtConcurrent>
#include "dataobject.h"
#include "sql_class.h"
#include <QSortFilterProxyModel>
#include <QGeoRectangle>
#include <QValidator>
#include <QObjectBindableProperty>
#if defined(Q_OS_ANDROID) || defined(Q_OS_IOS)
// GDAL missing
#elif defined(Q_OS_LINUX)
#include "gdal/ogrsf_frmts.h"
#include "gdal/gdal_alg.h"
#elif defined(Q_OS_MAC) || defined(Q_OS_WIN)
#include "ogrsf_frmts.h"
#include "gdal_alg.h"
#endif

class TerritoryStreet : public QObject
{
    Q_OBJECT
    Q_PROPERTY(int streetId READ streetId CONSTANT)
    Q_PROPERTY(int territoryId READ territoryId WRITE setTerritoryId BINDABLE bindableTerritoryId)
    Q_PROPERTY(QString streetName READ streetName WRITE setStreetName BINDABLE bindableStreetName)
    Q_PROPERTY(QString fromNumber READ fromNumber WRITE setFromNumber BINDABLE bindableFromNumber)
    Q_PROPERTY(QString toNumber READ toNumber WRITE setToNumber BINDABLE bindableToNumber)
    Q_PROPERTY(int quantity READ quantity WRITE setQuantity BINDABLE bindableQuantity)
    Q_PROPERTY(int streetTypeId READ streetTypeId WRITE setStreetTypeId BINDABLE bindableStreetTypeId)
    Q_PROPERTY(QString wktGeometry READ wktGeometry BINDABLE bindableWktGeometry)
    Q_PROPERTY(QList<QList<QGeoCoordinate>> multiLine READ multiLine WRITE setMultiLine BINDABLE bindableMultiLine)

public:
    explicit TerritoryStreet(QObject *parent = nullptr);
    TerritoryStreet(const int territoryId, QObject *parent = nullptr);
    TerritoryStreet(const int territoryId, const QString streetName, const QString fromNumber, const QString toNumber, const int quantity, const int streetTypeId,
                    const QString wktGeometry, QObject *parent = nullptr);
    TerritoryStreet(const int streetId, const int territoryId, const QString streetName, const QString fromNumber, const QString toNumber, const int quantity, const int streetTypeId,
                    const QString wktGeometry, QObject *parent = nullptr);
    ~TerritoryStreet();

    int streetId() const;

    int territoryId() { return b_territoryId.value(); }
    void setTerritoryId(int newValue) { b_territoryId = newValue; }
    QBindable<int> bindableTerritoryId() { return &b_territoryId; }

    QString streetName() { return b_streetName.value(); }
    void setStreetName(QString newValue) { b_streetName = newValue; }
    QBindable<QString> bindableStreetName() { return &b_streetName; }

    QString fromNumber() { return b_fromNumber.value(); }
    void setFromNumber(QString newValue) { b_fromNumber = newValue; }
    QBindable<QString> bindableFromNumber() { return &b_fromNumber; }

    QString toNumber() { return b_toNumber.value(); }
    void setToNumber(QString newValue) { b_toNumber = newValue; }
    QBindable<QString> bindableToNumber() { return &b_toNumber; }

    int quantity() { return b_quantity.value(); }
    void setQuantity(int newValue) { b_quantity = newValue; }
    QBindable<int> bindableQuantity() { return &b_quantity; }

    int streetTypeId() { return b_streetTypeId.value(); }
    void setStreetTypeId(int newValue) { b_streetTypeId = newValue; }
    QBindable<int> bindableStreetTypeId() { return &b_streetTypeId; }

    QString wktGeometry() { return b_wktGeometry.value(); }
    void setWktGeometry(QString newValue) { b_wktGeometry = newValue; }
    QBindable<QString> bindableWktGeometry() { return &b_wktGeometry; }

    QList<QList<QGeoCoordinate>> multiLine() { return b_multiLine.value(); }
    void setMultiLine(const QList<QList<QGeoCoordinate>> &newValue);
    QBindable<QList<QList<QGeoCoordinate>>> bindableMultiLine() { return &b_multiLine; }

#if defined(Q_OS_ANDROID) || defined(Q_OS_IOS)
    // GDAL missing
#elif defined(Q_OS_LINUX) || defined(Q_OS_MAC) || defined(Q_OS_WIN)
    OGRGeometry *getOGRGeometry();
    bool setOGRGeometry(OGRGeometry *newGeometry);
#endif

    bool isDirty() { return b_isDirty.value(); }
    void setIsDirty(bool newValue) { b_isDirty = newValue; }
    QBindable<bool> bindableIsDirty() { return &b_isDirty; }

    bool save();
    Q_INVOKABLE QGeoRectangle boundingGeoRectangle();
    /**
     * @brief generateMultiLine - Converts the WKT geometry value and sets the MultiLine property
     * This can be executed as necessary, e.g. when the data needs to be displayed on the map.
     */
    void generateMultiLine();

signals:
    void territoryIdChanged();
    void streetNameChanged();
    void fromNumberChanged();
    void toNumberChanged();
    void quantityChanged();
    void streetTypeIdChanged();
    void wktGeometryChanged();
    void multiLineChanged();

private:
    int m_streetId;
    Q_OBJECT_BINDABLE_PROPERTY(TerritoryStreet, int, b_territoryId, &TerritoryStreet::territoryIdChanged);
    Q_OBJECT_BINDABLE_PROPERTY(TerritoryStreet, QString, b_streetName, &TerritoryStreet::streetNameChanged);
    Q_OBJECT_BINDABLE_PROPERTY(TerritoryStreet, QString, b_fromNumber, &TerritoryStreet::fromNumberChanged);
    Q_OBJECT_BINDABLE_PROPERTY(TerritoryStreet, QString, b_toNumber, &TerritoryStreet::toNumberChanged);
    Q_OBJECT_BINDABLE_PROPERTY(TerritoryStreet, int, b_quantity, &TerritoryStreet::quantityChanged);
    Q_OBJECT_BINDABLE_PROPERTY(TerritoryStreet, int, b_streetTypeId, &TerritoryStreet::streetTypeIdChanged);
    QProperty<QString> b_wktGeometry;
    QProperty<QList<QList<QGeoCoordinate>>> b_multiLine;
    QProperty<bool> b_isDirty { false };
};

class TerritoryStreetModel : public QAbstractTableModel
{
    Q_OBJECT
    Q_PROPERTY(int count READ count NOTIFY countChanged)

public:
    enum Roles {
        None = 0,
        StreetIdRole = Qt::UserRole,
        StreetRole,
        TerritoryIdRole,
        StreetNameRole,
        FromNumberRole,
        ToNumberRole,
        QuantityRole,
        StreetTypeIdRole,
        StreetTypeNameRole,
        StreetTypeColorRole,
        WktGeometryRole,
        MultiLineRole
    };
    Q_ENUM(Roles)

    int count() { return rowCount(); }

    QHash<int, QByteArray> roleNames() const override;

    TerritoryStreetModel();
    TerritoryStreetModel(QObject *parent);
    Q_INVOKABLE void initialize();

    int rowCount(const QModelIndex &parent = QModelIndex()) const override;
    int columnCount(const QModelIndex &parent = QModelIndex()) const override;
    Q_INVOKABLE QVariantMap get(int row);
    QVariant data(const QModelIndex &index, int role = Qt::DisplayRole) const override;
    Q_INVOKABLE bool setData(const QModelIndex &index, const QVariant &value, int role = Qt::EditRole) override;
    Qt::ItemFlags flags(const QModelIndex &index) const override;
    QModelIndex index(int row, int column, const QModelIndex &parent = QModelIndex()) const override;
    Q_INVOKABLE QModelIndex getStreetIndex(int streetId) const;
    int addStreet(TerritoryStreet *territoryStreet);
    Q_INVOKABLE int addStreet(int territoryId, const QString streetName, QString wktGeometry, int streetTypeId = 1);
    Q_INVOKABLE QVariant findStreet(int streetId);
    Q_INVOKABLE void removeStreet(int streetId);
    Q_INVOKABLE bool removeRows(int row, int count, const QModelIndex &parent = QModelIndex()) override;
    Q_INVOKABLE void loadStreets(int territoryId = 0);
    Q_INVOKABLE void saveStreets();
#if defined(Q_OS_ANDROID) || defined(Q_OS_IOS)
    // GDAL missing
#elif defined(Q_OS_LINUX) || defined(Q_OS_MAC) || defined(Q_OS_WIN)
    Q_INVOKABLE QString cutoffStreetSection(int streetId, QList<QGeoCoordinate> cutAreaPath);
#endif
    Q_INVOKABLE bool updateStreet(int streetId, QString wktGeometry);
    Q_INVOKABLE bool updateStreet(int streetId, int territoryId);
    Q_INVOKABLE bool updateStreet(int streetId, int territoryId, const QString streetName, QString wktGeometry);

    Q_INVOKABLE int getDefaultStreetTypeId();

private:
    TerritoryStreet *getItem(const QModelIndex &index) const;

    QList<TerritoryStreet *> territoryStreets;
    DataObjectListModel *streetTypeListModel;

signals:
    void modelChanged();
    void countChanged();
    void notification();
    void editCompleted();
};

class TerritoryStreetSortFilterProxyModel : public QSortFilterProxyModel
{
    Q_OBJECT
    Q_PROPERTY(QObject *source READ source WRITE setSource NOTIFY sourceChanged)
    Q_PROPERTY(int filterTerritoryId READ filterTerritoryId WRITE setFilterTerritoryId NOTIFY filterTerritoryIdChanged)
    Q_PROPERTY(QString filterText READ filterText WRITE setFilterText NOTIFY filterTextChanged)
    Q_PROPERTY(QByteArray sortRole READ sortRole WRITE setSortRole NOTIFY sortChanged)
    Q_PROPERTY(QByteArray groupByRole READ groupByRole WRITE setGroupByRole NOTIFY groupByChanged)

public:
    TerritoryStreetSortFilterProxyModel(QObject *parent = nullptr);

    QObject *source() const;
    void setSource(QObject *source);

    int filterTerritoryId() const;
    void setFilterTerritoryId(int newValue);

    QString filterText() const;
    void setFilterText(QString newValue);

    QByteArray sortRole() const;
    void setSortRole(const QByteArray &role);

    QByteArray groupByRole() const;
    void setGroupByRole(const QByteArray &role);

    Q_INVOKABLE virtual void sort(int column, Qt::SortOrder order = Qt::AscendingOrder) override
    {
        QSortFilterProxyModel::sort(column, order);
    }

protected:
    bool filterAcceptsRow(int sourceRow, const QModelIndex &sourceParent) const override;
    bool lessThan(const QModelIndex &left, const QModelIndex &right) const override;
    int roleKey(const QByteArray &role) const;

signals:
    void sourceChanged();
    void filterTerritoryIdChanged();
    void filterTextChanged();
    void sortChanged();
    void groupByChanged();

private:
    int m_filterTerritoryId;
    QString m_filterText;
    QByteArray m_groupByRole;
};

class TerritoryStreetValidator : public QValidator
{
    Q_OBJECT
    Q_PROPERTY(TerritoryStreetModel *model READ model WRITE setModel NOTIFY modelChanged)
    Q_PROPERTY(int streetId READ streetId WRITE setStreetId NOTIFY streetIdChanged)
    Q_PROPERTY(TerritoryStreetModel::Roles role READ role WRITE setRole NOTIFY roleChanged)
public:
    TerritoryStreetValidator(QObject *parent = nullptr);
    ~TerritoryStreetValidator();

    QValidator::State validate(QString &input, int &pos) const override;

    TerritoryStreetModel *model() const;
    void setModel(TerritoryStreetModel *newModel);

    TerritoryStreetModel::Roles role() const;
    void setRole(TerritoryStreetModel::Roles newRole);

    int streetId() const;
    void setStreetId(int newStreetId);

Q_SIGNALS:
    void modelChanged();
    void streetIdChanged();
    void errorChanged(const QString &error) const;
    void roleChanged();

private:
    TerritoryStreetModel *m_model;
    int m_streetId;
    TerritoryStreetModel::Roles m_role;
};

#endif // TERRITORYSTREET_H
