/**
 * This file is part of TheocBase.
 *
 * Copyright (C) 2011-2019, TheocBase Development Team, see AUTHORS.
 *
 * TheocBase is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * TheocBase is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with TheocBase.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "territorystreet.h"
#include "cterritories.h"

TerritoryStreet::TerritoryStreet(QObject *parent)
    : TerritoryStreet(0, -1, "", "", "", 0, 0, "", parent)
{
}

TerritoryStreet::TerritoryStreet(const int territoryId, const QString streetName,
                                 const QString fromNumber, const QString toNumber, const int quantity, const int streetTypeId,
                                 const QString wktGeometry, QObject *parent)
    : TerritoryStreet(0, territoryId, streetName, fromNumber, toNumber, quantity, streetTypeId, wktGeometry, parent)
{
}

TerritoryStreet::TerritoryStreet(const int streetId, const int territoryId, const QString streetName, const QString fromNumber, const QString toNumber, const int quantity, const int streetTypeId, const QString wktGeometry, QObject *parent)
    : QObject(parent), m_streetId(streetId), b_territoryId(territoryId), b_streetName(streetName), b_fromNumber(fromNumber), b_toNumber(toNumber), b_quantity(quantity), b_streetTypeId(streetTypeId), b_wktGeometry(wktGeometry)
{
    QObject::connect(this, &TerritoryStreet::territoryIdChanged, [&]() { b_isDirty = true; });
    QObject::connect(this, &TerritoryStreet::streetNameChanged, [&]() { b_isDirty = true; });
    QObject::connect(this, &TerritoryStreet::fromNumberChanged, [&]() { b_isDirty = true; });
    QObject::connect(this, &TerritoryStreet::toNumberChanged, [&]() { b_isDirty = true; });
    QObject::connect(this, &TerritoryStreet::quantityChanged, [&]() { b_isDirty = true; });
    QObject::connect(this, &TerritoryStreet::streetTypeIdChanged, [&]() { b_isDirty = true; });
    QObject::connect(this, &TerritoryStreet::wktGeometryChanged, [&]() { b_isDirty = true; });
    QObject::connect(this, &TerritoryStreet::streetTypeIdChanged, [&]() { b_isDirty = true; });
}

TerritoryStreet::~TerritoryStreet()
{
}

int TerritoryStreet::streetId() const
{
    return m_streetId;
}

void TerritoryStreet::generateMultiLine()
{
    // catch MULTILINESTRING and LINESTRING
    QRegularExpressionMatchIterator iMultiLineString = multiLineStringRegExp->globalMatch(b_wktGeometry);

    QList<QList<QGeoCoordinate>> sectionList;
    while (iMultiLineString.hasNext()) {
        QRegularExpressionMatch lineStringMatch = iMultiLineString.next();
        QString lineString = lineStringMatch.captured(0);

        QGeoPath geoPath;
        QRegularExpressionMatchIterator iMatch = coordinateRegExp->globalMatch(lineString);
        while (iMatch.hasNext()) {
            QRegularExpressionMatch match = iMatch.next();
            double longitude = match.captured(1).toDouble();
            double latitude = match.captured(2).toDouble();
            QGeoCoordinate coord(latitude, longitude);
            geoPath.addCoordinate(coord);
        }
        sectionList.append(geoPath.path());
    }
    b_multiLine = sectionList;
}

void TerritoryStreet::setMultiLine(const QList<QList<QGeoCoordinate>> &newValue)
{
#if defined(Q_OS_ANDROID) || defined(Q_OS_IOS)
    // GDAL missing
    Q_UNUSED(newValue)
#elif defined(Q_OS_LINUX) || defined(Q_OS_MAC) || defined(Q_OS_WIN)
    OGRMultiLineString *multiLineString = new OGRMultiLineString();
    foreach (QList<QGeoCoordinate> section, newValue) {
        OGRLineString *lineString = new OGRLineString();
        foreach (QGeoCoordinate coordinate, section) {
            lineString->addPoint(coordinate.latitude(),
                                 coordinate.longitude());
        }
        multiLineString->addGeometry(lineString);
    }
    char *wkt_new = nullptr;
    multiLineString->exportToWkt(&wkt_new);

    // update properties
    Qt::beginPropertyUpdateGroup();
    b_wktGeometry = QString(wkt_new);
    b_multiLine = newValue;
    Qt::endPropertyUpdateGroup();
#endif
}

#if defined(Q_OS_ANDROID) || defined(Q_OS_IOS)
// GDAL missing
#elif defined(Q_OS_LINUX) || defined(Q_OS_MAC) || defined(Q_OS_WIN)
OGRGeometry *TerritoryStreet::getOGRGeometry()
{
    //    OGRSpatialReference osr;
    //    osr.SetWellKnownGeogCS("EPSG:4326");
    OGRGeometry *streetGeometry = nullptr;

    // Parse WKT
    if (wktGeometry().isNull())
        return nullptr;
    QByteArray bytes = wktGeometry().toUtf8();
    const char *data = bytes.constData();
    OGRErr err = OGRGeometryFactory::createFromWkt(data, nullptr, &streetGeometry);
    if (err != OGRERR_NONE) {
        // process error, like emit signal
        return nullptr;
    }
    if (!streetGeometry->IsValid()) {
        qDebug() << "Trying to make WKT geometry of territory street" << streetId() << "valid";
        streetGeometry = streetGeometry->MakeValid();
    }
    if (!streetGeometry->IsValid()) {
        qDebug() << "Saved WKT geometry of territory street" << streetId() << "is invalid!";
    }
    return streetGeometry;
}

bool TerritoryStreet::setOGRGeometry(OGRGeometry *newGeometry)
{
    char *txt;
    if (newGeometry != nullptr) {
        OGRGeometry *geometry = newGeometry->MakeValid();
        if (!geometry->IsEmpty() && !geometry->IsValid()) {
            qDebug() << "Geometry of territory street" << streetId() << "cannot be set because it is not valid!";
            return false;
        }
        if (geometry->exportToWkt(&txt) == OGRERR_NONE) {
            // test geometry before saving
            OGRSpatialReference osr;
            OGRGeometry *testGeometry = nullptr;

            // Parse WKT
            OGRErr err = OGRGeometryFactory::createFromWkt(txt, &osr, &testGeometry);
            if (err != OGRERR_NONE) {
                // process error, like emit signal
                qDebug() << "Geometry check for territory street" << streetId() << "failed!";
                return false;
            }
            if (!testGeometry->IsValid()) {
                qDebug() << "Geometry check for territory street" << streetId() << "failed because it is invalid!";
                return false;
            }

            qDebug() << "Geometry of territory street" << streetId() << "is valid. Set WKT geometry.";
            setWktGeometry(QString(txt));
            CPLFree(txt);
            return true;
        }
    }
    return false;
}
#endif

bool TerritoryStreet::save()
{
    if (streetName() == "")
        return false;

    AccessControl *ac = &Singleton<AccessControl>::Instance();
    if (!ac->user() || !ac->user()->hasPermission(PermissionRule::CanEditTerritories))
        return false;

    // save changes to database
    if (!isDirty())
        return true;
    sql_class *sql = &Singleton<sql_class>::Instance();

    sql_item queryitems;
    queryitems.insert(":id", m_streetId);
    int streetId = m_streetId > 0 ? sql->selectScalar("SELECT id FROM territory_street WHERE id = :id", &queryitems, -1).toInt() : -1;

    sql_item insertItems;
    insertItems.insert("territory_id", territoryId());
    insertItems.insert("street_name", streetName());
    insertItems.insert("from_number", fromNumber());
    insertItems.insert("to_number", toNumber());
    insertItems.insert("quantity", quantity());
    insertItems.insert("streettype_id", streetTypeId());
    insertItems.insert("wkt_geometry", wktGeometry());

    bool ret = false;
    if (streetId > 0) {
        // update
        ret = sql->updateSql("territory_street", "id", QString::number(streetId), &insertItems);
    } else {
        // insert new row
        int newId = sql->insertSql("territory_street", &insertItems, "id");
        ret = newId > 0;
        if (newId > 0)
            m_streetId = newId;
    }
    setIsDirty(!ret);

    return ret;
}

QGeoRectangle TerritoryStreet::boundingGeoRectangle()
{
    QGeoRectangle r;
#if defined(Q_OS_ANDROID) || defined(Q_OS_IOS)
    // catch MULTILINESTRING and LINESTRING
    QRegularExpressionMatchIterator iMultiLineString = multiLineStringRegExp->globalMatch(b_wktGeometry);

    QGeoPath geoPath;
    while (iMultiLineString.hasNext()) {
        QRegularExpressionMatch lineStringMatch = iMultiLineString.next();
        QString lineString = lineStringMatch.captured(0);

        QRegularExpressionMatchIterator iMatch = coordinateRegExp->globalMatch(lineString);
        while (iMatch.hasNext()) {
            QRegularExpressionMatch match = iMatch.next();
            double longitude = match.captured(1).toDouble();
            double latitude = match.captured(2).toDouble();
            QGeoCoordinate coord(latitude, longitude);
            geoPath.addCoordinate(coord);
        }
    }
    r = geoPath.boundingGeoRectangle();

#elif defined(Q_OS_LINUX) || defined(Q_OS_MAC) || defined(Q_OS_WIN)
    // Parse WKT street geometry
    OGRSpatialReference osr;
    OGRGeometry *streetGeometry = nullptr;
    QByteArray bytes = b_wktGeometry.value().toUtf8();
    const char *data = bytes.constData();
    OGRErr err = OGRGeometryFactory::createFromWkt(data, &osr, &streetGeometry);
    if (err != OGRERR_NONE) {
        // process error, like emit signal
    } else {
        OGREnvelope streetEnvelope;
        streetGeometry->getEnvelope(&streetEnvelope);
        r.setBottomLeft(QGeoCoordinate(streetEnvelope.MinY, streetEnvelope.MinX));
        r.setTopRight(QGeoCoordinate(streetEnvelope.MaxY, streetEnvelope.MaxX));
        return r;
    }
#endif
    return r;
}

TerritoryStreetModel::TerritoryStreetModel()
{
    initialize();
}

TerritoryStreetModel::TerritoryStreetModel(QObject *parent)
    : QAbstractTableModel(parent)
{
    initialize();
}

void TerritoryStreetModel::initialize()
{
    cterritories *ct = new cterritories;
    streetTypeListModel = ct->getStreetTypes();
}

QHash<int, QByteArray> TerritoryStreetModel::roleNames() const
{
    QHash<int, QByteArray> items;
    items[StreetIdRole] = "streetId";
    items[StreetRole] = "street";
    items[TerritoryIdRole] = "territoryId";
    items[StreetNameRole] = "streetName";
    items[FromNumberRole] = "fromNumber";
    items[ToNumberRole] = "toNumber";
    items[QuantityRole] = "quantity";
    items[StreetTypeIdRole] = "streetTypeId";
    items[StreetTypeNameRole] = "streetTypeName";
    items[StreetTypeColorRole] = "streetTypeColor";
    items[WktGeometryRole] = "wktGeometry";
    items[MultiLineRole] = "multiLine";
    return items;
}

int TerritoryStreetModel::rowCount(const QModelIndex & /*parent*/) const
{
    return territoryStreets.count();
}

int TerritoryStreetModel::columnCount(const QModelIndex & /*parent*/) const
{
    return 8;
}

QVariantMap TerritoryStreetModel::get(int row)
{
    QHash<int, QByteArray> names = roleNames();
    QHashIterator<int, QByteArray> i(names);
    QVariantMap res;
    QModelIndex idx = index(row, 0);
    while (i.hasNext()) {
        i.next();
        QVariant data = idx.data(i.key());
        res[i.value()] = data;
    }
    return res;
}

QVariant TerritoryStreetModel::data(const QModelIndex &index, int role) const
{
    if (!index.isValid())
        return QVariant();

    if (index.row() < 0 || index.row() > territoryStreets.count())
        return QVariant();

    switch (role) {
    case StreetIdRole:
        return territoryStreets[index.row()]->streetId();
    case StreetRole:
        return QVariant::fromValue(territoryStreets[index.row()]);
    case TerritoryIdRole:
        return territoryStreets[index.row()]->territoryId();
    case StreetNameRole:
        return territoryStreets[index.row()]->streetName();
    case FromNumberRole:
        return territoryStreets[index.row()]->fromNumber();
    case ToNumberRole:
        return territoryStreets[index.row()]->toNumber();
    case QuantityRole:
        return territoryStreets[index.row()]->quantity();
    case StreetTypeIdRole:
        return territoryStreets[index.row()]->streetTypeId();
    case StreetTypeNameRole: {
        QModelIndex typeIndex = streetTypeListModel->index(streetTypeListModel->getIndex(territoryStreets[index.row()]->streetTypeId()), 0);
        return streetTypeListModel->data(typeIndex, DataObjectListModel::NameRole).toString();
    }
    case StreetTypeColorRole: {
        QModelIndex typeIndex = streetTypeListModel->index(streetTypeListModel->getIndex(territoryStreets[index.row()]->streetTypeId()), 0);
        QString color = streetTypeListModel->data(typeIndex, DataObjectListModel::ColorRole).toString();
        return color.isEmpty() ? "#000000" : color;
    }
    case WktGeometryRole:
        return territoryStreets[index.row()]->wktGeometry();
    case MultiLineRole:
        return QVariant::fromValue(territoryStreets[index.row()]->multiLine());
    }
    return QVariant();
}

bool TerritoryStreetModel::setData(const QModelIndex &index, const QVariant &value, int role)
{
    TerritoryStreet *currTerritoryStreet = territoryStreets[index.row()];

    switch (role) {
    case Roles::TerritoryIdRole:
        currTerritoryStreet->setTerritoryId(value.toInt());
        emit dataChanged(index, index, { role });
        break;
    case Roles::StreetNameRole:
        currTerritoryStreet->setStreetName(value.toString());
        emit dataChanged(index, index, { role });
        break;
    case Roles::FromNumberRole:
        currTerritoryStreet->setFromNumber(value.toString());
        emit dataChanged(index, index, { role });
        break;
    case Roles::ToNumberRole:
        currTerritoryStreet->setToNumber(value.toString());
        emit dataChanged(index, index, { role });
        break;
    case Roles::QuantityRole:
        currTerritoryStreet->setQuantity(value.toInt());
        emit dataChanged(index, index, { role });
        break;
    case Roles::StreetTypeIdRole:
        currTerritoryStreet->setStreetTypeId(value.toInt());
        emit dataChanged(index, index, { role, Roles::StreetTypeIdRole, Roles::StreetTypeNameRole, Roles::StreetTypeColorRole });
        break;
    case Roles::MultiLineRole:
        currTerritoryStreet->setMultiLine(value.value<QList<QList<QGeoCoordinate>>>());
        emit dataChanged(index, index, { role, Roles::WktGeometryRole });
        break;
    case Roles::WktGeometryRole:
        currTerritoryStreet->setWktGeometry(value.toString());
        currTerritoryStreet->generateMultiLine();
        currTerritoryStreet->setIsDirty(true);
        emit dataChanged(index, index, { role, Roles::MultiLineRole });
        break;
    default:
        break;
    }
    return true;
}

Qt::ItemFlags TerritoryStreetModel::flags(const QModelIndex &index) const
{
    if (!index.isValid())
        return Qt::ItemIsEnabled;
    return Qt::ItemIsEditable | QAbstractTableModel::flags(index);
}

QModelIndex TerritoryStreetModel::index(int row, int column, const QModelIndex &parent)
        const
{
    if (hasIndex(row, column, parent)) {
        return createIndex(row, column);
    }
    return QModelIndex();
}

QModelIndex TerritoryStreetModel::getStreetIndex(int streetId) const
{
    for (int row = 0; row < this->rowCount(); ++row) {
        QModelIndex rowIndex = this->index(row, 0);
        if (rowIndex.data(StreetIdRole) == streetId)
            return rowIndex;
    }
    return QModelIndex();
}

int TerritoryStreetModel::addStreet(TerritoryStreet *territoryStreet)
{
    beginInsertRows(QModelIndex(), rowCount(), rowCount());
    territoryStreets << territoryStreet;
    endInsertRows();
    emit modelChanged();
    emit countChanged();
    return territoryStreet->streetId();
}

int TerritoryStreetModel::addStreet(int territoryId, const QString streetName, QString wktGeometry, int streetTypeId)
{
    beginInsertRows(QModelIndex(), rowCount(), rowCount());
    TerritoryStreet *newTerritoryStreet = new TerritoryStreet(territoryId, streetName, "", "", 1, streetTypeId, wktGeometry, this);
    newTerritoryStreet->generateMultiLine();
    newTerritoryStreet->setIsDirty(true);
    newTerritoryStreet->save();
    territoryStreets << newTerritoryStreet;
    endInsertRows();
    emit modelChanged();
    emit countChanged();
    return newTerritoryStreet->streetId();
}

QVariant TerritoryStreetModel::findStreet(int streetId)
{
    QModelIndex streetIndex = getStreetIndex(streetId);
    if (streetIndex.isValid()) {
        return streetIndex.data(StreetRole);
    }
    return QVariant();
}

void TerritoryStreetModel::removeStreet(int streetId)
{
    if (streetId > 0) {
        sql_class *sql = &Singleton<sql_class>::Instance();

        sql_item s;
        s.insert("active", 0);
        if (sql->updateSql("territory_street", "id", QString::number(streetId), &s)) {
            QModelIndex streetIndex = getStreetIndex(streetId);
            if (streetIndex.isValid()) {
                int row = streetIndex.row();
                beginRemoveRows(QModelIndex(), row, row);
                territoryStreets.erase(std::next(territoryStreets.begin(), row));
                endRemoveRows();
                emit modelChanged();
                emit countChanged();
            }
        }
    }
}

bool TerritoryStreetModel::removeRows(int row, int count, const QModelIndex &parent)
{
    Q_UNUSED(parent)
    if (row < 0 || count < 1 || (row + count) > territoryStreets.size())
        return false;
    beginRemoveRows(QModelIndex(), row, row + count - 1);
    for (int i = 0; i < count; i++) {
        territoryStreets.removeAt(row);
    }
    endRemoveRows();
    emit modelChanged();
    emit countChanged();
    return true;
}

void TerritoryStreetModel::loadStreets(int territoryId)
{
    AccessControl *ac = &Singleton<AccessControl>::Instance();
    if (!ac->user() || !ac->user()->hasPermission(PermissionRule::CanViewTerritories))
        return;

    beginResetModel();
    removeRows(0, territoryStreets.count());
    sql_class *sql = &Singleton<sql_class>::Instance();

    QString sqlQuery(QString("SELECT s.* FROM territory_street s "
                             "JOIN territories t ON s.territory_id = t.id "
                             "WHERE (%1 OR t.person_id = %2) AND ")
                             .arg(!ac->isActive() || ac->user()->hasPermission(PermissionRule::CanViewTerritoryAddresses))
                             .arg(ac->user()->personId()));
    if (territoryId != 0)
        sqlQuery.append("s.territory_id = " + QVariant(territoryId).toString() + " AND ");
    sqlQuery.append("s.active");
    sql_items territoryStreetRows = sql->selectSql(sqlQuery);

    if (!territoryStreetRows.empty()) {
        for (unsigned int i = 0; i < territoryStreetRows.size(); i++) {
            sql_item s = territoryStreetRows[i];
            addStreet(new TerritoryStreet(s.value("id").toInt(),
                                          s.value("territory_id").toInt(),
                                          s.value("street_name").toString(),
                                          s.value("from_number").toString(),
                                          s.value("to_number").toString(),
                                          s.value("quantity").toInt(),
                                          s.value("streettype_id").toInt(),
                                          s.value("wkt_geometry").toString(),
                                          this));
        }
    }
    // Fill multiline properties for the map
    QFuture<void> future = QtConcurrent::map(territoryStreets, &TerritoryStreet::generateMultiLine);
    endResetModel();
    emit modelChanged();
    emit countChanged();
}

void TerritoryStreetModel::saveStreets()
{
    sql_class *sql = &Singleton<sql_class>::Instance();
    sql->startTransaction();
    for (int i = 0; i < territoryStreets.count(); i++) {
        TerritoryStreet *s = territoryStreets[i];
        if (s->isDirty()) {
            s->save();
        }
    }
    sql->commitTransaction();
}

#if defined(Q_OS_ANDROID) || defined(Q_OS_IOS)
// GDAL missing
#elif defined(Q_OS_LINUX) || defined(Q_OS_MAC) || defined(Q_OS_WIN)
QString TerritoryStreetModel::cutoffStreetSection(int streetId, QList<QGeoCoordinate> cutAreaPath)
{
    QModelIndex rowIndex = getStreetIndex(streetId);
    if (!rowIndex.isValid())
        return "";

    // convert polygon path
    OGRLinearRing poRing;
    for (auto c = cutAreaPath.begin(), end = cutAreaPath.end(); c != end; ++c)
        poRing.addPoint((*c).longitude(), (*c).latitude());
    poRing.closeRings();
    OGRPolygon cutAreaPolygon;
    OGRGeometry *cutAreaGeometry = nullptr;
    cutAreaPolygon.addRing(&poRing);
    cutAreaGeometry = cutAreaPolygon.getLinearGeometry();

    OGRGeometry *streetGeometry = territoryStreets[rowIndex.row()]->getOGRGeometry();
    if (streetGeometry == nullptr)
        return "";

    // create section which is outside the cut area and will remain
    OGRGeometry *outsideAreaGeometry = nullptr;
    outsideAreaGeometry = streetGeometry->Difference(cutAreaGeometry);

    char *wkt_new = nullptr;
    // update the street's geometry
    outsideAreaGeometry->exportToWkt(&wkt_new);
    setData(rowIndex, QString(wkt_new), WktGeometryRole);

    // create section which is inside the cut area
    OGRGeometry *insideCutAreaGeometry = nullptr;
    insideCutAreaGeometry = streetGeometry->Intersection(cutAreaGeometry);
    // return the section that was cut off
    wkt_new = nullptr;
    if (insideCutAreaGeometry->IsValid())
        insideCutAreaGeometry->exportToWkt(&wkt_new);
    return wkt_new;
}
#endif

bool TerritoryStreetModel::updateStreet(int streetId, QString wktGeometry)
{
    QModelIndex rowIndex = getStreetIndex(streetId);
    if (rowIndex.isValid()) {
        setData(rowIndex, wktGeometry, WktGeometryRole);
        return true;
    }
    return false;
}

bool TerritoryStreetModel::updateStreet(int streetId, int territoryId)
{
    QModelIndex rowIndex = getStreetIndex(streetId);
    if (rowIndex.isValid()) {
        setData(rowIndex, territoryId, TerritoryIdRole);
        return true;
    }
    return false;
}

bool TerritoryStreetModel::updateStreet(int streetId, int territoryId, const QString streetName, QString wktGeometry)
{
    QModelIndex rowIndex = getStreetIndex(streetId);
    if (rowIndex.isValid()) {
        setData(rowIndex, territoryId, TerritoryIdRole);
        setData(rowIndex, streetName, StreetNameRole);
        setData(rowIndex, wktGeometry, WktGeometryRole);
        return true;
    }
    return false;
}

int TerritoryStreetModel::getDefaultStreetTypeId()
{
    sql_class *sql = &Singleton<sql_class>::Instance();
    return QVariant(sql->getSetting("territory_default_streettype", "1")).toInt();
}

TerritoryStreet *TerritoryStreetModel::getItem(const QModelIndex &index) const
{
    if (index.isValid()) {
        TerritoryStreet *item = territoryStreets[index.row()];
        if (item)
            return item;
    }
    return nullptr;
}

TerritoryStreetSortFilterProxyModel::TerritoryStreetSortFilterProxyModel(QObject *parent)
    : QSortFilterProxyModel(parent)
{
}

QObject *TerritoryStreetSortFilterProxyModel::source() const
{
    return sourceModel();
}

void TerritoryStreetSortFilterProxyModel::setSource(QObject *source)
{
    setSourceModel(qobject_cast<QAbstractItemModel *>(source));
    emit sourceChanged();
}

int TerritoryStreetSortFilterProxyModel::filterTerritoryId() const
{
    return m_filterTerritoryId;
}

void TerritoryStreetSortFilterProxyModel::setFilterTerritoryId(int newValue)
{
    beginFilterChange();
    m_filterTerritoryId = newValue;
    endFilterChange(QSortFilterProxyModel::Direction::Rows);
    emit filterTerritoryIdChanged();
}

QString TerritoryStreetSortFilterProxyModel::filterText() const
{
    return m_filterText;
}

void TerritoryStreetSortFilterProxyModel::setFilterText(QString newValue)
{
    beginFilterChange();
    m_filterText = newValue;
    endFilterChange(QSortFilterProxyModel::Direction::Rows);
    emit filterTextChanged();
}

QByteArray TerritoryStreetSortFilterProxyModel::sortRole() const
{
    return roleNames().value(QSortFilterProxyModel::sortRole());
}

void TerritoryStreetSortFilterProxyModel::setSortRole(const QByteArray &role)
{
    QSortFilterProxyModel::setSortRole(roleKey(role));
    emit sortChanged();
}

QByteArray TerritoryStreetSortFilterProxyModel::groupByRole() const
{
    return m_groupByRole;
}

void TerritoryStreetSortFilterProxyModel::setGroupByRole(const QByteArray &role)
{
    beginResetModel();
    m_groupByRole = role;
    endResetModel();
    emit groupByChanged();
    sort(0, sortOrder());
}

bool TerritoryStreetSortFilterProxyModel::filterAcceptsRow(int sourceRow,
                                                           const QModelIndex &sourceParent) const
{
    QModelIndex indexTerritoryId = sourceModel()->index(sourceRow, 0, sourceParent);
    int territoryId = sourceModel()->data(indexTerritoryId, TerritoryStreetModel::Roles::TerritoryIdRole).toInt();
    return (territoryId == filterTerritoryId())
            && sourceModel()->data(indexTerritoryId, TerritoryStreetModel::Roles::StreetNameRole).toString().contains(filterText(), Qt::CaseInsensitive);
}

bool TerritoryStreetSortFilterProxyModel::lessThan(const QModelIndex &left, const QModelIndex &right) const
{
    // compare sections first
    switch (roleKey(groupByRole())) {
    case TerritoryStreetModel::Roles::StreetTypeNameRole: {
        QVariant val1 = sourceModel()->data(left, TerritoryStreetModel::Roles::StreetTypeNameRole);
        QVariant val2 = sourceModel()->data(right, TerritoryStreetModel::Roles::StreetTypeNameRole);
        int compResult = QString::compare(val1.toString(), val2.toString(), Qt::CaseInsensitive);
        if (compResult != 0)
            return compResult < 0;
        break;
    }
    case TerritoryStreetModel::Roles::QuantityRole: {
        QVariant val1 = sourceModel()->data(left, TerritoryStreetModel::Roles::QuantityRole);
        QVariant val2 = sourceModel()->data(right, TerritoryStreetModel::Roles::QuantityRole);
        if (!val1.isNull() && !val2.isNull()) {
            int quantity1 = val1.toInt();
            int quantity2 = val2.toInt();
            if (quantity1 != quantity2)
                return quantity1 < quantity2;
        }
        break;
    }
    default:
        break;
    }

    // sort by street name within sections
    QVariant leftData = sourceModel()->data(left, TerritoryStreetModel::Roles::StreetNameRole);
    QVariant rightData = sourceModel()->data(right, TerritoryStreetModel::Roles::StreetNameRole);
    int compResult = QString::compare(leftData.toString(), rightData.toString(), Qt::CaseInsensitive);
    return compResult < 0;
}

int TerritoryStreetSortFilterProxyModel::roleKey(const QByteArray &role) const
{
    QHash<int, QByteArray> roles = roleNames();
    QHashIterator<int, QByteArray> it(roles);
    while (it.hasNext()) {
        it.next();
        if (it.value() == role)
            return it.key();
    }
    return -1;
}

TerritoryStreetValidator::TerritoryStreetValidator(QObject *parent)
    : QValidator(parent), m_model(nullptr), m_streetId(-1), m_role(TerritoryStreetModel::Roles::None)
{
}

TerritoryStreetValidator::~TerritoryStreetValidator()
{
}

QValidator::State TerritoryStreetValidator::validate(QString &input, int &pos) const
{
    Q_UNUSED(pos)

    if (input.isEmpty()) {
        emit errorChanged("");
        return Acceptable;
    }
    if (!model()) {
        emit errorChanged("");
        return Acceptable;
    }
    if (role() == TerritoryStreetModel::Roles::None) {
        emit errorChanged("");
        return Acceptable;
    }
    QModelIndex modelIndex = model()->getStreetIndex(streetId());
    if (!modelIndex.isValid()) {
        emit errorChanged("");
        return Acceptable;
    }

    // add validation here...

    emit errorChanged("");
    return Acceptable;
}

TerritoryStreetModel *TerritoryStreetValidator::model() const
{
    return m_model;
}

void TerritoryStreetValidator::setModel(TerritoryStreetModel *newModel)
{
    if (m_model == newModel)
        return;
    m_model = newModel;
    emit modelChanged();
}

int TerritoryStreetValidator::streetId() const
{
    return m_streetId;
}

void TerritoryStreetValidator::setStreetId(int newStreetId)
{
    if (m_streetId == newStreetId)
        return;
    m_streetId = newStreetId;
    emit streetIdChanged();
}

TerritoryStreetModel::Roles TerritoryStreetValidator::role() const
{
    return m_role;
}

void TerritoryStreetValidator::setRole(TerritoryStreetModel::Roles newRole)
{
    if (m_role == newRole)
        return;
    m_role = newRole;
    emit roleChanged();
}
