/**
 * This file is part of TheocBase.
 *
 * Copyright (C) 2011-2016, TheocBase Development Team, see AUTHORS.
 *
 * TheocBase is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * TheocBase is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with TheocBase.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef TERRITORYASSIGNMENT_H
#define TERRITORYASSIGNMENT_H

#include <QObject>
#include <QAbstractTableModel>
#include <QDate>
#include <QValidator>
#include "cpersons.h"
#include "dataobject.h"
#include "sortfilterproxymodel.h"
#include "territory.h"

class TerritoryAssignment : public QObject
{
    Q_OBJECT
    Q_PROPERTY(int id READ id NOTIFY idChanged FINAL)
    Q_PROPERTY(int territoryId READ territoryId WRITE setTerritoryId BINDABLE bindableTerritoryId)
    Q_PROPERTY(int personId READ personId WRITE setPersonId BINDABLE bindablePersonId)
    Q_PROPERTY(QString personFullName READ personFullName BINDABLE bindablePersonFullName)
    Q_PROPERTY(QDate assignedDate READ assignedDate WRITE setAssignedDate BINDABLE bindableAssignedDate)
    Q_PROPERTY(QDate completedDate READ completedDate WRITE setCompletedDate BINDABLE bindableCompletedDate)
    Q_PROPERTY(bool isDirty READ isDirty WRITE setIsDirty BINDABLE bindableIsDirty)
public:
    explicit TerritoryAssignment(QObject *parent = nullptr);
    TerritoryAssignment(int territoryId, QObject *parent = nullptr);
    TerritoryAssignment(const int id, const int territoryId, const int personId, const QDate assignedDate, const QDate completedDate, QObject *parent = nullptr);
    ~TerritoryAssignment();

    int id() const;

    int territoryId() { return b_territoryId.value(); }
    void setTerritoryId(int newValue) { b_territoryId = newValue; }
    QBindable<int> bindableTerritoryId() { return &b_territoryId; }

    int personId() { return b_personId.value(); }
    void setPersonId(int newValue) { b_personId = newValue; }
    QBindable<int> bindablePersonId() { return &b_personId; }

    QString personFullName() { return b_personFullName.value(); }
    QBindable<QString> bindablePersonFullName() { return &b_personFullName; }

    QDate assignedDate() { return b_assignedDate.value(); }
    void setAssignedDate(QDate newValue) { b_assignedDate = newValue; }
    QBindable<QDate> bindableAssignedDate() { return &b_assignedDate; }

    QDate completedDate() { return b_completedDate.value(); }
    void setCompletedDate(QDate newValue) { b_completedDate = newValue; }
    QBindable<QDate> bindableCompletedDate() { return &b_completedDate; }

    bool isDirty() { return b_isDirty.value(); }
    void setIsDirty(bool newValue) { b_isDirty = newValue; }
    QBindable<bool> bindableIsDirty() { return &b_isDirty; }

    bool save();

signals:
    void idChanged();
    void territoryIdChanged();
    void personIdChanged();
    void assignedDateChanged();
    void completedDateChanged();
    void isDirtyChanged();

private:
    int m_id;
    Q_OBJECT_BINDABLE_PROPERTY(TerritoryAssignment, int, b_territoryId, &TerritoryAssignment::territoryIdChanged);
    Q_OBJECT_BINDABLE_PROPERTY(TerritoryAssignment, int, b_personId, &TerritoryAssignment::personIdChanged);
    QProperty<QString> b_personFullName;
    // Q_OBJECT_BINDABLE_PROPERTY(TerritoryAssignment, QString, b_personFullName, &TerritoryAssignment::personFullNameChanged);
    Q_OBJECT_BINDABLE_PROPERTY(TerritoryAssignment, QDate, b_assignedDate, &TerritoryAssignment::assignedDateChanged);
    Q_OBJECT_BINDABLE_PROPERTY(TerritoryAssignment, QDate, b_completedDate, &TerritoryAssignment::completedDateChanged);
    QProperty<bool> b_isDirty { false };
};

class TerritoryAssignmentModel : public QAbstractTableModel
{
    Q_OBJECT
    Q_PROPERTY(int currentAssigneeId READ currentAssigneeId WRITE setCurrentAssigneeId NOTIFY currentAssigneeIdChanged FINAL)

public:
    enum Roles {
        None = 0,
        AssignmentIdRole = Qt::UserRole,
        AssignmentRole,
        PersonIdRole,
        PersonFullNameRole,
        AssignedDateRole,
        AssignedYearRole,
        CompletedDateRole
    };
    Q_ENUM(Roles)

    QHash<int, QByteArray> roleNames() const override;

    TerritoryAssignmentModel(QObject *parent = nullptr);
    Q_INVOKABLE void initialize();

    int currentAssigneeId() const;
    void setCurrentAssigneeId(int newCurrentAssigneeId);

    int rowCount(const QModelIndex &parent = QModelIndex()) const override;
    int columnCount(const QModelIndex &parent = QModelIndex()) const override;
    Q_INVOKABLE QVariantMap get(int row);
    QVariant data(const QModelIndex &index, int role = Qt::DisplayRole) const override;
    Q_INVOKABLE bool setData(const QModelIndex &index, const QVariant &value, int role = Qt::EditRole) override;
    Qt::ItemFlags flags(const QModelIndex &index) const override;
    QModelIndex index(int row, int column, const QModelIndex &parent = QModelIndex()) const override;
    Q_INVOKABLE QModelIndex getAssignmentIndex(int assignmentId) const;
    bool addAssignment(TerritoryAssignment *territoryAssignment);
    Q_INVOKABLE int addAssignment(int territoryId);
    Q_INVOKABLE void removeAssignment(int row);
    Q_INVOKABLE bool removeRows(int row, int count, const QModelIndex &parent = QModelIndex()) override;
    Q_INVOKABLE void loadAssignments(int territoryId);
    Q_INVOKABLE void saveAssignments();

    Q_INVOKABLE PersonsModel *getPublisherList() const;

private:
    QList<TerritoryAssignment *> territoryAssignments;
    PersonsModel *m_publisherList;
    int m_currentAssigneeId;
    QString displayNameFormat;

    void updateCurrentAssignee();

signals:
    void modelChanged();
    void notification();
    void editCompleted();
    void currentAssigneeIdChanged();
};

class TerritoryAssignmentSortFilterProxyModel : public QSortFilterProxyModel
{
    Q_OBJECT
    Q_PROPERTY(QObject *source READ source WRITE setSource NOTIFY sourceChanged)
    Q_PROPERTY(QString filterText READ filterText WRITE setFilterText NOTIFY filterTextChanged)
    Q_PROPERTY(QByteArray sortRole READ sortRole WRITE setSortRole NOTIFY sortChanged)
    Q_PROPERTY(QByteArray groupByRole READ groupByRole WRITE setGroupByRole NOTIFY groupByChanged)

public:
    TerritoryAssignmentSortFilterProxyModel(QObject *parent = nullptr);

    QObject *source() const;
    void setSource(QObject *source);

    QString filterText() const;
    void setFilterText(QString newValue);

    QByteArray sortRole() const;
    void setSortRole(const QByteArray &role);

    QByteArray groupByRole() const;
    void setGroupByRole(const QByteArray &role);

    Q_INVOKABLE virtual void sort(int column, Qt::SortOrder order = Qt::AscendingOrder) override
    {
        QSortFilterProxyModel::sort(column, order);
    }

protected:
    bool filterAcceptsRow(int sourceRow, const QModelIndex &sourceParent) const override;
    bool lessThan(const QModelIndex &left, const QModelIndex &right) const override;
    int roleKey(const QByteArray &role) const;

signals:
    void sourceChanged();
    void filterTextChanged();
    void sortChanged();
    void groupByChanged();

private:
    QString m_filterText;
    QByteArray m_groupByRole;
};

class TerritoryAssignmentValidator : public QValidator
{
    Q_OBJECT
    Q_PROPERTY(TerritoryAssignmentModel *model READ model WRITE setModel NOTIFY modelChanged)
    Q_PROPERTY(int assignmentId READ assignmentId WRITE setAssignmentId NOTIFY assignmentIdChanged)
    Q_PROPERTY(TerritoryAssignmentModel::Roles role READ role WRITE setRole NOTIFY roleChanged)
public:
    TerritoryAssignmentValidator(QObject *parent = nullptr);
    ~TerritoryAssignmentValidator();

    QValidator::State validate(QString &input, int &pos) const override;

    TerritoryAssignmentModel *model() const;
    void setModel(TerritoryAssignmentModel *newModel);

    TerritoryAssignmentModel::Roles role() const;
    void setRole(TerritoryAssignmentModel::Roles newRole);

    int assignmentId() const;
    void setAssignmentId(int newAssignmentId);

Q_SIGNALS:
    void modelChanged();
    void assignmentIdChanged();
    void errorChanged(const QString &error) const;
    void roleChanged();

private:
    TerritoryAssignmentModel *m_model;
    int m_assignmentId;
    TerritoryAssignmentModel::Roles m_role;
};

#endif // TERRITORYASSIGNMENT_H
