/**
 * This file is part of TheocBase.
 *
 * Copyright (C) 2011-2016, TheocBase Development Team, see AUTHORS.
 *
 * TheocBase is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * TheocBase is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with TheocBase.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "territoryassignment.h"
#include <QDebug>

TerritoryAssignment::TerritoryAssignment(QObject *parent)
    : TerritoryAssignment(0, 0, 0, QDate(), QDate(), parent)
{
}

TerritoryAssignment::TerritoryAssignment(int territoryId, QObject *parent)
    : TerritoryAssignment(0, territoryId, 0, QDate(), QDate(), parent)
{
}

TerritoryAssignment::TerritoryAssignment(const int id, const int territoryId, const int personId, const QDate assignedDate, const QDate completedDate, QObject *parent)
    : QObject(parent), m_id(id), b_territoryId(territoryId), b_personId(personId), b_assignedDate(assignedDate), b_completedDate(completedDate), b_isDirty(false)
{
    QObject::connect(this, &TerritoryAssignment::territoryIdChanged, [&]() { b_isDirty = true; });
    QObject::connect(this, &TerritoryAssignment::personIdChanged, [&]() { b_isDirty = true; });
    QObject::connect(this, &TerritoryAssignment::assignedDateChanged, [&]() { b_isDirty = true; });
    QObject::connect(this, &TerritoryAssignment::completedDateChanged, [&]() { b_isDirty = true; });

    b_personFullName.setBinding([&]() {
        cpersons persons;
        Person *publisher = persons.getPerson(b_personId);
        return publisher ? publisher->fullName() : "";
    });
}

TerritoryAssignment::~TerritoryAssignment()
{
}

int TerritoryAssignment::id() const
{
    return m_id;
}

bool TerritoryAssignment::save()
{
    if (personId() < 1)
        return false;

    // save changes to database
    sql_class *sql = &Singleton<sql_class>::Instance();

    int lang_id = sql->getLanguageDefaultId();

    sql_item queryitems;
    queryitems.insert(":id", m_id);
    int assignmentId = m_id > 0 ? sql->selectScalar("SELECT id FROM territory_assignment WHERE id = :id", &queryitems, -1).toInt() : -1;

    sql_item insertItems;
    insertItems.insert("territory_id", territoryId());
    insertItems.insert("person_id", personId());
    insertItems.insert("checkedout_date", assignedDate());
    insertItems.insert("checkedbackin_date", completedDate());
    insertItems.insert("lang_id", lang_id);

    bool ret = false;
    if (assignmentId > 0) {
        // update
        ret = sql->updateSql("territory_assignment", "id", QString::number(assignmentId), &insertItems);
    } else {
        // insert new row
        int newId = sql->insertSql("territory_assignment", &insertItems, "id");
        ret = newId > 0;
        if (newId > 0)
            m_id = newId;
    }
    setIsDirty(!ret);

    return ret;
}

TerritoryAssignmentModel::TerritoryAssignmentModel(QObject *parent)
    : QAbstractTableModel(parent)
{
    initialize();
}

void TerritoryAssignmentModel::initialize()
{
    // make sure the list of publishers is initialized after synchronization
    m_publisherList = new PersonsModel();
    m_publisherList->loadList();
}

QHash<int, QByteArray> TerritoryAssignmentModel::roleNames() const
{
    QHash<int, QByteArray> items;
    items[AssignmentIdRole] = "id";
    items[AssignmentRole] = "assignment";
    items[PersonIdRole] = "personId";
    items[PersonFullNameRole] = "personFullName";
    items[AssignedDateRole] = "assignedDate";
    items[AssignedYearRole] = "assignedYear";
    items[CompletedDateRole] = "completedDate";
    return items;
}

int TerritoryAssignmentModel::rowCount(const QModelIndex &parent) const
{
    Q_UNUSED(parent)
    return territoryAssignments.empty() ? 0 : territoryAssignments.count();
}

int TerritoryAssignmentModel::columnCount(const QModelIndex & /*parent*/) const
{
    return 5;
}

QVariantMap TerritoryAssignmentModel::get(int row)
{
    QHash<int, QByteArray> names = roleNames();
    QHashIterator<int, QByteArray> i(names);
    QVariantMap res;
    QModelIndex idx = index(row, 0);
    while (i.hasNext()) {
        i.next();
        QVariant data = idx.data(i.key());
        res[i.value()] = data;
    }
    return res;
}

QVariant TerritoryAssignmentModel::data(const QModelIndex &index, int role) const
{
    if (!index.isValid())
        return QVariant();

    if (index.row() < 0 || index.row() > territoryAssignments.count())
        return QVariant();

    switch (role) {
    case AssignmentIdRole:
        return territoryAssignments[index.row()]->id();
    case AssignmentRole:
        return QVariant::fromValue(territoryAssignments[index.row()]);
    case PersonIdRole:
        return territoryAssignments[index.row()]->personId();
    case PersonFullNameRole:
        return territoryAssignments[index.row()]->personFullName();
    case AssignedDateRole:
        return territoryAssignments[index.row()]->assignedDate().startOfDay();
    case AssignedYearRole:
        return territoryAssignments[index.row()]->assignedDate().year();
    case CompletedDateRole:
        return territoryAssignments[index.row()]->completedDate().startOfDay();
    }

    return QVariant();
}

bool TerritoryAssignmentModel::setData(const QModelIndex &index, const QVariant &value, int role)
{
    if (index.row() < 0 || index.row() > territoryAssignments.count())
        return false;
    TerritoryAssignment *currTerritoryAssignment = territoryAssignments[index.row()];

    switch (role) {
    case Roles::PersonIdRole:
        currTerritoryAssignment->setPersonId(value.toInt());
        emit dataChanged(index, index, { role, Roles::PersonFullNameRole });
        break;
    case Roles::AssignedDateRole: {
        QDate newDate = value.toDate();
        if (newDate.addYears(99).year() <= QDate::currentDate().year())
            newDate = newDate.addYears(100);
        currTerritoryAssignment->setAssignedDate(newDate);
        emit dataChanged(this->index(index.row(), 3), this->index(index.row(), 3));
        emit dataChanged(index, index, { role });
        break;
    }
    case Roles::CompletedDateRole: {
        QDate newDate = value.toDate();
        if (newDate.addYears(99).year() <= QDate::currentDate().year())
            newDate = newDate.addYears(100);
        currTerritoryAssignment->setCompletedDate(newDate);
        emit dataChanged(this->index(index.row(), 4), this->index(index.row(), 4));
        emit dataChanged(index, index, { role });
        break;
    }
    default:
        break;
    }

    return true;
}

Qt::ItemFlags TerritoryAssignmentModel::flags(const QModelIndex &index) const
{
    if (!index.isValid())
        return Qt::ItemIsEnabled;
    return Qt::ItemIsEditable | QAbstractTableModel::flags(index);
}

QModelIndex TerritoryAssignmentModel::index(int row, int column, const QModelIndex &parent)
        const
{
    if (hasIndex(row, column, parent)) {
        return createIndex(row, column);
    }
    return QModelIndex();
}

QModelIndex TerritoryAssignmentModel::getAssignmentIndex(int assignmentId) const
{
    for (int row = 0; row < this->rowCount(); ++row) {
        QModelIndex rowIndex = this->index(row, 0);
        if (rowIndex.data(AssignmentIdRole) == assignmentId)
            return rowIndex;
    }
    return QModelIndex();
}

bool TerritoryAssignmentModel::addAssignment(TerritoryAssignment *territoryAssignment)
{
    beginInsertRows(QModelIndex(), rowCount(), rowCount());
    territoryAssignments << territoryAssignment;
    endInsertRows();
    emit modelChanged();
    return true;
}

int TerritoryAssignmentModel::addAssignment(int territoryId)
{
    beginInsertRows(QModelIndex(), rowCount(), rowCount());
    TerritoryAssignment *newTerritoryAssignment = new TerritoryAssignment(territoryId, this);

    if (!territoryAssignments.isEmpty()) {
        // sort by assigned date
        std::sort(territoryAssignments.begin(), territoryAssignments.end(),
                  [](TerritoryAssignment *a, TerritoryAssignment *b) {
                      return a->assignedDate() < b->assignedDate();
                  });

        TerritoryAssignment *lastAssignment = territoryAssignments.last();
        if (!lastAssignment->completedDate().isValid()) {
            QModelIndex lastAssignmentIndex = this->index(territoryAssignments.count() - 1, 0);
            this->setData(lastAssignmentIndex, QDate::currentDate(), Roles::CompletedDateRole);
        }

        newTerritoryAssignment->setPersonId(lastAssignment->personId());
        newTerritoryAssignment->setAssignedDate(lastAssignment->completedDate().addDays(1));
    }

    territoryAssignments << newTerritoryAssignment;
    endInsertRows();
    emit modelChanged();
    updateCurrentAssignee();
    return newTerritoryAssignment->id();
}

void TerritoryAssignmentModel::removeAssignment(int id)
{
    if (id > 0) {
        sql_class *sql = &Singleton<sql_class>::Instance();

        sql_item s;
        s.insert("active", 0);
        if (sql->updateSql("territory_assignment", "id", QString::number(id), &s)) {
            QModelIndex assignmentIndex = getAssignmentIndex(id);
            if (assignmentIndex.isValid()) {
                int row = assignmentIndex.row();
                beginRemoveRows(QModelIndex(), row, row);
                territoryAssignments.erase(std::next(territoryAssignments.begin(), row));
                endRemoveRows();
            }
        }
    }
    emit modelChanged();
    updateCurrentAssignee();
}

bool TerritoryAssignmentModel::removeRows(int row, int count, const QModelIndex &parent)
{
    Q_UNUSED(parent)
    if (row < 0 || count < 1 || (row + count) > territoryAssignments.size())
        return false;
    beginRemoveRows(QModelIndex(), row, row + count - 1);
    for (int i = 0; i < count; i++) {
        territoryAssignments.removeAt(row);
    }
    endRemoveRows();
    emit modelChanged();
    updateCurrentAssignee();
    return true;
}

void TerritoryAssignmentModel::loadAssignments(int territoryId)
{
    beginResetModel();
    removeRows(0, territoryAssignments.count());
    sql_class *sql = &Singleton<sql_class>::Instance();
    sql_items territoryAssignmentRows = sql->selectSql("SELECT * FROM territoryassignments WHERE territory_id = " + QVariant(territoryId).toString() + " ORDER BY checkedout_date");

    if (!territoryAssignmentRows.empty()) {
        for (unsigned int i = 0; i < territoryAssignmentRows.size(); i++) {
            sql_item s = territoryAssignmentRows[i];
            addAssignment(new TerritoryAssignment(s.value("id").toInt(),
                                                  s.value("territory_id").toInt(),
                                                  s.value("person_id").toInt(),
                                                  s.value("checkedout_date").toDate(),
                                                  s.value("checkedbackin_date").toDate(),
                                                  this));
        }
    }
    endResetModel();
    emit modelChanged();
    updateCurrentAssignee();
}

void TerritoryAssignmentModel::saveAssignments()
{
    sql_class *sql = &Singleton<sql_class>::Instance();
    sql->startTransaction();
    for (int i = 0; i < territoryAssignments.count(); i++) {
        TerritoryAssignment *t = territoryAssignments[i];
        if (t->isDirty()) {
            t->save();
        }
    }
    sql->commitTransaction();
}

PersonsModel *TerritoryAssignmentModel::getPublisherList() const
{
    return m_publisherList;
}

void TerritoryAssignmentModel::updateCurrentAssignee()
{
    int currentAssigneeId = 0;
    if (territoryAssignments.count() > 0) {
        TerritoryAssignment *lastAssignment = territoryAssignments.last();
        if (!lastAssignment->completedDate().isValid())
            currentAssigneeId = territoryAssignments.last()->personId();
    }
    setCurrentAssigneeId(currentAssigneeId);
}

int TerritoryAssignmentModel::currentAssigneeId() const
{
    return m_currentAssigneeId;
}

void TerritoryAssignmentModel::setCurrentAssigneeId(int newCurrentAssigneeId)
{
    if (m_currentAssigneeId == newCurrentAssigneeId)
        return;
    m_currentAssigneeId = newCurrentAssigneeId;
    emit currentAssigneeIdChanged();
}

TerritoryAssignmentSortFilterProxyModel::TerritoryAssignmentSortFilterProxyModel(QObject *parent)
    : QSortFilterProxyModel(parent)
{
}

QObject *TerritoryAssignmentSortFilterProxyModel::source() const
{
    return sourceModel();
}

void TerritoryAssignmentSortFilterProxyModel::setSource(QObject *source)
{
    setSourceModel(qobject_cast<QAbstractItemModel *>(source));
    emit sourceChanged();
}

QString TerritoryAssignmentSortFilterProxyModel::filterText() const
{
    return m_filterText;
}

void TerritoryAssignmentSortFilterProxyModel::setFilterText(QString newValue)
{
    beginFilterChange();
    m_filterText = newValue;
    endFilterChange(QSortFilterProxyModel::Direction::Rows);
    emit filterTextChanged();
}

QByteArray TerritoryAssignmentSortFilterProxyModel::sortRole() const
{
    return roleNames().value(QSortFilterProxyModel::sortRole());
}

void TerritoryAssignmentSortFilterProxyModel::setSortRole(const QByteArray &role)
{
    QSortFilterProxyModel::setSortRole(roleKey(role));
    emit sortChanged();
}

QByteArray TerritoryAssignmentSortFilterProxyModel::groupByRole() const
{
    return m_groupByRole;
}

void TerritoryAssignmentSortFilterProxyModel::setGroupByRole(const QByteArray &role)
{
    beginResetModel();
    m_groupByRole = role;
    endResetModel();
    emit groupByChanged();
}

bool TerritoryAssignmentSortFilterProxyModel::filterAcceptsRow(int sourceRow,
                                                               const QModelIndex &sourceParent) const
{
    QModelIndex index = sourceModel()->index(sourceRow, 0, sourceParent);
    return sourceModel()->data(index, TerritoryAssignmentModel::Roles::PersonFullNameRole).toString().contains(filterText(), Qt::CaseInsensitive);
}

bool TerritoryAssignmentSortFilterProxyModel::lessThan(const QModelIndex &left, const QModelIndex &right) const
{
    // compare sections first
    switch (roleKey(groupByRole())) {
    case TerritoryAssignmentModel::Roles::PersonFullNameRole: {
        QVariant val1 = sourceModel()->data(left, TerritoryAssignmentModel::Roles::PersonFullNameRole);
        QVariant val2 = sourceModel()->data(right, TerritoryAssignmentModel::Roles::PersonFullNameRole);
        int compResult = QString::compare(val1.toString(), val2.toString(), Qt::CaseInsensitive);
        if (compResult != 0)
            return compResult < 0;
        break;
    }
    default:
        break;
    }

    // sort by date within sections
    QVariant leftData = sourceModel()->data(left, TerritoryAssignmentModel::Roles::AssignedDateRole);
    QVariant rightData = sourceModel()->data(right, TerritoryAssignmentModel::Roles::AssignedDateRole);
    if (leftData.userType() == QMetaType::QDate) {
        return leftData.toDate() < rightData.toDate();
    } else
        return leftData.toString() < rightData.toString();
}

int TerritoryAssignmentSortFilterProxyModel::roleKey(const QByteArray &role) const
{
    QHash<int, QByteArray> roles = roleNames();
    QHashIterator<int, QByteArray> it(roles);
    while (it.hasNext()) {
        it.next();
        if (it.value() == role)
            return it.key();
    }
    return -1;
}

TerritoryAssignmentValidator::TerritoryAssignmentValidator(QObject *parent)
    : QValidator(parent), m_model(nullptr), m_assignmentId(-1), m_role(TerritoryAssignmentModel::Roles::None)
{
}

TerritoryAssignmentValidator::~TerritoryAssignmentValidator()
{
}

QValidator::State TerritoryAssignmentValidator::validate(QString &input, int &pos) const
{
    Q_UNUSED(pos)

    if (input.isEmpty()) {
        emit errorChanged("");
        return Acceptable;
    }
    if (!model()) {
        emit errorChanged("");
        return Acceptable;
    }
    if (role() == TerritoryAssignmentModel::Roles::None) {
        emit errorChanged("");
        return Acceptable;
    }
    QModelIndex modelIndex = model()->getAssignmentIndex(assignmentId());
    if (!modelIndex.isValid()) {
        emit errorChanged("");
        return Acceptable;
    }

    if (role() == TerritoryAssignmentModel::Roles::PersonFullNameRole) {
        // is in own congregation
        ccongregation cc;
        int ownCongregationId = cc.getMyCongregation().id;
        int publisherId = model()->data(modelIndex, TerritoryAssignmentModel::Roles::PersonIdRole).toInt();
        Person *publisher = cpersons::getPerson(publisherId);
        bool isNoMemberOfOwnCongregation = publisher
                ? publisher->congregationId() != ownCongregationId
                : false;
        if (isNoMemberOfOwnCongregation) {
            emit errorChanged(tr("The assigned person is no member of the congregation."));
            return Invalid;
        }
    }

    // check date
    QDate assignedDate = model()->data(modelIndex, TerritoryAssignmentModel::Roles::AssignedDateRole).toDate();
    QDate completedDate = model()->data(modelIndex, TerritoryAssignmentModel::Roles::CompletedDateRole).toDate();
    if (role() == TerritoryAssignmentModel::Roles::AssignedDateRole) {
        if (!assignedDate.isValid()) {
            emit errorChanged("");
            return Intermediate;
        }

        QModelIndex prevModelIndex = model()->index(modelIndex.row() - 1, modelIndex.column());
        if (prevModelIndex.isValid()) {
            QDate prevCompletedDate = model()->data(prevModelIndex, TerritoryAssignmentModel::Roles::CompletedDateRole).toDate();
            if (assignedDate < prevCompletedDate) {
                emit errorChanged(tr("Reassigned before completion. Please correct the dates.", "Territory assignment date"));
                return Invalid;
            }
        }

        emit errorChanged("");
        return Acceptable;
    }
    if (role() == TerritoryAssignmentModel::Roles::CompletedDateRole) {
        if (!assignedDate.isValid()) {
            emit errorChanged("");
            return Intermediate;
        }
        if (!completedDate.isValid()) {
            emit errorChanged("");
            return Intermediate;
        }
        if (completedDate < assignedDate) {
            emit errorChanged(tr("Completion date before date of assignment. Please correct the dates.", "Territory assignment date"));
            return Invalid;
        }

        QModelIndex nextModelIndex = model()->index(modelIndex.row() + 1, modelIndex.column());
        if (nextModelIndex.isValid()) {
            QDate nextAssignedDate = model()->data(nextModelIndex, TerritoryAssignmentModel::Roles::AssignedDateRole).toDate();
            if (completedDate > nextAssignedDate) {
                emit errorChanged(tr("Reassigned before completion. Please correct the dates.", "Territory assignment date"));
                return Invalid;
            }
        }

        emit errorChanged("");
        return Acceptable;
    }

    emit errorChanged("");
    return Acceptable;
}

TerritoryAssignmentModel *TerritoryAssignmentValidator::model() const
{
    return m_model;
}

void TerritoryAssignmentValidator::setModel(TerritoryAssignmentModel *newModel)
{
    if (m_model == newModel)
        return;
    m_model = newModel;
    emit modelChanged();
}

int TerritoryAssignmentValidator::assignmentId() const
{
    return m_assignmentId;
}

void TerritoryAssignmentValidator::setAssignmentId(int newAssignmentId)
{
    if (m_assignmentId == newAssignmentId)
        return;
    m_assignmentId = newAssignmentId;
    emit assignmentIdChanged();
}

TerritoryAssignmentModel::Roles TerritoryAssignmentValidator::role() const
{
    return m_role;
}

void TerritoryAssignmentValidator::setRole(TerritoryAssignmentModel::Roles newRole)
{
    if (m_role == newRole)
        return;
    m_role = newRole;
    emit roleChanged();
}
