﻿/**
 * This file is part of TheocBase.
 *
 * Copyright (C) 2011-2017, TheocBase Development Team, see AUTHORS.
 *
 * TheocBase is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * TheocBase is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with TheocBase.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef TERRITORYADDRESS_H
#define TERRITORYADDRESS_H

#include <QObject>
#include <QAbstractTableModel>
#include <QSortFilterProxyModel>
#include <QValidator>
#include <QGeoCoordinate>
#include "dataobject.h"
#include "sql_class.h"

class TerritoryAddress : public QObject
{
    Q_OBJECT
    Q_PROPERTY(int id READ id NOTIFY notification)
    Q_PROPERTY(int territoryId READ territoryId WRITE setTerritoryId NOTIFY territoryIdChanged FINAL)
    Q_PROPERTY(QString country READ country WRITE setCountry NOTIFY countryChanged FINAL)
    Q_PROPERTY(QString state READ state WRITE setState NOTIFY stateChanged FINAL)
    Q_PROPERTY(QString county READ county WRITE setCounty NOTIFY countyChanged FINAL)
    Q_PROPERTY(QString city READ city WRITE setCity NOTIFY cityChanged FINAL)
    Q_PROPERTY(QString district READ district WRITE setDistrict NOTIFY districtChanged FINAL)
    Q_PROPERTY(QString street READ street WRITE setStreet NOTIFY streetChanged FINAL)
    Q_PROPERTY(QString houseNumber READ houseNumber WRITE setHouseNumber NOTIFY houseNumberChanged FINAL)
    Q_PROPERTY(QString postalCode READ postalCode WRITE setPostalCode NOTIFY postalCodeChanged FINAL)
    Q_PROPERTY(QString wktGeometry READ wktGeometry WRITE setWktGeometry NOTIFY wktGeometryChanged FINAL)
    Q_PROPERTY(QGeoCoordinate coordinate READ coordinate WRITE setCoordinate NOTIFY coordinateChanged FINAL)
    Q_PROPERTY(QString name READ name WRITE setName NOTIFY nameChanged FINAL)
    Q_PROPERTY(int addressTypeNumber READ addressTypeNumber WRITE setAddressTypeNumber NOTIFY addressTypeNumberChanged FINAL)
    Q_PROPERTY(QDate requestDate READ requestDate WRITE setRequestDate NOTIFY requestDateChanged FINAL)

public:
    explicit TerritoryAddress(QObject *parent = nullptr);
    TerritoryAddress(const int territoryId, QObject *parent = nullptr);
    TerritoryAddress(const int territoryId, const QString country,
                     const QString state, const QString county, const QString city,
                     const QString district, const QString street, const QString houseNumber,
                     const QString postalCode, const QString wktGeometry, QObject *parent = nullptr);
    TerritoryAddress(const int id, const int territoryId, const QString country,
                     const QString state, const QString county, const QString city,
                     const QString district, const QString street, const QString houseNumber,
                     const QString postalCode, const QString wktGeometry, const QString name,
                     const int addressTypeNumber, const QDate requestDate, QObject *parent = nullptr);
    ~TerritoryAddress();

    int id() const;

    int territoryId() const;
    void setTerritoryId(int newTerritoryId);
    QString country() const;
    void setCountry(const QString &newCountry);
    QString state() const;
    void setState(const QString &newState);
    QString county() const;
    void setCounty(const QString &newCounty);
    QString city() const;
    void setCity(const QString &newCity);
    QString district() const;
    void setDistrict(const QString &newDistrict);
    QString street() const;
    void setStreet(const QString &newStreet);
    QString houseNumber() const;
    void setHouseNumber(const QString &newHouseNumber);
    QString postalCode() const;
    void setPostalCode(const QString &newPostalCode);
    QString wktGeometry() const;
    void setWktGeometry(const QString &newWktGeometry);
    QGeoCoordinate coordinate() const;
    void setCoordinate(const QGeoCoordinate &newCoordinate);
    double latitude() const;
    double longitude() const;
    QString name() const;
    void setName(const QString &newName);
    int addressTypeNumber() const;
    void setAddressTypeNumber(int newAddressTypeNumber);
    QDate requestDate() const;
    void setRequestDate(const QDate &newRequestDate);

    bool isDirty() const { return m_isDirty; }
    void setIsDirty(const bool value);
    bool save();

private:
    int m_id;
    int m_territoryId;
    QString m_country;
    QString m_state;
    QString m_county;
    QString m_city;
    QString m_district;
    QString m_street;
    QString m_houseNumber;
    QString m_postalCode;
    QString m_wktGeometry;
    double m_latitude;
    double m_longitude;
    QString m_name;
    int m_addressTypeNumber;
    QDate m_requestDate;
    bool m_isDirty;

    void setLatLon();

signals:
    void notification();
    void requestDateChanged();
    void territoryIdChanged();
    void countryChanged();
    void stateChanged();
    void countyChanged();
    void cityChanged();
    void districtChanged();
    void streetChanged();
    void houseNumberChanged();
    void postalCodeChanged();
    void wktGeometryChanged();
    void coordinateChanged();
    void latitudeChanged();
    void longitudeChanged();
    void nameChanged();
    void addressTypeNumberChanged();
};

class TerritoryAddressModel : public QAbstractTableModel
{
    Q_OBJECT
    Q_PROPERTY(int count READ count NOTIFY countChanged)

public:
    enum Roles {
        None = 0,
        AddressIdRole = Qt::UserRole,
        AddressRole,
        TerritoryIdRole,
        CountryRole,
        StateRole,
        CountyRole,
        CityRole,
        DistrictRole,
        StreetRole,
        HouseNumberRole,
        PostalCodeRole,
        WktGeometryRole,
        NameRole,
        AddressTypeNumberRole,
        AddressTypeNameRole,
        AddressTypeColorRole,
        RequestDateRole,
        LatitudeRole,
        LongitudeRole,
        CoordinateRole
    };
    Q_ENUM(Roles)

    int count() { return rowCount(); }

    QHash<int, QByteArray> roleNames() const override;

    TerritoryAddressModel();
    TerritoryAddressModel(QObject *parent);
    Q_INVOKABLE void initialize();

    int rowCount(const QModelIndex &parent = QModelIndex()) const override;
    int columnCount(const QModelIndex &parent = QModelIndex()) const override;
    Q_INVOKABLE QVariantMap get(int row);
    QVariant data(const QModelIndex &index, int role = Qt::DisplayRole) const override;
    Q_INVOKABLE bool setData(const QModelIndex &index, const QVariant &value, int role = Qt::EditRole) override;
    Qt::ItemFlags flags(const QModelIndex &index) const override;
    QModelIndex index(int row, int column, const QModelIndex &parent = QModelIndex()) const override;
    Q_INVOKABLE QModelIndex getAddressIndex(int addressId) const;
    int addAddress(TerritoryAddress *territoryAddress);
    Q_INVOKABLE int addAddress(int territoryId, const QString country, const QString state, const QString county,
                               const QString city, const QString district, const QString street,
                               const QString houseNumber, const QString postalCode, QString wktGeometry);
    Q_INVOKABLE QVariant findAddress(int addressId);
    Q_INVOKABLE void removeAddress(int id);
    Q_INVOKABLE bool removeRows(int row, int count, const QModelIndex &parent = QModelIndex()) override;
    Q_INVOKABLE void loadAddresses(int territoryId = 0);
    Q_INVOKABLE void saveAddresses();
    Q_INVOKABLE bool updateAddress(int addressId, QString wktGeometry);
    Q_INVOKABLE bool updateAddress(int addressId, int territoryId);
    Q_INVOKABLE bool updateAddress(int addressId, int territoryId, const QString country, const QString state, const QString county,
                                   const QString city, const QString district, const QString street,
                                   const QString houseNumber, const QString postalCode, QString wktGeometry);

    Q_INVOKABLE double getMarkerScale();
    Q_INVOKABLE int getDefaultAddressTypeNumber();

private:
    TerritoryAddress *getItem(const QModelIndex &index) const;

    QList<TerritoryAddress *> territoryAddresses;
    DataObjectListModel *addressTypeListModel;

signals:
    void modelChanged();
    void countChanged();
    void notification();
    void editCompleted();
};

class TerritoryAddressSortFilterProxyModel : public QSortFilterProxyModel
{
    Q_OBJECT
    Q_PROPERTY(QObject *source READ source WRITE setSource NOTIFY sourceChanged)
    Q_PROPERTY(int filterTerritoryId READ filterTerritoryId WRITE setFilterTerritoryId NOTIFY filterTerritoryIdChanged)
    Q_PROPERTY(QString filterText READ filterText WRITE setFilterText NOTIFY filterTextChanged)
    Q_PROPERTY(QByteArray sortRole READ sortRole WRITE setSortRole NOTIFY sortChanged)
    Q_PROPERTY(QByteArray groupByRole READ groupByRole WRITE setGroupByRole NOTIFY groupByChanged)

public:
    TerritoryAddressSortFilterProxyModel(QObject *parent = nullptr);

    QObject *source() const;
    void setSource(QObject *source);

    int filterTerritoryId() const;
    void setFilterTerritoryId(int newValue);

    QString filterText() const;
    void setFilterText(QString newValue);

    QByteArray sortRole() const;
    void setSortRole(const QByteArray &role);

    QByteArray groupByRole() const;
    void setGroupByRole(const QByteArray &role);

    Q_INVOKABLE virtual void sort(int column, Qt::SortOrder order = Qt::AscendingOrder) override
    {
        QSortFilterProxyModel::sort(column, order);
    }

protected:
    bool filterAcceptsRow(int sourceRow, const QModelIndex &sourceParent) const override;
    bool lessThan(const QModelIndex &left, const QModelIndex &right) const override;
    int roleKey(const QByteArray &role) const;

signals:
    void sourceChanged();
    void filterTerritoryIdChanged();
    void filterTextChanged();
    void sortChanged();
    void groupByChanged();

private:
    int m_filterTerritoryId;
    QString m_filterText;
    QByteArray m_groupByRole;
};

class TerritoryAddressValidator : public QValidator
{
    Q_OBJECT
    Q_PROPERTY(TerritoryAddressModel *model READ model WRITE setModel NOTIFY modelChanged)
    Q_PROPERTY(int addressId READ addressId WRITE setAddressId NOTIFY addressIdChanged)
    Q_PROPERTY(TerritoryAddressModel::Roles role READ role WRITE setRole NOTIFY roleChanged)
public:
    TerritoryAddressValidator(QObject *parent = nullptr);
    ~TerritoryAddressValidator();

    QValidator::State validate(QString &input, int &pos) const override;

    TerritoryAddressModel *model() const;
    void setModel(TerritoryAddressModel *newModel);

    TerritoryAddressModel::Roles role() const;
    void setRole(TerritoryAddressModel::Roles newRole);

    int addressId() const;
    void setAddressId(int newAddressId);

Q_SIGNALS:
    void modelChanged();
    void addressIdChanged();
    void errorChanged(const QString &error) const;
    void roleChanged();

private:
    TerritoryAddressModel *m_model;
    int m_addressId;
    TerritoryAddressModel::Roles m_role;
};

#endif // TERRITORYADDRESS_H
