/**
 * This file is part of TheocBase.
 *
 * Copyright (C) 2011-2017, TheocBase Development Team, see AUTHORS.
 *
 * TheocBase is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * TheocBase is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with TheocBase.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "territoryaddress.h"
#include "cterritories.h"

TerritoryAddress::TerritoryAddress(QObject *parent)
    : QObject(parent), m_id(0), m_territoryId(-1), m_country(""), m_state(""), m_county(""), m_city(""), m_district(""), m_street(""), m_houseNumber(""), m_postalCode(""), m_wktGeometry(""), m_name(""), m_isDirty(false)
{
    sql_class *sql = &Singleton<sql_class>::Instance();
    m_addressTypeNumber = QVariant(sql->getSetting("territory_default_addresstype", "1")).toInt();
}

TerritoryAddress::TerritoryAddress(const int territoryId, const QString country,
                                   const QString state, const QString county, const QString city,
                                   const QString district, const QString street, const QString houseNumber,
                                   const QString postalCode, const QString wktGeometry, QObject *parent)
    : QObject(parent), m_id(0), m_territoryId(territoryId), m_country(country), m_state(state), m_county(county), m_city(city), m_district(district), m_street(street), m_houseNumber(houseNumber), m_postalCode(postalCode), m_wktGeometry(wktGeometry), m_name(""), m_isDirty(true)
{
    sql_class *sql = &Singleton<sql_class>::Instance();
    m_addressTypeNumber = QVariant(sql->getSetting("territory_default_addresstype", "1")).toInt();
    setLatLon();
}

TerritoryAddress::TerritoryAddress(const int id, const int territoryId, const QString country,
                                   const QString state, const QString county, const QString city,
                                   const QString district, const QString street, const QString houseNumber,
                                   const QString postalCode, const QString wktGeometry, const QString name,
                                   const int addressTypeNumber, const QDate requestDate, QObject *parent)
    : QObject(parent), m_id(id), m_territoryId(territoryId), m_country(country), m_state(state), m_county(county), m_city(city), m_district(district), m_street(street), m_houseNumber(houseNumber), m_postalCode(postalCode), m_wktGeometry(wktGeometry), m_name(name), m_addressTypeNumber(addressTypeNumber), m_requestDate(requestDate), m_isDirty(false)
{
    setLatLon();
}

TerritoryAddress::~TerritoryAddress()
{
}

int TerritoryAddress::id() const
{
    return m_id;
}

int TerritoryAddress::territoryId() const
{
    return m_territoryId;
}

void TerritoryAddress::setTerritoryId(int newTerritoryId)
{
    if (m_territoryId == newTerritoryId)
        return;
    m_territoryId = newTerritoryId;
    setIsDirty(true);
    emit territoryIdChanged();
}

QString TerritoryAddress::country() const
{
    return m_country;
}

void TerritoryAddress::setCountry(const QString &newCountry)
{
    if (m_country == newCountry)
        return;
    m_country = newCountry;
    setIsDirty(true);
    emit countryChanged();
}

QString TerritoryAddress::state() const
{
    return m_state;
}

void TerritoryAddress::setState(const QString &newState)
{
    if (m_state == newState)
        return;
    m_state = newState;
    setIsDirty(true);
    emit stateChanged();
}

QString TerritoryAddress::county() const
{
    return m_county;
}

void TerritoryAddress::setCounty(const QString &newCounty)
{
    if (m_county == newCounty)
        return;
    m_county = newCounty;
    setIsDirty(true);
    emit countyChanged();
}

QString TerritoryAddress::city() const
{
    return m_city;
}

void TerritoryAddress::setCity(const QString &newCity)
{
    if (m_city == newCity)
        return;
    m_city = newCity;
    setIsDirty(true);
    emit cityChanged();
}

QString TerritoryAddress::district() const
{
    return m_district;
}

void TerritoryAddress::setDistrict(const QString &newDistrict)
{
    if (m_district == newDistrict)
        return;
    m_district = newDistrict;
    setIsDirty(true);
    emit districtChanged();
}

QString TerritoryAddress::street() const
{
    return m_street;
}

void TerritoryAddress::setStreet(const QString &newStreet)
{
    if (m_street == newStreet)
        return;
    m_street = newStreet;
    setIsDirty(true);
    emit streetChanged();
}

QString TerritoryAddress::houseNumber() const
{
    return m_houseNumber;
}

void TerritoryAddress::setHouseNumber(const QString &newHouseNumber)
{
    if (m_houseNumber == newHouseNumber)
        return;
    m_houseNumber = newHouseNumber;
    setIsDirty(true);
    emit houseNumberChanged();
}

QString TerritoryAddress::postalCode() const
{
    return m_postalCode;
}

void TerritoryAddress::setPostalCode(const QString &newPostalCode)
{
    if (m_postalCode == newPostalCode)
        return;
    m_postalCode = newPostalCode;
    setIsDirty(true);
    emit postalCodeChanged();
}

QString TerritoryAddress::wktGeometry() const
{
    return m_wktGeometry;
}

void TerritoryAddress::setWktGeometry(const QString &newWktGeometry)
{
    if (m_wktGeometry == newWktGeometry)
        return;
    m_wktGeometry = newWktGeometry;
    setLatLon();
    setIsDirty(true);
    emit wktGeometryChanged();
}

QGeoCoordinate TerritoryAddress::coordinate() const
{
    return QGeoCoordinate(m_latitude, m_longitude);
}

void TerritoryAddress::setCoordinate(const QGeoCoordinate &newCoordinate)
{
    m_longitude = newCoordinate.longitude();
    m_latitude = newCoordinate.latitude();
    m_wktGeometry = "POINT(" + QVariant(m_longitude).toString() + " " + QVariant(m_latitude).toString() + ")";
    emit latitudeChanged();
    emit longitudeChanged();
    emit wktGeometryChanged();
    emit coordinateChanged();
}

void TerritoryAddress::setLatLon()
{
    QRegularExpressionMatch match = coordinateRegExp->match(m_wktGeometry);
    if (match.hasMatch()) {
        m_longitude = match.captured(1).toDouble();
        m_latitude = match.captured(2).toDouble();
        emit latitudeChanged();
        emit longitudeChanged();
    }
}

double TerritoryAddress::latitude() const
{
    return m_latitude;
}

double TerritoryAddress::longitude() const
{
    return m_longitude;
}

QString TerritoryAddress::name() const
{
    return m_name;
}

void TerritoryAddress::setName(const QString &newName)
{
    if (m_name == newName)
        return;
    m_name = newName;
    setIsDirty(true);
    emit nameChanged();
}

int TerritoryAddress::addressTypeNumber() const
{
    return m_addressTypeNumber;
}

void TerritoryAddress::setAddressTypeNumber(int newAddressTypeNumber)
{
    if (m_addressTypeNumber == newAddressTypeNumber)
        return;
    m_addressTypeNumber = newAddressTypeNumber;
    setIsDirty(true);
    emit addressTypeNumberChanged();
}

QDate TerritoryAddress::requestDate() const
{
    return m_requestDate;
}

void TerritoryAddress::setRequestDate(const QDate &newRequestDate)
{
    if (m_requestDate == newRequestDate)
        return;
    m_requestDate = newRequestDate;
    setIsDirty(true);
    emit requestDateChanged();
}

void TerritoryAddress::setIsDirty(const bool value)
{
    m_isDirty = value;
}

bool TerritoryAddress::save()
{
    if (street() == "")
        return false;

    // save changes to database
    sql_class *sql = &Singleton<sql_class>::Instance();

    int lang_id = sql->getLanguageDefaultId();

    sql_item queryitems;
    queryitems.insert(":id", m_id);
    int addressId = m_id > 0 ? sql->selectScalar("SELECT id FROM territory_address WHERE id = :id", &queryitems, -1).toInt() : -1;

    sql_item insertItems;
    insertItems.insert("territory_id", territoryId());
    insertItems.insert("country", country());
    insertItems.insert("state", state());
    insertItems.insert("county", county());
    insertItems.insert("city", city());
    insertItems.insert("district", district());
    insertItems.insert("street", street());
    insertItems.insert("housenumber", houseNumber());
    insertItems.insert("postalcode", postalCode());
    insertItems.insert("wkt_geometry", wktGeometry());
    insertItems.insert("name", name());
    insertItems.insert("addresstype_number", addressTypeNumber());
    insertItems.insert("request_date", requestDate());
    insertItems.insert("lang_id", lang_id);

    bool ret = false;
    if (addressId > 0) {
        // update
        ret = sql->updateSql("territory_address", "id", QString::number(addressId), &insertItems);
    } else {
        // insert new row
        int newId = sql->insertSql("territory_address", &insertItems, "id");
        ret = newId > 0;
        if (newId > 0)
            m_id = newId;
    }
    setIsDirty(!ret);

    return ret;
}

TerritoryAddressModel::TerritoryAddressModel()
{
    initialize();
}

TerritoryAddressModel::TerritoryAddressModel(QObject *parent)
    : QAbstractTableModel(parent)
{
    initialize();
}

void TerritoryAddressModel::initialize()
{
    cterritories *ct = new cterritories;
    addressTypeListModel = ct->getAddressTypes();
}

QHash<int, QByteArray> TerritoryAddressModel::roleNames() const
{
    QHash<int, QByteArray> items;
    items[AddressIdRole] = "id";
    items[AddressRole] = "address";
    items[TerritoryIdRole] = "territoryId";
    items[CountryRole] = "country";
    items[StateRole] = "state";
    items[CountyRole] = "county";
    items[CityRole] = "city";
    items[DistrictRole] = "district";
    items[StreetRole] = "street";
    items[HouseNumberRole] = "houseNumber";
    items[PostalCodeRole] = "postalCode";
    items[WktGeometryRole] = "wktGeometry";
    items[NameRole] = "name";
    items[AddressTypeNumberRole] = "addressTypeNumber";
    items[AddressTypeNameRole] = "addressTypeName";
    items[AddressTypeColorRole] = "addressTypeColor";
    items[RequestDateRole] = "requestDate";
    items[LatitudeRole] = "latitude";
    items[LongitudeRole] = "longitude";
    items[CoordinateRole] = "coordinate";
    return items;
}

int TerritoryAddressModel::rowCount(const QModelIndex & /*parent*/) const
{
    return territoryAddresses.count();
}

int TerritoryAddressModel::columnCount(const QModelIndex & /*parent*/) const
{
    return 14;
}

QVariantMap TerritoryAddressModel::get(int row)
{
    QHash<int, QByteArray> names = roleNames();
    QHashIterator<int, QByteArray> i(names);
    QVariantMap res;
    QModelIndex idx = index(row, 0);
    while (i.hasNext()) {
        i.next();
        QVariant data = idx.data(i.key());
        res[i.value()] = data;
    }
    return res;
}

QVariant TerritoryAddressModel::data(const QModelIndex &index, int role) const
{
    if (!index.isValid())
        return QVariant();

    if (index.row() < 0 || index.row() > territoryAddresses.count())
        return QVariant();

    if (role == Qt::DisplayRole || role == Qt::EditRole) {
        switch (index.column()) {
        case 0:
            return territoryAddresses[index.row()]->id();
        case 1:
            return territoryAddresses[index.row()]->territoryId();
        case 2:
            return territoryAddresses[index.row()]->country();
        case 3:
            return territoryAddresses[index.row()]->state();
        case 4:
            return territoryAddresses[index.row()]->county();
        case 5:
            return territoryAddresses[index.row()]->city();
        case 6:
            return territoryAddresses[index.row()]->district();
        case 7:
            return territoryAddresses[index.row()]->street();
        case 8:
            return territoryAddresses[index.row()]->houseNumber();
        case 9:
            return territoryAddresses[index.row()]->postalCode();
        case 10:
            return territoryAddresses[index.row()]->wktGeometry();
        case 11:
            return territoryAddresses[index.row()]->name();
        case 12:
            return territoryAddresses[index.row()]->addressTypeNumber();
        case 13: {
            QModelIndex typeIndex = addressTypeListModel->index(addressTypeListModel->getIndex(territoryAddresses[index.row()]->addressTypeNumber()), 0);
            return addressTypeListModel->data(typeIndex, DataObjectListModel::NameRole).toString();
        }
        case 14:
            return territoryAddresses[index.row()]->requestDate();
        case 15:
            return territoryAddresses[index.row()]->latitude();
        case 16:
            return territoryAddresses[index.row()]->longitude();
        }
    }

    switch (role) {
    case AddressIdRole:
        return territoryAddresses[index.row()]->id();
    case AddressRole:
        return QVariant::fromValue(territoryAddresses[index.row()]);
    case TerritoryIdRole:
        return territoryAddresses[index.row()]->territoryId();
    case CountryRole:
        return territoryAddresses[index.row()]->country();
    case StateRole:
        return territoryAddresses[index.row()]->state();
    case CountyRole:
        return territoryAddresses[index.row()]->county();
    case CityRole:
        return territoryAddresses[index.row()]->city();
    case DistrictRole:
        return territoryAddresses[index.row()]->district();
    case StreetRole:
        return territoryAddresses[index.row()]->street();
    case HouseNumberRole:
        return territoryAddresses[index.row()]->houseNumber();
    case PostalCodeRole:
        return territoryAddresses[index.row()]->postalCode();
    case WktGeometryRole:
        return territoryAddresses[index.row()]->wktGeometry();
    case NameRole:
        return territoryAddresses[index.row()]->name();
    case AddressTypeNumberRole:
        return territoryAddresses[index.row()]->addressTypeNumber();
    case AddressTypeNameRole: {
        QModelIndex typeIndex = addressTypeListModel->index(addressTypeListModel->getIndex(territoryAddresses[index.row()]->addressTypeNumber()), 0);
        return addressTypeListModel->data(typeIndex, DataObjectListModel::NameRole).toString();
    }
    case AddressTypeColorRole: {
        QModelIndex typeIndex = addressTypeListModel->index(addressTypeListModel->getIndex(territoryAddresses[index.row()]->addressTypeNumber()), 0);
        QString color = addressTypeListModel->data(typeIndex, DataObjectListModel::ColorRole).toString();
        return color.isEmpty() ? "#000000" : color;
    }
    case RequestDateRole:
        return territoryAddresses[index.row()]->requestDate();
    case LatitudeRole:
        return territoryAddresses[index.row()]->latitude();
    case LongitudeRole:
        return territoryAddresses[index.row()]->longitude();
    case CoordinateRole:
        return QVariant::fromValue(territoryAddresses[index.row()]->coordinate());
    }
    return QVariant();
}

bool TerritoryAddressModel::setData(const QModelIndex &index, const QVariant &value, int role)
{
    if (!index.isValid())
        return false;

    if (index.row() < 0 || index.row() > territoryAddresses.count())
        return false;

    TerritoryAddress *currTerritoryAddress = territoryAddresses[index.row()];

    if (role == Qt::EditRole) {
        int column = index.column();

        switch (column) {
        case 0:
            // addressId
            break;
        case 1:
            currTerritoryAddress->setTerritoryId(value.toInt());
            break;
        case 2:
            currTerritoryAddress->setCountry(value.toString());
            break;
        case 3:
            currTerritoryAddress->setState(value.toString());
            break;
        case 4:
            currTerritoryAddress->setCounty(value.toString());
            break;
        case 5:
            currTerritoryAddress->setCity(value.toString());
            break;
        case 6:
            currTerritoryAddress->setDistrict(value.toString());
            break;
        case 7:
            currTerritoryAddress->setStreet(value.toString());
            break;
        case 8:
            currTerritoryAddress->setHouseNumber(value.toString());
            break;
        case 9:
            currTerritoryAddress->setPostalCode(value.toString());
            break;
        case 10:
            currTerritoryAddress->setWktGeometry(value.toString());
            break;
        case 11:
            currTerritoryAddress->setName(value.toString());
            break;
        case 12:
            currTerritoryAddress->setAddressTypeNumber(value.toInt());
            break;
        case 13: {
            QString dateFormat = QLocale().dateFormat(QLocale::ShortFormat);
            QDate date = QDate::fromString(value.toString(), dateFormat);
            if (!dateFormat.contains("yyyy"))
                date = date.addYears(99).year() > QDate::currentDate().year() ? date : date.addYears(100);
            currTerritoryAddress->setRequestDate(date);
            break;
        }
        default:
            break;
        }

        emit editCompleted();
    } else {
        switch (role) {
        case Roles::TerritoryIdRole:
            currTerritoryAddress->setTerritoryId(value.toInt());
            emit dataChanged(index, index, { role });
            break;
        case Roles::CountryRole:
            currTerritoryAddress->setCountry(value.toString());
            emit dataChanged(index, index, { role });
            break;
        case Roles::StateRole:
            currTerritoryAddress->setState(value.toString());
            emit dataChanged(index, index, { role });
            break;
        case Roles::CountyRole:
            currTerritoryAddress->setCounty(value.toString());
            emit dataChanged(index, index, { role });
            break;
        case Roles::CityRole:
            currTerritoryAddress->setCity(value.toString());
            emit dataChanged(index, index, { role });
            break;
        case Roles::DistrictRole:
            currTerritoryAddress->setDistrict(value.toString());
            emit dataChanged(index, index, { role });
            break;
        case Roles::StreetRole:
            currTerritoryAddress->setStreet(value.toString());
            emit dataChanged(index, index, { role });
            break;
        case Roles::HouseNumberRole:
            currTerritoryAddress->setHouseNumber(value.toString());
            emit dataChanged(index, index, { role });
            break;
        case Roles::PostalCodeRole:
            currTerritoryAddress->setPostalCode(value.toString());
            emit dataChanged(index, index, { role });
            break;
        case Roles::WktGeometryRole:
            currTerritoryAddress->setWktGeometry(value.toString());
            emit dataChanged(index, index, { role, Roles::LatitudeRole, Roles::LongitudeRole, Roles::CoordinateRole });
            break;
        case Roles::NameRole:
            currTerritoryAddress->setName(value.toString());
            emit dataChanged(index, index, { role });
            break;
        case Roles::AddressTypeNumberRole:
            currTerritoryAddress->setAddressTypeNumber(value.toInt());
            emit dataChanged(index, index, { role, Roles::AddressTypeNameRole, Roles::AddressTypeColorRole });
            break;
        case Roles::RequestDateRole: {
            QString dateFormat = QLocale().dateFormat(QLocale::ShortFormat);
            QDate date = QDate::fromString(value.toString(), dateFormat);
            if (!dateFormat.contains("yyyy"))
                date = date.addYears(99).year() > QDate::currentDate().year() ? date : date.addYears(100);
            currTerritoryAddress->setRequestDate(date);
            emit dataChanged(index, index, { role });
            break;
        }
        default:
            break;
        }
    }

    return true;
}

Qt::ItemFlags TerritoryAddressModel::flags(const QModelIndex &index) const
{
    if (!index.isValid())
        return Qt::ItemIsEnabled;
    return Qt::ItemIsEditable | QAbstractTableModel::flags(index);
}

QModelIndex TerritoryAddressModel::index(int row, int column, const QModelIndex &parent)
        const
{
    if (hasIndex(row, column, parent)) {
        return createIndex(row, column);
    }
    return QModelIndex();
}

QModelIndex TerritoryAddressModel::getAddressIndex(int addressId) const
{
    for (int row = 0; row < this->rowCount(); ++row) {
        QModelIndex rowIndex = this->index(row, 0);
        if (rowIndex.data(AddressIdRole) == addressId)
            return rowIndex;
    }
    return QModelIndex();
}

int TerritoryAddressModel::addAddress(TerritoryAddress *territoryAddress)
{
    beginInsertRows(QModelIndex(), rowCount(), rowCount());
    territoryAddresses << territoryAddress;
    endInsertRows();
    emit modelChanged();
    emit countChanged();
    return territoryAddress->id();
}

int TerritoryAddressModel::addAddress(int territoryId, const QString country,
                                      const QString state, const QString county, const QString city,
                                      const QString district, const QString street, const QString houseNumber,
                                      const QString postalCode, QString wktGeometry)
{
    beginInsertRows(QModelIndex(), rowCount(), rowCount());
    TerritoryAddress *newTerritoryAddress = new TerritoryAddress(territoryId, country, state, county,
                                                                 city, district, street, houseNumber,
                                                                 postalCode, wktGeometry, this);
    newTerritoryAddress->save();
    territoryAddresses << newTerritoryAddress;
    endInsertRows();
    emit modelChanged();
    emit countChanged();
    return newTerritoryAddress->id();
}

QVariant TerritoryAddressModel::findAddress(int addressId)
{
    QModelIndex addressIndex = getAddressIndex(addressId);
    if (addressIndex.isValid()) {
        return addressIndex.data(AddressRole);
    }
    return QVariant();
}

void TerritoryAddressModel::removeAddress(int id)
{
    if (id > 0) {
        sql_class *sql = &Singleton<sql_class>::Instance();

        sql_item s;
        s.insert("active", 0);
        if (sql->updateSql("territory_address", "id", QString::number(id), &s)) {
            QModelIndex addressIndex = getAddressIndex(id);
            if (addressIndex.isValid()) {
                int row = addressIndex.row();
                beginRemoveRows(QModelIndex(), row, row);
                territoryAddresses.erase(std::next(territoryAddresses.begin(), row));
                endRemoveRows();
                emit modelChanged();
                emit countChanged();
            }
        }
    }
}

bool TerritoryAddressModel::removeRows(int row, int count, const QModelIndex &parent)
{
    Q_UNUSED(parent)
    if (row < 0 || count < 1 || (row + count) > territoryAddresses.size())
        return false;
    beginRemoveRows(QModelIndex(), row, row + count - 1);
    for (int i = 0; i < count; i++) {
        territoryAddresses.removeAt(row);
    }
    endRemoveRows();
    emit modelChanged();
    emit countChanged();
    return true;
}

void TerritoryAddressModel::loadAddresses(int territoryId)
{
    AccessControl *ac = &Singleton<AccessControl>::Instance();
    if (!ac->user() || !ac->user()->hasPermission(PermissionRule::CanViewTerritories))
        return;

    beginResetModel();
    removeRows(0, territoryAddresses.count());
    sql_class *sql = &Singleton<sql_class>::Instance();

    QString sqlQuery(QString("SELECT a.* FROM territory_address a "
                             "JOIN territories t ON a.territory_id = t.id "
                             "WHERE (%1 OR t.person_id = %2) AND ")
                             .arg(!ac->isActive() || ac->user()->hasPermission(PermissionRule::CanViewTerritoryAddresses))
                             .arg(ac->user()->personId()));
    if (territoryId != 0)
        sqlQuery.append("a.territory_id = " + QVariant(territoryId).toString() + " AND ");
    sqlQuery.append("a.active ORDER BY a.street, housenumber");
    sql_items territoryAddressRows = sql->selectSql(sqlQuery);

    if (!territoryAddressRows.empty()) {
        for (unsigned int i = 0; i < territoryAddressRows.size(); i++) {
            sql_item s = territoryAddressRows[i];

            addAddress(new TerritoryAddress(s.value("id").toInt(),
                                            s.value("territory_id").toInt(),
                                            s.value("country").toString(),
                                            s.value("state").toString(),
                                            s.value("county").toString(),
                                            s.value("city").toString(),
                                            s.value("district").toString(),
                                            s.value("street").toString(),
                                            s.value("housenumber").toString(),
                                            s.value("postalcode").toString(),
                                            s.value("wkt_geometry").toString(),
                                            s.value("name").toString(),
                                            s.value("addresstype_number").toInt(),
                                            s.value("request_date").toDate(),
                                            this));
        }
    }
    endResetModel();
    emit modelChanged();
    emit countChanged();
}

void TerritoryAddressModel::saveAddresses()
{
    sql_class *sql = &Singleton<sql_class>::Instance();
    sql->startTransaction();
    for (int i = 0; i < territoryAddresses.count(); i++) {
        TerritoryAddress *t = territoryAddresses[i];
        if (t->isDirty()) {
            t->save();
        }
    }
    sql->commitTransaction();
}

bool TerritoryAddressModel::updateAddress(int addressId, QString wktGeometry)
{
    QModelIndex rowIndex = getAddressIndex(addressId);
    if (rowIndex.isValid()) {
        setData(rowIndex, wktGeometry, WktGeometryRole);
        return true;
    }
    return false;
}

bool TerritoryAddressModel::updateAddress(int addressId, int territoryId)
{
    QModelIndex rowIndex = getAddressIndex(addressId);
    if (rowIndex.isValid()) {
        setData(rowIndex, territoryId, TerritoryIdRole);
        return true;
    }
    return false;
}

bool TerritoryAddressModel::updateAddress(int addressId, int territoryId, const QString country,
                                          const QString state, const QString county,
                                          const QString city, const QString district,
                                          const QString street, const QString houseNumber,
                                          const QString postalCode, QString wktGeometry)
{
    QModelIndex rowIndex = getAddressIndex(addressId);
    if (rowIndex.isValid()) {
        setData(rowIndex, territoryId, TerritoryIdRole);
        setData(rowIndex, country, CountryRole);
        setData(rowIndex, state, StateRole);
        setData(rowIndex, county, CountyRole);
        setData(rowIndex, city, CityRole);
        setData(rowIndex, district, DistrictRole);
        setData(rowIndex, street, StreetRole);
        setData(rowIndex, houseNumber, HouseNumberRole);
        setData(rowIndex, postalCode, PostalCodeRole);
        setData(rowIndex, wktGeometry, WktGeometryRole);
        return true;
    }
    return false;
}

double TerritoryAddressModel::getMarkerScale()
{
    sql_class *sql = &Singleton<sql_class>::Instance();
    return QVariant(sql->getSetting("territory_map_markerscale", "0.5")).toDouble();
}

int TerritoryAddressModel::getDefaultAddressTypeNumber()
{
    sql_class *sql = &Singleton<sql_class>::Instance();
    return QVariant(sql->getSetting("territory_default_addresstype", "1")).toInt();
}

TerritoryAddress *TerritoryAddressModel::getItem(const QModelIndex &index) const
{
    if (index.isValid()) {
        TerritoryAddress *item = territoryAddresses[index.row()];
        if (item)
            return item;
    }
    return nullptr;
}

TerritoryAddressSortFilterProxyModel::TerritoryAddressSortFilterProxyModel(QObject *parent)
    : QSortFilterProxyModel(parent)
{
}

QObject *TerritoryAddressSortFilterProxyModel::source() const
{
    return sourceModel();
}

void TerritoryAddressSortFilterProxyModel::setSource(QObject *source)
{
    setSourceModel(qobject_cast<QAbstractItemModel *>(source));
    emit sourceChanged();
}

int TerritoryAddressSortFilterProxyModel::filterTerritoryId() const
{
    return m_filterTerritoryId;
}

void TerritoryAddressSortFilterProxyModel::setFilterTerritoryId(int newValue)
{
    beginFilterChange();
    m_filterTerritoryId = newValue;
    endFilterChange(QSortFilterProxyModel::Direction::Rows);
    emit filterTerritoryIdChanged();
}

QString TerritoryAddressSortFilterProxyModel::filterText() const
{
    return m_filterText;
}

void TerritoryAddressSortFilterProxyModel::setFilterText(QString newValue)
{
    beginFilterChange();
    m_filterText = newValue;
    endFilterChange(QSortFilterProxyModel::Direction::Rows);
    emit filterTextChanged();
}

QByteArray TerritoryAddressSortFilterProxyModel::sortRole() const
{
    return roleNames().value(QSortFilterProxyModel::sortRole());
}

void TerritoryAddressSortFilterProxyModel::setSortRole(const QByteArray &role)
{
    QSortFilterProxyModel::setSortRole(roleKey(role));
    emit sortChanged();
}

QByteArray TerritoryAddressSortFilterProxyModel::groupByRole() const
{
    return m_groupByRole;
}

void TerritoryAddressSortFilterProxyModel::setGroupByRole(const QByteArray &role)
{
    if (m_groupByRole == role)
        return;
    beginResetModel();
    m_groupByRole = role;
    endResetModel();
    emit groupByChanged();
    sort(0, sortOrder());
}

bool TerritoryAddressSortFilterProxyModel::filterAcceptsRow(int sourceRow,
                                                            const QModelIndex &sourceParent) const
{
    QModelIndex indexTerritoryId = sourceModel()->index(sourceRow, 0, sourceParent);
    int territoryId = sourceModel()->data(indexTerritoryId, TerritoryAddressModel::Roles::TerritoryIdRole).toInt();
    return (territoryId == filterTerritoryId())
            && (sourceModel()->data(indexTerritoryId, TerritoryAddressModel::Roles::StreetRole).toString().contains(filterText(), Qt::CaseInsensitive)
                || sourceModel()->data(indexTerritoryId, TerritoryAddressModel::Roles::NameRole).toString().contains(filterText(), Qt::CaseInsensitive));
}

bool TerritoryAddressSortFilterProxyModel::lessThan(const QModelIndex &left, const QModelIndex &right) const
{
    // compare sections first
    switch (roleKey(groupByRole())) {
    case TerritoryAddressModel::Roles::AddressTypeNameRole: {
        QVariant val1 = sourceModel()->data(left, TerritoryAddressModel::Roles::AddressTypeNameRole);
        QVariant val2 = sourceModel()->data(right, TerritoryAddressModel::Roles::AddressTypeNameRole);
        int compResult = QString::compare(val1.toString(), val2.toString(), Qt::CaseInsensitive);
        if (compResult != 0)
            return compResult < 0;
        break;
    }
    default:
        break;
    }

    // sort by address within sections
    QVariant leftData = sourceModel()->data(left, TerritoryAddressModel::Roles::StreetRole);
    QVariant rightData = sourceModel()->data(right, TerritoryAddressModel::Roles::StreetRole);
    int compResult = QString::compare(leftData.toString(), rightData.toString(), Qt::CaseInsensitive);
    if (compResult != 0)
        return compResult < 0;
    leftData = sourceModel()->data(left, TerritoryAddressModel::Roles::HouseNumberRole);
    rightData = sourceModel()->data(right, TerritoryAddressModel::Roles::HouseNumberRole);
    return leftData.toString() < rightData.toString();
}

int TerritoryAddressSortFilterProxyModel::roleKey(const QByteArray &role) const
{
    QHash<int, QByteArray> roles = roleNames();
    QHashIterator<int, QByteArray> it(roles);
    while (it.hasNext()) {
        it.next();
        if (it.value() == role)
            return it.key();
    }
    return -1;
}

TerritoryAddressValidator::TerritoryAddressValidator(QObject *parent)
    : QValidator(parent), m_model(nullptr), m_addressId(-1), m_role(TerritoryAddressModel::Roles::None)
{
}

TerritoryAddressValidator::~TerritoryAddressValidator()
{
}

QValidator::State TerritoryAddressValidator::validate(QString &input, int &pos) const
{
    Q_UNUSED(pos)

    if (input.isEmpty()) {
        emit errorChanged("");
        return Acceptable;
    }
    if (!model()) {
        emit errorChanged("");
        return Acceptable;
    }
    if (role() == TerritoryAddressModel::Roles::None) {
        emit errorChanged("");
        return Acceptable;
    }
    QModelIndex modelIndex = model()->getAddressIndex(addressId());
    if (!modelIndex.isValid()) {
        emit errorChanged("");
        return Acceptable;
    }

    // add validation here...

    emit errorChanged("");
    return Acceptable;
}

TerritoryAddressModel *TerritoryAddressValidator::model() const
{
    return m_model;
}

void TerritoryAddressValidator::setModel(TerritoryAddressModel *newModel)
{
    if (m_model == newModel)
        return;
    m_model = newModel;
    emit modelChanged();
}

int TerritoryAddressValidator::addressId() const
{
    return m_addressId;
}

void TerritoryAddressValidator::setAddressId(int newAddressId)
{
    if (m_addressId == newAddressId)
        return;
    m_addressId = newAddressId;
    emit addressIdChanged();
}

TerritoryAddressModel::Roles TerritoryAddressValidator::role() const
{
    return m_role;
}

void TerritoryAddressValidator::setRole(TerritoryAddressModel::Roles newRole)
{
    if (m_role == newRole)
        return;
    m_role = newRole;
    emit roleChanged();
}
