/**
 * This file is part of TheocBase.
 *
 * Copyright (C) 2011-2016, TheocBase Development Team, see AUTHORS.
 *
 * TheocBase is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * TheocBase is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with TheocBase.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef TERRITORY_H
#define TERRITORY_H

#include <QObject>
#include <QAbstractListModel>
#include <QtCore>
#include <QGeoCoordinate>
#include <QGeoRectangle>
#include <QGeoPolygon>
#include "accesscontrol.h"
#include "cpersons.h"
#include "sql_class.h"

#if defined(Q_OS_ANDROID) || defined(Q_OS_IOS)

#elif defined(Q_OS_LINUX)
#include "gdal/ogrsf_frmts.h"
#include "gdal/gdal_alg.h"
#elif defined(Q_OS_MAC) || defined(Q_OS_WIN)
#include "ogrsf_frmts.h"
#include "gdal_alg.h"
#endif

class Territory : public QObject
{
    Q_OBJECT

    Q_PROPERTY(int territoryId READ territoryId CONSTANT)
    Q_PROPERTY(int territoryNumber READ territoryNumber WRITE setTerritoryNumber BINDABLE bindableTerritoryNumber)
    Q_PROPERTY(QString locality READ locality WRITE setLocality BINDABLE bindableLocality)
    Q_PROPERTY(int cityId READ cityId WRITE setCityId BINDABLE bindableCityId)
    Q_PROPERTY(int typeId READ typeId WRITE setTypeId BINDABLE bindableTypeId)
    Q_PROPERTY(int priority READ priority WRITE setPriority BINDABLE bindablePriority)
    Q_PROPERTY(QString remark READ remark WRITE setRemark BINDABLE bindableRemark)
    Q_PROPERTY(QString wktGeometry READ wktGeometry WRITE setWktGeometry BINDABLE bindableWktGeometry)
    Q_PROPERTY(QList<QGeoPolygon> multiPolygon READ multiPolygon WRITE setMultiPolygon BINDABLE bindableMultiPolygon)
    Q_PROPERTY(QGeoRectangle boundingGeoRectangle READ boundingGeoRectangle BINDABLE bindableBoundingGeoRectangle)
    Q_PROPERTY(bool isDirty READ isDirty WRITE setIsDirty NOTIFY isDirtyChanged FINAL)

public:
    explicit Territory(QObject *parent = nullptr);
    Territory(int territoryId, int territoryNumber, QString locality, int cityId, int typeId, int priority, QString remark, QString wktGeometry, QString uuid, QObject *parent = nullptr);
    ~Territory();

    int territoryId() const;

    int territoryNumber() { return b_territoryNumber.value(); }
    void setTerritoryNumber(int newValue);
    QBindable<int> bindableTerritoryNumber() { return &b_territoryNumber; }

    QString locality() { return b_locality.value(); }
    void setLocality(QString newValue) { b_locality = newValue; }
    QBindable<QString> bindableLocality() { return &b_locality; }

    int cityId() { return b_cityId.value(); }
    void setCityId(int newValue) { b_cityId = newValue; }
    QBindable<int> bindableCityId() { return &b_cityId; }

    int typeId() { return b_typeId.value(); }
    void setTypeId(int newValue) { b_typeId = newValue; }
    QBindable<int> bindableTypeId() { return &b_typeId; }

    int priority() { return b_priority.value(); }
    void setPriority(int newValue) { b_priority = newValue; }
    QBindable<int> bindablePriority() { return &b_priority; }

    QString remark() { return b_remark.value(); }
    void setRemark(QString newValue) { b_remark = newValue; }
    QBindable<QString> bindableRemark() { return &b_remark; }

    QString wktGeometry() { return b_wktGeometry.value(); }
    void setWktGeometry(QString newValue) { b_wktGeometry = newValue; }
    QBindable<QString> bindableWktGeometry() { return &b_wktGeometry; }

    QList<QGeoPolygon> multiPolygon() { return b_multiPolygon.value(); }
    void setMultiPolygon(const QList<QGeoPolygon> &newValue) { b_multiPolygon = newValue; }
    QBindable<QList<QGeoPolygon>> bindableMultiPolygon() { return &b_multiPolygon; }

    QGeoRectangle boundingGeoRectangle() { return b_boundingGeoRectangle.value(); }
    QBindable<QGeoRectangle> bindableBoundingGeoRectangle() { return &b_boundingGeoRectangle; }

#if defined(Q_OS_ANDROID) || defined(Q_OS_IOS)
    // GDAL missing
#elif defined(Q_OS_LINUX) || defined(Q_OS_MAC) || defined(Q_OS_WIN)
    OGRGeometry *getOGRGeometry();
    bool setOGRGeometry(OGRGeometry *newGeometry);
    Q_INVOKABLE bool crosses(QList<QGeoCoordinate> path);
#endif

    bool isDirty() { return b_isDirty.value(); }
    void setIsDirty(bool newValue) { b_isDirty = newValue; }
    QBindable<bool> bindableIsDirty() { return &b_isDirty; }

    QString uuid() { return m_uuid; }

    Q_INVOKABLE int getNewNumber();
    Q_INVOKABLE bool reloadTerritory();
    Q_INVOKABLE bool remove();
    Q_INVOKABLE bool save();

    /**
     * @brief generateMultiPolygon - Converts the WKT geometry value and sets the MultiPolygon property
     * This can be executed as necessary, e.g. when the data needs to be displayed on the map.
     */
    void generateMultiPolygon();

signals:
    void territoryNumberChanged();
    void localityChanged();
    void cityIdChanged();
    void typeIdChanged();
    void priorityChanged();
    void remarkChanged();
    void wktGeometryChanged();
    void multiPolygonChanged();
    void boundingGeoRectangleChanged();
    void isDirtyChanged();

public slots:

private:
    int m_territoryId;
    Q_OBJECT_BINDABLE_PROPERTY(Territory, int, b_territoryNumber, &Territory::territoryNumberChanged);
    Q_OBJECT_BINDABLE_PROPERTY(Territory, QString, b_locality, &Territory::localityChanged);
    Q_OBJECT_BINDABLE_PROPERTY(Territory, int, b_cityId, &Territory::cityIdChanged);
    Q_OBJECT_BINDABLE_PROPERTY(Territory, int, b_typeId, &Territory::typeIdChanged);
    Q_OBJECT_BINDABLE_PROPERTY(Territory, int, b_priority, &Territory::priorityChanged);
    Q_OBJECT_BINDABLE_PROPERTY(Territory, QString, b_remark, &Territory::remarkChanged);
    Q_OBJECT_BINDABLE_PROPERTY(Territory, QString, b_wktGeometry, &Territory::wktGeometryChanged);
    Q_OBJECT_BINDABLE_PROPERTY(Territory, QList<QGeoPolygon>, b_multiPolygon, &Territory::multiPolygonChanged);
    QProperty<QGeoRectangle> b_boundingGeoRectangle;
    QProperty<bool> b_isDirty { false };
    QString m_uuid;
};

Q_DECLARE_METATYPE(Territory *)

class TerritoryType : public QObject
{
    Q_OBJECT
    Q_PROPERTY(int id READ territoryTypeId CONSTANT)
    Q_PROPERTY(QString name READ territoryTypeName WRITE setTerritoryTypeName NOTIFY notification)

public:
    TerritoryType(int id, QString typeName, QObject *parent = nullptr);
    ~TerritoryType();

    int territoryTypeId() { return m_territoryTypeId; }

    QString territoryTypeName() { return m_territoryTypeName; }
    void setTerritoryTypeName(QString value) { m_territoryTypeName = value; }

signals:
    void notification();

public slots:

private:
    int m_territoryTypeId;
    QString m_territoryTypeName;
};

class TerritoryTreeItem : public QObject
{
    Q_OBJECT
    Q_PROPERTY(int territoryId READ territoryId WRITE setTerritoryId NOTIFY territoryIdChanged FINAL)
    Q_PROPERTY(Territory *territory READ territory WRITE setTerritory NOTIFY territoryChanged FINAL)

public:
    explicit TerritoryTreeItem(const QList<QVariant> &data, TerritoryTreeItem *parentItem = nullptr);
    ~TerritoryTreeItem();

    TerritoryTreeItem *child(int row);
    Q_INVOKABLE int childCount() const;
    int columnCount() const;
    Q_INVOKABLE QVariant data(int column) const;
    Q_INVOKABLE QVariant value(int column) const;
    int row() const;
    void appendChild(TerritoryTreeItem *child);
    bool insertChildren(int position, int count);
    bool removeChildren(int position, int count);
    TerritoryTreeItem *parentItem();
    bool setData(int column, const QVariant &value);

    int territoryId() const;
    void setTerritoryId(int newTerritoryId);

    Territory *territory() const;
    void setTerritory(Territory *newTerritory);

signals:
    void territoryIdChanged();
    void territoryChanged();

private:
    QList<TerritoryTreeItem *> m_childItems;
    QList<QVariant> m_itemData;
    TerritoryTreeItem *m_parentItem;

    int m_territoryId;
    Territory *m_territory;
};

class TerritoryTreeModel : public QAbstractItemModel
{
    Q_OBJECT
    Q_PROPERTY(int groupByIndex READ groupByIndex WRITE setGroupByIndex NOTIFY groupByIndexChanged FINAL)
    Q_PROPERTY(Qt::SortOrder groupBySortOrder READ groupBySortOrder WRITE setGroupBySortOrder NOTIFY groupBySortOrderChanged FINAL)
    Q_PROPERTY(QGeoRectangle boundingGeoRectangle READ boundingGeoRectangle NOTIFY boundingGeoRectangleChanged FINAL)
    Q_PROPERTY(QList<QGeoCoordinate> snapPoints READ snapPoints NOTIFY snapPointsChanged FINAL)

public:
    enum Role {
        TitleRole = Qt::UserRole,
        SummaryRole,
        TerritoryIdRole,
        TerritoryRole,
        TerritoryNumberRole,
        LocalityRole,
        MultiPolygonRole,
        ChildCountRole
    };
    Q_ENUM(Role)

    enum GroupByTypes {
        GroupByCity,
        GroupByPublisher,
        GroupByTerritoryType,
        GroupByWorkedThrough
    };
    Q_ENUM(GroupByTypes)

    explicit TerritoryTreeModel(QObject *parent = nullptr);
    ~TerritoryTreeModel();

    QHash<int, QByteArray> roleNames() const override;
    QHash<int, QByteArray> groupByNames() const;

    QVariant data(const QModelIndex &index, int role) const override;
    Qt::ItemFlags flags(const QModelIndex &index) const override;
    bool setData(const QModelIndex &index, const QVariant &value,
                 int role = Qt::EditRole) override;
    Q_INVOKABLE QVariantMap get(int row) const;
    QVariant headerData(int section, Qt::Orientation orientation,
                        int role = Qt::DisplayRole) const override;
    Q_INVOKABLE QModelIndex index(int row, int column,
                                  const QModelIndex &parent = QModelIndex()) const override;
    Q_INVOKABLE QModelIndex index(int territoryId) const;
    Q_INVOKABLE QModelIndex parent(const QModelIndex &index) const override;

    int rowCount(const QModelIndex &parent = QModelIndex()) const override;
    int columnCount(const QModelIndex &parent = QModelIndex()) const override;

    bool insertRows(int position, int rows,
                    const QModelIndex &parent = QModelIndex()) override;

    bool removeRows(int position, int rows,
                    const QModelIndex &parent = QModelIndex()) override;

    int groupByIndex() const;
    void setGroupByIndex(int newGroupByIndex);
    Qt::SortOrder groupBySortOrder() const;
    void setGroupBySortOrder(Qt::SortOrder newGroupBySortOrder);
    Q_INVOKABLE void updateModel();
    Q_INVOKABLE void saveTerritories();

    Q_INVOKABLE int addNewTerritory();
    Q_INVOKABLE QVariant findTerritory(int territoryId);
    Q_INVOKABLE void removeTerritory(int territoryId);
    QModelIndex getGroupNodeIndex(QVariant value) const;

    QGeoRectangle boundingGeoRectangle() const;
    QList<QGeoCoordinate> snapPoints() const;

signals:
    void modelChanged();
    void notification();
    void groupByIndexChanged();
    void groupBySortOrderChanged();
    void territoryRemoved();
    void boundingGeoRectangleChanged();
    void snapPointsChanged();

private slots:
    void changeCity(int territoryId, int cityId);
    void changeType(int territoryId, int typeId);

private:
    void setupModelData(TerritoryTreeItem *parent);
    TerritoryTreeItem *getItem(const QModelIndex &index) const;
    TerritoryTreeItem *rootItem;
    int m_groupByIndex;
    Qt::SortOrder m_groupBySortOrder;
    QHash<int, QString> workedThroughNames;
    QGeoRectangle m_boundingGeoRectangle;
    QList<QGeoCoordinate> m_snapPoints;
};

class TerritoryTreeSFProxyModel : public QSortFilterProxyModel
{
    Q_OBJECT
    Q_PROPERTY(QObject *source READ source WRITE setSource NOTIFY sourceChanged)
    Q_PROPERTY(QString filterText READ filterText WRITE setFilterText NOTIFY filterTextChanged FINAL)
    Q_PROPERTY(QByteArray sortRole READ sortRole WRITE setSortRole NOTIFY sortRoleChanged FINAL)
    Q_PROPERTY(Qt::SortOrder sortOrder READ sortOrder WRITE setSortOrder NOTIFY sortOrderChanged FINAL)

public:
    explicit TerritoryTreeSFProxyModel(QObject *parent = nullptr);

    QObject *source() const;
    void setSource(QObject *source);

    QString filterText() const;
    void setFilterText(QString newValue);

    QByteArray sortRole() const;
    void setSortRole(const QByteArray &role);

    void setSortOrder(Qt::SortOrder order);

    Q_INVOKABLE QVariantMap get(int row);

    Q_INVOKABLE virtual void sort(int column, Qt::SortOrder order = Qt::AscendingOrder) override
    {
        QSortFilterProxyModel::sort(column, order);
    }

protected:
    bool filterAcceptsRow(int sourceRow, const QModelIndex &sourceParent) const override;
    bool lessThan(const QModelIndex &left, const QModelIndex &right) const override;
    int roleKey(const QByteArray &role) const;

signals:
    void sourceChanged();
    void filterTextChanged();
    void sortRoleChanged();
    void sortOrderChanged();

private:
    QString m_filterText;
};
#endif // TERRITORY_H
