/**
 * This file is part of TheocBase.
 *
 * Copyright (C) 2011-2016, TheocBase Development Team, see AUTHORS.
 *
 * TheocBase is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * TheocBase is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with TheocBase.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "territory.h"
#include "cterritories.h"

Territory::Territory(QObject *parent)
    : Territory(-1, -1, "", -1, -1, -1, "", "", "", parent)
{
}

Territory::Territory(int territoryId, int territoryNumber, QString locality, int cityId, int typeId, int priority, QString remark, QString wktGeometry, QString uuid, QObject *parent)
    : QObject(parent), m_territoryId(territoryId), b_territoryNumber(territoryNumber), b_locality(locality), b_cityId(cityId), b_typeId(typeId), b_priority(priority), b_remark(remark), b_wktGeometry(wktGeometry), b_isDirty(false), m_uuid(uuid)
{
    QObject::connect(this, &Territory::territoryNumberChanged, [&]() { b_isDirty = true; });
    QObject::connect(this, &Territory::localityChanged, [&]() { b_isDirty = true; });
    QObject::connect(this, &Territory::cityIdChanged, [&]() { b_isDirty = true; });
    QObject::connect(this, &Territory::typeIdChanged, [&]() { b_isDirty = true; });
    QObject::connect(this, &Territory::priorityChanged, [&]() { b_isDirty = true; });
    QObject::connect(this, &Territory::remarkChanged, [&]() { b_isDirty = true; });
    QObject::connect(this, &Territory::wktGeometryChanged, [&]() { b_isDirty = true; });

    b_boundingGeoRectangle.setBinding([&]() {
        QGeoRectangle boundingGeoRectangle;
        foreach (QGeoPolygon polygon, multiPolygon()) {
            if (boundingGeoRectangle.isValid()) {
                QGeoRectangle polygonBoundingBox = polygon.boundingGeoRectangle();
                boundingGeoRectangle.extendRectangle(polygonBoundingBox.bottomLeft());
                boundingGeoRectangle.extendRectangle(polygonBoundingBox.topRight());
            } else {
                boundingGeoRectangle = polygon.boundingGeoRectangle();
            }
        }
        return boundingGeoRectangle;
    });

    generateMultiPolygon();
}

Territory::~Territory()
{
}

int Territory::territoryId() const
{
    return m_territoryId;
}

void Territory::setTerritoryNumber(int newValue)
{
    if (newValue < 0) {
        newValue = 1;
        sql_class *sql = &Singleton<sql_class>::Instance();
        sql_items items = sql->selectSql("SELECT MAX(territory_number) AS maxNum FROM territory");
        if (items.size() > 0) {
            sql_item s = items[0];
            int maxvalue = s.value("maxNum").toInt();
            newValue = maxvalue + 1;
        }
    }
    b_territoryNumber = newValue;
}

void Territory::generateMultiPolygon()
{
    // update multi polygon list
    QRegularExpressionMatchIterator iMultiPolygon = multiPolygonRegExp->globalMatch(wktGeometry());
    QList<QGeoPolygon> newMultiPolygon;
    while (iMultiPolygon.hasNext()) {
        QRegularExpressionMatch multiPolygonMatch = iMultiPolygon.next();
        QString multiPolygonString = multiPolygonMatch.captured(0);
        QRegularExpressionMatchIterator iPolygon = polygonRegExp->globalMatch(multiPolygonString);

        QGeoPolygon geoPolygon;
        int innerPolygonIndex = 0;
        while (iPolygon.hasNext()) {
            QRegularExpressionMatch polygonMatch = iPolygon.next();
            QString polygon = polygonMatch.captured(0);
            QRegularExpressionMatchIterator iMatch = coordinateRegExp->globalMatch(polygon);
            QList<QGeoCoordinate> coordinates;
            while (iMatch.hasNext()) {
                QRegularExpressionMatch match = iMatch.next();
                double longitude = match.captured(1).toDouble();
                double latitude = match.captured(2).toDouble();
                QGeoCoordinate coordinate(latitude, longitude);
                coordinates.append(coordinate);
            }
            if (innerPolygonIndex == 0) {
                // fetched coordinates belong to current polygon's perimeter
                geoPolygon.setPerimeter(coordinates);
            } else {
                // fetched coordinates belong to a hole in the current polygon
                geoPolygon.addHole(coordinates);
            }
            innerPolygonIndex += 1;
        }
        newMultiPolygon.append(geoPolygon);
    }
    setMultiPolygon(newMultiPolygon);
}

#if defined(Q_OS_ANDROID) || defined(Q_OS_IOS)
// GDAL missing
#elif defined(Q_OS_LINUX) || defined(Q_OS_MAC) || defined(Q_OS_WIN)
OGRGeometry *Territory::getOGRGeometry()
{
    //    OGRSpatialReference osr;
    //    osr.SetWellKnownGeogCS("EPSG:4326");
    OGRGeometry *territoryGeometry = nullptr;

    // Parse WKT
    if (wktGeometry().isNull())
        return nullptr;
    QByteArray bytes = wktGeometry().toUtf8();
    const char *data = bytes.constData();
    OGRErr err = OGRGeometryFactory::createFromWkt(data, nullptr, &territoryGeometry);
    if (err != OGRERR_NONE) {
        // process error, like emit signal
        return nullptr;
    }
    if (!territoryGeometry->IsValid()) {
        qDebug() << "Trying to make WKT geometry of territory" << territoryNumber() << "valid";
        territoryGeometry = territoryGeometry->MakeValid();
    }
    if (!territoryGeometry->IsValid()) {
        qDebug() << "Saved WKT geometry of territory" << territoryNumber() << "is invalid!";
    }
    return territoryGeometry;
}

bool Territory::setOGRGeometry(OGRGeometry *newGeometry)
{
    char *txt;
    if (newGeometry != nullptr) {
        OGRGeometry *geometry = newGeometry->MakeValid();
        if (!geometry->IsEmpty() && !geometry->IsValid()) {
            qDebug() << "Geometry of territory" << territoryNumber() << "cannot be set because it is not valid!";
            return false;
        }
        if (geometry->exportToWkt(&txt) == OGRERR_NONE) {
            // test geometry before saving
            OGRSpatialReference osr;
            OGRGeometry *testGeometry = nullptr;

            // Parse WKT
            OGRErr err = OGRGeometryFactory::createFromWkt(txt, &osr, &testGeometry);
            if (err != OGRERR_NONE) {
                // process error, like emit signal
                qDebug() << "Geometry check for territory" << territoryNumber() << "failed!";
                return false;
            }
            if (!testGeometry->IsValid())
                testGeometry = testGeometry->Buffer(-0.001);
            if (!testGeometry->IsValid()) {
                qDebug() << "Geometry check for territory" << territoryNumber() << "failed because it is invalid!";
                return false;
            }

            qDebug() << "Geometry of territory" << territoryNumber() << "is valid. Set WKT geometry.";
            setWktGeometry(QString(txt));
            CPLFree(txt);
            return true;
        }
    }
    return false;
}
#endif

#if defined(Q_OS_ANDROID) || defined(Q_OS_IOS)
// GDAL missing
#elif defined(Q_OS_LINUX) || defined(Q_OS_MAC) || defined(Q_OS_WIN)
bool Territory::crosses(QList<QGeoCoordinate> path)
{
    OGRGeometry *territoryGeometry = getOGRGeometry();
    if (territoryGeometry != nullptr && territoryGeometry->IsEmpty())
        return true;

    if (territoryGeometry == nullptr || territoryGeometry->IsEmpty() || !territoryGeometry->IsValid())
        return false;

    OGRPolygon polygon;
    OGRLinearRing poRing;
    for (auto c = path.begin(), end = path.end(); c != end; ++c) {
        // check if point is within territory geometry, because check via Crosses doesn't work yet
        // nevertheless point check is not enough, since a line segment can also cross the territory
        OGRPoint pt;
        pt.setX((*c).longitude());
        pt.setY((*c).latitude());
        if (pt.Within(territoryGeometry))
            return true;
        poRing.addPoint((*c).longitude(), (*c).latitude());
    }
    poRing.closeRings();
    polygon.addRing(&poRing);
    OGRGeometry *testGeometry = polygon.getLinearGeometry();
    if (territoryGeometry->Intersects(testGeometry)) {
        OGRGeometry *intersection = territoryGeometry->Intersection(testGeometry);
        if (intersection == nullptr)
            return false;
        if (intersection->getGeometryType() == wkbPolygon || intersection->getGeometryType() == wkbMultiPolygon)
            return true;
        if (intersection->getGeometryType() == wkbGeometryCollection) {
            OGRGeometryCollection *poColl = intersection->toGeometryCollection();
            for (auto &&poSubGeom : *poColl) {
                switch (poSubGeom->getGeometryType()) {
                case wkbPolygon:
                    return true;
                case wkbMultiPolygon:
                    return true;
                default:
                    break;
                }
            }
        }
    }
    return false;
}
#endif

int Territory::getNewNumber()
{
    int newNumber = 1;
    sql_class *sql = &Singleton<sql_class>::Instance();
    sql_items items = sql->selectSql("SELECT MAX(territory_number) FROM territory");
    if (items.size() > 0) {
        sql_item s = items[0];
        int maxvalue = s.value(nullptr).toInt();
        newNumber = maxvalue + 1;
    }

    return newNumber;
}

bool Territory::reloadTerritory()
{
    sql_class *sql = &Singleton<sql_class>::Instance();
    sql_item queryitem;
    queryitem.insert(":id", m_territoryId);
    sql_items items = sql->selectSql("SELECT * FROM territory WHERE id = :id and active",
                                     &queryitem);
    if (items.size() > 0) {
        sql_item s = items[0];
        setTerritoryNumber(s.value("territory_number").toInt());
        setLocality(s.value("locality").toString());
        setCityId(s.value("city_id").toInt());
        setTypeId(s.value("type_id").toInt());
        setPriority(s.value("priority").toInt());
        setRemark(s.value("remark").toString());
        setWktGeometry(s.value("wkt_geometry").toString());
        generateMultiPolygon();

        if (b_isDirty)
            save();
        return true;
    }
    return false;
}

bool Territory::remove()
{
    sql_class *sql = &Singleton<sql_class>::Instance();

    sql_item s;
    s.insert("active", 0);
    return sql->updateSql("territory", "id", QString::number(m_territoryId), &s);
}

bool Territory::save()
{
    if (territoryNumber() < 1)
        return false;
    AccessControl *ac = &Singleton<AccessControl>::Instance();
    if (!ac->user() || !ac->user()->hasPermission(PermissionRule::CanEditTerritories))
        return false;

    // save changes to database
    if (!isDirty())
        return true;
    sql_class *sql = &Singleton<sql_class>::Instance();
    int lang_id = sql->getLanguageDefaultId();

    sql_item queryitems;
    queryitems.insert(":id", m_territoryId);
    int id = m_territoryId > 0 ? sql->selectScalar("SELECT id FROM territory WHERE id = :id", &queryitems, -1).toInt() : -1;

    sql_item insertItems;
    insertItems.insert("territory_number", territoryNumber());
    insertItems.insert("locality", locality());
    insertItems.insert("city_id", cityId());
    insertItems.insert("type_id", typeId());
    insertItems.insert("priority", priority());
    insertItems.insert("remark", remark());
    insertItems.insert("wkt_geometry", wktGeometry());
    insertItems.insert("lang_id", lang_id);

    bool ret = false;
    if (id > 0) {
        // update
        ret = sql->updateSql("territory", "id", QString::number(id), &insertItems);
    } else {
        // insert new row
        int newId = sql->insertSql("territory", &insertItems, "id");
        ret = newId > 0;
        if (newId > 0)
            m_territoryId = newId;
    }
    setIsDirty(!ret);

    return ret;
}

TerritoryType::TerritoryType(int id, QString typeName, QObject *parent)
    : QObject(parent), m_territoryTypeId(id), m_territoryTypeName(typeName)
{
}

TerritoryType::~TerritoryType()
{
}

TerritoryTreeItem::TerritoryTreeItem(const QList<QVariant> &data, TerritoryTreeItem *parentItem)
    : m_itemData(data), m_parentItem(parentItem)
{
}

TerritoryTreeItem::~TerritoryTreeItem()
{
    qDeleteAll(m_childItems);
}

TerritoryTreeItem *TerritoryTreeItem::child(int row)
{
    if (row < 0 || row >= m_childItems.size())
        return nullptr;
    return m_childItems.at(row);
}

int TerritoryTreeItem::childCount() const
{
    return m_childItems.count();
}

int TerritoryTreeItem::columnCount() const
{
    return m_itemData.count();
}

QVariant TerritoryTreeItem::data(int column) const
{
    if (column < 0 || column >= m_itemData.size())
        return QVariant();
    return m_itemData.at(column);
}

QVariant TerritoryTreeItem::value(int column) const
{
    if (column < 0 || column >= m_itemData.size())
        return QVariant();
    return m_itemData.at(column);
}

void TerritoryTreeItem::appendChild(TerritoryTreeItem *item)
{
    m_childItems.append(item);
}

bool TerritoryTreeItem::insertChildren(int position, int count)
{
    if (position < 0 || position > m_childItems.size())
        return false;

    for (int row = 0; row < count; ++row) {
        QList<QVariant> columnData;
        columnData << "";
        columnData << "";
        columnData << 0;
        columnData << 0;
        TerritoryTreeItem *item = new TerritoryTreeItem(columnData, this);
        m_childItems.insert(position, item);
    }

    return true;
}

bool TerritoryTreeItem::removeChildren(int position, int count)
{
    if (position < 0 || position + count > m_childItems.size())
        return false;

    for (int row = 0; row < count; ++row)
        delete m_childItems.takeAt(position);

    return true;
}

TerritoryTreeItem *TerritoryTreeItem::parentItem()
{
    return m_parentItem;
}

int TerritoryTreeItem::row() const
{
    if (m_parentItem)
        return m_parentItem->m_childItems.indexOf(const_cast<TerritoryTreeItem *>(this));

    return 0;
}

bool TerritoryTreeItem::setData(int column, const QVariant &value)
{
    if (column < 0 || column >= m_itemData.size())
        return false;

    m_itemData[column] = value;
    return true;
}

int TerritoryTreeItem::territoryId() const
{
    return data(2).toInt();
}

void TerritoryTreeItem::setTerritoryId(int newTerritoryId)
{
    setData(2, newTerritoryId);
    emit territoryIdChanged();
}
Territory *TerritoryTreeItem::territory() const
{
    return data(3).value<Territory *>();
}

void TerritoryTreeItem::setTerritory(Territory *newTerritory)
{
    setData(3, QVariant::fromValue(newTerritory));
    emit territoryChanged();
}

TerritoryTreeModel::TerritoryTreeModel(QObject *parent)
    : QAbstractItemModel(parent), m_groupByIndex(0), m_groupBySortOrder(Qt::AscendingOrder)
{
    // load settings such as sort order for grouping
    QSettings settings;
    m_groupByIndex = settings.value("LookupControl/territories_groupByIndex", m_groupByIndex).toInt();
    bool sortOrder = settings.value("LookupControl/territories_groupBySortOrder", m_groupBySortOrder).toBool();
    m_groupBySortOrder = static_cast<Qt::SortOrder>(sortOrder);

    QList<QVariant> rootData;
    rootData << "Title"
             << "Summary"
             << "ID"
             << "Territory";
    rootItem = new TerritoryTreeItem(rootData);
    workedThroughNames[0] = tr("Other territories");
    workedThroughNames[1] = tr("< 6 months", "territory worked");
    workedThroughNames[2] = tr("6 to 12 months", "territory worked");
    workedThroughNames[3] = tr("> 12 months ago", "territory worked");
    setupModelData(rootItem);
}

TerritoryTreeModel::~TerritoryTreeModel()
{
    delete rootItem;
}

int TerritoryTreeModel::columnCount(const QModelIndex &parent) const
{
    if (parent.isValid())
        return static_cast<TerritoryTreeItem *>(parent.internalPointer())->columnCount();
    return rootItem->columnCount();
}

QVariant TerritoryTreeModel::data(const QModelIndex &index, int role) const
{
    if (!index.isValid())
        return QVariant();

    TerritoryTreeItem *item = static_cast<TerritoryTreeItem *>(index.internalPointer());

    switch (role) {
    case TitleRole:
        return item->data(0);
    case SummaryRole:
        return item->data(1);
    case TerritoryIdRole:
        return item->data(2);
    case TerritoryRole:
        return item->data(3);
    case TerritoryNumberRole: {
        Territory *t = item->data(3).value<Territory *>();
        if (t != nullptr)
            return t->territoryNumber();
        else
            return 0;
        break;
    }
    case LocalityRole: {
        Territory *t = item->data(3).value<Territory *>();
        if (t != nullptr)
            return t->locality();
        else
            return "";
        break;
    }
    case MultiPolygonRole: {
        Territory *t = item->data(3).value<Territory *>();
        if (t != nullptr)
            return QVariant::fromValue(t->multiPolygon());
        else
            return QVariant::fromValue(QList<QGeoPolygon>());
        break;
    }
    case ChildCountRole:
        return item->childCount();
    default:
        return QVariant();
    }
    return item->data(index.column());
}

Qt::ItemFlags TerritoryTreeModel::flags(const QModelIndex &index) const
{
    if (!index.isValid())
        return Qt::NoItemFlags;

    return QAbstractItemModel::flags(index);
}

bool TerritoryTreeModel::setData(const QModelIndex &index, const QVariant &value, int role)
{
    TerritoryTreeItem *item = getItem(index);

    bool result = false;
    switch (role) {
    case Qt::EditRole:
        result = item->setData(index.column(), value);
        break;
    case TitleRole:
        result = item->setData(0, value);
        break;
    case SummaryRole:
        result = item->setData(1, value);
        break;
    case TerritoryIdRole:
        result = item->setData(2, value);
        break;
    case TerritoryRole:
        result = item->setData(3, value);
        break;
    }

    if (result)
        emit dataChanged(index, index);

    return result;
}

QVariantMap TerritoryTreeModel::get(int row) const
{
    QHash<int, QByteArray> names = roleNames();
    QHashIterator<int, QByteArray> i(names);
    QVariantMap res;
    QModelIndex idx = index(row, 0);
    while (i.hasNext()) {
        i.next();
        QVariant data = idx.data(i.key());
        res[i.value()] = data;
    }
    return res;
}

QVariant TerritoryTreeModel::headerData(int section, Qt::Orientation orientation,
                                        int role) const
{
    if (role == Qt::DisplayRole && orientation == Qt::Horizontal) {
        switch (section) {
        case 0:
            return tr("Number", "Territory number according to the S-12 Territory Map Card");
        case 1:
            return tr("Locality", "Locality according to the S-12 Territory Map Card");
        default:
            return QVariant();
        }
    }
    return QVariant();
}

QModelIndex TerritoryTreeModel::index(int row, int column, const QModelIndex &parent)
        const
{
    //    if (!hasIndex(row, column, parent))
    //        return QModelIndex();

    if (!rootItem || row < 0 || column < 0)
        return QModelIndex();

    TerritoryTreeItem *parentItem;

    if (!parent.isValid())
        parentItem = rootItem;
    else
        parentItem = static_cast<TerritoryTreeItem *>(parent.internalPointer());

    TerritoryTreeItem *childItem = parentItem->child(row);
    if (childItem)
        return createIndex(row, column, childItem);
    return QModelIndex();
}

QModelIndex TerritoryTreeModel::index(int territoryId) const
{
    for (int parentRow = 0; parentRow < this->rowCount(); ++parentRow) {
        QModelIndex parent = this->index(parentRow, 0);

        for (int childRow = 0; childRow < this->rowCount(parent); ++childRow) {
            QModelIndex child = this->index(childRow, 0, parent);
            if (child.data(TerritoryIdRole) == territoryId)
                return child;
        }
    }
    return QModelIndex();
}

QModelIndex TerritoryTreeModel::parent(const QModelIndex &index) const
{
    if (!index.isValid())
        return QModelIndex();

    TerritoryTreeItem *childItem = static_cast<TerritoryTreeItem *>(index.internalPointer());
    TerritoryTreeItem *parentItem = childItem->parentItem();

    if (parentItem == rootItem)
        return QModelIndex();

    return createIndex(parentItem->row(), 0, parentItem);
}

int TerritoryTreeModel::rowCount(const QModelIndex &parent) const
{
    TerritoryTreeItem *parentItem;
    if (parent.column() > 0)
        return 0;

    if (!parent.isValid())
        parentItem = rootItem;
    else
        parentItem = static_cast<TerritoryTreeItem *>(parent.internalPointer());

    return parentItem->childCount();
}

void TerritoryTreeModel::setupModelData(TerritoryTreeItem *parent)
{
    AccessControl *ac = &Singleton<AccessControl>::Instance();
    if (!ac->user() || !ac->user()->hasPermission(PermissionRule::CanViewTerritories))
        return;

    TerritoryTreeItem *rootItem = parent;
    TerritoryTreeItem *currentParent = nullptr;

    sql_class *sql = &Singleton<sql_class>::Instance();
    QByteArray groupByFieldName = groupByNames().value(m_groupByIndex);
    QString sqlCommand;
    // in the grouping of the territories, display data such as assignee and work progress only if the user
    // 1) has permissions to view all territory assignments or 2) is assigned to the territory
    switch (m_groupByIndex) {
    case GroupByPublisher: {
        // prepare full name with name order according to the settings
        QString nameOrderFormat = sql->getSetting("nameFormat", "%2, %1");
        nameOrderFormat = QString(nameOrderFormat).arg("p.firstname").arg("p.lastname");
        nameOrderFormat.replace(' ', " || ' ' || ");
        nameOrderFormat.replace(',', " || ','");
        // TODO: include firstname and lastname columns in territories to query without additional join
        sqlCommand = QString("SELECT t.*, "
                             "  CASE WHEN %1 OR (t.person_id = %2) THEN %3"
                             "  ELSE '%4' END as groupByField "
                             "FROM territories t "
                             "LEFT JOIN persons p on t.person_id = p.id")
                             .arg(!ac->isActive() || ac->user()->hasPermission(PermissionRule::CanViewTerritoryAssignments))
                             .arg(ac->user()->personId())
                             .arg(nameOrderFormat)
                             .arg(tr("Other territories"));
        break;
    }
    case GroupByWorkedThrough:
        sqlCommand = QString("SELECT *, "
                             "  CASE WHEN %1 OR (person_id = %2) THEN %3"
                             "  ELSE '%4' END as groupByField "
                             "FROM territories")
                             .arg(!ac->isActive() || ac->user()->hasPermission(PermissionRule::CanViewTerritoryAssignments))
                             .arg(ac->user()->personId())
                             .arg(groupByFieldName)
                             .arg(tr("Other territories"));
        break;
    default:
        sqlCommand = QString("SELECT *, %1 as groupByField FROM territories")
                             .arg(groupByFieldName);
        break;
    }
    sqlCommand.append(QString(" ORDER BY groupByField %1, locality;")
                              .arg(m_groupBySortOrder == Qt::AscendingOrder ? " ASC" : " DESC"));
    sql_items territories = sql->selectSql(sqlCommand);
    // reset bounding box and snap points
    m_boundingGeoRectangle = QGeoRectangle();
    m_snapPoints.clear();
    if (!territories.empty()) {
        QString currentGroupByFieldValue = nullptr;

        for (unsigned int i = 0; i < territories.size(); i++) {
            const sql_item item = territories[i];

            QString newGroupByFieldValue = item.value("groupByField").toString();
            if (newGroupByFieldValue == nullptr) {
                if (m_groupByIndex == GroupByWorkedThrough)
                    newGroupByFieldValue = tr("Not worked", "Group text for territories that are not worked yet.");
                else
                    newGroupByFieldValue = tr("Not assigned", "Value of the field, the territories are grouped by, is empty.");
            } else if (m_groupByIndex == GroupByWorkedThrough) {
                newGroupByFieldValue = workedThroughNames.value(newGroupByFieldValue.toInt());
            }

            QList<QVariant> columnData;

            if (currentGroupByFieldValue != newGroupByFieldValue) {
                if (currentParent != nullptr) {
                    int childCount = currentParent->childCount();
                    // update current group header before the next will be added
                    currentParent->setData(1, tr("Count=%1", "Lookup control").arg(childCount));
                }
                // add new group header
                columnData << newGroupByFieldValue;
                columnData << "";
                columnData << 0;
                columnData << 0;
                currentParent = new TerritoryTreeItem(columnData, rootItem);
                rootItem->appendChild(currentParent);

                currentGroupByFieldValue = newGroupByFieldValue;
            }

            int territoryId = item.value("id").toInt();
            columnData.clear();
            columnData << "";
            columnData << "";
            columnData << territoryId;
            // NOTE: parent must be set in order to protect it from destroying by qml
            Territory *t = new Territory(
                    territoryId,
                    item.value("territory_number").toInt(),
                    item.value("locality").toString(),
                    item.value("city_id").toInt(),
                    item.value("type_id").toInt(),
                    item.value("priority").toInt(),
                    item.value("remark").toString(),
                    item.value("wkt_geometry").toString(),
                    item.value("uuid").toString(),
                    this);
            columnData << QVariant::fromValue(t);

            // extend bounding box
            QGeoRectangle territoryBoundingBox = t->boundingGeoRectangle();
            if (m_boundingGeoRectangle.isValid()) {
                m_boundingGeoRectangle.extendRectangle(territoryBoundingBox.bottomLeft());
                m_boundingGeoRectangle.extendRectangle(territoryBoundingBox.topRight());
            } else {
                m_boundingGeoRectangle = territoryBoundingBox.boundingGeoRectangle();
            }

            // add snap points
            foreach (QGeoPolygon polygon, t->multiPolygon()) {
                m_snapPoints.append(polygon.perimeter());
                for (qsizetype iHole = 0; iHole < polygon.holesCount(); ++iHole) {
                    m_snapPoints.append(polygon.holePath(iHole));
                }
            }

            // Append a new item to the current parent's list of children.
            if (currentParent != nullptr)
                currentParent->appendChild(new TerritoryTreeItem(columnData, currentParent));
        }
        if (currentParent != nullptr) {
            // update last group header
            int childCount = currentParent->childCount();
            currentParent->setData(1, tr("Count=%1", "Lookup control").arg(childCount));
        }
    }
}

bool TerritoryTreeModel::insertRows(int position, int rows, const QModelIndex &parent)
{
    TerritoryTreeItem *parentItem = getItem(parent);
    bool success;

    beginInsertRows(parent, position, position + rows - 1);
    success = parentItem->insertChildren(position, rows);
    endInsertRows();
    emit modelChanged();
    return success;
}

bool TerritoryTreeModel::removeRows(int position, int rows,
                                    const QModelIndex &parent)
{
    TerritoryTreeItem *parentItem = getItem(parent);
    bool success;

    if (position < 0 || rows < 1 || position + rows > parentItem->childCount())
        return false;

    beginRemoveRows(parent, position, position + rows - 1);
    success = parentItem->removeChildren(position, rows);
    endRemoveRows();
    emit modelChanged();

    return success;
}

QHash<int, QByteArray> TerritoryTreeModel::roleNames() const
{
    auto roles = QAbstractItemModel::roleNames();
    roles[TitleRole] = "title";
    roles[SummaryRole] = "summary";
    roles[TerritoryIdRole] = "territoryId";
    roles[TerritoryRole] = "territory";
    roles[TerritoryNumberRole] = "territoryNumber";
    roles[LocalityRole] = "locality";
    roles[MultiPolygonRole] = "multiPolygon";
    roles[ChildCountRole] = "childCount";
    return roles;
}

QHash<int, QByteArray> TerritoryTreeModel::groupByNames() const
{
    QHash<int, QByteArray> items;
    items[GroupByCity] = "city";
    items[GroupByPublisher] = "publisher";
    items[GroupByTerritoryType] = "type_name";
    items[GroupByWorkedThrough] = "workedthrough";
    return items;
}

int TerritoryTreeModel::groupByIndex() const
{
    return m_groupByIndex;
}

void TerritoryTreeModel::setGroupByIndex(int newGroupByIndex)
{
    if (m_groupByIndex == newGroupByIndex)
        return;
    m_groupByIndex = newGroupByIndex;
    updateModel();
    emit groupByIndexChanged();
}

Qt::SortOrder TerritoryTreeModel::groupBySortOrder() const
{
    return m_groupBySortOrder;
}

void TerritoryTreeModel::setGroupBySortOrder(Qt::SortOrder newGroupBySortOrder)
{
    if (m_groupBySortOrder == newGroupBySortOrder)
        return;
    m_groupBySortOrder = newGroupBySortOrder;
    updateModel();
    emit groupBySortOrderChanged();
}

void TerritoryTreeModel::updateModel()
{
    beginResetModel();

    if (rootItem->childCount() > 0) {
        beginRemoveRows(QModelIndex(), 0, rootItem->childCount() - 1);
        rootItem->removeChildren(0, rootItem->childCount());
        endRemoveRows();
    }

    setupModelData(rootItem);

    endResetModel();
    emit modelChanged();
    emit boundingGeoRectangleChanged();
    emit snapPointsChanged();
}

void TerritoryTreeModel::saveTerritories()
{
    sql_class *sql = &Singleton<sql_class>::Instance();
    sql->startTransaction();
    for (int parentRow = 0; parentRow < this->rowCount(); ++parentRow) {
        QModelIndex parent = this->index(parentRow, 0);

        for (int childRow = 0; childRow < this->rowCount(parent); ++childRow) {
            QModelIndex child = this->index(childRow, 0, parent);
            Territory *territory = child.data(TerritoryRole).value<Territory *>();
            territory->save();
        }
    }
    sql->commitTransaction();
}

TerritoryTreeItem *TerritoryTreeModel::getItem(const QModelIndex &index) const
{
    if (index.isValid()) {
        TerritoryTreeItem *item = static_cast<TerritoryTreeItem *>(index.internalPointer());
        if (item)
            return item;
    }
    return rootItem;
}

QGeoRectangle TerritoryTreeModel::boundingGeoRectangle() const
{
    return m_boundingGeoRectangle;
}

QList<QGeoCoordinate> TerritoryTreeModel::snapPoints() const
{
    return m_snapPoints;
}

int TerritoryTreeModel::addNewTerritory()
{
    // create new territory datarow
    Territory *newTerritory = new Territory(this);
    newTerritory->setTerritoryNumber(-1); // get next number from db
    newTerritory->save();
    int newTerritoryId = newTerritory->territoryId();

    if (newTerritoryId > 0) {
        QModelIndex parentIndex;

        // find "Not assigned" group
        for (int row = 0; row < this->rowCount(); ++row) {
            QModelIndex nodeIndex = this->index(row, 0);
            if (nodeIndex.data(TitleRole) == tr("Not assigned", "Value of the field, the territories are grouped by, is empty.")) {
                parentIndex = nodeIndex;
                break;
            }
        }

        if (!parentIndex.isValid()) {
            // insert new row for the "Not assigned" group, in case the list is empty or all territories are assigned
            QModelIndex rootIndex = createIndex(rootItem->row(), 0, rootItem);
            if (this->insertRow(0)) {
                parentIndex = this->index(0, 0, rootIndex);
                this->setData(
                        parentIndex,
                        tr("Not assigned", "Value of the field, the territories are grouped by, is empty."),
                        TitleRole);
            }
        }

        if (parentIndex.isValid()) {
            // insert new row for the territory
            int pos = this->rowCount(parentIndex);
            if (insertRows(pos, 1, parentIndex)) {
                QModelIndex childIndex = this->index(pos, 0, parentIndex);
                // set territory data
                this->setData(childIndex, QVariant::fromValue(newTerritoryId), TerritoryIdRole);
                this->setData(childIndex, QVariant::fromValue(newTerritory), TerritoryRole);
                // update parent row's summary
                int childCount = this->rowCount(parentIndex);
                this->setData(parentIndex, tr("Count=%1", "Lookup control").arg(childCount), SummaryRole);
            }
        }
    }
    return newTerritoryId;
    emit modelChanged();
}

QVariant TerritoryTreeModel::findTerritory(int territoryId)
{
    QModelIndex territoryIndex = index(territoryId);
    if (territoryIndex.isValid()) {
        return territoryIndex.data(TerritoryRole);
    }
    return QVariant();
}

void TerritoryTreeModel::removeTerritory(int territoryId)
{
    cterritories *ct = new cterritories;
    if (ct->removeTerritory(territoryId)) {
        // update tree
        QModelIndex territoryIndex = index(territoryId);
        QModelIndex parent = territoryIndex.parent();
        if (territoryIndex.isValid())
            if (removeRows(territoryIndex.row(), 1, territoryIndex.parent())) {
                if (this->rowCount(parent) == 0)
                    removeRows(parent.row(), 1, parent.parent());
            }
    }
    emit territoryRemoved();
    emit modelChanged();
}

QModelIndex TerritoryTreeModel::getGroupNodeIndex(QVariant value) const
{
    int grpRows = this->rowCount();
    for (int i = 0; i < grpRows; i++) {
        QModelIndex idx = index(i, 0);

        if (this->data(idx, TerritoryIdRole) == value)
            return idx;
    }
    return QModelIndex();
}

void TerritoryTreeModel::changeCity(int territoryId, int cityId)
{
    // reload; TODO: move treeitem
    Q_UNUSED(territoryId)
    Q_UNUSED(cityId)
    if (m_groupByIndex == GroupByCity)
        setGroupByIndex(GroupByCity);
}

void TerritoryTreeModel::changeType(int territoryId, int typeId)
{
    // reload; TODO: move treeitem
    Q_UNUSED(territoryId)
    Q_UNUSED(typeId)
    if (m_groupByIndex == GroupByTerritoryType)
        setGroupByIndex(GroupByTerritoryType);
}

TerritoryTreeSFProxyModel::TerritoryTreeSFProxyModel(QObject *parent)
    : QSortFilterProxyModel(parent)
{
    setRecursiveFilteringEnabled(true);
}

QObject *TerritoryTreeSFProxyModel::source() const
{
    return sourceModel();
}

void TerritoryTreeSFProxyModel::setSource(QObject *source)
{
    setSourceModel(qobject_cast<QAbstractItemModel *>(source));
    emit sourceChanged();
}

QString TerritoryTreeSFProxyModel::filterText() const
{
    return m_filterText;
}

void TerritoryTreeSFProxyModel::setFilterText(QString newValue)
{
    beginFilterChange();
    m_filterText = newValue;
    endFilterChange(QSortFilterProxyModel::Direction::Rows);
    emit filterTextChanged();
}

QByteArray TerritoryTreeSFProxyModel::sortRole() const
{
    return roleNames().value(QSortFilterProxyModel::sortRole());
}

void TerritoryTreeSFProxyModel::setSortRole(const QByteArray &role)
{
    QSortFilterProxyModel::setSortRole(roleKey(role));
    emit sortRoleChanged();
    QSortFilterProxyModel::invalidate();
}

void TerritoryTreeSFProxyModel::setSortOrder(Qt::SortOrder order)
{
    QSortFilterProxyModel::sort(0, order);
    emit sortOrderChanged();
}

bool TerritoryTreeSFProxyModel::filterAcceptsRow(int sourceRow,
                                                 const QModelIndex &sourceParent) const
{
    QModelIndex index = sourceModel()->index(sourceRow, 0, sourceParent);
    Territory *t = sourceModel()->data(index, TerritoryTreeModel::TerritoryRole).value<Territory *>();
    QString title = sourceModel()->data(index, TerritoryTreeModel::TitleRole).toString();
    return (t != nullptr
            && (QVariant(t->territoryNumber()).toString().contains(filterText(), Qt::CaseInsensitive)
                || t->locality().contains(filterText(), Qt::CaseInsensitive)))
            || title.contains(filterText(), Qt::CaseInsensitive);
}

QVariantMap TerritoryTreeSFProxyModel::get(int row)
{
    QHash<int, QByteArray> names = roleNames();
    QHashIterator<int, QByteArray> i(names);
    QVariantMap res;
    QModelIndex idx = index(row, 0);
    while (i.hasNext()) {
        i.next();
        QVariant data = idx.data(i.key());
        res[i.value()] = data;
    }
    return res;
}

int TerritoryTreeSFProxyModel::roleKey(const QByteArray &role) const
{
    QHash<int, QByteArray> roles = roleNames();
    QHashIterator<int, QByteArray> it(roles);
    while (it.hasNext()) {
        it.next();
        if (it.value() == role)
            return it.key();
    }
    return -1;
}

bool TerritoryTreeSFProxyModel::lessThan(const QModelIndex &left, const QModelIndex &right) const
{
    QVariant id = sourceModel()->data(left, TerritoryTreeModel::TerritoryIdRole);
    if (id.toInt() == 0) {
        // group node; keep sort order from the model
    } else {
        switch (roleKey(sortRole())) {
        case TerritoryTreeModel::TerritoryNumberRole: {
            QVariant val1 = sourceModel()->data(left, roleKey(sortRole()));
            QVariant val2 = sourceModel()->data(right, roleKey(sortRole()));
            if (!val1.isNull() && !val2.isNull()) {
                return val1.toInt() < val2.toInt();
            }
            break;
        }
        case TerritoryTreeModel::LocalityRole: {
            QVariant val1 = sourceModel()->data(left, roleKey(sortRole()));
            QVariant val2 = sourceModel()->data(right, roleKey(sortRole()));
            int compResult = QString::compare(val1.toString(), val2.toString(), Qt::CaseInsensitive);
            return compResult < 0;
            break;
        }
        default:
            break;
        }
    }
    return false;
}
