/**
 * This file is part of TheocBase.
 *
 * Copyright (C) 2011-2022, TheocBase Development Team, see AUTHORS.
 *
 * TheocBase is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * TheocBase is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with TheocBase.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef TBSTYLE_H
#define TBSTYLE_H

#include <QObject>
#include <QColor>
#include <QFont>
#include <QQmlEngine>
#include <QApplication>
#include <QPalette>
#include <QtMath>
#include <QProperty>
#include <QObjectBindableProperty>
#include "sql_class.h"

class TBStyle : public QObject
{
    Q_OBJECT
    Q_DISABLE_COPY(TBStyle)
    // QML_ELEMENT

    // colors
    Q_PROPERTY(Qt::ColorScheme colorScheme READ colorScheme BINDABLE bindableColorScheme)
    Q_PROPERTY(QColor primaryColor READ primaryColor BINDABLE bindablePrimaryColor)
    Q_PROPERTY(QColor primaryTextColor READ primaryTextColor BINDABLE bindablePrimaryTextColor)
    Q_PROPERTY(QColor onPrimaryColor READ onPrimaryColor BINDABLE bindableOnPrimaryColor)
    Q_PROPERTY(QColor mediumColor READ mediumColor BINDABLE bindableMediumColor)
    Q_PROPERTY(QColor alertColor READ alertColor BINDABLE bindableAlertColor)
    Q_PROPERTY(QColor neutralColor READ neutralColor BINDABLE bindableNeutralColor)
    Q_PROPERTY(QColor neutralTextColor READ neutralTextColor BINDABLE bindableNeutralTextColor)
    Q_PROPERTY(QColor lmmSection1Color READ lmmSection1Color BINDABLE bindableLmmSection1Color)
    Q_PROPERTY(QColor lmmSection1TextColor READ lmmSection1TextColor BINDABLE bindableLmmSection1TextColor)
    Q_PROPERTY(QColor lmmSection2Color READ lmmSection2Color BINDABLE bindableLmmSection2Color)
    Q_PROPERTY(QColor lmmSection2TextColor READ lmmSection2TextColor BINDABLE bindableLmmSection2TextColor)
    Q_PROPERTY(QColor lmmSection3Color READ lmmSection3Color BINDABLE bindableLmmSection3Color)
    Q_PROPERTY(QColor lmmSection3TextColor READ lmmSection3TextColor BINDABLE bindableLmmSection3TextColor)
    Q_PROPERTY(QColor publicTalkColor READ publicTalkColor BINDABLE bindablePublicTalkColor)
    Q_PROPERTY(QColor publicTalkTextColor READ publicTalkTextColor BINDABLE bindablePublicTalkTextColor)
    Q_PROPERTY(QColor watchtowerStudyColor READ watchtowerStudyColor BINDABLE bindableWatchtowerStudyColor)
    Q_PROPERTY(QColor watchtowerStudyTextColor READ watchtowerStudyTextColor BINDABLE bindableWatchtowerStudyTextColor)
    Q_PROPERTY(QColor outgoingSpeakersColor READ outgoingSpeakersColor BINDABLE bindableOutgoingSpeakersColor)
    Q_PROPERTY(QColor outgoingSpeakersTextColor READ outgoingSpeakersTextColor BINDABLE bindableOutgoingSpeakersTextColor)
    // text styles
    Q_PROPERTY(QFont bodySmallFont READ bodySmallFont BINDABLE bindableBodySmallFont)
    Q_PROPERTY(QFont bodyMediumFont READ bodyMediumFont BINDABLE bindableBodyMediumFont)
    Q_PROPERTY(QFont bodyLargeFont READ bodyLargeFont BINDABLE bindableBodyLargeFont)
    Q_PROPERTY(QFont titleSmallFont READ titleSmallFont BINDABLE bindableTitleSmallFont)
    Q_PROPERTY(QFont titleMediumFont READ titleMediumFont BINDABLE bindableTitleMediumFont)
    Q_PROPERTY(QFont titleLargeFont READ titleLargeFont BINDABLE bindableTitleLargeFont)
    Q_PROPERTY(QFont headlineSmallFont READ headlineSmallFont BINDABLE bindableHeadlineSmallFont)
    // formats
    Q_PROPERTY(QString fullNameFormat READ fullNameFormat WRITE setFullNameFormat BINDABLE bindableFullNameFormat)

public:
    enum ColorRole {
        Primary,
        PrimaryText,
        OnPrimary,
        Medium,
        Alert,
        Neutral,
        NeutralText,
        LMMSection1,
        LMMSection1Text,
        LMMSection2,
        LMMSection2Text,
        LMMSection3,
        LMMSection3Text,
        PublicTalk,
        PublicTalkText,
        WatchtowerStudy,
        WatchtowerStudyText,
        OutgoingSpeakers,
        OutgoingSpeakersText,
        NColorRoles = OutgoingSpeakersText + 1
    };
    Q_ENUM(ColorRole)

    struct OKLCh {
        qreal L;
        qreal C;
        qreal h;
    };

    struct OKLab {
        qreal L;
        qreal a;
        qreal b;
    };

    enum TextStyleRole {
        BodySmall,
        BodyMedium,
        BodyLarge,
        TitleSmall,
        TitleMedium,
        TitleLarge,
        HeadlineSmall,
        NTextStyleRoles = HeadlineSmall + 1
    };
    Q_ENUM(TextStyleRole)

    static TBStyle &Instance()
    {
        static TBStyle s_instance;
        QQmlEngine::setObjectOwnership(&s_instance, QQmlEngine::CppOwnership);
        return s_instance;
    }
    static QObject *qmlInstance(QQmlEngine *engine, QJSEngine *scriptEngine)
    {
        Q_UNUSED(engine)
        Q_UNUSED(scriptEngine)

        static TBStyle *s_instance = &TBStyle::Instance();
        return s_instance;
    }
    bool event(QEvent *event) override;

    // colors
    Qt::ColorScheme colorScheme() const { return b_colorScheme; }
    QBindable<Qt::ColorScheme> bindableColorScheme() { return &b_colorScheme; }
    QColor primaryColor() const { return b_primaryColor; }
    QBindable<QColor> bindablePrimaryColor() { return &b_primaryColor; }
    QColor primaryTextColor() const { return b_primaryTextColor; }
    QBindable<QColor> bindablePrimaryTextColor() { return &b_primaryTextColor; }
    QColor onPrimaryColor() const { return b_onPrimaryColor; }
    QBindable<QColor> bindableOnPrimaryColor() { return &b_onPrimaryColor; }
    QColor mediumColor() const { return b_mediumColor; }
    QBindable<QColor> bindableMediumColor() { return &b_mediumColor; }
    QColor alertColor() const { return b_alertColor; }
    QBindable<QColor> bindableAlertColor() { return &b_alertColor; }
    QColor neutralColor() const { return b_neutralColor; }
    QBindable<QColor> bindableNeutralColor() { return &b_neutralColor; }
    QColor neutralTextColor() const { return b_neutralTextColor; }
    QBindable<QColor> bindableNeutralTextColor() { return &b_neutralTextColor; }
    QColor lmmSection1Color() const { return b_lmmSection1Color; }
    QBindable<QColor> bindableLmmSection1Color() { return &b_lmmSection1Color; }
    QColor lmmSection1TextColor() const { return b_lmmSection1TextColor; }
    QBindable<QColor> bindableLmmSection1TextColor() { return &b_lmmSection1TextColor; }
    QColor lmmSection2Color() const { return b_lmmSection2Color; }
    QBindable<QColor> bindableLmmSection2Color() { return &b_lmmSection2Color; }
    QColor lmmSection2TextColor() const { return b_lmmSection2TextColor; }
    QBindable<QColor> bindableLmmSection2TextColor() { return &b_lmmSection2TextColor; }
    QColor lmmSection3Color() const { return b_lmmSection3Color; }
    QBindable<QColor> bindableLmmSection3Color() { return &b_lmmSection3Color; }
    QColor lmmSection3TextColor() const { return b_lmmSection3TextColor; }
    QBindable<QColor> bindableLmmSection3TextColor() { return &b_lmmSection3TextColor; }
    QColor publicTalkColor() const { return b_publicTalkColor; }
    QBindable<QColor> bindablePublicTalkColor() { return &b_publicTalkColor; }
    QColor publicTalkTextColor() const { return b_publicTalkTextColor; }
    QBindable<QColor> bindablePublicTalkTextColor() { return &b_publicTalkTextColor; }
    QColor watchtowerStudyColor() const { return b_watchtowerStudyColor; }
    QBindable<QColor> bindableWatchtowerStudyColor() { return &b_watchtowerStudyColor; }
    QColor watchtowerStudyTextColor() const { return b_watchtowerStudyTextColor; }
    QBindable<QColor> bindableWatchtowerStudyTextColor() { return &b_watchtowerStudyTextColor; }
    QColor outgoingSpeakersColor() const { return b_outgoingSpeakersColor; }
    QBindable<QColor> bindableOutgoingSpeakersColor() { return &b_outgoingSpeakersColor; }
    QColor outgoingSpeakersTextColor() const { return b_outgoingSpeakersTextColor; }
    QBindable<QColor> bindableOutgoingSpeakersTextColor() { return &b_outgoingSpeakersTextColor; }

    // text styles
    QFont bodySmallFont() const { return b_bodySmallFont; }
    QBindable<QFont> bindableBodySmallFont() { return &b_bodySmallFont; }
    QFont bodyMediumFont() const { return b_bodyMediumFont; }
    QBindable<QFont> bindableBodyMediumFont() { return &b_bodyMediumFont; }
    QFont bodyLargeFont() const { return b_bodyLargeFont; }
    QBindable<QFont> bindableBodyLargeFont() { return &b_bodyLargeFont; }
    QFont titleSmallFont() const { return b_titleSmallFont; }
    QBindable<QFont> bindableTitleSmallFont() { return &b_titleSmallFont; }
    QFont titleMediumFont() const { return b_titleMediumFont; }
    QBindable<QFont> bindableTitleMediumFont() { return &b_titleMediumFont; }
    QFont titleLargeFont() const { return b_titleLargeFont; }
    QBindable<QFont> bindableTitleLargeFont() { return &b_titleLargeFont; }
    QFont headlineSmallFont() const { return b_headlineSmallFont; }
    QBindable<QFont> bindableHeadlineSmallFont() { return &b_headlineSmallFont; }

    // formats
    QString fullNameFormat() const { return b_fullNameFormat; }
    void setFullNameFormat(QString newValue)
    {
        if (b_fullNameFormat != newValue) {
            sql_class *sql = &Singleton<sql_class>::Instance();
            sql->saveSetting("nameFormat", newValue);
        }
        b_fullNameFormat = newValue;
    }
    QBindable<QString> bindableFullNameFormat() { return &b_fullNameFormat; }

signals:
    void colorSchemeChanged();
    void primaryColorChanged();
    void primaryTextColorChanged();
    void onPrimaryColorChanged();
    void mediumColorChanged();
    void alertColorChanged();
    void neutralColorChanged();
    void neutralTextColorChanged();
    void lmmSection1ColorChanged();
    void lmmSection1TextColorChanged();
    void lmmSection2ColorChanged();
    void lmmSection2TextColorChanged();
    void lmmSection3ColorChanged();
    void lmmSection3TextColorChanged();
    void publicTalkColorChanged();
    void publicTalkTextColorChanged();
    void watchtowerStudyColorChanged();
    void watchtowerStudyTextColorChanged();
    void outgoingSpeakersColorChanged();
    void outgoingSpeakersTextColorChanged();
    void bodySmallFontChanged();
    void bodyMediumFontChanged();
    void bodyLargeFontChanged();
    void titleSmallFontChanged();
    void titleMediumFontChanged();
    void titleLargeFontChanged();
    void headlineSmallFontChanged();
    void stylesChanged();
    void fullNameFormatChanged();

private:
    void initialize();

    Q_OBJECT_BINDABLE_PROPERTY(TBStyle, Qt::ColorScheme, b_colorScheme, &TBStyle::colorSchemeChanged)
    Q_OBJECT_BINDABLE_PROPERTY(TBStyle, QColor, b_primaryColor, &TBStyle::primaryColorChanged)
    Q_OBJECT_BINDABLE_PROPERTY(TBStyle, QColor, b_primaryTextColor, &TBStyle::primaryTextColorChanged)
    Q_OBJECT_BINDABLE_PROPERTY(TBStyle, QColor, b_onPrimaryColor, &TBStyle::onPrimaryColorChanged)
    Q_OBJECT_BINDABLE_PROPERTY(TBStyle, QColor, b_mediumColor, &TBStyle::mediumColorChanged)
    Q_OBJECT_BINDABLE_PROPERTY(TBStyle, QColor, b_alertColor, &TBStyle::alertColorChanged)
    Q_OBJECT_BINDABLE_PROPERTY(TBStyle, QColor, b_neutralColor, &TBStyle::neutralColorChanged)
    Q_OBJECT_BINDABLE_PROPERTY(TBStyle, QColor, b_neutralTextColor, &TBStyle::neutralTextColorChanged)
    Q_OBJECT_BINDABLE_PROPERTY(TBStyle, QColor, b_lmmSection1Color, &TBStyle::lmmSection1ColorChanged)
    Q_OBJECT_BINDABLE_PROPERTY(TBStyle, QColor, b_lmmSection1TextColor, &TBStyle::lmmSection1TextColorChanged)
    Q_OBJECT_BINDABLE_PROPERTY(TBStyle, QColor, b_lmmSection2Color, &TBStyle::lmmSection2ColorChanged)
    Q_OBJECT_BINDABLE_PROPERTY(TBStyle, QColor, b_lmmSection2TextColor, &TBStyle::lmmSection2TextColorChanged)
    Q_OBJECT_BINDABLE_PROPERTY(TBStyle, QColor, b_lmmSection3Color, &TBStyle::lmmSection3ColorChanged)
    Q_OBJECT_BINDABLE_PROPERTY(TBStyle, QColor, b_lmmSection3TextColor, &TBStyle::lmmSection3TextColorChanged)
    Q_OBJECT_BINDABLE_PROPERTY(TBStyle, QColor, b_publicTalkColor, &TBStyle::publicTalkColorChanged)
    Q_OBJECT_BINDABLE_PROPERTY(TBStyle, QColor, b_publicTalkTextColor, &TBStyle::publicTalkTextColorChanged)
    Q_OBJECT_BINDABLE_PROPERTY(TBStyle, QColor, b_watchtowerStudyColor, &TBStyle::watchtowerStudyColorChanged)
    Q_OBJECT_BINDABLE_PROPERTY(TBStyle, QColor, b_watchtowerStudyTextColor, &TBStyle::watchtowerStudyTextColorChanged)
    Q_OBJECT_BINDABLE_PROPERTY(TBStyle, QColor, b_outgoingSpeakersColor, &TBStyle::outgoingSpeakersColorChanged)
    Q_OBJECT_BINDABLE_PROPERTY(TBStyle, QColor, b_outgoingSpeakersTextColor, &TBStyle::outgoingSpeakersTextColorChanged)
    Q_OBJECT_BINDABLE_PROPERTY(TBStyle, QFont, b_bodySmallFont, &TBStyle::bodySmallFontChanged)
    Q_OBJECT_BINDABLE_PROPERTY(TBStyle, QFont, b_bodyMediumFont, &TBStyle::bodyMediumFontChanged)
    Q_OBJECT_BINDABLE_PROPERTY(TBStyle, QFont, b_bodyLargeFont, &TBStyle::bodyLargeFontChanged)
    Q_OBJECT_BINDABLE_PROPERTY(TBStyle, QFont, b_titleSmallFont, &TBStyle::titleSmallFontChanged)
    Q_OBJECT_BINDABLE_PROPERTY(TBStyle, QFont, b_titleMediumFont, &TBStyle::titleMediumFontChanged)
    Q_OBJECT_BINDABLE_PROPERTY(TBStyle, QFont, b_titleLargeFont, &TBStyle::titleLargeFontChanged)
    Q_OBJECT_BINDABLE_PROPERTY(TBStyle, QFont, b_headlineSmallFont, &TBStyle::headlineSmallFontChanged)
    Q_OBJECT_BINDABLE_PROPERTY(TBStyle, QString, b_fullNameFormat, &TBStyle::fullNameFormatChanged)

    // colors
    static QColor getCompositeColor(QColor backgroundColor, QColor foregroundColor);
    /**
     * @brief getContrastRatio - Calculate the contrast ratio between two colors using the WCAG formula
     * @param lightColor - lighter color
     * @param darkColor - darker color
     * @return - contrast ratio [1-21] (21 = constrast ratio betweeen black and white)
     */
    qreal getContrastRatio(QColor lightColor, QColor darkColor);
    OKLCh convertOKLab2OKLCh(OKLab color);
    OKLab convertOKLCh2OKLab(OKLCh color);
    OKLCh convertColor2OKLCh(QColor color);
    QColor convertOKLCh2Color(OKLCh color);
    qreal getOKLChColorDifference(OKLCh color1, OKLCh color2);
    /**
     * @brief findForegroundColor - Find a new foreground color for the given background color with good contrast
     * @param backgroundColor - background color
     * @param sourceColor - color to start from
     * @param minimumContrastRatio - minimum level for acceptable contrast
     * @return - closest color to the source, which meets the required contrast ratio
     */
    QColor findForegroundColor(QColor backgroundColor, QColor sourceColor, qreal minimumContrastRatio = 4.5);
    TBStyle(QObject *parent = nullptr);

    // text styles
    inline const QFont &font(TextStyleRole tr) const
    {
        Q_ASSERT(tr < NTextStyleRoles);
        return fontMap[tr];
    };
    QFont fontMap[NTextStyleRoles];
};

#endif // TBSTYLE_H
