/**
 * This file is part of TheocBase.
 *
 * Copyright (C) 2011-2015, TheocBase Development Team, see AUTHORS.
 *
 * TheocBase is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * TheocBase is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with TheocBase.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "single_application.h"

#include <QLocalSocket>

#include "single_application.h"
#include "tbstyle.h"

SingleApplication::SingleApplication(int &argc, char *argv[], const QString uniqueKey)
    : QApplication(argc, argv), _uniqueKey(uniqueKey)
{
    sharedMemory.setKey(_uniqueKey);
    if (sharedMemory.attach())
        _isRunning = true;
    else {
        _isRunning = false;
        // create shared memory.
        if (!sharedMemory.create(1)) {
            qDebug("Unable to create single instance.");
            return;
        }
        // create local server and listen to incomming messages from other instances.
        localServer = new QLocalServer(this);
        connect(localServer, SIGNAL(newConnection()), this, SLOT(receiveMessage()));
        localServer->listen(_uniqueKey);
    }
}

// public slots.

void SingleApplication::receiveMessage()
{
    QLocalSocket *localSocket = localServer->nextPendingConnection();
    if (!localSocket->waitForReadyRead(timeout)) {
        // qDebug(localSocket->errorString().toLatin1());
        return;
    }
    QByteArray byteArray = localSocket->readAll();
    QString message = QString::fromUtf8(byteArray.constData());
    emit messageAvailable(message);
    localSocket->disconnectFromServer();
}

// public functions.

bool SingleApplication::isRunning()
{
    return _isRunning;
}

bool SingleApplication::sendMessage(const QString &message)
{
    if (!_isRunning)
        return false;
    QLocalSocket localSocket(this);
    localSocket.connectToServer(_uniqueKey, QIODevice::WriteOnly);
    if (!localSocket.waitForConnected(timeout)) {
        // qDebug(localSocket.errorString().toLatin1());
        return false;
    }
    localSocket.write(message.toUtf8());
    if (!localSocket.waitForBytesWritten(timeout)) {
        // qDebug(localSocket.errorString().toLatin1());
        return false;
    }
    localSocket.disconnectFromServer();
    return true;
}

bool SingleApplication::event(QEvent *event)
{
    switch (event->type()) {
    case QEvent::ApplicationPaletteChange: {
        TBStyle *tbStyle = &TBStyle::Instance();
        QEvent paletteChangeEvent(QEvent::ApplicationPaletteChange);
        QApplication::sendEvent(tbStyle, &paletteChangeEvent);
        return QApplication::event(event);
    }
    case QEvent::FileOpen:
        emit messageAvailable(static_cast<QFileOpenEvent *>(event)->file());
        return true;
    default:
        return QApplication::event(event);
    }
}
