import QtQuick
import QtQuick.Controls
import QtQuick.Controls.impl
import QtQuick.Layouts
import net.theocbase 1.0
import "./"

Page {
    id: control

    property bool showSearchField: true
    property bool showFilterControls: true
    property bool showGroupControls: false
    property int groupByIndex
    property bool groupBySortOrder
    property bool sortOrder
    property bool showAddButton: false
    property bool showEditButton: false
    property bool isEditing: false
    property ListModel filterModel
    property ListModel groupByModel
    property Menu moreMenu
    property variant model
    property int selectionBehavior: TableView.SelectCells
    property int selectionMode: TableView.ExtendedSelection
    property variant selectionModel
    property Component delegate: MenuItem {
        width: TreeView.view.width
        text: modelData
        highlighted: TreeView.view.currentIndex === index

        onClicked: {
            TreeView.view.currentIndex = index
        }
    }
    property Component headerDelegate: Button {
        text: model.display
        onClicked: {
            console.log("sort column " + index)
        }
    }
    property bool keyNavigationEnabled
    property var columnWidthProvider
    property var rowHeightProvider: function (row) {
        return 30;
    }
    property bool alternatingRows

    signal searchTextChanged(var text)
    signal filterChanged(int index, bool checked)
    signal addItem()

    ColumnLayout {
        anchors.fill: parent

        HorizontalHeaderView {
            id: horizontalHeader
            Layout.fillWidth: true
            Layout.alignment: Qt.AlignLeft | Qt.AlignTop
            syncView: treeView
            clip: false
            delegate: control.headerDelegate
        }
        TreeView {
            id: treeView

            Layout.fillHeight: true
            Layout.fillWidth: true
            model: control.model
            delegate: control.delegate
            keyNavigationEnabled: control.keyNavigationEnabled
            columnWidthProvider: control.columnWidthProvider
            rowHeightProvider: control.rowHeightProvider
            alternatingRows: control.alternatingRows

            clip: true

            selectionBehavior: control.selectionBehavior
            selectionMode: control.selectionMode
            selectionModel: control.selectionModel

            ScrollBar.vertical: ScrollBar {
                focus: true
            }
        }
    }

    //headerPositioning: TreeView.OverlayHeader
    header: FocusScope {
        width: control.width; height: childrenRect.height
        x: childrenRect.x; y: childrenRect.y
        z: 3

        Pane {
            width: parent.width
            GridLayout {
                width: parent.width
                rows: control.showSearchField && control.showFilterControls && control.showGroupControls ? 2 : 1
                columns: 2

                TextField {
                    id: searchTextField
                    placeholderText: qsTr("Search")
                    visible: control.showSearchField
                    Layout.fillWidth: true
                    Layout.columnSpan: control.showSearchField && control.showFilterControls && control.showGroupControls ? 2 : 1
                    horizontalAlignment: Text.AlignLeft

                    onVisibleChanged: text = "";
                    Keys.onEnterPressed: searchTextChanged(text);
                    Keys.onReturnPressed: searchTextChanged(text);

                    IconImage {
                        anchors { top: parent.top; right: parent.right; margins: 5 }
                        id: clearText
                        fillMode: Image.PreserveAspectFit
                        smooth: true
                        visible: searchTextField.text
                        color: myPalette.windowText
                        source: "qrc:///icons/backspace.svg"
                        mirror: Qt.application.layoutDirection === Qt.RightToLeft

                        height: 16
                        width: 16

                        MouseArea {
                            id: clear
                            anchors.fill: parent
                            onClicked: {
                                searchTextField.text = "";
                                searchTextField.forceActiveFocus();
                                searchTextChanged(searchTextField.text);
                            }
                        }
                    }
                }

                RowLayout {
                    visible: control.showFilterControls
                    Repeater {
                        model: control.filterModel

                        ToolButton {
                            visible: model.visible
                            ToolTip.visible: hovered
                            ToolTip.delay: Qt.styleHints.mousePressAndHoldInterval
                            ToolTip.text: model.text
                            checkable: true
                            icon.source: model.icon
                            checked: model.checked // read initial value from the model
                            focusPolicy: Qt.NoFocus

                            onCheckedChanged: {
                                if (model.checked !== checked)
                                {
                                    filterModel.setProperty(index, "checked", checked) // update model
                                    filterChanged(index, checked)
                                }
                                checked = Qt.binding(function() { return model.checked }) // reestablish binding and continue auto-updating from the model
                            }
                        }
                    }
                }

                RowLayout {
                    visible: control.showGroupControls
                    Layout.alignment: Qt.AlignRight | Qt.AlignVCenter
                    Layout.fillWidth: true

                    ComboBox {
                        id: groupByComboBox
                        Layout.maximumWidth: 100
                        model: control.groupByModel
                        textRole: "key"
                        //: Arrange list by the selected criteria
                        //~ Context the '%1'-part will be replaced by terms, such as date, name, ...
                        displayText: qsTr("By %1", "Lookup control").arg(currentText)
                        currentIndex: control.groupByIndex
                        focusPolicy: Qt.NoFocus

                        onActivated: {
                            if (control.groupByIndex !== currentIndex)
                                control.groupByIndex = currentIndex; // TODO: avoid qt.qml.binding.removal
                            currentIndex = Qt.binding(function() { return control.groupByIndex }); // reestablish binding and continue auto-updating from the model
                        }
                    }

                    ToolButton {
                        id: groupBySortOrderToolButton
                        checkable: true
                        icon.source: checked ? "qrc:/icons/arrow_drop_down.svg" : "qrc:/icons/arrow_drop_up.svg"
                        checked: control.groupBySortOrder
                        focusPolicy: Qt.NoFocus

                        onCheckedChanged: {
                            if (control.groupBySortOrder !== checked)
                                control.groupBySortOrder = checked; // TODO: avoid qt.qml.binding.removal
                            checked = Qt.binding(function() { return control.groupBySortOrder }); // reestablish binding and continue auto-updating from the model
                        }
                    }
                }
            }
        }
    }


    //footerPositioning: TreeView.OverlayHeader
    footer: Pane {
        width: control.width;
        z: 3

        RowLayout {
            anchors.right: parent.right
            Label {
                verticalAlignment: Text.AlignVCenter
                //: Number of rows in a list
                //~ Context the '%1'-part will be replaced by the actual number
                text: qsTr("Count=%1", "Lookup control").arg(treeView.rows)
            }

            ToolButton {
                id: addButton
                Layout.alignment: Qt.AlignRight | Qt.AlignVCenter
                visible: control.showAddButton
                icon.source: "qrc:///icons/add.svg"
                onClicked: addItem()
            }

            ToolButton {
                id: toggleEditButton
                Layout.alignment: Qt.AlignRight | Qt.AlignVCenter
                visible: control.showEditButton
                highlighted: true
                checkable: true
                icon.source: checked ? "qrc:///icons/check.svg" : "qrc:///icons/edit.svg"
                checked: control.isEditing
                Binding { target: control; property: "isEditing"; value: toggleEditButton.checked }
            }

            ToolButton {
                id: moreMenuButton
                Layout.alignment: Qt.AlignRight | Qt.AlignVCenter
                visible: moreMenu
                ToolTip.visible: hovered
                ToolTip.delay: Qt.styleHints.mousePressAndHoldInterval
                ToolTip.text: qsTr("Settings", "Lookup control")
                icon.source: "qrc:///icons/more.svg"

                onClicked: moreMenu.popup()
            }
        }
    }
}


