/****************************************************************************
**
** Copyright (C) 2017 The Qt Company Ltd.
** Contact: http://www.qt.io/licensing/
**
** This file is part of the Qt Quick Controls 2 module of the Qt Toolkit.
**
** $QT_BEGIN_LICENSE:LGPL3$
** Commercial License Usage
** Licensees holding valid commercial Qt licenses may use this file in
** accordance with the commercial license agreement provided with the
** Software or, alternatively, in accordance with the terms contained in
** a written agreement between you and The Qt Company. For licensing terms
** and conditions see http://www.qt.io/terms-conditions. For further
** information use the contact form at http://www.qt.io/contact-us.
**
** GNU Lesser General Public License Usage
** Alternatively, this file may be used under the terms of the GNU Lesser
** General Public License version 3 as published by the Free Software
** Foundation and appearing in the file LICENSE.LGPLv3 included in the
** packaging of this file. Please review the following information to
** ensure the GNU Lesser General Public License version 3 requirements
** will be met: https://www.gnu.org/licenses/lgpl.html.
**
** GNU General Public License Usage
** Alternatively, this file may be used under the terms of the GNU
** General Public License version 2.0 or later as published by the Free
** Software Foundation and appearing in the file LICENSE.GPL included in
** the packaging of this file. Please review the following information to
** ensure the GNU General Public License version 2.0 requirements will be
** met: http://www.gnu.org/licenses/gpl-2.0.html.
**
** $QT_END_LICENSE$
**
****************************************************************************/

import QtQuick
import QtQuick.Window
import QtQuick.Templates as T
import QtQuick.Controls
import QtQuick.Controls.impl
import QtQuick.Controls.Fusion
import QtQuick.Controls.Fusion.impl
import net.theocbase 1.0
import "./"

T.ComboBox {
    id: control

    property bool showSearchField: true
    property bool showFilterControls: true
    property bool showGroupControls: true
    property int groupByIndex
    property bool sortOrder
    property bool showEditButton: true
    property bool isEditing: true
    property bool isButtonVisible: true
    property ListModel filterModel
    property ListModel groupByModel
    property Menu moreMenu

    property string sectionProperty: ""
    property int sectionCriteria: ViewSection.FullString
    property Component sectionDelegate

    property string error

    signal searchTextChanged(var text)
    signal filterChanged(int index, bool checked)
    signal cleared()

    implicitWidth: Math.max(implicitBackgroundWidth + leftInset + rightInset,
                            implicitContentWidth + leftPadding + rightPadding)
    implicitHeight: Math.max(implicitBackgroundHeight + topInset + bottomInset,
                             implicitContentHeight + topPadding + bottomPadding,
                             implicitIndicatorHeight + topPadding + bottomPadding)

    leftPadding: padding + (!control.mirrored || !indicator || !indicator.visible ? 0 : indicator.width + spacing)
    rightPadding: padding + (control.mirrored || !indicator || !indicator.visible ? 0 : indicator.width + spacing)

    // workaround to use the right color on Windows with dark theme
    SystemPalette { id: myPalette; colorGroup: SystemPalette.Active }

    ToolTip.visible: ToolTip.text !== "" && hovered
    ToolTip.text: error

    IconImage {
        id: clearText
        anchors { top: parent.top; right: parent.right; margins: 4; rightMargin: indicator.width }
        fillMode: Image.PreserveAspectFit
        smooth: true
        visible: control.enabled && (control.editable ? control.editText : control.displayText)
        color: myPalette.windowText
        source: "qrc:///icons/backspace.svg"
        mirror: Qt.application.layoutDirection === Qt.RightToLeft

        height: 16
        width: 16

        MouseArea {
            id: clear
            anchors.fill: parent
            onClicked: {
                control.currentIndex = -1;
                cleared();
            }
        }
    }

    delegate: MenuItem {
        width: ListView.view.width
        text: control.textRole ? (Array.isArray(control.model) ? modelData[control.textRole] : model[control.textRole]) : modelData
        font.weight: control.currentIndex === index ? Font.DemiBold : Font.Normal
        highlighted: control.highlightedIndex === index
        hoverEnabled: control.hoverEnabled
    }

    indicator: ColorImage {
        x: control.mirrored ? control.padding : control.width - width - control.padding
        y: control.topPadding + (control.availableHeight - height) / 2
        //color: control.editable ? control.palette.text : control.palette.buttonText
        color: control.editable ? myPalette.text : myPalette.buttonText
        source: "qrc:/qt-project.org/imports/QtQuick/Controls/Fusion/images/arrow.png"
        width: 20
        fillMode: Image.Pad
        visible: control.isButtonVisible
    }

    contentItem: T.TextField {
        topPadding: 4
        leftPadding: (!control.mirrored
                     ? (acceptableInput ? 4 : 24)
                     : ((control.enabled && (control.editable ? control.editText : control.displayText)) ? 24 : 4))
                     - control.padding
        rightPadding: (control.mirrored
                      ? (acceptableInput ? 4 : 24)
                      : ((control.enabled && (control.editable ? control.editText : control.displayText)) ? 24 : 4))
                      - control.padding
        bottomPadding: 4

        text: control.editable ? control.editText : control.displayText

        enabled: control.editable
        autoScroll: control.editable
        readOnly: control.down
        inputMethodHints: control.inputMethodHints
        validator: control.validator
        selectByMouse: control.selectTextByMouse

        font: control.font
        //color: control.editable ? control.palette.text : control.palette.buttonText
        color: control.editable ? myPalette.text : myPalette.buttonText
        //selectionColor: control.palette.highlight
        selectionColor: myPalette.highlight
        //selectedTextColor: control.palette.highlightedText
        selectedTextColor: myPalette.highlightedText
        verticalAlignment: Text.AlignVCenter
        horizontalAlignment: Text.AlignLeft

        background: PaddedRectangle {
            clip: true
            radius: 2
            padding: 1
            leftPadding: control.mirrored ? -2 : padding
            rightPadding: !control.mirrored ? -2 : padding
            color: control.palette.base
            visible: control.editable && !control.flat

            Rectangle {
                x: parent.width - width
                y: 1
                width: 1
                height: parent.height - 2
                color: Fusion.buttonOutline(control.palette, control.activeFocus, control.enabled)
            }

            Rectangle {
                x: 1
                y: 1
                width: parent.width - 3
                height: 1
                color: Fusion.topShadow
            }
        }

        AlertIndicator {
            id: errorIndicator
            x: 2
            y: 2
            visible: !parent.acceptableInput

            states:
                [
                State {
                    name: "Valid"; when: !visible
                    PropertyChanges { target: errorIndicator; scale: 1.2; }
                },
                State {
                    name: "Invalid"; when: visible
                    PropertyChanges { target: errorIndicator; scale: 1; }
                }
            ]

            transitions: Transition {
                NumberAnimation {
                    properties: "scale"
                    from: 1.2;
                    to: 1;
                    duration: 100
                    loops: 2
                }
            }
        }

        Rectangle {
            x: 1 - control.leftPadding
            y: 1
            width: control.width - 2
            height: control.height - 2
            color: "transparent"
            border.color: Color.transparent(Fusion.highlightedOutline(control.palette), 40 / 255)
            visible: control.activeFocus
            radius: 1.7
        }
    }

    background: ButtonPanel {
        implicitWidth: 120
        implicitHeight: 24

        control: control
        visible: !control.flat || control.down
        // ### TODO: fix control.contentItem.activeFocus
        highlighted: control.visualFocus || control.contentItem.activeFocus
    }

    popup: T.Popup {
        width: control.width
        height: Math.min(contentItem.implicitHeight + 2, control.Window.height - topMargin - bottomMargin)
        topMargin: 6
        bottomMargin: 6
        palette: control.palette
        padding: 1

        contentItem: LookupControl {
            id: lookupControl
            clip: true
            implicitHeight: contentHeight
            model: control.delegateModel
            currentIndex: control.highlightedIndex
            highlightRangeMode: ListView.ApplyRange
            highlightMoveDuration: 0

            showSearchField: control.showSearchField
            showFilterControls: control.showFilterControls
            showGroupControls: control.showGroupControls
            groupByIndex: control.groupByIndex
            sortOrder: control.sortOrder
            showEditButton: control.showEditButton
            isEditing: control.isEditing
            filterModel: control.filterModel
            groupByModel: control.groupByModel
            moreMenu: control.moreMenu

            section.property: control.sectionProperty
            section.criteria: control.sectionCriteria
            section.delegate: control.sectionDelegate

            onGroupByIndexChanged: {
                if (control.groupByIndex !== groupByIndex)
                    control.groupByIndex = groupByIndex // update property
                groupByIndex = Qt.binding(function() { return control.groupByIndex }) // reestablish binding and continue auto-updating from the model
            }
            onSortOrderChanged: {
                if (control.sortOrder !== sortOrder)
                    control.sortOrder = sortOrder; // TODO: avoid qt.qml.binding.removal
                sortOrder = Qt.binding(function() { return control.sortOrder }); // reestablish binding and continue auto-updating from the model
            }

            Binding { target: control; property: "isEditing"; value: lookupControl.isEditing }

            onSearchTextChanged: text => control.searchTextChanged(text)
            onFilterChanged: (index, checked) => control.filterChanged(index, checked)

            T.ScrollBar.vertical: ScrollBar { }
        }

        background: Rectangle {
            color: control.popup.palette.window
            border.color: Fusion.outline(control.palette)

            Rectangle {
                z: -1
                x: 1; y: 1
                width: parent.width
                height: parent.height
                color: control.palette.shadow
                opacity: 0.2
            }
        }
    }
}
