/**
 * This file is part of TheocBase.
 *
 * Copyright (C) 2011-2024, TheocBase Development Team, see AUTHORS.
 *
 * TheocBase is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * TheocBase is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with TheocBase.  If not, see <http://www.gnu.org/licenses/>.
 */
import QtCore
import QtQuick
import QtQuick.Controls
import QtQuick.Layouts
import QtQuick.Dialogs
import QtWebEngine
import net.theocbase 1.0
import "controls"

Page {
    id: webBrowser
    width: 480
    height: 560
    header: RowLayout {
        //anchors.fill: parent
        Layout.minimumHeight: 40
        Label {
            Layout.fillWidth: true
            text: webView.title ? webView.title : qsTr("Watchtower ONLINE LIBRARY")
            font: TBStyle.titleSmallFont
            padding: 10
        }
        BusyIndicator {
            running: loading
            visible: running
            Layout.alignment: Qt.AlignVCenter
        }
    }

    property date date
    property int meetingType
    property bool loading

    function importHtml(url, html) {
        buttonImport.enabled = false;
        root.showBusy = true;
        var result = "";
        if (meetingType === MeetingType.MidweekMeeting) {
            result = mwbImport.importHtml(url, html);
        } else {
            result = wtImport.importHtml(url, html);
        }
        console.log(result);
        root.showBusy = false
        messagebox.text = result;
        messagebox.open();
        buttonImport.enabled = true;
    }

    function saveLastWtLocale(classList) {
        var regExp = /ml-(?<wtlocale>\w+)/g, match;
        var wtlocale;
        while (match = regExp.exec(classList))
        {
            wtlocale = match[1];
        }
        settings.lastWtLocale = wtlocale;
    }

    function resetDefaultSettings() {
        settings.url = "https://wol.jw.org/wol/finder?srctype=wol";
        settings.lastLanguage = "";
        settings.lastWtLocale = "";
        settings.sync();
        setUrl(settings.url);
    }

    function setUrl(baseUrl) {
        moreMenu.currentUrl = baseUrl;
        if (!isNaN(date)) {
            var urlstr = baseUrl;
            // try to set document id of the given date
            var documentId = meetingType === MeetingType.WeekendMeeting
                    ? wtImport.getDocumentId(date)
                    : mwbImport.getDocumentId(date);
            if (documentId > 0) {
                urlstr += "&docid=" + documentId;
            } else {
                // set 'meetings' and 'date' to jump to the week and let the user go to schedule
                urlstr += "&alias=meetings&date=" + date.toISOString().substring(0, 10);
            }
            // set locale (may be ignored)
            let lang = Qt.locale().name;
            // set wt locale
            var wtLocale = settings.lastWtLocale;
            if (wtLocale === "")
                wtLocale = mwbImport.getWtLocale(lang);
            if (wtLocale != "") {
                urlstr += "&wtlocale=" + wtLocale;
            }
            if (settings.lastLanguage != "") {
                urlstr += "&locale=" + settings.lastLanguage;
            } else {
                let match = lang.match("_|-");
                if (match.length > 0) {
                    lang = lang.substring(0, lang.indexOf(match[0]));
                }
                urlstr += "&locale=" + lang;
            }
            console.log("URL: "+ urlstr);
            webView.url = urlstr;
        }
    }

    Component.onCompleted: {
        setUrl(settings.url);
    }

    Component.onDestruction: {
        settings.url = moreMenu.currentUrl;
        settings.sync();
    }

    MessageDialog {
        id: messagebox
        //icon: StandardIcon.Information
        onButtonClicked: {
            root.setDate(date)
        }
    }

    Menu {
        id: moreMenu
        property string currentUrl

        onCurrentUrlChanged: {
            setUrl(currentUrl);
        }

        MenuItem {
            text: "Official Website (jw.org)"
            checkable: true
            checked: (moreMenu.currentUrl?.search("www.jw.org") ?? -1) > 0
            onTriggered: {
                moreMenu.currentUrl = "https://www.jw.org/finder?srctype=wol"
                // restore binding, since the checked property was set to a
                // static value (true or false) when the CheckBox was clicked
                checked = Qt.binding(function() { return (moreMenu.currentUrl?.search("www.jw.org") ?? -1) > 0 });
            }
        }
        MenuItem {
            text: "Online Library (wol.jw.org)"
            checkable: true
            checked: (moreMenu.currentUrl?.search("wol.jw.org") ?? -1) > 0
            onTriggered: {
                moreMenu.currentUrl = "https://wol.jw.org/wol/finder?srctype=wol"
                // restore binding, since the checked property was set to a
                // static value (true or false) when the CheckBox was clicked
                checked = Qt.binding(function() { return (moreMenu.currentUrl?.search("wol.jw.org") ?? -1) > 0 });
            }
        }
        MenuItem { text: qsTr("Reset default settings", "Webbrowser settings"); onTriggered: resetDefaultSettings() }
    }

    Settings {
        id: settings
        category: "WebBrowser"
        property string lastLanguage: ""
        property string lastWtLocale: ""
        property string url: "https://wol.jw.org/wol/finder?srctype=wol"
    }
    WorkbookImport { id: mwbImport }
    WTImport { id: wtImport }

    WebEngineView {
        id: webView
        anchors.fill: parent
        url: popup.url
        onUrlChanged: webBrowser.loading = true

        onLoadingChanged: function(loadRequest) {
            if (loadRequest.status === WebEngineView.LoadFailedStatus
                    || loadRequest.status === WebEngineView.LoadStoppedStatus) {
                webBrowser.loading = false;
            } else if (loadRequest.status === WebEngineView.LoadSucceededStatus) {
                webBrowser.loading = false;
                console.log("Page loaded: " + loadRequest.url + " " + webView.loading);
                // save last language
                let urlparts = loadRequest.url.toString().split("/");
                urlparts.forEach(function(item, i) {
                    if (item === "wol.jw.org") {
                        settings.lastLanguage = urlparts[i + 1];
                        settings.sync();
                    }
                })
                // save last wt locale
                var js = "document.getElementsByTagName(\"article\")[0].classList.value";
                webView.runJavaScript(js, function(result) {
                    saveLastWtLocale(result);
                });
            }
        }
    }

    footer: ColumnLayout {
        RowLayout {
            Layout.alignment: Qt.AlignRight
            Layout.margins: 10
            ToolButton {
                id: buttonImport
                icon.source: "qrc:/icons/import_contacts.svg"
                visible: canEditMidweekMeetingSchedule
                ToolTip.text: qsTr("Import midweek meeting schedule from the browser cache")
                ToolTip.visible: hovered

                onClicked: {
                    var js = "document.getElementsByTagName(\"article\")[0].outerHTML";
                    webView.runJavaScript(js, function(result) {
                        let html = "<html><body>" + result + "</body></html>";
                        importHtml(webView.url, html);
                    });
                }
            }
            ToolButton {
                id: moreMenuButton
                Layout.alignment: Qt.AlignRight | Qt.AlignVCenter
                visible: moreMenu
                ToolTip.visible: hovered
                ToolTip.delay: Qt.styleHints.mousePressAndHoldInterval
                ToolTip.text: qsTr("Settings")
                icon.source: "qrc:///icons/more.svg"

                onClicked: moreMenu.popup()
            }
        }
    }
}
