import QtCore
import QtQuick
import QtQuick.Window
import QtQuick.Controls
import QtQuick.Layouts
import net.theocbase 1.0
import "controls"
import "sidePanelScripts.js" as SPScripts

Page {
    id: streetPage

    // PROPERTY DECLARATIONS
    property var street
    property bool isValidationTextVisible: false
    property bool isStreetNameLoading: true
    property bool isStreetTypeLoading: true
    property bool isHouseNumberLoading: true

    signal pageCanceled()

    // OBJECT PROPERTIES
    header: RowLayout {
        ToolButton {
            Layout.alignment: Qt.AlignRight
            icon.source: "qrc:/icons/chevron_left.svg"
            onClicked: {
                streetModel.saveStreets();
                streetsEditStack.pop();
            }
        }
        Label {
            Layout.fillWidth: true
            text: qsTr("Edit territory street")
            font: TBStyle.titleSmallFont
        }
        ToolButton {
            Layout.alignment: Qt.AlignRight
            icon.source: "qrc:/icons/close.svg"
            onClicked: {
                streetsEditStack.pop();
                pageCanceled();
            }
        }
    }

    Component.onCompleted: {
        streetTypeListProxyModel.sort(0, Qt.AscendingOrder);
    }

    onStreetChanged: {
        if (street) {
            streetTypeComboBox.currentIndex = streetTypeComboBox.indexOfValue(street.streetTypeId)
        } else {
            streetTypeComboBox.currentIndex = -1;
        }
    }

    // CHILD OBJECTS
    TerritoryStreetValidator {
        id: streetNameValidator
        model: streetModel
        streetId: street ? street.streetId : 0
        role: TerritoryStreetModel.StreetNameRole
        onErrorChanged: function(error) { streetNameErrorLabel.text = error }
        onStreetIdChanged: {
            isValidationTextVisible = false;
            isStreetNameLoading = true;
            isStreetNameLoading = false;
        }
    }
    TerritoryStreetValidator {
        id: streetTypeValidator
        model: streetModel
        streetId: street ? street.streetId : 0
        role: TerritoryStreetModel.StreetTypeIdRole
        onErrorChanged: function(error) { streetTypeErrorLabel.text = error }
        onStreetIdChanged: {
            isValidationTextVisible = false;
            isStreetTypeLoading = true;
            isStreetTypeLoading = false;
        }
    }
    TerritoryStreetValidator {
        id: fromNumberValidator
        model: streetModel
        streetId: street ? street.streetId : 0
        role: TerritoryStreetModel.FromNumberRole
        onErrorChanged: function(error) { fromNumberErrorLabel.text = error }
        onStreetIdChanged: {
            isValidationTextVisible = false;
            isHouseNumberLoading = true;
            isHouseNumberLoading = false;
        }
    }
    TerritoryStreetValidator {
        id: toNumberValidator
        model: streetModel
        streetId: street ? street.streetId : 0
        role: TerritoryStreetModel.ToNumberRole
        onErrorChanged: function(error) { toNumberErrorLabel.text = error }
        onStreetIdChanged: {
            isValidationTextVisible = false;
            isHouseNumberLoading = true;
            isHouseNumberLoading = false;
        }
    }

    DataObjectListSFProxyModel {
        id: streetTypeListProxyModel
        source: streetTypeListModel
    }

    Pane {
        anchors.fill: parent
        padding: 10

        Column {
            id: delegateContentLayout2
            anchors.fill: parent
            spacing: 5

            RowLayout {
                width: parent.width
                InterimIcon {
                    Layout.alignment: Qt.AlignVCenter | Qt.AlignLeft
                    Layout.margins: 4
                    source: "qrc:/icons/road.svg"
                    ToolTip.text: qsTr("Street name")
                    ToolTip.visible: hovered
                }
                ValidationTextField {
                    id: streetNameTextField
                    Layout.fillWidth: true
                    Layout.alignment: Qt.AlignTop
                    Layout.margins: 0
                    horizontalAlignment: Text.AlignLeft
                    verticalAlignment: Text.AlignVCenter

                    text: street ? street.streetName : ""
                    wrapMode: TextInput.WordWrap
                    validator: isStreetNameLoading ? null : streetNameValidator

                    onTextEdited: {
                        var modelIndex = streetModel.getStreetIndex(street.streetId);
                        streetModel.setData(modelIndex, text, TerritoryStreetModel.StreetNameRole);
                        // trigger revalidation by switching isLoaded value
                        isStreetNameLoading = true;
                        isStreetNameLoading = false;
                    }
                }
            }
            Label {
                id: streetNameErrorLabel
                font: TBStyle.bodySmallFont
                color: TBStyle.alertColor
                visible: text
            }
            RowLayout {
                width: parent.width
                InterimIcon {
                    Layout.alignment: Qt.AlignVCenter | Qt.AlignLeft
                    Layout.margins: 4
                    source: "qrc:/icons/home.svg"
                    ToolTip.text: qsTr("From number", "Number range of addresses in the street")
                    ToolTip.visible: hovered
                }
                ValidationTextField {
                    id: fromNumberTextField
                    Layout.alignment: Qt.AlignTop
                    Layout.fillWidth: true
                    Layout.margins: 0
                    horizontalAlignment: Text.AlignLeft
                    verticalAlignment: Text.AlignVCenter

                    text: street ? street.fromNumber : ""
                    wrapMode: TextInput.WordWrap
                    validator: isHouseNumberLoading ? null : fromNumberValidator

                    onTextEdited: {
                        var modelIndex = streetModel.getStreetIndex(street.streetId);
                        streetModel.setData(modelIndex, text, TerritoryStreetModel.FromNumberRole);
                        // trigger revalidation by switching isLoaded value
                        isHouseNumberLoading = true;
                        isHouseNumberLoading = false;
                    }
                }
                Label {
                    Layout.fillWidth: true
                    horizontalAlignment: Text.AlignHCenter
                    text: " - "
                }
                ValidationTextField {
                    id: toNumberTextField
                    Layout.alignment: Qt.AlignTop
                    Layout.fillWidth: true
                    Layout.margins: 0
                    horizontalAlignment: Text.AlignLeft
                    verticalAlignment: Text.AlignVCenter

                    text: street ? street.toNumber : ""
                    wrapMode: TextInput.WordWrap
                    validator: isHouseNumberLoading ? null : toNumberValidator

                    onTextEdited: {
                        var modelIndex = streetModel.getStreetIndex(street.streetId);
                        streetModel.setData(modelIndex, text, TerritoryStreetModel.ToNumberRole);
                        // trigger revalidation by switching isLoaded value
                        isHouseNumberLoading = true;
                        isHouseNumberLoading = false;
                    }
                }
            }
            Label {
                id: fromNumberErrorLabel
                font: TBStyle.bodySmallFont
                color: TBStyle.alertColor
                visible: text
            }
            Label {
                id: toNumberErrorLabel
                font: TBStyle.bodySmallFont
                color: TBStyle.alertColor
                visible: text
            }
            RowLayout {
                width: parent.width
                InterimIcon {
                    Layout.alignment: Qt.AlignVCenter | Qt.AlignLeft
                    Layout.margins: 4
                    source: "qrc:/icons/category.svg"
                    ToolTip.text: qsTr("Type", "Type of street")
                    ToolTip.visible: hovered
                }
                ComboBox {
                    id: streetTypeComboBox
                    Layout.fillWidth: true
                    Layout.alignment: Qt.AlignVCenter
                    Layout.margins: 0

                    model: streetTypeListProxyModel

                    textRole: "name"
                    valueRole: "id"

                    validator: isStreetTypeLoading ? null : streetTypeValidator

                    // When an item is selected, update the backend.
                    onActivated: {
                        var modelIndex = streetModel.getStreetIndex(street.streetId);
                        streetModel.setData(modelIndex, currentValue, TerritoryStreetModel.StreetTypeIdRole);
                        // trigger revalidation by switching isLoaded value
                        isStreetTypeLoading = true;
                        isStreetTypeLoading = false;
                    }
                    // Set the initial currentIndex to the value stored in the backend.
                    Component.onCompleted: {
                        if (street) {
                            currentIndex = indexOfValue(street.streetTypeId);
                        }
                    }
                }
            }
            Label {
                id: streetTypeErrorLabel
                font: TBStyle.bodySmallFont
                color: TBStyle.alertColor
                visible: text
            }
            RowLayout {
                width: parent.width
                InterimIcon {
                    Layout.alignment: Qt.AlignVCenter | Qt.AlignLeft
                    Layout.margins: 4
                    source: "qrc:/icons/location_all.svg"
                    ToolTip.text: qsTr("Quantity", "Quantity of addresses in the street")
                    ToolTip.visible: hovered
                }
                Row {
                    Layout.fillWidth: true
                    Layout.alignment: Qt.AlignVCenter
                    Layout.margins: 0
                    SpinBox {
                        id: quantitySpinBox

                        implicitWidth: 80
                        from: 0
                        value: street ? street.quantity : 0

                        onValueModified: {
                            var modelIndex = streetModel.getStreetIndex(street.streetId);
                            streetModel.setData(modelIndex, value, TerritoryStreetModel.QuantityRole);
                        }
                    }
                }
            }
        }
    }
}
