/**
* This file is part of TheocBase.
*
* Copyright (C) 2011-2016, TheocBase Development Team, see AUTHORS.
*
* TheocBase is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* TheocBase is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with TheocBase.  If not, see <http://www.gnu.org/licenses/>.
*/
import QtQuick
import QtQuick.Window
import QtQuick.Controls
import QtQml.Models
import QtQuick.Layouts
import QtPositioning
import QtLocation
import net.theocbase 1.0
import "controls"

Item {
    id: item1

    property alias baseMap: _baseMap
    property alias vectorLayer: vectorOverlay
    property alias addressLayer: addressItemView
    property alias streetLayer: streetItemView
    property alias boundaryLayer: boundaryItemView
    property alias addressTextField: addressTextField
    property alias importBoundariesButton: importBoundariesButton
    property alias createBoundaryButton: createBoundaryButton
    property alias splitTerritoryButton: splitTerritoryButton
    property alias removeBoundaryButton: removeBoundaryButton
    property alias cutoffStreetSectionButton: cutoffStreetSectionButton
    property alias zoomFullButton: zoomFullButton
    property alias zoomSlider: zoomSlider
    property alias showBoundariesButton: showBoundariesButton
    property alias showMarkersButton: showMarkersButton
    property alias showStreetsButton: showStreetsButton
    property alias editModeButton: editModeButton
    property alias hoverHandler: _hoverHandler
    property alias tapHandler: _tapHandler
    property alias wheelHandler: _wheelHandler
    property alias dragHandler: _dragHandler

    Map {
        id: _baseMap

        property variant pluginParameters

        anchors.fill: parent
        zoomLevel: 15

        WheelHandler {
            id: _wheelHandler
        }
        DragHandler {
            id: _dragHandler
            signal flickStarted // for autotests only
            signal flickEnded
            target: null
        }

        HoverHandler {
            id: _hoverHandler
        }

        TapHandler {
            id: _tapHandler
            acceptedButtons: Qt.LeftButton | Qt.RightButton
            grabPermissions: PointerHandler.CanTakeOverFromAnything
        }
    }

    Map {
        id: streetOverlay

        anchors.fill: parent
        plugin: Plugin { name: "itemsoverlay" }
        //gesture.enabled: false
        center: baseMap.center
        color: 'transparent' // Necessary to make this map transparent
        minimumFieldOfView: baseMap.minimumFieldOfView
        maximumFieldOfView: baseMap.maximumFieldOfView
        minimumTilt: baseMap.minimumTilt
        maximumTilt: baseMap.maximumTilt
        minimumZoomLevel: baseMap.minimumZoomLevel
        maximumZoomLevel: baseMap.maximumZoomLevel
        zoomLevel: baseMap.zoomLevel
        tilt: baseMap.tilt;
        bearing: baseMap.bearing
        fieldOfView: baseMap.fieldOfView
        z: _baseMap.z + 1

        // The code below enables SSAA
        layer.enabled: true
        layer.smooth: true
        property int w : vectorOverlay.width
        property int h : vectorOverlay.height
        property int pr: Screen.devicePixelRatio
        layer.textureSize: Qt.size(w  * 2 * pr, h * 2 * pr)

        MapItemView {
            id: streetItemView

            property int streetDisplayOption: 2

            model: streetModel
            delegate: territoryStreetDelegate
        }
    }

    Map {
        id: addressOverlay

        anchors.fill: parent
        plugin: Plugin { name: "itemsoverlay" }
        //gesture.enabled: false
        center: baseMap.center
        color: 'transparent' // Necessary to make this map transparent
        minimumFieldOfView: baseMap.minimumFieldOfView
        maximumFieldOfView: baseMap.maximumFieldOfView
        minimumTilt: baseMap.minimumTilt
        maximumTilt: baseMap.maximumTilt
        minimumZoomLevel: baseMap.minimumZoomLevel
        maximumZoomLevel: baseMap.maximumZoomLevel
        zoomLevel: baseMap.zoomLevel
        tilt: baseMap.tilt;
        bearing: baseMap.bearing
        fieldOfView: baseMap.fieldOfView
        z: streetOverlay.z + 1

        // The code below enables SSAA
        layer.enabled: true
        layer.smooth: true
        property int w : vectorOverlay.width
        property int h : vectorOverlay.height
        property int pr: Screen.devicePixelRatio
        layer.textureSize: Qt.size(w  * 2 * pr, h * 2 * pr)

        MapItemView {
            id: addressItemView

            property int markerDisplayOption: 2

            model: addressModel
            delegate: territoryMarkerDelegate
        }
    }

    Map {
        id: boundaryOverlay

        anchors.fill: parent
        plugin: Plugin { name: "itemsoverlay" }
        //gesture.enabled: false
        center: baseMap.center
        color: 'transparent' // Necessary to make this map transparent
        minimumFieldOfView: baseMap.minimumFieldOfView
        maximumFieldOfView: baseMap.maximumFieldOfView
        minimumTilt: baseMap.minimumTilt
        maximumTilt: baseMap.maximumTilt
        minimumZoomLevel: baseMap.minimumZoomLevel
        maximumZoomLevel: baseMap.maximumZoomLevel
        zoomLevel: baseMap.zoomLevel
        tilt: baseMap.tilt;
        bearing: baseMap.bearing
        fieldOfView: baseMap.fieldOfView
        z: addressOverlay.z + 1

        // The code below enables SSAA
        layer.enabled: true
        layer.smooth: true
        property int w : vectorOverlay.width
        property int h : vectorOverlay.height
        property int pr: Screen.devicePixelRatio
        layer.textureSize: Qt.size(w  * 2 * pr, h * 2 * pr)

        MapItemView {
            id: boundaryItemView

            property int boundaryDisplayOption: 2

            model: territoryTreeModel
            delegate: territoryBoundaryDelegate
        }
    }

    Map {
        id: vectorOverlay

        // This property indicates the mode
        //   0 = view
        //   1 = edit address
        //   2 = edit boundary
        //   3 = edit street
        property int editMode: 0
        property bool isDigitizing: false
        property TerritoryMapPolygon polygon
        property TerritoryMapMarker marker
        property TerritoryMapMarker addressSearchMarker
        property var addressSearchMarkers: []
        property var boundaries: []
        property double markerScale: 1.0

        // This property indicates the mode
        //   0 = polyline
        //   1 = polygon
        property int digitizeMode: 1
        property bool isLineComplete: false
        property SimpleMarker snapMarker
        property SimplePolygon newPolygon
        property SimplePolyline newPolyline
        property SimplePolyline rubberBand

        anchors.fill: parent
        plugin: Plugin { name: "itemsoverlay" }
        //gesture.enabled: false
        center: baseMap.center
        color: 'transparent' // Necessary to make this map transparent
        minimumFieldOfView: baseMap.minimumFieldOfView
        maximumFieldOfView: baseMap.maximumFieldOfView
        minimumTilt: baseMap.minimumTilt
        maximumTilt: baseMap.maximumTilt
        minimumZoomLevel: baseMap.minimumZoomLevel
        maximumZoomLevel: baseMap.maximumZoomLevel
        zoomLevel: baseMap.zoomLevel
        tilt: baseMap.tilt;
        bearing: baseMap.bearing
        fieldOfView: baseMap.fieldOfView
        z: boundaryOverlay.z + 1

        // The code below enables SSAA
        layer.enabled: true
        layer.smooth: true
        property int w : vectorOverlay.width
        property int h : vectorOverlay.height
        property int pr: Screen.devicePixelRatio
        layer.textureSize: Qt.size(w  * 2 * pr, h * 2 * pr)
    }

    Column {
        topPadding: 30
        bottomPadding: 30
        leftPadding: Qt.application.layoutDirection === Qt.LeftToRight ? 10 : 0
        rightPadding: Qt.application.layoutDirection === Qt.LeftToRight ? 0 : 10
        height: parent.height
        anchors.left: parent.left
        z: vectorLayer.z + 1

        Slider {
            id: zoomSlider
            orientation: Qt.Vertical
            height: parent.height - parent.topPadding - parent.bottomPadding
            from: baseMap.minimumZoomLevel
            to: baseMap.maximumZoomLevel
            value: baseMap.zoomLevel
        }
    }
    Row {
        topPadding: 10
        leftPadding: Qt.application.layoutDirection === Qt.LeftToRight ? 90 : 30
        rightPadding: Qt.application.layoutDirection === Qt.LeftToRight ? 30 : 90
        width: parent.width
        z: vectorLayer.z + 1

        TextField {
            id: addressTextField
            width: 400
            placeholderText: qsTr("search", "Territory map")
            selectByMouse: true
        }
    }
    Column {
        topPadding: 30
        bottomPadding: 30
        leftPadding: Qt.application.layoutDirection === Qt.LeftToRight ? 0 : 10
        rightPadding: Qt.application.layoutDirection === Qt.LeftToRight ? 10 : 0
        height: parent.height
        spacing: 5
        anchors.right: parent.right
        z: vectorLayer.z + 1

        RoundButton {
            id: importBoundariesButton
            ToolTip.text: qsTr("Import data",
                               "Territory map")
            ToolTip.visible: hovered
            icon.source: "qrc:///icons/input.svg"
            icon.width: 20
            icon.height: 20
            opacity: parent.enabled ? 1.0 : 0.5
            visible: canEditTerritories
        }

        RoundButton {
            id: zoomFullButton
            ToolTip.text: qsTr("Zoom full",
                               "Territory map")
            ToolTip.visible: hovered
            icon.source: "qrc:///icons/fullscreen.svg"
            icon.width: 20
            icon.height: 20
            opacity: parent.enabled ? 1.0 : 0.5
        }

        RoundButton {
            id: showBoundariesButton
            ToolTip.text: qsTr("Show/hide territories",
                               "Territory map")
            ToolTip.visible: hovered
            checkable: false
            icon.source: "qrc:///icons/boundary_all.svg"
            icon.width: 20
            icon.height: 20
            opacity: parent.enabled ? 1.0 : 0.5
        }

        RoundButton {
            id: showMarkersButton
            ToolTip.text: qsTr("Show/hide markers",
                               "Territory map")
            ToolTip.visible: hovered
            icon.source: "qrc:///icons/location_all.svg"
            icon.width: 20
            icon.height: 20
            opacity: parent.enabled ? 1.0 : 0.5
            checkable: false
        }

        RoundButton {
            id: showStreetsButton
            ToolTip.text: qsTr("Show/hide streets",
                               "Territory map")
            ToolTip.visible: hovered
            icon.source: "qrc:///icons/road_all.svg"
            icon.width: 20
            icon.height: 20
            opacity: parent.enabled ? 1.0 : 0.5
            checkable: false
        }

        RoundButton {
            id: editModeButton
            ToolTip.text: qsTr("Switch edit mode",
                               "Territory map")
            ToolTip.visible: hovered
            icon.source: "qrc:///icons/visibility.svg"
            icon.width: 20
            icon.height: 20
            opacity: parent.enabled ? 1.0 : 0.5
            checkable: false
            visible: canEditTerritories
        }

        RoundButton {
            id: cutoffStreetSectionButton
            visible: vectorLayer.editMode === 2
            enabled: currentStreet && isGdalAvailable
            ToolTip.text: qsTr("Cut off a street section", "Territory map")
            ToolTip.visible: hovered
            icon.source: "qrc:///icons/road_cut.svg"
            icon.width: 20
            icon.height: 20
            opacity: parent.enabled ? 1.0 : 0.5
        }

        RoundButton {
            id: createBoundaryButton
            visible: vectorLayer.editMode === 3
            enabled: currentTerritory && isGdalAvailable
            ToolTip.text: qsTr("Create boundary",
                               "Territory map")
            ToolTip.visible: hovered
            icon.source: "qrc:///icons/add_boundary.svg"
            icon.width: 20
            icon.height: 20
            opacity: parent.enabled ? 1.0 : 0.5
        }

        RoundButton {
            id: removeBoundaryButton
            visible: canEditTerritories && vectorLayer.editMode === 3
            enabled: currentTerritory
            ToolTip.text: qsTr("Remove boundary",
                               "Territory map")
            ToolTip.visible: hovered
            icon.source: "qrc:///icons/remove_boundary.svg"
            icon.width: 20
            icon.height: 20
            opacity: parent.enabled ? 1.0 : 0.5
        }

        RoundButton {
            id: splitTerritoryButton
            visible: vectorLayer.editMode === 3
            enabled: currentTerritory && isGdalAvailable
            ToolTip.text: qsTr("Split territory", "Territory map")
            ToolTip.visible: hovered
            icon.source: "qrc:///icons/boundary_split.svg"
            icon.width: 20
            icon.height: 20
            opacity: parent.enabled ? 1.0 : 0.5
        }
    }
}
