import QtCore
import QtQuick
import QtQuick.Controls
import QtQuick.Dialogs
import QtQuick.Window
import QtPositioning
import QtLocation
import net.theocbase 1.0
import "controls"

TerritoryMapForm {
    id: territoryMapForm

    // PROPERTY DECLARATIONS
    property variant territoryBoundaries
    property variant territoryAddresses
    property int geocodeRetryCount: 0
    property string geocodeQuery: ""

    // SIGNAL DECLARATIONS
    signal refreshTerritoryMapPolygon(var showAllBoundaries)

    // JAVASCRIPT FUNCTIONS
    function importBoundaries() {
        importDialog.x = mainWindow.x + mainWindow.width / 2 - width / 2;
        importDialog.y = mainWindow.y + mainWindow.height / 2 - height / 2;
        importDialog.show();
    }

    function createBoundary() {
        vectorLayer.digitizeMode = 1;
        vectorLayer.isLineComplete = false;
        vectorLayer.isDigitizing = true;
    }

    function splitTerritory() {
        vectorLayer.digitizeMode = 0;
        vectorLayer.isLineComplete = false;
        vectorLayer.isDigitizing = true;
    }

    function cutoffStreetSection() {
        vectorLayer.digitizeMode = 0;
        vectorLayer.isLineComplete = false;
        vectorLayer.isDigitizing = true;
    }

    function setBoundary(overlappedTerritoryIds, annexOverlappingAreas) {
        if (territories.setTerritoryBoundary(currentTerritory.territoryId, vectorLayer.newPolygon.path, true, overlappedTerritoryIds, annexOverlappingAreas)) {
            console.log("Territory boundaries changed")
        }
        // clean up map
        vectorLayer.removeMapItem(vectorLayer.newPolygon);
        vectorLayer.newPolygon = null;
    }

    function removeBoundary() {
        if (currentTerritory) {
            currentTerritory.wktGeometry = "";
            // TODO: Check if refreshing the map is possible without reloading all territories
            territoryManager.reloadTerritories();
        }
    }

    function zoomSelectedTerritory() {
        if (currentTerritory)
            baseMap.visibleRegion = currentTerritory.boundingGeoRectangle;
    }

    function zoomSelectedStreet() {
        if (currentStreet) {
            baseMap.visibleRegion = currentStreet.boundingGeoRectangle();
        }
    }

    function zoomSelectedAddress() {
        if (currentAddress) {
            var regExp = /((?:[-+]?\.\d+|[-+]?\d+(?:\.\d*)?))\s*((?:[-+]?\.\d+|[-+]?\d+(?:\.\d*)?))/g, match;
            var lat;
            var lon;
            while (match = regExp.exec(currentAddress.wktGeometry))
            {
                lat = match[2];
                lon = match[1];
            }
            zoomCoordinate(lat, lon);
        }
    }

    function zoomCoordinate(lat, lon) {
        baseMap.center = QtPositioning.coordinate(lat, lon);
        if (baseMap.zoomLevel < 17)
            baseMap.zoomLevel = 17;
    }

    function zoomFull() {
        if (addressModel.count === 0 && !territoryTreeModel.boundingGeoRectangle.isValid) {
            // zoom to congregation's address
            var congrAddress = territories.getCongregationAddress();
            if (congrAddress !== "")
                geocodeAddress(congrAddress);
        } else {
            baseMap.visibleRegion = territoryTreeModel.boundingGeoRectangle;
        }
    }

    function getRoot() {
        var par = parent;
        while ( par.parent !== null ) {
            par = par.parent;
        }
        return par;
    }

    function clearAddressSearchMarkers() {
        // remove previous markers
        var count = vectorLayer.addressSearchMarkers.length;
        for (var i = 0; i < count; i++) {
            vectorLayer.removeMapItem(vectorLayer.addressSearchMarkers[i]);
            vectorLayer.addressSearchMarkers[i].destroy();
        }
        vectorLayer.addressSearchMarkers = [];
    }

    function geocodeAddress(query) {
        // send the geocode request
        geocodeRetryCount = 0;
        geocodeQuery = query;
        territories.onGeocodingFinished.connect(geocodingFinished);
        territories.onGeocodingError.connect(geocodingFailed);
        territories.geocodeAddress(geocodeQuery);
    }

    function geocodingFinished(geocodeResults) {
        if (geocodeResults.length < 1 && geocodeRetryCount < 3 && geocodeQuery !== "") {
            // try again
            geocodeRetryCount++;
            territories.geocodeAddress(geocodeQuery);
            return;
        }

        territories.onGeocodingFinished.disconnect(geocodingFinished);
        territories.onGeocodingError.disconnect(geocodingFailed);

        displayGeocodeResults(geocodeResults);
    }

    function geoLocationToLocation(geoLocation) {
        var location = Object.create(geoLocationPrototype);
        location.location = geoLocation;
        return location;
    }

    function displayGeocodeResults(geocodeResults) {
        clearAddressSearchMarkers();

        var count = geocodeResults.length;
        if (count > 0) {
            var markerScale = addressModel.getMarkerScale();
            var minLat = Number.POSITIVE_INFINITY;
            var minLon = Number.POSITIVE_INFINITY;
            var maxLat = Number.NEGATIVE_INFINITY;
            var maxLon = Number.NEGATIVE_INFINITY;

            for (var j = 0; j < geocodeResults.length; j++) {
                var territoryId = 0;
                var addressTypeNumber = 0;
                var addressTypeIndex = 0;
                var color = "#000000";

                var location = geoLocationToLocation(geocodeResults[j]);
                var lat = location.coordinate.latitude;
                var lon = location.coordinate.longitude;
                minLat = Math.min(lat, minLat);
                minLon = Math.min(lon, minLon);
                maxLat = Math.max(lat, maxLat);
                maxLon = Math.max(lon, maxLon);

                var component = Qt.createComponent('TerritoryMapMarker.qml');
                if (component.status === Component.Ready) {
                    vectorLayer.addressSearchMarker = component.createObject(vectorLayer);
                    vectorLayer.addressSearchMarker.territoryId = 0;
                    vectorLayer.addressSearchMarker.isTerritorySelected = false;
                    vectorLayer.addressSearchMarker.addressId = 0;
                    vectorLayer.addressSearchMarker.coordinate = location.coordinate;
                    vectorLayer.addressSearchMarker.color = "#000000";
                    vectorLayer.addressSearchMarker.markerScale = markerScale < 1 ? 1 : markerScale;
                    vectorLayer.addressSearchMarker.location.location = geocodeResults[j];

                    vectorLayer.addressSearchMarkers.push(vectorLayer.addressSearchMarker);
                    vectorLayer.addMapItem(vectorLayer.addressSearchMarker);
                }
            }

            baseMap.visibleRegion = QtPositioning.rectangle(QtPositioning.coordinate(maxLat, minLon), QtPositioning.coordinate(minLat, maxLon));
            if (count > 1)
                baseMap.zoomLevel = baseMap.zoomLevel - 1;
        }
    }

    function geocodingFailed(errorMessage) {
        territories.onGeocodingFinished.disconnect(geocodingFinished);
        territories.onGeocodingError.disconnect(geocodingFailed);
        message.title = qsTr("Search address", "Add or edit territory address");
        message.text = errorMessage;
        message.visible = true;
        return;
    }

    function addAddress(location) {
        var newAddressId = addressModel.addAddress(currentTerritory.territoryId, location.address.country,
                                    location.address.state, location.address.county, location.address.city, location.address.district,
                                    location.address.street, location.address.streetNumber, location.address.postalCode,
                                    "POINT(" + location.coordinate.longitude + " " + location.coordinate.latitude + ")");
        if (newAddressId > 0) {
            clearAddressSearchMarkers();
            currentAddress = addressModel.findAddress(newAddressId);
        }
    }

    function reassignAddress(addressId) {
        if (addressId > 0 && currentTerritory !== null) {
            addressModel.updateAddress(addressId, currentTerritory.territoryId);
        }
    }

    function removeAddress(addressId) {
        if (addressId > 0) {
            addressModel.removeAddress(addressId);
        }
    }

    function createSnapMarker(point) {
        if (vectorLayer.snapMarker === null) {
            var componentMarker = Qt.createComponent("SimpleMarker.qml");
            if (componentMarker.status === Component.Ready) {
                vectorLayer.snapMarker = componentMarker.createObject(vectorLayer);
                vectorLayer.snapMarker.coordinate = point;
                vectorLayer.snapMarker.markerType = 3;
                vectorLayer.snapMarker.visible = false;
                vectorLayer.snapMarker.isClickable = false;
                vectorLayer.addMapItem(vectorLayer.snapMarker);
            } else {
                console.log("Marker not created");
            }
        }
    }

    function createElements(point) {
        createSnapMarker(point);

        switch (vectorLayer.digitizeMode) {
        case 0:
            // Polyline mode
            if (vectorLayer.newPolyline === null) {
                createLine(point);
                createRubberBand(point);
            }
            vectorLayer.newPolyline.mainPolyline.insertCoordinate(vectorLayer.newPolyline.path.length - 1, point);
            break;
        default:
            // Polygon mode
            if (vectorLayer.newPolygon === null) {
                createPolygon(point);
                createRubberBand(point);
            }
            vectorLayer.newPolygon.mainPolygon.addCoordinate(point);

            // remove existing polyline
            if (vectorLayer.newPolyline !== null) {
                vectorLayer.removeMapItem(vectorLayer.newPolyline);
                vectorLayer.newPolyline = null;
            }
            // create line if polygon has 2 different points (besides start and end point, which have the same position)
            if (vectorLayer.newPolygon.mainPolygon.path.length === 3) {
                createLine(vectorLayer.newPolygon.mainPolygon.path[0]);
                vectorLayer.newPolyline.mainPolyline.addCoordinate(point);
            }
            break;
        }
    }

    function createLine(point) {
        var componentLine = Qt.createComponent("SimplePolyline.qml");
        if (componentLine.status === Component.Ready) {
            vectorLayer.newPolyline = componentLine.createObject(vectorLayer);
            vectorLayer.newPolyline.lineWidth = 2;
            vectorLayer.newPolyline.mainPolyline.addCoordinate(point);

            vectorLayer.addMapItem(vectorLayer.newPolyline);
        } else {
            console.log("Line not created");
        }
    }

    function createPolygon(point) {
        var componentPolygon = Qt.createComponent("SimplePolygon.qml");
        if (componentPolygon.status === Component.Ready) {
            vectorLayer.newPolygon = componentPolygon.createObject(vectorLayer);
            vectorLayer.newPolygon.mainPolygon.addCoordinate(point);

            vectorLayer.addMapItem(vectorLayer.newPolygon);
        } else {
            console.log("Polygon not created");
        }
    }

    function createRubberBand(point) {
        vectorLayer.removeMapItem(vectorLayer.rubberBand);
        var component = Qt.createComponent("SimplePolyline.qml");
        if (component.status === Component.Ready) {
            vectorLayer.rubberBand = component.createObject(vectorLayer);
            vectorLayer.rubberBand.mainPolyline.addCoordinate(point);
            vectorLayer.rubberBand.mainPolyline.addCoordinate(point);
            vectorLayer.rubberBand.mainPolyline.addCoordinate(point);
            vectorLayer.addMapItem(vectorLayer.rubberBand);
        } else {
            console.log("Rubber band not created");
        }
    }

    // OBJECT PROPERTIES
    Keys.onPressed: (event)=> {
        if (event.key === Qt.Key_Plus) {
            baseMap.zoomLevel++;
        } else if (event.key === Qt.Key_Minus) {
            baseMap.zoomLevel--;
        } else if (event.key === Qt.Key_Left || event.key === Qt.Key_Right
                   || event.key === Qt.Key_Up   || event.key === Qt.Key_Down) {
            var dx = 0;
            var dy = 0;

            switch (event.key) {
            case Qt.Key_Left: dx = baseMap.width / 4; break;
            case Qt.Key_Right: dx = -baseMap.width / 4; break;
            case Qt.Key_Up: dy = baseMap.height / 4; break;
            case Qt.Key_Down: dy = -baseMap.height / 4; break;
            }

            var mapCenterPoint = Qt.point(baseMap.width / 2.0 - dx, baseMap.height / 2.0 - dy);
            baseMap.center = baseMap.toCoordinate(mapCenterPoint);
        } else if (event.key === Qt.Key_Escape) {
            if (vectorLayer.isDigitizing) {
                // cancel digitizing
                vectorLayer.isLineComplete = true;
                vectorLayer.isDigitizing = false;
                if (vectorLayer.rubberBand !== null) {
                    vectorLayer.removeMapItem(vectorLayer.rubberBand);
                    vectorLayer.rubberBand = null;
                }
                if (vectorLayer.snapMarker !== null) {
                    vectorLayer.removeMapItem(vectorLayer.snapMarker);
                    vectorLayer.snapMarker = null;
                }

                switch (vectorLayer.digitizeMode) {
                case 0: {
                    if (vectorLayer.newPolyline !== null) {
                        vectorLayer.removeMapItem(vectorLayer.newPolyline);
                        vectorLayer.newPolyline = null;
                    }
                    break;
                }
                default: {
                    if (vectorLayer.newPolygon !== null) {
                        vectorLayer.removeMapItem(vectorLayer.newPolyline);
                        vectorLayer.newPolyline = null;
                        vectorLayer.removeMapItem(vectorLayer.newPolygon);
                        vectorLayer.newPolygon = null;
                    }
                    break;
                }
                }
            }
        } else if (event.key === Qt.Key_Backspace) {
            if (vectorLayer.isDigitizing) {
                var point;
                switch (vectorLayer.digitizeMode) {
                case 0: {
                    // remove last but one point (last point = first point)
                    if (vectorLayer.newPolyline !== null && vectorLayer.newPolyline.mainPolyline.pathLength() > 2) {
                        vectorLayer.newPolyline.mainPolyline.removeCoordinate(vectorLayer.newPolyline.mainPolyline.pathLength() - 2);
                        point = vectorLayer.newPolyline.mainPolyline.path[vectorLayer.newPolyline.mainPolyline.pathLength() - 2];
                    }
                    break;
                }
                default: {
                    if (vectorLayer.newPolygon !== null && vectorLayer.newPolygon.mainPolygon.path.length > 2) {
                        point = vectorLayer.newPolygon.mainPolygon.path[vectorLayer.newPolygon.mainPolygon.path.length - 1];
                        vectorLayer.newPolygon.mainPolygon.removeCoordinate(point);
                        point = vectorLayer.newPolygon.mainPolygon.path[vectorLayer.newPolygon.mainPolygon.path.length - 1];
                    }

                    // remove existing polyline
                    if (vectorLayer.newPolyline !== null) {
                        vectorLayer.removeMapItem(vectorLayer.newPolyline);
                        vectorLayer.newPolyline = null;
                    }
                    // create line if polygon has 2 different points (besides start and end point, which have the same position)
                    if (vectorLayer.newPolygon.mainPolygon.path.length === 3) {
                        createLine(vectorLayer.newPolygon.mainPolygon.path[0]);
                        vectorLayer.newPolyline.mainPolyline.addCoordinate(point);
                    }
                    break;
                }
                }
                if (vectorLayer.rubberBand !== null && point !== undefined) {
                    // update rubber band
                    vectorLayer.rubberBand.mainPolyline.replaceCoordinate(0, point);
                }
            }
        }
    }

    addressTextField.onAccepted: {
        geocodeAddress(addressTextField.text);
    }

    cutoffStreetSectionButton.onClicked: cutoffStreetSection()

    importBoundariesButton.onClicked: importBoundaries()

    createBoundaryButton.onClicked: createBoundary()
    splitTerritoryButton.onClicked: splitTerritory()
    removeBoundaryButton.onClicked: removeBoundary()

    showBoundariesButton.states: [
        State {
            name: "ShowAllBoundaries"; when: boundaryLayer.boundaryDisplayOption === 2
            PropertyChanges { target: showBoundariesButton; icon.source: "qrc:///icons/boundary_all.svg"; }
        },
        State {
            name: "ShowSelectedBoundaries"; when: boundaryLayer.boundaryDisplayOption === 1
            PropertyChanges { target: showBoundariesButton; icon.source: "qrc:///icons/boundary.svg"; }
        },
        State {
            name: "HideBoundaries"; when: boundaryLayer.boundaryDisplayOption === 0
            PropertyChanges { target: showBoundariesButton; icon.source: "qrc:///icons/boundary_off.svg"; }
        }
    ]

    showBoundariesButton.onClicked: boundaryLayer.boundaryDisplayOption = (boundaryLayer.boundaryDisplayOption + 1) % 3

    showMarkersButton.states: [
        State {
            name: "ShowAllAddresses"; when: addressLayer.markerDisplayOption === 2
            PropertyChanges { target: showMarkersButton; icon.source: "qrc:///icons/location_all.svg"; }
        },
        State {
            name: "ShowTerritoryAddresses"; when: addressLayer.markerDisplayOption === 1
            PropertyChanges { target: showMarkersButton; icon.source: "qrc:///icons/location.svg"; }
        },
        State {
            name: "HideAddresses"; when: addressLayer.markerDisplayOption === 0
            PropertyChanges { target: showMarkersButton; icon.source: "qrc:///icons/location_off.svg"; }
        }
    ]
    showMarkersButton.onClicked: addressLayer.markerDisplayOption = (addressLayer.markerDisplayOption + 1) % 3

    showStreetsButton.states: [
        State {
            name: "ShowAllStreets"; when: streetLayer.streetDisplayOption === 2
            PropertyChanges { target: showStreetsButton; icon.source: "qrc:///icons/road_all.svg"; }
        },
        State {
            name: "ShowTerritoryStreets"; when: streetLayer.streetDisplayOption === 1
            PropertyChanges { target: showStreetsButton; icon.source: "qrc:///icons/road.svg"; }
        },
        State {
            name: "HideStreets"; when: streetLayer.streetDisplayOption === 0
            PropertyChanges { target: showStreetsButton; icon.source: "qrc:///icons/road_off.svg"; }
        }
    ]
    showStreetsButton.onClicked: streetLayer.streetDisplayOption = (streetLayer.streetDisplayOption + 1) % 3

    editModeButton.states: [
        State {
            name: "View"; when: vectorLayer.editMode === 0
            PropertyChanges { target: editModeButton; icon.source: "qrc:///icons/visibility.svg"; }
        },
        State {
            name: "EditAddresses"; when: vectorLayer.editMode === 1
            PropertyChanges { target: editModeButton; icon.source: "qrc:///icons/edit_location.svg"; }
        },
        State {
            name: "EditStreets"; when: vectorLayer.editMode === 2
            PropertyChanges { target: editModeButton; icon.source: "qrc:///icons/edit_road.svg"; }
        },
        State {
            name: "EditBoundaries"; when: vectorLayer.editMode === 3
            PropertyChanges { target: editModeButton; icon.source: "qrc:///icons/edit_boundary.svg"; }
        }
    ]
    editModeButton.onClicked: vectorLayer.editMode = (vectorLayer.editMode + 1) % 4

    zoomFullButton.onClicked: zoomFull()

    zoomSlider.onMoved: {
        baseMap.zoomLevel = zoomSlider.value;
    }

    // baseMap.plugin: osmPlugin

    baseMap.onSupportedMapTypesChanged: {
        if (baseMap.supportedMapTypes.length > 0) {
            var newActiveMapType = baseMap.supportedMapTypes[0];
            if (settings.osm_map_type !== "") {
                for (var mapType in baseMap.supportedMapTypes) {
                    if (baseMap.supportedMapTypes[mapType].name === settings.osm_map_type) {
                        newActiveMapType = baseMap.supportedMapTypes[mapType];
                        break;
                    }
                }
            }
            if (newActiveMapType !== undefined)
                baseMap.activeMapType = newActiveMapType;
        }
    }

    baseMap.onActiveMapTypeChanged: {
        settings.osm_map_type = baseMap.activeMapType.name;
    }

    baseMap.onOpacityChanged: {
        settings.basemap_opacity = baseMap.opacity;
    }

    // workaround for QTBUG-87646 / QTBUG-112394 / QTBUG-112432:
    // Magic Mouse pretends to be a trackpad but doesn't work with PinchHandler
    // and we don't yet distinguish mice and trackpads on Wayland either
    wheelHandler.acceptedDevices: Qt.platform.pluginName === "cocoa" || Qt.platform.pluginName === "wayland"
                     ? PointerDevice.Mouse | PointerDevice.TouchPad
                     : PointerDevice.Mouse
    wheelHandler.onWheel: (event) => {
        const loc = baseMap.toCoordinate(wheelHandler.point.position)
        switch (event.modifiers) {
            case Qt.NoModifier:
                baseMap.zoomLevel += event.angleDelta.y / 120
                break
            case Qt.ShiftModifier:
                baseMap.bearing += event.angleDelta.y / 15
                break
            case Qt.ControlModifier:
                baseMap.tilt += event.angleDelta.y / 15
                break
        }
        baseMap.alignCoordinateToPoint(loc, wheelHandler.point.position)
    }

    dragHandler.onTranslationChanged: (delta) => baseMap.pan(-delta.x, -delta.y)

    // CHILD OBJECTS
    Settings {
        id: settings
        category: "map_view"
        property string osm_map_type: ""
        property double basemap_opacity: 1.0
        property int boundary_display_mode: 2
        property double boundary_layer_opacity: 1.0
        property int street_display_mode: 2
        property double street_layer_opacity: 1.0
        property int address_display_mode: 2
        property double address_layer_opacity: 1.0
    }

    hoverHandler.onPointChanged: {
        if (!vectorLayer.isDigitizing)
            return;

        var mapPoint = vectorLayer.toCoordinate(hoverHandler.point.position);
        var delta = vectorLayer.toCoordinate(Qt.point(0, 0)).distanceTo(vectorLayer.toCoordinate(Qt.point(10, 0)));

        var snapPoint = territories.getClosestPoint(mapPoint, delta, territoryTreeModel.snapPoints);
        createSnapMarker(mapPoint);
        if (vectorLayer.snapMarker !== null) {
            if (snapPoint.isValid) {
                mapPoint = snapPoint;
                vectorLayer.snapMarker.coordinate = snapPoint;
                vectorLayer.snapMarker.visible = true;
            } else {
                vectorLayer.snapMarker.visible = false;
            }
        }

        if (vectorLayer.rubberBand !== null)
            vectorLayer.rubberBand.mainPolyline.replaceCoordinate(1, mapPoint);
    }

    tapHandler.onTapped: (eventPoint, button)=> {
        Qt.callLater(vectorLayer.forceActiveFocus);
        if (!vectorLayer.isDigitizing)
            return;

        if (button === Qt.LeftButton) {
            if (vectorLayer.isLineComplete)
                return;
            var mapPoint = vectorLayer.toCoordinate(eventPoint.position);
            // replace clicked point if a point is snapped
            if (vectorLayer.snapMarker !== null && vectorLayer.snapMarker.visible)
                mapPoint = vectorLayer.snapMarker.coordinate;

            if (vectorLayer.rubberBand !== null) {
                // update rubber band
                vectorLayer.rubberBand.mainPolyline.replaceCoordinate(0, mapPoint);
                vectorLayer.rubberBand.mainPolyline.replaceCoordinate(1, mapPoint);
            }
            createElements(mapPoint);
        }
        if (button === Qt.RightButton) {
            vectorLayer.isLineComplete = true;
            vectorLayer.isDigitizing = false;
            if (vectorLayer.rubberBand !== null) {
                vectorLayer.removeMapItem(vectorLayer.rubberBand);
                vectorLayer.rubberBand = null;
            }
            switch (vectorLayer.digitizeMode) {
            case 0:
                if (vectorLayer.editMode === 2 && vectorLayer.newPolyline !== null) {
                    // cut street
                    if (currentStreet) {
                        //streetModel.saveStreets();
                        var cutoffSection = streetModel.cutoffStreetSection(currentStreet.streetId, vectorLayer.newPolyline.path);
                        //var cutOffSection = territories.cutTerritoryStreet(currentStreet.streetId, vectorLayer.newPolyline.path);
                        if (cutoffSection !== "") {
                            addStreetMessageDialog.text = qsTr("Do you want to create a new street from the section that was cut off?\nSelect 'No' to discard the section.", "Add a new street")
                            addStreetMessageDialog.newStreetWktGeometry = cutoffSection;
                            addStreetMessageDialog.visible = true;
                        }
                        vectorLayer.removeMapItem(vectorLayer.newPolyline);
                        vectorLayer.newPolyline = null;
                    }
                } else if (vectorLayer.editMode === 3 && vectorLayer.newPolyline !== null) {
                    // split territory
                    if (currentTerritory) {
                        currentTerritory.save();
                        var newTerritoryId = territories.splitTerritory(currentTerritory.territoryId, vectorLayer.newPolyline.path);
                        if (newTerritoryId !== 0) {
                            currentTerritory = territoryTreeModel.findTerritory(newTerritoryId);
                        }
                        vectorLayer.removeMapItem(vectorLayer.newPolyline);
                        vectorLayer.newPolyline = null;
                    }
                }
                break;
            default:
                if (vectorLayer.newPolygon !== null) {
                    if (currentTerritory) {
                        currentTerritory.save();
                        // check for overlapping territories
                        var territoryList = territories.getAllTerritories();
                        var count = territoryList.length;
                        var overlappedTerritories = [];
                        var overlappedTerritoryNumbers = [];
                        for (var i = 0; i < count; i++) {
                            if (typeof territoryList[i] !== "undefined")
                                if (territoryList[i].territoryId !== currentTerritory.territoryId && territoryList[i].crosses(vectorLayer.newPolygon.path)) {
                                    overlappedTerritories.push(territoryList[i].territoryId);
                                    overlappedTerritoryNumbers.push(territoryList[i].territoryNumber);
                                }
                        }

                        if (overlappedTerritories.length > 0) {
                            addBoundaryMessageDialog.text = qsTr("The new boundary overlaps %n territory(ies):", "Add or edit territory boundary", overlappedTerritories.length) +
                                    "\n" + overlappedTerritoryNumbers.join(",") + "\n" +
                                    qsTr("Do you want to assign overlapping areas to the current territory?\nSelect 'No' if overlapping areas should remain in their territories and to add only the part, that doesn't overlap other territories.", "Add or edit territory boundary")
                            addBoundaryMessageDialog.overlappedTerritoryIds = overlappedTerritories;
                            addBoundaryMessageDialog.visible = true;
                        } else
                            setBoundary(overlappedTerritories, false);
                    }
                }
            }
        }
    }

    Connections {
        target: territoryManager
        function onZoomSelectedTerritory() { zoomSelectedTerritory(); }
        function onZoomSelectedAddress() { zoomSelectedAddress(); }
        function onZoomSelectedStreet() { zoomSelectedStreet(); }
        function onZoomCoordinate(lat, lon) { zoomCoordinate(lat, lon); }
        function onDisplayGeocodeResults(geocodeResults) { displayGeocodeResults(geocodeResults); }
        function onClearAddressSearchMarkers() { clearAddressSearchMarkers(); }
    }

    Location { id: geoLocationPrototype } // for conversion from QGeoLocation to QML Location

    Component {
        id: territoryBoundaryDelegate

        // nested MapItemView for grouped territories in tree model
        MapItemView {
            id: groupDelegate
            required property int index
            required property int childCount

            model: childCount
            delegate: TerritoryMapPolygon {
                id: mapPolygon
                required property int index
                property var territory: territoryTreeModel.data(
                                                  territoryTreeModel.index(mapPolygon.index, 0, territoryTreeModel.index(groupDelegate.index, 0)),
                                                  TerritoryTreeModel.TerritoryRole)
                // model data is set via required properties
                isSelected: false
                territoryId: territory ? territory.territoryId : 0
                multiPolygon: territoryTreeModel.data(
                           territoryTreeModel.index(mapPolygon.index, 0, territoryTreeModel.index(groupDelegate.index, 0)),
                           TerritoryTreeModel.MultiPolygonRole)  // mapPolygon.multiPolygon
                model: multiPolygon
            }
        }
    }

    Component {
        id: territoryStreetDelegate

        TerritoryMapPolyline {
            // model data is set via required properties
            isTerritorySelected: currentTerritory?.territoryId === street.territoryId ? true : false
            isStreetSelected: currentStreet?.streetId === street.streetId ? true : false
            model: multiLine
        }
    }

    Component {
        id: territoryMarkerDelegate

        TerritoryMapMarker {
            territoryId: model.territoryId
            isTerritorySelected: currentTerritory?.territoryId === model.territoryId ? true : false
            isMarkerSelected: currentAddress?.id === model.id ? true : false
            addressId: model.id
            coordinate: model.coordinate
            color: model.addressTypeColor
            markerScale: vectorLayer.markerScale
        }
    }

    Plugin {
        id: osmPlugin
        name: "osm"
        // specify plugin parameters if necessary
        // PluginParameter {
        //     name:
        //     value:
        // }
        //PluginParameter{ name: "style"; value: "CycleMap"}
    }

    Menu {
        id: contextMenu
        property var marker
        property int featureTerritoryId
        property int markerAddressId

        function show(sender, territoryId, addressId) {
            marker = sender;
            featureTerritoryId = territoryId;
            markerAddressId = addressId;

            if (canEditTerritories
                    && ((contextMenu.featureTerritoryId !== 0 && contextMenu.markerAddressId === -1
                         && currentTerritory !== null && currentTerritory.territoryId !== contextMenu.featureTerritoryId)
                        || (contextMenu.featureTerritoryId === 0 && contextMenu.markerAddressId > -1)
                        || (contextMenu.featureTerritoryId > 0 && contextMenu.markerAddressId > -1))) {
                popup();
            }
        }

        onAboutToShow: {
            addAddressMenuItem.enabled = currentTerritory;
            reassignAddressMenuItem.enabled = currentTerritory !== null
                    && currentTerritory.territoryId !== featureTerritoryId;
        }

        MenuItem {
            id: annexAreaMenuItem
            text: qsTr("Join to the selected territory", "Join two territories into one")
            icon.source: "qrc:///icons/boundary_join.svg"
            visible: canEditTerritories && contextMenu.featureTerritoryId !== 0 && contextMenu.markerAddressId === -1
                     && currentTerritory !== null && currentTerritory.territoryId !== contextMenu.featureTerritoryId
            enabled: visible
            height: visible ? implicitHeight : 0
            onTriggered: {
                var territoryList = territories.getAllTerritories();
                var count = territoryList.length;
                for (var i = 0; i < count; i++) {
                    if (typeof territoryList[i] !== "undefined")
                        if (territoryList[i].territoryId === contextMenu.featureTerritoryId) {
                            var overlappedTerritoryNumbers = [];
                            overlappedTerritoryNumbers.push(contextMenu.featureTerritoryId);
                            if (territories.setTerritoryBoundary(currentTerritory.territoryId, territoryList[i].wktGeometry, true, overlappedTerritoryNumbers, true)) {
                                console.log("Territories joined");
                            }
                            break;
                        }
                }
            }
        }

        MenuItem {
            id: addAddressMenuItem
            text: qsTr("Add address to selected territory")
            icon.source: "qrc:///icons/add_location.svg"
            visible: canEditTerritories && contextMenu.featureTerritoryId === 0 && contextMenu.markerAddressId > -1
            enabled: visible
            height: visible ? implicitHeight : 0
            onTriggered: addAddress(contextMenu.marker.location)
        }

        MenuItem {
            id: reassignAddressMenuItem
            text: qsTr("Assign to selected territory", "Reassign territory address")
            icon.source: "qrc:///icons/move_location.svg"
            visible: canEditTerritories && contextMenu.featureTerritoryId > 0 && contextMenu.markerAddressId > -1
            enabled: visible
            height: visible ? implicitHeight : 0
            onTriggered: reassignAddress(contextMenu.markerAddressId)
        }

        MenuItem {
            id: removeAddressMenuItem
            text: qsTr("Remove address", "Delete territory address")
            icon.source: "qrc:///icons/remove_location.svg"
            visible: canEditTerritories && contextMenu.featureTerritoryId > 0 && contextMenu.markerAddressId > -1
            enabled: visible
            height: visible ? implicitHeight : 0
            onTriggered: removeAddress(contextMenu.markerAddressId)
        }
    }

    MessageDialog {
        id: addBoundaryMessageDialog

        property var overlappedTerritoryIds

        buttons: MessageDialog.Yes | MessageDialog.No | MessageDialog.Cancel
        // icon: StandardIcon.Question
        modality: Qt.ApplicationModal
        onButtonClicked: function (button, role) {
            switch (button) {
            case MessageDialog.Yes:
                setBoundary(overlappedTerritoryIds, true);
                break;
            case MessageDialog.No:
                setBoundary(overlappedTerritoryIds, false);
                break;
            case MessageDialog.Cancel:
                vectorLayer.removeMapItem(vectorLayer.newPolygon);
                vectorLayer.newPolygon = null;
                break;
            }
        }
    }

    MessageDialog {
        id: addStreetMessageDialog

        property var newStreetWktGeometry

        buttons: MessageDialog.Yes | MessageDialog.No
        // icon: StandardIcon.Question
        modality: Qt.ApplicationModal
        onAccepted: {
            var newStreetId = streetModel.addStreet(currentTerritory.territoryId, currentStreet.streetName, newStreetWktGeometry, currentStreet.streetTypeId);
            if (newStreetId > 0) {
                // select the new street
                territoryManager.currentStreet = streetModel.findStreet(newStreetId);
            }
        }
    }

    Window {
        id: importDialog
        title: qsTr("Import territory data", "Territory data import dialog")
        width: 400
        minimumWidth: 400
        minimumHeight: 330
        height: 270
        flags: Qt.Popup | Qt.Dialog
        modality: Qt.WindowModal
        color: myPalette.window

        onVisibleChanged: {
            x = mainWindow.x + mainWindow.width / 2 - width / 2;
            y = mainWindow.y + mainWindow.height / 2 - height / 2;
        }

        MessageDialog {
            id: importMessage
            buttons: MessageDialog.Ok
            //icon: StandardIcon.Warning
            modality: Qt.ApplicationModal

            onAccepted: {
                importForm.displayProgress = false;
            }
        }

        TerritoryImportForm {
            id: importForm
            anchors.leftMargin: 10
            anchors.rightMargin: 10
            anchors.topMargin: 10
            anchors.bottomMargin: 10

            property DataObjectListModel addressTypeListModel: territories.getAddressTypes()
            property url fileName: importOption == 1 ? Qt.resolvedUrl(boundariesFileDialog.selectedFile) : Qt.resolvedUrl(addressesFileDialog.selectedFile)
            property url failedAddressesFileName: Qt.resolvedUrl(failedAddressesFileDialog.selectedFile)
            property string csvDelimiter: ""
            property int importedItemCount: 0
            property int itemCount: 0

            importButton.enabled: false
            addressTypeComboBox.model: addressTypeListModel
            addressTypeComboBox.currentIndex: addressTypeListModel.getIndex(addressModel.getDefaultAddressTypeNumber())

            function selectFile() {
                switch (importOption) {
                case 1:
                    boundariesFileDialog.open();
                    break;
                case 2:
                    addressesFileDialog.open();
                    break;
                }
            }

            function addressImportProgressChanged(rowCount, importCount) {
                itemCount = rowCount;
                importedItemCount = importCount;
                progressLabel.text = qsTr("%1 of %2 address(es) imported.", "Territory address import progress").arg(importCount).arg(rowCount);
            }

            importButton.onClicked: {
                switch (importOption) {
                case 1: {
                    var nameMatchField = importForm.boundaryNameComboBox.currentIndex;
                    var descriptionMatchField = importForm.boundaryDescriptionComboBox.currentIndex;
                    var searchByDescription = importForm.searchByDescriptionCheckBox.checked;

                    if (nameMatchField !== 0 && descriptionMatchField !== 0 && nameMatchField === descriptionMatchField) {
                        importMessage.title = qsTr("Import territory boundaries", "Territory import dialog");
                        importMessage.text = qsTr("The selected fields should be different.", "Territory import from KML file");
                        // importMessage.icon = StandardIcon.Warning;
                        importMessage.visible = true;
                        return;
                    }

                    var importRes = territories.importKmlGeometry(importForm.fileName, nameMatchField, descriptionMatchField, searchByDescription);
                    if (importRes >= 0) {
                        importMessage.title = qsTr("Import territory data", "Territory import dialog");
                        importMessage.text = qsTr("%n territory(ies) imported or updated.", "Number of territories imported or updated", importRes);
                        // importMessage.icon = StandardIcon.Information;
                        importMessage.visible = true;
                        zoomFull();
                    } else {
                        importMessage.title = qsTr("Import territory data", "Territory import dialog");
                        importMessage.text = qsTr("The import file could not be read.", "Territory address import from file");
                        // importMessage.icon = StandardIcon.Critical;
                        importMessage.visible = true;
                    }
                    break;
                }
                case 2: {
                    var addressField = importForm.addressComboBox.currentIndex;
                    var nameField = importForm.addressNameComboBox.currentIndex;

                    if (addressField === -1 || nameField === -1) {
                        importMessage.title = qsTr("Import territory data", "Territory import dialog");
                        importMessage.text = qsTr("Please select the address and name fields.", "Fields for territory address import from file");
                        // importMessage.icon = StandardIcon.Warning;
                        importMessage.visible = true;
                        return;
                    }

                    if (addressField >= 0 && nameField >= 0 && addressField === nameField) {
                        importMessage.title = qsTr("Import territory data", "Territory import dialog");
                        importMessage.text = qsTr("The selected fields should be different.", "Fields for territory address import from file");
                        // importMessage.icon = StandardIcon.Warning;
                        importMessage.visible = true;
                        return;
                    }

                    if (currentTerritory === null) {
                        importMessage.title = qsTr("Import territory addresses", "Territory import dialog");
                        importMessage.text = qsTr("The addresses will be added to the current territory. Please select a territory first.", "Territory address import from file");
                        // importMessage.icon = StandardIcon.Warning;
                        importMessage.visible = true;
                        return;
                    }

                    var addressTypeNumber = addressTypeListModel.get(importForm.addressTypeComboBox.currentIndex).id;
                    if (addressTypeNumber < 1)
                        addressTypeNumber = 1;

                    territories.onAddressImportProgressChanged.connect(addressImportProgressChanged);

                    progressLabel.text = "";
                    displayProgress = true;
                    var addressImportRes = territories.importAddresses(
                                importForm.fileName, addressField, nameField, csvDelimiter,
                                currentTerritory.territoryId,
                                addressTypeNumber, importForm.failedAddressesFileName);

                    territories.onAddressImportProgressChanged.disconnect(addressImportProgressChanged);

                    if (addressImportRes >= 0) {
                        importMessage.title = qsTr("Import territory data", "Territory import dialog");
                        importMessage.text = qsTr("%n address(es) imported.", "Number of addresses imported", addressImportRes);
                        // importMessage.icon = StandardIcon.Information;
                        importMessage.visible = true;
                    } else {
                        importMessage.title = qsTr("Import territory data", "Territory import dialog");
                        if (addressImportRes === -2)
                            importMessage.text = qsTr("No valid territory selected.", "Territory boundary import");
                        else
                            importMessage.text = qsTr("The import file could not be read.", "Territory boundary import");
                        // importMessage.icon = StandardIcon.Critical;
                        importMessage.visible = true;
                    }
                    break;
                }
                }
            }

            closeButton.onClicked: importDialog.close();

            onFileNameChanged: {
                var path = fileName.toString();
                importButton.enabled = path !== "";
                if (Qt.platform.os === "windows")
                    path = path.replace(/^(file:\/{3})|(qrc:\/{2})|(http:\/{2})/,"");
                else
                    path = path.replace(/^(file:\/{2})|(qrc:\/{2})|(http:\/{2})/,"");

                var pathText = decodeURIComponent(path);
                if (fileNameTextField.text !== pathText) {
                    fileNameTextField.text = pathText;
                    if (importOption == 2 && pathText !== "") {
                        var schema = territories.getCSVschema(importForm.fileName, "");
                        csvDelimiter = schema.delimiter;
                        importForm.csvFieldModel.clear();
                        var i = 0;
                        for (var fieldName in schema.fields) {
                            importForm.csvFieldModel.append({"name":schema.fields[fieldName]});
                            if (schema.fields[fieldName].toLocaleLowerCase() === qsTr("Address", "Default Address-field for territory address import").toLocaleLowerCase())
                                importForm.addressComboBox.currentIndex = i;
                            if (schema.fields[fieldName].toLocaleLowerCase() === qsTr("Name", "Default Name-field for territory address import").toLocaleLowerCase())
                                importForm.addressNameComboBox.currentIndex = i;
                            i++;
                        }
                    }
                }
            }

            onFailedAddressesFileNameChanged: {
                var path = failedAddressesFileName.toString();
                if (Qt.platform.os === "windows")
                    path = path.replace(/^(file:\/{3})|(qrc:\/{2})|(http:\/{2})/,"");
                else
                    path = path.replace(/^(file:\/{2})|(qrc:\/{2})|(http:\/{2})/,"");

                var pathText = decodeURIComponent(path);
                if (failedFileNameTextField.text !== pathText) {
                    failedFileNameTextField.text = pathText;
                }
            }

            selectFileButton.onClicked: selectFile()

            selectFailedFileButton.onClicked: failedAddressesFileDialog.open()
        }

        FileDialog {
            id: boundariesFileDialog
            title: qsTr("Open file")
            nameFilters: [ qsTr("KML files (*.kml)", "Filedialog pattern"), qsTr("All files (*)", "Filedialog pattern") ]
            selectedNameFilter.index: 0
            fileMode: FileDialog.OpenFile
        }

        FileDialog {
            id: addressesFileDialog
            title: qsTr("Open file")
            nameFilters: [ qsTr("CSV files (*.csv)", "Filedialog pattern"), qsTr("Text files (*.txt)", "Filedialog pattern"), qsTr("All files (*)", "Filedialog pattern") ]
            selectedNameFilter.index: 0
            fileMode: FileDialog.OpenFile
        }

        FileDialog {
            id: failedAddressesFileDialog
            title: qsTr("Save file")
            nameFilters: [ qsTr("CSV files (*.csv)", "Filedialog pattern"), qsTr("Text files (*.txt)", "Filedialog pattern"), qsTr("All files (*)", "Filedialog pattern") ]
            selectedNameFilter.index: 0
            fileMode: FileDialog.SaveFile
        }
    }

    Component.onCompleted: {
        // set map plugin
        var parameters = new Array();
        for (var prop in geoServiceParameters) {
            var parameter = Qt.createQmlObject('import QtLocation; PluginParameter{ name: "'+ prop + '"; value: "' + geoServiceParameters[prop] + '"}', baseMap);
            parameters.push(parameter);
        }
        baseMap.pluginParameters = parameters;
        var provider = defaultGeoServiceProvider;
        var plugin;
        if (parameters && parameters.length > 0)
            plugin = Qt.createQmlObject ('import QtLocation; Plugin{ name:"' + provider + '"; parameters: baseMap.pluginParameters}', baseMap);
        else
            plugin = Qt.createQmlObject ('import QtLocation; Plugin{ name:"' + provider + '"}', baseMap);
        baseMap.plugin = plugin;
        baseMap.opacity = settings.basemap_opacity;
        boundaryLayer.opacity = settings.boundary_layer_opacity;
        streetLayer.opacity = settings.street_layer_opacity;
        addressLayer.opacity = settings.address_layer_opacity;

        vectorLayer.markerScale = addressModel.getMarkerScale();

        boundaryLayer.boundaryDisplayOption = settings.boundary_display_mode;
        streetLayer.streetDisplayOption = settings.street_display_mode;
        addressLayer.markerDisplayOption = settings.address_display_mode;
        zoomFull();
    }

    Component.onDestruction: {
        settings.osm_map_type = baseMap.activeMapType.name;
        settings.basemap_opacity = baseMap.opacity;
        settings.boundary_display_mode = boundaryLayer.boundaryDisplayOption;
        settings.boundary_layer_opacity = boundaryLayer.opacity;
        settings.street_display_mode = streetLayer.streetDisplayOption;
        settings.street_layer_opacity = streetLayer.opacity;
        settings.address_display_mode = addressLayer.markerDisplayOption;
        settings.address_layer_opacity = addressLayer.opacity;
        settings.sync();
    }
}
