import QtCore
import QtQuick
import QtQuick.Controls
import QtQuick.Dialogs
import QtQuick.Layouts
import QtPositioning
import QtLocation
import net.theocbase 1.0
import "controls"
import "sidePanelScripts.js" as SPScripts

TerritoryManagerForm {
    id: territoryManager

    // PROPERTY DECLARATIONS
    property bool canEditTerritories: accessControl?.user?.hasPermission(PermissionRule.CanEditTerritories)
    property bool canViewTerritoryAssignments: accessControl?.user?.hasPermission(PermissionRule.CanViewTerritoryAssignments)
    property bool canViewTerritoryAddresses: accessControl?.user?.hasPermission(PermissionRule.CanViewTerritoryAddresses)
    property DataObjectListModel addressTypeListModel: territories.getAddressTypes()
    property DataObjectListModel streetTypeListModel: territories.getStreetTypes()
    property Territories territories: Territories {}
    property Territory currentTerritory
    property TerritoryStreet currentStreet
    property TerritoryAddress currentAddress

    // SIGNAL DECLARATIONS
    signal territorySelectionChanged()
    signal zoomSelectedTerritory()
    signal zoomSelectedStreet()
    signal zoomSelectedAddress()
    signal zoomCoordinate(var lat, var lon)
    signal displayGeocodeResults(var geocodeResults)
    signal clearAddressSearchMarkers()

    // JAVASCRIPT FUNCTIONS
    function resetDefaultTerritoryLookupTreeViewSettings() {
        settings.territories_groupByIndex = 1;
        territoryTreeView.groupByIndex = settings.territories_groupByIndex;
        settings.territories_groupBySortOrder = Qt.AscendingOrder;
        territoryTreeView.groupBySortOrder = settings.territories_groupBySortOrder;
        settings.territories_sortOrder = Qt.AscendingOrder;
        territoryTreeView.sortOrder = settings.territories_sortOrder;
        settings.territories_detailRowCount = 3;
        settings.territories_labelingMode = 1;
        territoryLookupTreeViewMoreMenu.detailRowCount = settings.territories_detailRowCount;
        territoryLookupTreeViewMoreMenu.labelingMode = settings.territories_labelingMode;
        settings.sync();
        reloadTerritories();
    }

    function reloadTerritories() {
        // save current selection
        var currentTerritoryId = currentTerritory?.territoryId ?? 0;
        if (canEditTerritories)
            territoryTreeModel.saveTerritories();
        territoryTreeModel.updateModel();
        // restore selection
        currentTerritory = territoryTreeModel.findTerritory(currentTerritoryId);
    }

    function refreshStreets(reload) {
        // save current selection
        var previousStreetId = currentStreet?.streetId ?? 0;
        if (canEditTerritories) {
            // save streets
            streetModel.saveStreets();
        }
        if (reload)
            streetModel.loadStreets();
        // restore selection
        var previousStreet = streetModel.findStreet(previousStreetId);
        if (previousStreet?.territoryId === currentTerritory.territoryId)
            currentStreet = previousStreet;
        else
            currentStreet = null;
    }

    function refreshAddresses(reload) {
        // save current selection
        var previousAddressId = currentAddress?.id ?? 0;
        if (canEditTerritories) {
            // save addresses
            addressModel.saveAddresses();
        }
        if (reload)
            addressModel.loadAddresses();
        // restore selection
        var previousAddress = addressModel.findAddress(previousAddressId);
        if (previousAddress?.territoryId === currentTerritory.territoryId)
            currentAddress = previousAddress;
        else
            currentAddress = null;
    }

    function save() {
        return true;
    }

    function switchSidePanel() {
        territoryManager.forceActiveFocus()

        if (!sidebar.isVisible)
            sidebar.loadPage("../TerritoryPanel.qml", {})
        sidebar.isVisible = !sidebar.isVisible;
    }

    // OBJECT PROPERTIES
    territoryTreeView.columnWidthProvider: function (column) {
        return territoryTreeView.columnWidths[column];
    }
    territoryTreeView.onSearchTextChanged: function(text) {
        territoriesProxyModel.filterText = text
    }

    territoryTreeView.onGroupByIndexChanged: {
        if (territoryTreeView.groupByIndex >= 0) {
            settings.territories_groupByIndex = territoryTreeView.groupByIndex;
            territoryTreeModel.groupByIndex = territoryTreeView.groupByIndex;
        }
    }

    territoryTreeView.onGroupBySortOrderChanged: {
        settings.territories_groupBySortOrder = territoryTreeView.groupBySortOrder;
        territoryTreeModel.groupBySortOrder = territoryTreeView.groupBySortOrder ? Qt.DescendingOrder : Qt.AscendingOrder;
    }

    territoryTreeView.onSortOrderChanged: {
        settings.territories_sortOrder = territoryTreeView.sortOrder;
        territoriesProxyModel.sort(0, territoryTreeView.sortOrder)
    }

    territoryTreeView.onAddItem: {
        var newTerritoryId = territoryTreeModel.addNewTerritory();
        if (newTerritoryId >= 0) {
            currentTerritory = territoryTreeModel.findTerritory(newTerritoryId);
            if (!sidebar.isVisible)
                switchSidePanel();
        }
    }

    onCurrentTerritoryChanged: {
        if (canEditTerritories) {
            territoryTreeModel.saveTerritories();
            if (currentTerritory) {
                // reload territory, in case boundaries etc. were changed in the map
                // to avoid saving an older values when other properties are edited
                currentTerritory.reloadTerritory();
            }
        }

        if (canEditTerritories)
            assignmentModel.saveAssignments();
        assignmentModel.loadAssignments(currentTerritory ? currentTerritory.territoryId : 0);

        refreshStreets();

        refreshAddresses();

        // set selection in treeview
        var territoryIndex = territoryTreeModel.index(currentTerritory ? currentTerritory.territoryId : 0);
        territoryTreeView.selectionModel.setCurrentIndex(territoriesProxyModel.mapFromSource(territoryIndex), ItemSelectionModel.ClearAndSelect);
        territorySelectionChanged();
    }

    onCurrentStreetChanged: {
        if (canEditTerritories)
            streetModel.saveStreets();
    }

    onCurrentAddressChanged: {
        if (canEditTerritories)
            addressModel.saveAddresses();
    }

    showSidePanelButton.onClicked: switchSidePanel()

    // CHILD OBJECTS
    SystemPalette { id: myPalette; colorGroup: SystemPalette.Active }

    MessageDialog {
        id: message
        buttons: MessageDialog.Ok
        //icon: StandardIcon.Warning
        modality: Qt.ApplicationModal
    }

    Settings {
        id: settings
        category: "LookupControl"
        property int territories_groupByIndex: 1
        property bool territories_groupBySortOrder: Qt.AscendingOrder
        property bool territories_sortOrder: Qt.AscendingOrder
        property int territories_detailRowCount: 3
        property int territories_labelingMode: 1
    }

    TerritoryTreeSFProxyModel {
        id: territoriesProxyModel
        source: territoryTreeModel
        sortCaseSensitivity: Qt.CaseInsensitive

        //sortRole: territoryTreeView.he
    }

    Connections {
        target: territories
        function onBoundaryGeometryChanged() {
            reloadTerritories();
        }
        function onStreetGeometryChanged() {
            refreshStreets(true);
        }
        function onAddressGeometryChanged() {
            refreshAddresses(true);
        }
    }

    PersonDetailSectionDelegate { id: personsSectionDelegate }

    Component {
        id: territoriesItemDelegate

        TreeViewDelegate {
            id: itemDelegate

            clip: true

            implicitWidth: 50

            contentItem: Item {
                anchors.top: itemDelegate.top
                anchors.bottom: itemDelegate.bottom

                Row {
                    anchors.top: parent.top
                    anchors.bottom: parent.bottom
                    anchors.left: parent.left
                    anchors.right: buttonRow.left

                    Label {
                        width: parent.width
                        anchors.verticalCenter: parent.verticalCenter
                        horizontalAlignment: (territoryId === 0 && column === 1) ? Qt.AlignRight : Qt.AlignLeft
                        text: {
                            if (column === 0)
                                (territoryId == 0) ? title : territory.territoryNumber
                            else if (column === 1)
                                (territoryId == 0) ? summary : territory.locality
                            else
                                ""
                        }
                        elide: (territoryId === 0 && column === 1) ? Qt.ElideLeft : Text.ElideRight
                    }
                }

                Row {
                    id: buttonRow
                    anchors.verticalCenter: parent.verticalCenter
                    anchors.right: parent.right
                    visible: territoryId > 0 && column === 1

                    HoverHandler {
                        id: hoverHandler
                        acceptedDevices: PointerDevice.AllDevices
                        cursorShape: Qt.PointingHandCursor
                    }

                    ToolButton {
                        id: buttonDelete
                        icon.source: "qrc:/icons/delete.svg"
                        z: 2
                        visible: canEditTerritories && hoverHandler.hovered
                        onClicked: {
                            if (territoryId >= 0)
                                territoryTreeModel.removeTerritory(territoryId);
                        }
                    }
                    ToolButton {
                        id: buttonEdit
                        icon.source: hoverHandler.hovered ? "qrc:/icons/edit.svg" : "qrc:/icons/more.svg"
                        z: 2
                        visible: canEditTerritories
                        onClicked: {
                            if (territoryId >= 0) {
                                currentTerritory = territoryTreeModel.findTerritory(territoryId);
                                if (!sidebar.isVisible)
                                    switchSidePanel();
                            }
                        }
                    }
                }
            }

            background: Rectangle {
                color: itemDelegate.row === itemDelegate.treeView.currentRow
                       ? myPalette.highlight
                       : (itemDelegate.treeView.alternatingRows && itemDelegate.row % 2 !== 0
                          ? myPalette.alternateBase
                          : myPalette.base)
            }

            onClicked: {
                if (territoryId > 0) {
                    currentTerritory = territory;
                }
            }

            onDoubleClicked: {
                if (territoryId > 0) {
                    zoomSelectedTerritory();
                }
            }
        }
    }

    Component {
        id: territoriesHeaderDelegate

        ToolButton {
            id: itemDelegate

            text: model.display
            checkable: false
            icon.source: ((column === 0 && territoriesProxyModel.sortRole.toString() === "territoryNumber")
                          || (column === 1 && territoriesProxyModel.sortRole.toString() === "locality"))
                         ? (territoriesProxyModel.sortOrder === Qt.AscendingOrder
                            ? "qrc:/icons/arrow_drop_down.svg"
                            : "qrc:/icons/arrow_drop_up.svg")
                         : ""
            focusPolicy: Qt.NoFocus
            onClicked: {
                if (column === 0)
                    territoriesProxyModel.sortRole = "territoryNumber"
                if (column === 1)
                    territoriesProxyModel.sortRole = "locality"

                territoriesProxyModel.sortOrder = territoriesProxyModel.sortOrder === Qt.AscendingOrder
                        ? Qt.DescendingOrder
                        : Qt.AscendingOrder;
            }
        }
    }

    Component {
        id: territoriesSectionDelegate

        Pane {
            id: territoriesSection
            width: ListView.view.width
            background: Rectangle {
                anchors.fill: parent
                color: TBStyle.primaryColor
            }
            SystemPalette { id: myPalette; colorGroup: SystemPalette.Active }
            Label {
                width: parent.width
                anchors.centerIn: parent
                text: switch(territoriesSection.ListView.view.section.property) {
                      case "assignmentFrequencyRange": return SPScripts.formatRating(section)
                      case "timeRange": return SPScripts.formatTimeRange(section)
                      case "theme": return SPScripts.formatTheme(section)
                      case "weeksIdle": return SPScripts.formatWeeksIdle(section)
                      default: return section;
                      }
                horizontalAlignment: Qt.AlignHCenter
                verticalAlignment: Qt.AlignVCenter
                font: TBStyle.bodyMediumFont
                color: TBStyle.onPrimaryColor
                elide: territoriesSection.ListView.view.section.property === "theme" ? Text.ElideMiddle : Text.ElideNone
            }
        }
    }

    Component.onCompleted: {
        territoryTreeView.groupByIndex = settings.territories_groupByIndex;
        territoryTreeView.groupBySortOrder = settings.territories_groupBySortOrder;
        territoryTreeView.sortOrder = settings.territories_sortOrder;

        // set binding to only display data from the current territory
        addressProxyModel.filterTerritoryId = Qt.binding(function() { return currentTerritory?.territoryId ?? 0; });
        streetProxyModel.filterTerritoryId = Qt.binding(function() { return currentTerritory?.territoryId ?? 0; });

        streetModel.loadStreets();
        addressModel.loadAddresses();
    }

    Component.onDestruction: {
        settings.territories_groupByIndex = territoryTreeView.groupByIndex;
        settings.territories_groupBySortOrder = territoryTreeView.groupBySortOrder;
        settings.territories_sortOrder = territoryTreeView.sortOrder;
        settings.sync();

        // save current changes
        if (currentTerritory)
            currentTerritory.save();
        assignmentModel.saveAssignments();
        streetModel.saveStreets();
        addressModel.saveAddresses();
    }
}
