import QtCore
import QtQuick
import QtQuick.Window
import QtQuick.Controls
import QtQuick.Layouts
import net.theocbase 1.0
import "controls"
import "sidePanelScripts.js" as SPScripts

Page {
    id: assignmentPage

    // PROPERTY DECLARATIONS
    property var assignment
    property PersonsModel publisherListModel: assignmentModel.getPublisherList()
    property bool isPublisherLoading: true
    property bool isDateLoading: true
    property string dateFormat: Qt.locale().dateFormat(Locale.ShortFormat)

    signal pageCanceled()

    // OBJECT PROPERTIES
    header: RowLayout {
        ToolButton {
            Layout.alignment: Qt.AlignRight
            icon.source: "qrc:/icons/chevron_left.svg"
            onClicked: {
                assignmentModel.saveAssignments();
                // reload list due to dependencies to other rows
                // when validating the time ranges of assignments
                assignmentModel.loadAssignments(territoryManager.currentTerritory.territoryId)
                assignmentEditStack.pop();
            }
        }
        Label {
            Layout.fillWidth: true
            text: qsTr("Edit territory assignment")
            font: TBStyle.titleSmallFont
        }
        ToolButton {
            Layout.alignment: Qt.AlignRight
            icon.source: "qrc:/icons/close.svg"
            onClicked: {
                assignmentEditStack.pop();
                pageCanceled();
            }
        }
    }

    Component.onCompleted: {
        publisherListProxyModel.sort(0, Qt.AscendingOrder);
    }

    onAssignmentChanged: {
        if (assignment) {
            if (!isNaN(assignment.assignedDate.getTime()))
                assignedDateEdit.selectedDate = assignment.assignedDate;
            if (!isNaN(assignment.completedDate.getTime()))
                completedDateEdit.selectedDate = assignment.completedDate;
            assigneeLookupComboBox.currentIndex = assigneeLookupComboBox.indexOfValue(assignment.personId);
            // var modelIndex = publisherListModel.index(publisherListModel.getIndex(assignment.personId), 0);
            // var proxyIndex = publisherListProxyModel.mapFromSource(modelIndex);
            // assigneeLookupComboBox.currentIndex = proxyIndex.row;
        } else {
            assigneeLookupComboBox.currentIndex = -1;
        }
    }

    // CHILD OBJECTS
    TerritoryAssignmentValidator {
        id: publisherValidator
        model: assignmentModel
        assignmentId: assignment ? assignment.id : 0
        role: TerritoryAssignmentModel.PersonFullNameRole
        onErrorChanged: function(error) { publisherErrorLabel.text = error }
        onAssignmentIdChanged: {
            isPublisherLoading = true;
            isPublisherLoading = false;
        }
    }
    TerritoryAssignmentValidator {
        id: assignedDateValidator
        model: assignmentModel
        assignmentId: assignment ? assignment.id : 0
        role: TerritoryAssignmentModel.AssignedDateRole
        onErrorChanged: function(error) { assignedDateErrorLabel.text = error }
        onAssignmentIdChanged: {
            isDateLoading = true;
            isDateLoading = false;
        }
    }
    TerritoryAssignmentValidator {
        id: completedDateValidator
        model: assignmentModel
        assignmentId: assignment ? assignment.id : 0
        role: TerritoryAssignmentModel.CompletedDateRole
        onErrorChanged: function(error) { completedDateErrorLabel.text = error }
        onAssignmentIdChanged: {
            isDateLoading = true;
            isDateLoading = false;
        }
    }

    PersonsSFProxyModel {
        id: publisherListProxyModel
        source: publisherListModel
    }

    Pane {
        anchors.fill: parent
        Column {
            anchors.fill: parent
            spacing: 5

            RowLayout {
                width: parent.width
                InterimIcon {
                    Layout.alignment: Qt.AlignLeft | Qt.AlignVCenter
                    Layout.margins: 4
                    source: "qrc:/icons/person.svg"
                    ToolTip.text: qsTr("Publisher")
                    ToolTip.visible: hovered
                }
                LookupComboBox {
                    id: assigneeLookupComboBox

                    Layout.fillWidth: true

                    showSearchField: true
                    showFilterControls: false
                    showGroupControls: false
                    groupByIndex: -1
                    showEditButton: false
                    isEditing: false

                    model: publisherListProxyModel

                    textRole: "personFullName"
                    valueRole: "personId"
                    displayText: currentIndex === -1
                                 ? assignment.personFullName
                                 : currentText

                    validator: isPublisherLoading ? null : publisherValidator

                    sectionProperty: "personFullName"
                    sectionCriteria: ViewSection.FirstCharacter
                    sectionDelegate: Pane {
                        id: publisherSection

                        width: assigneeLookupComboBox.width
                        background: Rectangle {
                            anchors.fill: parent
                            color: TBStyle.primaryColor
                        }

                        Label {
                            width: parent.width
                            anchors.centerIn: parent
                            text: section
                            horizontalAlignment: Qt.AlignHCenter
                            verticalAlignment: Qt.AlignVCenter
                            font: TBStyle.bodyMediumFont
                            color: TBStyle.onPrimaryColor
                            elide: Text.ElideMiddle
                        }
                    }

                    // When an item is selected, update the backend.
                    onActivated: {
                        var modelIndex = assignmentModel.getAssignmentIndex(assignment.id);
                        assignmentModel.setData(modelIndex, currentValue, TerritoryAssignmentModel.PersonIdRole);
                        // trigger revalidation by switching isLoaded value
                        isPublisherLoading = true;
                        isPublisherLoading = false;
                    }
                    // Set the initial currentIndex to the value stored in the backend.
                    Component.onCompleted: {
                        if (assignment) {
                            // var modelIndex = publisherListModel.index(publisherListModel.getIndex(assignment.personId), 0);
                            // var proxyIndex = publisherListProxyModel.mapFromSource(modelIndex);
                            // currentIndex = proxyIndex.row;
                            currentIndex = indexOfValue(assignment.personId)
                        }
                    }

                    onSearchTextChanged: text => publisherListProxyModel.filterText = text
                }
            }
            Label {
                id: publisherErrorLabel
                font: TBStyle.bodySmallFont
                color: TBStyle.alertColor
                visible: text
            }
            RowLayout {
                width: parent.width
                InterimIcon {
                    Layout.alignment: Qt.AlignLeft | Qt.AlignVCenter
                    Layout.margins: 4
                    source: "qrc:/icons/event_upcoming.svg"
                    ToolTip.text: qsTr("Assigned", "Territory assignment")
                    ToolTip.visible: hovered
                }
                DateEdit {
                    id: assignedDateEdit
                    Layout.fillWidth: true
                    text: assignment?.assignedDate.toLocaleDateString(Qt.locale(), Locale.ShortFormat)
                    placeholderText: dateFormat
                    // inputMask: dateFormat.replace(/[./-]/g, "[$&]").replace(/(dd|d|MM|M)/g, "00").replace(/(yy)/g, "00")
                    isClearButtonVisible: true
                    validator: isDateLoading ? null : assignedDateValidator

                    onTextEdited: {
                        var newDate = Date.fromLocaleString(Qt.locale(), assignedDateEdit.text, dateFormat);
                        if (!isNaN(newDate)) {
                            assignedDateEdit.selectedDate = newDate;
                        }
                    }

                    onSelectedDateChanged: {
                        if (assignment) {
                            var modelIndex = assignmentModel.getAssignmentIndex(assignment.id);
                            assignmentModel.setData(modelIndex, selectedDate, TerritoryAssignmentModel.AssignedDateRole);
                        }
                        isDateLoading = true;
                        isDateLoading = false;
                    }

                    onClearTriggered: {
                        var modelIndex = assignmentModel.getAssignmentIndex(assignment.id);
                        assignmentModel.setData(modelIndex, undefined, TerritoryAssignmentModel.AssignedDateRole);
                        isDateLoading = true;
                        isDateLoading = false;
                    }
                }
            }
            Label {
                id: assignedDateErrorLabel
                font: TBStyle.bodySmallFont
                color: TBStyle.alertColor
                visible: text
            }
            RowLayout {
                width: parent.width
                InterimIcon {
                    Layout.alignment: Qt.AlignLeft | Qt.AlignVCenter
                    Layout.margins: 4
                    source: "qrc:/icons/event_available.svg"
                    ToolTip.text: qsTr("Completed", "Territory assignment")
                    ToolTip.visible: hovered
                }
                DateEdit {
                    id: completedDateEdit
                    Layout.fillWidth: true
                    text: assignment?.completedDate.toLocaleDateString(Qt.locale(), Locale.ShortFormat)
                    placeholderText: dateFormat
                    // inputMask: dateFormat.replace(/[./-]/g, "[$&]").replace(/(dd|d|MM|M)/g, "00").replace(/(yy)/g, "00")
                    isClearButtonVisible: true
                    validator: isDateLoading ? null : completedDateValidator

                    onTextEdited: {
                        var newDate = Date.fromLocaleString(Qt.locale(), completedDateEdit.text, dateFormat);
                        if (!isNaN(newDate)) {
                            completedDateEdit.selectedDate = newDate;
                        }
                    }

                    onSelectedDateChanged: {
                        if (assignment) {
                            var modelIndex = assignmentModel.getAssignmentIndex(assignment.id);
                            assignmentModel.setData(modelIndex, selectedDate, TerritoryAssignmentModel.CompletedDateRole);
                        }
                        isDateLoading = true;
                        isDateLoading = false;
                    }

                    onClearTriggered: {
                        var modelIndex = assignmentModel.getAssignmentIndex(assignment.id);
                        assignmentModel.setData(modelIndex, undefined, TerritoryAssignmentModel.CompletedDateRole);
                        isDateLoading = true;
                        isDateLoading = false;
                    }
                }
            }
            Label {
                id: completedDateErrorLabel
                font: TBStyle.bodySmallFont
                color: TBStyle.alertColor
                visible: text
            }
        }
    }
}
