import QtCore
import QtQuick
import QtQuick.Window
import QtQuick.Controls
import QtQuick.Layouts
import net.theocbase 1.0
import "controls"
import "sidePanelScripts.js" as SPScripts

Page {
    id: addressPage

    // PROPERTY DECLARATIONS
    property var address
    property bool isValidationTextVisible: false
    property bool isStreetNameLoading: true
    property bool isAddressTypeLoading: true
    property bool isHouseNumberLoading: true
    property bool isNameLoading: true

    signal pageCanceled()

    // OBJECT PROPERTIES
    header: RowLayout {
        ToolButton {
            Layout.alignment: Qt.AlignRight
            icon.source: "qrc:/icons/chevron_left.svg"
            onClicked: {
                addressModel.saveAddresses();
                addressesEditStack.pop();
            }
        }
        Label {
            Layout.fillWidth: true
            text: qsTr("Edit territory address")
            font: TBStyle.titleSmallFont
        }
        ToolButton {
            Layout.alignment: Qt.AlignRight
            icon.source: "qrc:/icons/close.svg"
            onClicked: {
                addressesEditStack.pop();
                pageCanceled();
            }
        }
    }

    Component.onCompleted: {
        addressTypeListProxyModel.sort(0, Qt.AscendingOrder);
    }

    onAddressChanged: {
        if (address) {
            addressTypeComboBox.currentIndex = addressTypeComboBox.indexOfValue(address.addressTypeId)
        } else {
            addressTypeComboBox.currentIndex = -1;
        }
    }

    // CHILD OBJECTS
    TerritoryAddressValidator {
        id: streetNameValidator
        model: addressModel
        addressId: address ? address.id : 0
        role: TerritoryAddressModel.StreetRole
        onErrorChanged: function(error) { streetNameErrorLabel.text = error }
        onAddressIdChanged: {
            isValidationTextVisible = false;
            isStreetNameLoading = true;
            isStreetNameLoading = false;
        }
    }
    TerritoryAddressValidator {
        id: addressTypeValidator
        model: addressModel
        addressId: address ? address.id : 0
        role: TerritoryAddressModel.AddressTypeNumberRole
        onErrorChanged: function(error) { addressTypeErrorLabel.text = error }
        onAddressIdChanged: {
            isValidationTextVisible = false;
            isAddressTypeLoading = true;
            isAddressTypeLoading = false;
        }
    }
    TerritoryAddressValidator {
        id: houseNumberValidator
        model: addressModel
        addressId: address ? address.id : 0
        role: TerritoryAddressModel.HouseNumberRole
        onErrorChanged: function(error) { houseNumberErrorLabel.text = error }
        onAddressIdChanged: {
            isValidationTextVisible = false;
            isHouseNumberLoading = true;
            isHouseNumberLoading = false;
        }
    }
    TerritoryAddressValidator {
        id: nameValidator
        model: addressModel
        addressId: address ? address.id : 0
        role: TerritoryAddressModel.NameRole
        onErrorChanged: function(error) { nameErrorLabel.text = error }
        onAddressIdChanged: {
            isValidationTextVisible = false;
            isNameLoading = true;
            isNameLoading = false;
        }
    }

    DataObjectListSFProxyModel {
        id: addressTypeListProxyModel
        source: addressTypeListModel
    }

    Pane {
        anchors.fill: parent
        padding: 10

        Column {
            id: delegateContentLayout2
            width: parent.width

            RowLayout {
                width: parent.width
                InterimIcon {
                    Layout.alignment: Qt.AlignVCenter | Qt.AlignLeft
                    Layout.margins: 4
                    source: "qrc:/icons/road.svg"
                    ToolTip.text: qsTr("Street", "Street name")
                    ToolTip.visible: hovered
                }
                ValidationTextField {
                    id: streetNameTextField
                    Layout.fillWidth: true
                    Layout.alignment: Qt.AlignTop
                    Layout.margins: 0
                    horizontalAlignment: Text.AlignLeft
                    verticalAlignment: Text.AlignVCenter

                    text: address ? address.street : ""
                    wrapMode: TextInput.WordWrap
                    validator: isStreetNameLoading ? null : streetNameValidator

                    onTextEdited: {
                        var modelIndex = addressModel.getAddressIndex(address.id);
                        addressModel.setData(modelIndex, text, TerritoryAddressModel.StreetRole);
                        addressModel.saveAddresses();
                        // trigger revalidation by switching isLoaded value
                        isStreetNameLoading = true;
                        isStreetNameLoading = false;
                    }
                }
                InterimIcon {
                    Layout.alignment: Qt.AlignVCenter | Qt.AlignLeft
                    Layout.margins: 4
                    source: "qrc:/icons/home.svg"
                    ToolTip.text: qsTr("No.", "House or street number")
                    ToolTip.visible: hovered
                }
                ValidationTextField {
                    id: houseNumberTextField
                    Layout.fillWidth: true
                    Layout.alignment: Qt.AlignTop | Qt.AlignRight
                    Layout.margins: 0
                    horizontalAlignment: Text.AlignRight
                    verticalAlignment: Text.AlignVCenter

                    text: address ? address.houseNumber : ""
                    wrapMode: TextInput.WordWrap
                    validator: isHouseNumberLoading ? null : houseNumberValidator

                    onTextEdited: {
                        var modelIndex = addressModel.getAddressIndex(address.id);
                        addressModel.setData(modelIndex, text, TerritoryAddressModel.HouseNumberRole);
                        addressModel.saveAddresses();
                        // trigger revalidation by switching isLoaded value
                        isHouseNumberLoading = true;
                        isHouseNumberLoading = false;
                    }
                }
            }
            Label {
                id: streetNameErrorLabel
                font: TBStyle.bodySmallFont
                color: TBStyle.alertColor
                visible: text
            }
            Label {
                id: houseNumberErrorLabel
                font: TBStyle.bodySmallFont
                color: TBStyle.alertColor
                visible: text
            }

            RowLayout {
                width: parent.width
                InterimIcon {
                    Layout.alignment: Qt.AlignVCenter | Qt.AlignLeft
                    Layout.margins: 4
                    source: "qrc:/icons/person.svg"
                    ToolTip.text: qsTr("Name", "Name of person or building")
                    ToolTip.visible: hovered
                }
                ValidationTextField {
                    id: nameTextField
                    Layout.fillWidth: true
                    Layout.alignment: Qt.AlignTop | Qt.AlignLeft
                    Layout.margins: 0
                    horizontalAlignment: Text.AlignLeft
                    verticalAlignment: Text.AlignVCenter

                    text: address ? address.name : ""
                    wrapMode: TextInput.WordWrap
                    validator: isNameLoading ? null : nameValidator

                    onTextEdited: {
                        var modelIndex = addressModel.getAddressIndex(address.id);
                        addressModel.setData(modelIndex, text, TerritoryAddressModel.NameRole);
                        // trigger revalidation by switching isLoaded value
                        isNameLoading = true;
                        isNameLoading = false;
                    }
                }
            }
            Label {
                id: nameErrorLabel
                font: TBStyle.bodySmallFont
                color: TBStyle.alertColor
                visible: text
            }

            RowLayout {
                width: parent.width
                InterimIcon {
                    Layout.alignment: Qt.AlignVCenter | Qt.AlignLeft
                    Layout.margins: 4
                    source: "qrc:/icons/category.svg"
                    ToolTip.text: qsTr("Type", "Type of address")
                    ToolTip.visible: hovered
                }
                ComboBox {
                    id: addressTypeComboBox
                    Layout.fillWidth: true
                    Layout.alignment: Qt.AlignVCenter | Qt.AlignRight
                    Layout.margins: 0

                    model: addressTypeListProxyModel

                    textRole: "name"
                    valueRole: "id"

                    validator: isAddressTypeLoading ? null : addressTypeValidator

                    // When an item is selected, update the backend.
                    onActivated: {
                        var modelIndex = addressModel.getAddressIndex(address.id);
                        addressModel.setData(modelIndex, currentValue, TerritoryAddressModel.AddressTypeNumberRole);
                        // trigger revalidation by switching isLoaded value
                        isAddressTypeLoading = true;
                        isAddressTypeLoading = false;
                    }
                    // Set the initial currentIndex to the value stored in the backend.
                    Component.onCompleted: {
                        if (address) {
                            currentIndex = indexOfValue(address.addressTypeNumber)
                        }
                    }
                }
            }
            Label {
                id: addressTypeErrorLabel
                font: TBStyle.bodySmallFont
                color: TBStyle.alertColor
                visible: text
            }
        }
    }
}
