import QtCore
import QtQuick
import QtQuick.Controls
import QtQuick.Dialogs
import QtQuick.Layouts
import QtQuick.Window
import net.theocbase 1.0
import "controls"
import "sidePanelScripts.js" as SPScripts

Page {
    // PROPERTY DECLARATIONS

    // SIGNAL DECLARATIONS

    // JAVASCRIPT FUNCTIONS
    function streetListReceived(streetResults) {
        territories.onStreetListReceived.disconnect(streetListReceived);
        territories.onStreetRequestFailed.disconnect(streetRequestFailed);
        streetResultModel.removeRows(0, streetResultModel.rowCount());
        busyIndicator.running = false;
        if (streetResults.length > 0) {
            for (var i=0; i<streetResults.length; i++)
                streetResultModel.addStreetResult(streetResults[i]);
        }
        else
        {
            streetRequestMessage.title = qsTr("Add streets", "Add streets to a territory");
            streetRequestMessage.text = qsTr("No streets found.", "Add streets to a territory");
            streetRequestMessage.visible = true;
            return;
        }
    }

    function streetRequestFailed(errorMessage) {
        territories.onStreetListReceived.disconnect(streetListReceived);
        territories.onStreetRequestFailed.disconnect(streetRequestFailed);
        busyIndicator.running = false;
        streetRequestMessage.title = qsTr("Add streets", "Add streets to a territory");
        streetRequestMessage.text = errorMessage;
        streetRequestMessage.visible = true;
        return;
    }

    function copyStreetResultsToClipboard() {
        var streetNames = [];
        for( var i = 0; i < streetResultSFProxyModel.rowCount(); i++ ) {
            var currProxyIndex = streetResultSFProxyModel.index(i, 0);
            var currIndex = streetResultSFProxyModel.mapToSource(currProxyIndex);
            var streetName = streetResultModel.get(currIndex.row).streetName;
            streetNames.push(streetName);
        }
        shareUtils.copyToClipboard(streetNames.join('\n'));
    }

    function resetDefaultSettings() {
        settings.territoryStreetResult_sortOrder = Qt.AscendingOrder;
        streetResultLookupControl.sortOrder = settings.territoryStreetResult_sortOrder;
        settings.territoryStreetResult_hideAlreadyAdded = true;
        streetResultLookupControlMoreMenu.hideAlreadyAddedStreets = settings.territoryStreetResult_hideAlreadyAdded;
        settings.sync();
    }

    // OBJECT PROPERTIES
    StackView.onActivating: {
        busyIndicator.running = true;
        streetResultModel.removeRows(0, streetResultModel.rowCount());
        streetResultLookupControl.sortOrder = settings.territoryStreetResult_sortOrder;
        streetResultLookupControlMoreMenu.hideAlreadyAddedStreets = settings.territoryStreetResult_hideAlreadyAdded;

        if (territoryManager.currentTerritory)
        {
            territories.onStreetListReceived.connect(streetListReceived);
            territories.onStreetRequestFailed.connect(streetRequestFailed);
            territories.requestStreetList(territoryManager.currentTerritory.territoryId);
        }
    }

    StackView.onDeactivating: {
        settings.territoryStreetResult_sortOrder = streetResultLookupControl.sortOrder;
        settings.territoryStreetResult_hideAlreadyAdded = streetResultLookupControlMoreMenu.hideAlreadyAddedStreets;
        settings.sync();
    }

    // CHILD OBJECTS
    StreetResultModel {
        id: streetResultModel
    }
    StreetResultSFProxyModel {
        id: streetResultSFProxyModel
        source: streetResultModel
        //filterText: streetResultLookupControl.searchText
        hideAlreadyAdded: streetResultLookupControlMoreMenu.hideAlreadyAddedStreets
    }

    ShareUtils { id: shareUtils }
    Menu {
        id: streetResultLookupControlMoreMenu

        property bool hideAlreadyAddedStreets: true

        MenuItem {
            text: qsTr("Hide already added", "Lookup control settings");
            checkable: true
            checked: streetResultLookupControlMoreMenu.hideAlreadyAddedStreets
            onTriggered: streetResultLookupControlMoreMenu.hideAlreadyAddedStreets = checked
        }
        MenuItem { text: qsTr("Reset default settings", "Lookup control settings"); onTriggered: resetDefaultSettings() }
    }

    MessageDialog {
        id: streetRequestMessage
        buttons: MessageDialog.Ok
        //icon: StandardIcon.Warning
        modality: Qt.ApplicationModal
    }

    header: RowLayout {
        ToolButton {
            Layout.alignment: Qt.AlignRight
            icon.source: "qrc:/icons/chevron_left.svg"
            onClicked: {
                if (streetsTabPage.editTerritoryId > 0) {
                    var newStreetId = 0;
                    var defaultStreetTypeId = streetModel.getDefaultStreetTypeId();
                    // add selected streets
                    for (var i = 0; i < streetResultModel.rowCount(); ++i) {
                        if (streetResultModel.get(i).isChecked) {
                            var streetName = streetResultModel.get(i).streetName;
                            var wktGeometry = streetResultModel.get(i).wktGeometry;

                            newStreetId = streetModel.addStreet(streetsTabPage.editTerritoryId,
                                                                streetName, wktGeometry, defaultStreetTypeId);
                        }
                    }

                    // select last street, that was added
                    if (newStreetId > 0) {
                        var streetIndex = streetModel.getStreetIndex(newStreetId);
                        var streetProxyIndex = streetProxyModel.mapFromSource(streetIndex);
                        if (streetProxyIndex) {
                            streetLookupControl.currentIndex = streetProxyIndex.row;
                        }
                    }
                }
                streetsEditStack.pop();
            }
        }
        Label {
            Layout.fillWidth: true
            text: qsTr("Add streets")
            font: TBStyle.titleSmallFont
        }
        ToolButton {
            Layout.alignment: Qt.AlignRight
            icon.source: "qrc:/icons/close.svg"
            onClicked: {
                streetsEditStack.pop();
            }
        }
    }

    LookupControl {
        id: streetResultLookupControl

        anchors.fill: parent

        clip: true
        focus: true

        showFilterControls: true
        showGroupControls: false
        groupByIndex: -1
        showEditButton: false
        isEditing: true

        model: streetResultSFProxyModel

        moreMenu: streetResultLookupControlMoreMenu

        delegate: Component {
            id: streetResultItemDelegate

            MenuItem {
                id: itemDelegate
                width: LookupControl.view.width

                function updateStreetSelection(isChecked) {
                    var currProxyIndex = streetResultSFProxyModel.index(index, 0);
                    var currIndex = streetResultSFProxyModel.mapToSource(currProxyIndex);
                    streetResultModel.setData(currIndex, isChecked, StreetResultModel.IsCheckedRole);
                }

                onClicked: {
                    LookupControl.view.currentIndex = index;
                    updateStreetSelection(!isSelectedCheckBox.checked);
                }

                contentItem: ColumnLayout {
                    width: parent.width
                    RowLayout {
                        Layout.fillWidth: true

                        CheckBox {
                            id: isSelectedCheckBox
                            checked: isChecked
                            Layout.alignment: Qt.AlignLeft
                            width: 80
                            padding: 0

                            onClicked: {
                                updateStreetSelection(checked);
                                // restore binding, since the checked property was set to a
                                // static value (true or false) when the CheckBox was clicked
                                checked = Qt.binding(function() { return isChecked; });
                            }
                        }

                        Item {
                            Layout.fillWidth: true
                            Layout.alignment: Qt.AlignLeft | Qt.AlignVCenter

                            Label {
                                id: streetNameText
                                width: parent.width
                                anchors.centerIn: parent
                                text: streetName
                                elide: Text.ElideRight
                            }
                        }
                    }
                }
                background: Rectangle {
                    color: streetResultItemDelegate.down || streetResultItemDelegate.hovered
                           ? Qt.tint(index % 2 == 0
                                     ? myPalette.base
                                     : myPalette.alternateBase,
                                     Qt.rgba(myPalette.highlight.r,
                                             myPalette.highlight.g,
                                             myPalette.highlight.b,
                                             streetResultItemDelegate.down ? 1.0 : 0.2))
                           : index % 2 == 0
                             ? myPalette.base
                             : myPalette.alternateBase
                }
            }
        }

        section.property: "streetName"
        section.criteria: ViewSection.FirstCharacter
        section.delegate: Pane {
            id: streetResultSection

            width: LookupControl.view.width
            background: Rectangle {
                anchors.fill: parent
                color: TBStyle.primaryColor
            }

            Label {
                width: parent.width
                anchors.centerIn: parent
                text: section
                horizontalAlignment: Qt.AlignHCenter
                verticalAlignment: Qt.AlignVCenter
                font: TBStyle.bodyMediumFont
                color: TBStyle.onPrimaryColor
                elide: Text.ElideMiddle
            }
        }

        footer: Pane {
            width: LookupControl.view.width;
            z: 3

            RowLayout {
                anchors.right: parent.right
                Label {
                    verticalAlignment: Text.AlignVCenter
                    //: Number of rows in a list
                    //~ Context the '%1'-part will be replaced by the actual number
                    text: qsTr("Count=%1", "Lookup control").arg(streetResultLookupControl.count)
                }

                ToolButton {
                    id: copyToClipboardMenuButton
                    Layout.alignment: Qt.AlignRight | Qt.AlignVCenter
                    enabled: streetResultLookupControl.count > 0
                    ToolTip.visible: hovered
                    ToolTip.delay: Qt.styleHints.mousePressAndHoldInterval
                    ToolTip.text: qsTr("Copy to clipboard", "Lookup control")
                    icon.source: "qrc:///icons/content_copy.svg"

                    onClicked: copyStreetResultsToClipboard()
                }

                ToolButton {
                    id: streetMoreMenuButton
                    Layout.alignment: Qt.AlignRight | Qt.AlignVCenter
                    visible: streetResultLookupControl.moreMenu
                    ToolTip.visible: hovered
                    ToolTip.delay: Qt.styleHints.mousePressAndHoldInterval
                    ToolTip.text: qsTr("Settings", "Lookup control")
                    icon.source: "qrc:///icons/more.svg"

                    onClicked: streetResultLookupControl.moreMenu.popup()
                }
            }
        }

        onSearchTextChanged: text => streetResultSFProxyModel.filterText = text

        onSortOrderChanged: {
            settings.territoryStreetResult_sortOrder = sortOrder;
            streetResultSFProxyModel.sort(0, sortOrder);
        }
        ScrollBar.vertical: ScrollBar {}
    }

    BusyIndicator {
        id: busyIndicator
        anchors.centerIn: parent
        running: true
    }

    // STATES

    // TRANSITIONS
}
