import QtCore
import QtQuick
import QtQuick.Controls
import QtQuick.Layouts
import QtQuick.Dialogs
import net.theocbase 1.0
import "controls"
import "sidePanelScripts.js" as SPScripts

PersonsForm {
    id: personsForm

    // PROPERTY DECLARATIONS
    property bool canViewCongregationSettings: accessControl && accessControl.user && accessControl.user.hasPermission(PermissionRule.CanViewCongregationSettings)
    property bool canEditCongregationSettings: accessControl && accessControl.user && accessControl.user.hasPermission(PermissionRule.CanEditCongregationSettings)
    property bool canViewPublishers: accessControl && accessControl.user && accessControl.user.hasPermission(PermissionRule.CanViewPublishers)
    property bool canEditPublishers: accessControl && accessControl.user && accessControl.user.hasPermission(PermissionRule.CanEditPublishers)
    property bool canViewStudentData: accessControl && accessControl.user && accessControl.user.hasPermission(PermissionRule.CanViewStudentData)
    property bool canEditStudentData: accessControl && accessControl.user && accessControl.user.hasPermission(PermissionRule.CanEditStudentData)
    property bool canViewPrivileges: accessControl && accessControl.user && accessControl.user.hasPermission(PermissionRule.CanViewPrivileges)
    property bool canEditPrivileges: accessControl && accessControl.user && accessControl.user.hasPermission(PermissionRule.CanEditPrivileges)
    property bool canViewAvailabilities: accessControl && accessControl.user && accessControl.user.hasPermission(PermissionRule.CanViewAvailabilities)
    property bool canEditAvailabilities: accessControl && accessControl.user && accessControl.user.hasPermission(PermissionRule.CanEditAvailabilities)
    property bool canViewPublicSpeakers: accessControl && accessControl.user && accessControl.user.hasPermission(PermissionRule.CanViewPublicSpeakers)
    property bool canEditPublicSpeakers: accessControl && accessControl.user && accessControl.user.hasPermission(PermissionRule.CanEditPublicSpeakers)

    property date currentDate: new Date()
    property Congregation currentCongregation
    property Publisher currentPerson: null
    property var familyList
    property Family currentFamily

    // SIGNAL DECLARATIONS

    // JAVASCRIPT FUNCTIONS
    function save() {
        if (!(canEditCongregationSettings || canEditPublishers || canEditPrivileges || canEditStudentData || canEditPublicSpeakers))
            return;

        // save congregations and persons
        congregationTreeModel.save();

        // save additional data
        if (congregationDelegateModel.rootIndex.valid) {
            // person
            saveFamily();
            savePublicTalks();
        } else {
            // congregation
        }
    }

    function saveFamily() {
        if (!canEditPublishers)
            return;
        if (!currentPerson)
            return;

        if (isFamilyHeadSwitch.checked) {
            // family head
            if (!currentFamily || currentFamily.headId !== currentPerson.id) {
                family.getOrAddFamily(currentPerson.id);
            }
        } else {
            if (currentFamily && currentFamily.headId === currentPerson.id) {
                // remove family
                currentFamily.removeFamily();
            }
            // family member
            if (familyHeadComboBox.currentIndex > 0) {
                // add member
                if (!currentFamily || (currentFamily.headId !== familyList[familyHeadComboBox.currentIndex].headId)) {
                    familyList[familyHeadComboBox.currentIndex].addMember(currentPerson.id);
                }
            } else {
                // remove member
                if (currentFamily)
                    currentFamily.removeMember(currentPerson.id);
            }
        }
        familyList = family.getFamiliesVariantList();
    }

    function savePublicTalks() {
        if (!canEditPublicSpeakers)
            return;
        if (!currentPerson)
            return;
        publicTalkModel.saveSpeakersTalks(currentPerson.id);
    }

    function changePersonsCongregation(personId, newCongregationId) {
        congregationTreeModel.changeCongregation(currentPerson.id, newCongregationId);
        // switch selection in the treeview to the new congregation
        var newCongregationIndex = congregationTreeModel.getCongregationIndex(newCongregationId);
        var newCongregationProxyIndex = congregationTreeProxyModel.mapFromSource(newCongregationIndex);
        congregationDelegateModel.rootIndex = newCongregationProxyIndex;
        var currentPersonIndex = congregationTreeModel.getPersonIndex(personId);
        var personProxyIndex = congregationTreeProxyModel.mapFromSource(currentPersonIndex);
        congregationLookupControl.currentIndex = personProxyIndex.row;
        currentPerson = congregationTreeModel.data(currentPersonIndex, CongregationTreeModel.PersonRole);
    }

    function reloadHistoryList() {
        if (!currentPerson)
            return;

        var classnumber = 1;
        var meetingPart = MeetingPart.LMM_Chairman;

        historyModel.loadAssignmentDetails(currentPerson.id,
                                           isLoadingWeekendPartsCheckBox.checked,
                                           isLoadingMidweekPartsCheckBox.checked,
                                           historyDetailFilterModel.get(0).checked, // nonstudent parts
                                           historyDetailFilterModel.get(1).checked, // other assignments
                                           historyDetailFilterModel.get(2).checked, // student parts
                                           historyDetailFilterModel.get(3).checked); // assistant in student parts
        historyProxyModel.sort(0, historyLookupControl.sortOrder);
    }

    function resetDefaultHistoryLookupControlSettings() {
        historyDetailFilterModel.setProperty(0, "checked", true);
        historyDetailFilterModel.setProperty(1, "checked", true);
        historyDetailFilterModel.setProperty(2, "checked", true);
        historyDetailFilterModel.setProperty(3, "checked", true);
        settings.personsHistory_filter1 = historyDetailFilterModel.get(0).checked;
        settings.personsHistory_filter2 = historyDetailFilterModel.get(1).checked;
        settings.personsHistory_filter3 = historyDetailFilterModel.get(2).checked;
        settings.personsHistory_filter4 = historyDetailFilterModel.get(3).checked;
        settings.personsHistory_groupByIndex = 0;
        historyLookupControl.groupByIndex = settings.personsHistory_groupByIndex;
        settings.personsHistory_sortOrder = Qt.DescendingOrder;
        historyLookupControl.sortOrder = settings.personsHistory_sortOrder;
        settings.sync();
        reloadHistoryList();
    }

    // OBJECT PROPERTIES
    congregationPage.header: ColumnLayout {
        RowLayout {
            Layout.rightMargin: 10
            visible: !congregationDelegateModel.rootIndex.valid
                     && canViewPublicSpeakers
            Switch {
                id: isFilteredByCircuit
                Layout.alignment: Qt.AlignRight
                text: qsTr("Filter by circuit")
                checked: false
            }
            ComboBox {
                id: circuitFilterComboBox
                Layout.fillWidth: true
                model: congregationTreeModel.circuits
            }
        }
        RowLayout {
            visible: congregationDelegateModel.rootIndex.valid
            ToolButton {
                Layout.alignment: Qt.AlignRight
                icon.source: "qrc:/icons/chevron_left.svg"
                text: qsTr("Congregations")
                visible: canViewCongregationSettings
                onClicked: {
                    if (congregationDelegateModel.rootIndex.valid) {
                        // go back to the list of congregations
                        save();
                        congregationDelegateModel.rootIndex = undefined;
                        congregationLookupControl.currentIndex = -1;
                        currentCongregation = null;
                        currentPerson = null;
                    }
                }
            }
            Label {
                Layout.fillWidth: true
                text: (congregationDelegateModel.rootIndex.valid
                       ? congregationTreeModel.data(congregationTreeProxyModel.mapToSource(congregationDelegateModel.rootIndex), CongregationTreeModel.TitleRole)
                       : "")
            }
        }
    }

    congregationLookupControl.onAddItem: {
        save();
        if (congregationDelegateModel.rootIndex.valid) {
            // person
            var newPersonId = congregationTreeModel.addPerson(currentCongregation.congregationId);
            var currentPersonIndex = congregationTreeModel.getPersonIndex(newPersonId);
            var personProxyIndex = congregationTreeProxyModel.mapFromSource(currentPersonIndex);
            congregationLookupControl.currentIndex = personProxyIndex.row;
            currentPerson = congregationTreeModel.data(currentPersonIndex, CongregationTreeModel.PersonRole);
            if (!currentCongregation.isLocalCongregation && canEditPublicSpeakers) {
                // If public talk coordinator adds a person, it should be a speaker
                currentPerson.gender = Publisher.Male;
                currentPerson.servant = true;
                currentPerson.usefor += Publisher.PublicTalk;
            }
        } else {
            // congregation
            var newCongregationId = congregationTreeModel.addCongregation();
            var currentCongregationIndex = congregationTreeModel.getCongregationIndex(newCongregationId);
            var congregationProxyIndex = congregationTreeProxyModel.mapFromSource(currentCongregationIndex);
            congregationLookupControl.currentIndex = congregationProxyIndex.row;
            currentCongregation = congregationTreeModel.data(currentCongregationIndex, CongregationTreeModel.CongregationRole);
        }
    }

    congregationLookupControl.onSearchTextChanged: text => congregationTreeProxyModel.filterText = text

    onCurrentCongregationChanged: {
        currentPublicMeetingDayComboBox.currentIndex = currentCongregation?.currentPublicMeetingDayAndTime?.meetingDay
                ? currentPublicMeetingDayComboBox.indexOfValue(currentCongregation?.currentPublicMeetingDayAndTime?.meetingDay)
                : -1;
        nextPublicMeetingDayComboBox.currentIndex = currentCongregation?.nextPublicMeetingDayAndTime?.meetingDay
                ? currentPublicMeetingDayComboBox.indexOfValue(currentCongregation?.nextPublicMeetingDayAndTime?.meetingDay)
                : -1;
    }

    onCurrentPersonChanged: {
        isPersonLoading = true;
        personsCongregationComboBox.currentIndex = personsCongregationComboBox.indexOfValue(currentPerson?.congregationId);
        reloadHistoryList();
        currentFamily = family.getPersonFamily(currentPerson?.id);
        publicTalkModel.loadPublicTalks(currentPerson?.id);
        publicTalksLookupControl.isEditing = false;
        if (currentPerson) {
            unavailabilityEditor.loadUnavailabilities(currentPerson.id);
        }
        isPersonLoading = false;
    }

    onCurrentFamilyChanged: {
        var currentFamilyIndex = familyHeadComboBox.find(currentFamily?.name);
        familyHeadComboBox.currentIndex = currentFamilyIndex > 0 ? currentFamilyIndex : 0;
    }

    // congregation
    congregationNameTextField.onTextEdited: currentCongregation.congregationName = congregationNameTextField.text
    congregationNameTextField.onClearText: currentCongregation.congregationName = ""
    congregationInfoTextArea.onTextEdited: currentCongregation.info = congregationInfoTextArea.text
    congregationAddressTextArea.onTextEdited: currentCongregation.address = congregationAddressTextArea.text
    circuitTextField.onTextEdited: currentCongregation.circuit = circuitTextField.text
    circuitTextField.onClearText: currentCongregation.circuit = ""
    currentPublicMeetingDayComboBox.onActivated: currentCongregation.currentPublicMeetingDayAndTime.meetingDay = currentPublicMeetingDayComboBox.currentValue
    currentPublicMeetingTimeTextField.onTextEdited: currentCongregation.currentPublicMeetingDayAndTime.meetingTime = currentPublicMeetingTimeTextField.text
    nextPublicMeetingDayComboBox.onActivated: currentCongregation.nextPublicMeetingDayAndTime.meetingDay = nextPublicMeetingDayComboBox.currentValue
    nextPublicMeetingTimeTextField.onTextEdited: currentCongregation.nextPublicMeetingDayAndTime.meetingTime = nextPublicMeetingTimeTextField.text

    // person
    personsCongregationComboBox.onActivated: {
        // check for valid congregation id before changing the person's congregation
        if (personsCongregationComboBox.currentValue > -1) {
            var newCongregationId = personsCongregationComboBox.currentValue;
            if (historyModel.hasUpcomingPublicTalks()) {
                // get confirmation to move the person to another congregation and the scheduled talks to the To Do list
                var newCongregationIndex = congregationTreeModel.getCongregationIndex(newCongregationId);
                var newCongregation = congregationTreeModel.data(newCongregationIndex, CongregationTreeModel.CongregationRole);
                msgMovePerson.congregation = newCongregation;
                msgMovePerson.person = currentPerson;
                msgMovePerson.open();
            } else {
                changePersonsCongregation(currentPerson.id, newCongregationId);
            }
        }
    }

    firstNameTextField.onTextEdited: {
        isPersonLoading = true;
        currentPerson.firstName = firstNameTextField.text;
        isPersonLoading = false;
    }
    firstNameTextField.onClearText: {
        isPersonLoading = true;
        currentPerson.firstName = "";
        isPersonLoading = false;
    }
    lastNameTextField.onTextEdited: {
        isPersonLoading = true;
        currentPerson.lastName = lastNameTextField.text;
        isPersonLoading = false;
    }
    lastNameTextField.onClearText: {
        isPersonLoading = true;
        currentPerson.lastName = "";
        isPersonLoading = false;
    }
    phoneTextField.onTextEdited: currentPerson.phone = phoneTextField.text
    phoneTextField.onClearText: currentPerson.phone = ""
    mobileTextField.onTextEdited: currentPerson.mobile = mobileTextField.text
    mobileTextField.onClearText: currentPerson.mobile = ""
    emailTextField.onTextEdited: currentPerson.email = emailTextField.text
    emailTextField.onClearText: currentPerson.email = ""
    notesTextField.onTextEdited: currentPerson.info = notesTextField.text
    notesTextField.onClearText: currentPerson.info = ""
    isMaleRadioButton.onCheckedChanged: {
        if (currentPerson)
            currentPerson.gender = (isMaleRadioButton.checked ? Publisher.Male : Publisher.Female)
    }
    servantCheckBox.onClicked: currentPerson.servant = servantCheckBox.checked
    prayerCheckBox.onClicked: currentPerson.usefor += (prayerCheckBox.checked ? Publisher.Prayer : -Publisher.Prayer)

    midweekChairmanCheckBox.onClicked: currentPerson.usefor += (midweekChairmanCheckBox.checked ? Publisher.LMM_Chairman : -Publisher.LMM_Chairman)
    // auxiliaryClassCounselorCheckBox.onClicked: currentPerson.usefor += (auxiliaryClassCounselorCheckBox.checked ? Publisher.LMM_Chairman : -Publisher.LMM_Chairman)
    treasuresTalkCheckBox.onClicked: currentPerson.usefor += (treasuresTalkCheckBox.checked ? Publisher.LMM_TR_Talk : -Publisher.LMM_TR_Talk)
    spiritualGemsCheckBox.onClicked: currentPerson.usefor += (spiritualGemsCheckBox.checked ? Publisher.LMM_TR_SpiritualGems : -Publisher.LMM_TR_SpiritualGems)
    bibleReadingCheckBox.onClicked: currentPerson.usefor += (bibleReadingCheckBox.checked ? Publisher.LMM_TR_BibleReading : -Publisher.LMM_TR_BibleReading)
    startingConversationCheckBox.onClicked: currentPerson.usefor += (startingConversationCheckBox.checked ? Publisher.LMM_FM_StartingConversation : -Publisher.LMM_FM_StartingConversation)
    followingUpCheckBox.onClicked: currentPerson.usefor += (followingUpCheckBox.checked ? Publisher.LMM_FM_FollowingUp : -Publisher.LMM_FM_FollowingUp)
    makingDisciplesCheckBox.onClicked: currentPerson.usefor += (makingDisciplesCheckBox.checked ? Publisher.LMM_FM_MakingDisciples : -Publisher.LMM_FM_MakingDisciples)
    assistantCheckBox.onClicked: currentPerson.usefor += (assistantCheckBox.checked ? Publisher.Assistant : -Publisher.Assistant)
    explainingBeliefsCheckBox.onClicked: currentPerson.usefor += (explainingBeliefsCheckBox.checked ? Publisher.LMM_FM_ExplainingBeliefs : -Publisher.LMM_FM_ExplainingBeliefs)
    studentTalkCheckBox.onClicked: currentPerson.usefor += (studentTalkCheckBox.checked ? Publisher.LMM_FM_Talk : -Publisher.LMM_FM_Talk)
    fmDiscussionCheckBox.onClicked: currentPerson.usefor += (fmDiscussionCheckBox.checked ? Publisher.LMM_FM_Discussion : -Publisher.LMM_FM_Discussion)
    christianLivingTalkCheckBox.onClicked: currentPerson.usefor += (christianLivingTalkCheckBox.checked ? Publisher.LMM_CL_Talk : -Publisher.LMM_CL_Talk)
    cbsConductorCheckBox.onClicked: currentPerson.usefor += (cbsConductorCheckBox.checked ? Publisher.CBSConductor : -Publisher.CBSConductor)
    cbsReaderCheckBox.onClicked: currentPerson.usefor += (cbsReaderCheckBox.checked ? Publisher.CBSReader : -Publisher.CBSReader)

    allClassesRadioButton.onClicked: {
        if (currentPerson.usefor & Publisher.SchoolMain)
            currentPerson.usefor -= Publisher.SchoolMain
        if (currentPerson.usefor & Publisher.SchoolAux)
            currentPerson.usefor -= Publisher.SchoolAux
    }
    onlyMainClassRadioButton.onClicked: {
        if (currentPerson.usefor & Publisher.SchoolAux)
            currentPerson.usefor -= Publisher.SchoolAux
        currentPerson.usefor += Publisher.SchoolMain
    }
    onlyAuxClassRadioButton.onClicked: {
        if (currentPerson.usefor & Publisher.SchoolMain)
            currentPerson.usefor -= Publisher.SchoolMain
        currentPerson.usefor += Publisher.SchoolAux
    }

    weekendChairmanCheckBox.onClicked: currentPerson.usefor += (weekendChairmanCheckBox.checked ? Publisher.Chairman : -Publisher.Chairman)
    publicTalkCheckBox.onClicked: currentPerson.usefor += (publicTalkCheckBox.checked ? Publisher.PublicTalk : -Publisher.PublicTalk)
    hostCheckBox.onClicked: currentPerson.usefor += (hostCheckBox.checked ? Publisher.Hospitality : -Publisher.Hospitality)
    watchtowerConductorCheckBox.onClicked: currentPerson.usefor += (watchtowerConductorCheckBox.checked ? Publisher.WtCondoctor : - Publisher.WtCondoctor)
    watchtowerReaderCheckBox.onClicked: currentPerson.usefor += (watchtowerReaderCheckBox.checked ? Publisher.WtReader : -Publisher.WtReader)

    publicTalksLookupControl.onSearchTextChanged: text => publicTalkProxyModel.filterText = text
    publicTalksLookupControl.onGroupByIndexChanged: publicTalkProxyModel.groupByRole = publicTalksGroupByModel.get(publicTalksLookupControl.groupByIndex).value
    publicTalksLookupControl.onSortOrderChanged: publicTalkProxyModel.sort(0, publicTalksLookupControl.sortOrder)

    historyLookupControl.onSearchTextChanged: text => historyProxyModel.filterText = text
    historyLookupControl.onGroupByIndexChanged: historyProxyModel.groupByRole = historyDetailGroupByModel.get(historyLookupControl.groupByIndex).value
    historyLookupControl.onSortOrderChanged: historyProxyModel.sort(0, historyLookupControl.sortOrder)
    isLoadingMidweekPartsCheckBox.onCheckedChanged: reloadHistoryList()
    isLoadingWeekendPartsCheckBox.onCheckedChanged: reloadHistoryList()
    historyLookupControl.onFilterChanged: (index, checked) => {
        settings.setValue("personsHistory_filter" + (index + 1), checked);
        settings.sync();
        reloadHistoryList();
    }

    isBreakCheckBox.onClicked: currentPerson.usefor += (isBreakCheckBox.checked ? Publisher.IsBreak : -Publisher.IsBreak)

    Component.onCompleted: {
        historyDetailGroupByModel.remove(6); // weeks idle
        historyDetailGroupByModel.remove(2); // frequency
        historyDetailGroupByModel.setProperty(1, "value", "year");
        historyDetailGroupByModel.remove(0); // name

        historyDetailFilterModel.setProperty(0, "checked", settings.personsHistory_filter1);
        historyDetailFilterModel.setProperty(1, "checked", settings.personsHistory_filter2);
        historyDetailFilterModel.setProperty(2, "checked", settings.personsHistory_filter3);
        historyDetailFilterModel.setProperty(3, "checked", settings.personsHistory_filter4);
        historyDetailFilterModel.remove(4);
        historyLookupControl.groupByIndex = settings.personsHistory_groupByIndex;
        historyLookupControl.sortOrder = settings.personsHistory_sortOrder;
        historyLookupControl.filterModel = historyDetailFilterModel;

        publicTalksLookupControl.groupByIndex = 1;

        // setup list of weekend days
        weekendDayModel.append({'key': 6, 'dayName': Qt.locale().dayName(Qt.Saturday, Locale.LongFormat) });
        weekendDayModel.append({'key': 7, 'dayName': Qt.locale().dayName(Qt.Sunday, Locale.LongFormat) });

        // update congregation model
        congregationTreeModel.updateModel();
        //  select own congregation and display persons
        var myCongregationIndex = congregationTreeModel.getMyCongregationIndex();
        var myCongregationProxyIndex = congregationTreeProxyModel.mapFromSource(myCongregationIndex);
        congregationDelegateModel.rootIndex = myCongregationProxyIndex;
        congregationLookupControl.currentIndex = -1;

        // setup family list
        familyList = family.getFamiliesVariantList();
        // assignment history
        reloadHistoryList();
    }

    Component.onDestruction: {
        settings.personsHistory_groupByIndex = historyLookupControl.groupByIndex;
        settings.personsHistory_sortOrder = historyLookupControl.sortOrder;
        settings.sync();

        // save current changes
        save();

        // remove model binding to avoid crash in qquicklistview.cpp when leaving the page
        congregationLookupControl.model = null;
    }

    // CHILD OBJECTS
    SystemPalette { id: myPalette; colorGroup: SystemPalette.Active }

    Settings {
        id: settings
        category: "LookupControl"
        property bool personsHistory_filter1: true
        property bool personsHistory_filter2: true
        property bool personsHistory_filter3: true
        property bool personsHistory_filter4: true
        property int personsHistory_groupByIndex: 1
        property bool personsHistory_sortOrder: Qt.AscendingOrder
    }

    // congregation
    ListModel {id: weekendDayModel }
    CongregationValidator {
        id: congregationNameValidator
        model: congregationTreeModel
        congregation: currentCongregation
        field: CongregationValidator.CongregationName
        onErrorChanged: error => congregationNameErrorLabel.text = error
    }
    CongregationTreeModel {
        id: congregationTreeModel
        onModelChanged: {
            congregationTreeProxyModel.sort(1)
        }
    }
    CongregationTreeSFProxyModel {
        id: congregationTreeProxyModel
        source: congregationTreeModel
        sortCaseSensitivity: Qt.CaseInsensitive
        filterRootIndex: congregationDelegateModel.rootIndex
        filterByCircuit: isFilteredByCircuit.checked
        circuit: circuitFilterComboBox.currentText
    }

    Component {
        id: moveToCongregationPage

        Page {
            Layout.fillWidth: true
            Layout.fillHeight: true

            signal pageCanceled()

            CongregationTreeSFProxyModel {
                id: moveToCongregationSFProxyModel
                sourceModel: congregationTreeModel
            }

            header: RowLayout {
                ToolButton {
                    Layout.alignment: Qt.AlignRight
                    icon.source: "qrc:/icons/chevron_left.svg"
                    onClicked: {
                        // check for valid congregation id before changing the person's congregation
                        if (personsCongregationLookupControl.currentIndex > -1) {
                            var congregationProxyIndex = moveToCongregationSFProxyModel.index(personsCongregationLookupControl.currentIndex, 0);
                            var newCongregationIndex = moveToCongregationSFProxyModel.mapToSource(congregationProxyIndex);
                            var newCongregationId = congregationTreeModel.data(newCongregationIndex, CongregationTreeModel.CongregationIdRole);
                            if (historyModel.hasUpcomingPublicTalks()) {
                                // get confirmation to move the person to another congregation and the scheduled talks to the To Do list
                                var newCongregation = congregationTreeModel.data(newCongregationIndex, CongregationTreeModel.CongregationRole);
                                msgMovePerson.congregation = newCongregation;
                                msgMovePerson.person = currentPerson;
                                msgMovePerson.open();
                            } else {
                                changePersonsCongregation(currentPerson.id, newCongregationId);
                            }
                        }
                        congregationStack.pop();
                    }
                }
                Label {
                    Layout.fillWidth: true
                    text: qsTr("Select new congregation")
                    font: TBStyle.titleSmallFont
                }
                ToolButton {
                    Layout.alignment: Qt.AlignRight
                    icon.source: "qrc:/icons/close.svg"
                    onClicked: {
                        congregationStack.pop();
                        pageCanceled();
                    }
                }
            }

            contentItem: LookupControl {
                id: personsCongregationLookupControl

                Layout.fillWidth: true
                Layout.fillHeight: true
                clip: true
                focus: true

                showFilterControls: true
                showGroupControls: false
                groupByIndex: -1
                showEditButton: false
                isEditing: false

                model: moveToCongregationSFProxyModel

                delegate: MenuItem {
                    width: LookupControl.view.width
                    text: title
                    highlighted: LookupControl.view.currentIndex === index

                    onClicked: {
                        LookupControl.view.currentIndex = index
                    }
                }

                onSearchTextChanged: text => moveToCongregationSFProxyModel.filterText = text
            }
        }
    }

    DelegateModel {
        id: congregationDelegateModel
        model: congregationTreeProxyModel

        delegate: ItemDelegate {
            id: congregationItemDelegate

            // use required properties to avoid ambiguity when accessing these properties from within the validator
            required property int index
            required property int type
            required property int congregationId
            required property Congregation congregation
            required property int personId
            required property Publisher person

            function setCurrentItem() {
                console.log("congregationItemDelegate.onClicked");
                save();
                congregationLookupControl.focus = true;
                congregationItemDelegate.LookupControl.view.currentIndex = congregationItemDelegate.index;
                if (congregationDelegateModel.rootIndex.valid) {
                    // person
                    if (congregationItemDelegate.personId > 0) {
                        if (currentPerson !== congregationItemDelegate.person)
                            currentPerson = congregationItemDelegate.person;
                    }
                } else {
                    currentPerson = null;
                    // congregation
                    if (congregationItemDelegate.congregationId > 0) {
                        if (currentCongregation !== congregationItemDelegate.congregation)
                            currentCongregation = congregationItemDelegate.congregation;
                    }
                }
            }

            function switchToPersonsList() {
                if (congregationDelegateModel.rootIndex.valid) {
                    // if the list displays persons already, select the person
                    if (congregationItemDelegate.personId > 0) {
                        if (currentPerson !== congregationItemDelegate.person)
                            currentPerson = congregationItemDelegate.person;
                    }
                } else {
                    // go to the list of persons of the selected congregation
                    congregationDelegateModel.rootIndex = congregationTreeProxyModel.index(congregationItemDelegate.index, 0);
                    congregationLookupControl.currentIndex = -1;
                }
            }

            width: LookupControl.view.width
            highlighted: LookupControl.view.currentIndex === index

            contentItem: RowLayout {
                width: parent.width
                spacing: 6

                ValidationTextField {
                    Layout.fillWidth: true
                    Layout.alignment: Qt.AlignVCenter | Qt.AlignLeft

                    horizontalAlignment: Text.AlignLeft
                    text: congregationItemDelegate.type === CongregationTreeModel.CongregationNode
                          ? congregationItemDelegate.congregation.congregationName
                          : congregationItemDelegate.person.fullName
                    validator: CongregationValidator {
                        model: congregationTreeModel
                        congregation: congregationItemDelegate.congregation
                        person: congregationItemDelegate.person
                        field: congregationItemDelegate.type === CongregationTreeModel.CongregationNode
                               ? CongregationValidator.CongregationName
                               : CongregationValidator.FullName
                    }
                    readOnly: true
                    wrapMode: TextInput.Wrap
                    color: myPalette.text
                    background.visible: false
                    activeFocusOnPress: false
                    selectByMouse: false

                    TapHandler {
                        acceptedDevices: PointerDevice.AllDevices
                        onTapped: setCurrentItem()
                        onDoubleTapped: switchToPersonsList()
                    }
                }

                Item {
                    implicitWidth: Math.max(buttonDelete.implicitBackgroundWidth + buttonDelete.leftInset + buttonDelete.rightInset,
                                            buttonDelete.implicitContentWidth + buttonDelete.leftPadding + buttonDelete.rightPadding) * 3
                    Layout.alignment: Qt.AlignVCenter | Qt.AlignRight
                    Layout.margins: 0

                    RowLayout {
                        anchors.right: parent.right
                        anchors.verticalCenter: parent.verticalCenter
                        ToolButton {
                            id: buttonMove
                            icon.source: "qrc:/icons/location_away.svg"
                            Layout.fillWidth: true
                            Layout.alignment: Qt.AlignVCenter
                            z: 2
                            ToolTip.text: qsTr("Change congregation")
                            ToolTip.visible: hovered

                            visible: congregationItemDelegate.hovered
                                     && canEditPublishers
                                     && congregationItemDelegate.type === CongregationTreeModel.PersonNode
                                     && congregationItemDelegate.person.usefor & Publisher.PublicTalk
                            onClicked: {
                                congregationItemDelegate.setCurrentItem();
                                congregationStack.push(moveToCongregationPage);
                            }
                        }
                        ToolButton {
                            id: buttonDelete
                            icon.source: "qrc:/icons/delete.svg"
                            Layout.fillWidth: true
                            Layout.alignment: Qt.AlignVCenter
                            z: 2
                            ToolTip.text: qsTr("Delete")
                            ToolTip.visible: hovered

                            visible: congregationItemDelegate.hovered
                                     && (congregationItemDelegate.type === CongregationTreeModel.CongregationNode
                                         ? canEditCongregationSettings
                                         : (currentCongregation.isLocalCongregation
                                            ? canEditPublishers
                                            : canEditPublicSpeakers))
                            onClicked: {
                                if (congregationItemDelegate.type === CongregationTreeModel.CongregationNode) {
                                    msgRemoveCongregation.congregation = congregationItemDelegate.congregation;
                                    msgRemoveCongregation.rowIndex = congregationItemDelegate.index;
                                    msgRemoveCongregation.open();
                                } else {
                                    msgRemovePerson.person = congregationItemDelegate.person;
                                    msgRemovePerson.rowIndex = congregationItemDelegate.index;
                                    msgRemovePerson.open();
                                }
                            }
                        }
                    }
                }
            }

            onClicked: setCurrentItem()

            onDoubleClicked: switchToPersonsList()

            background: Rectangle {
                width: congregationItemDelegate.LookupControl.view.width
                color: congregationItemDelegate.index === congregationItemDelegate.LookupControl.view.currentIndex
                       ? myPalette.highlight
                       : congregationItemDelegate.down || congregationItemDelegate.hovered
                         ? Qt.tint(congregationItemDelegate.index % 2 == 0
                                   ? myPalette.base
                                   : myPalette.alternateBase,
                                   Qt.rgba(myPalette.highlight.r,
                                           myPalette.highlight.g,
                                           myPalette.highlight.b,
                                           congregationItemDelegate.down ? 1.0 : 0.2))
                         : congregationItemDelegate.index % 2 == 0
                           ? myPalette.base
                           : myPalette.alternateBase
            }
        }
        onRootIndexChanged: {
            if (rootIndex.valid) {
                currentCongregation = congregationTreeModel.data(congregationTreeProxyModel.mapToSource(rootIndex), CongregationTreeModel.CongregationRole);
            }
        }
    }

    // person
    Family{
        id: family
    }
    CongregationValidator {
        id: firstNameValidator
        model: congregationTreeModel
        congregation: currentCongregation
        person: currentPerson
        field: CongregationValidator.FirstName
        onErrorChanged: error => firstNameErrorLabel.text = error
    }
    CongregationValidator {
        id: lastNameValidator
        model: congregationTreeModel
        congregation: currentCongregation
        person: currentPerson
        field: CongregationValidator.LastName
        onErrorChanged: error => lastNameErrorLabel.text = error
    }
    PersonDetailSectionDelegate { id: personsSectionDelegate }
    PersonDetailModel {
        id: historyModel
    }
    PersonDetailSFProxyModel {
        id: historyProxyModel
        source: historyModel
        onGroupByChanged: {
            historyProxyModel.sort(0, historyLookupControl.sortOrder);
            historyLookupControl.section.property = historyDetailGroupByModel.get(historyLookupControl.groupByIndex).value;
        }
    }
    Component {
        id: historyItemDelegate

        ItemDelegate {
            id: itemDelegate

            property int labelingMode: 2
            property bool includePartsOfBothMeetings: isLoadingMidweekPartsCheckBox.checked && isLoadingWeekendPartsCheckBox.checked
            property bool displayAssigneeLabel: true

            width: LookupControl.view.width - LookupControl.view.leftMargin - LookupControl.view.rightMargin
            height: gridLayout.implicitHeight

            function formatAssignmentLabel(theme, assignmentInfo, congregationName, labelingMode, weekOfDate) {
                if (!assignmentInfo)
                    return "";
                return (includePartsOfBothMeetings
                        ? SPScripts.getMeetingIndicator(assignmentInfo.meetingType) + " "
                        : "")
                        + (labelingMode === 0
                           ? assignmentInfo.assignmentTypeName
                           : (labelingMode === 1
                              ? assignmentInfo.meetingPartName(weekOfDate)
                              : (labelingMode === 2
                                 ? theme
                                 : congregationName)))
                        + (assignmentInfo.roomNumber > 1
                           ? " [" + SPScripts.getClassName(assignmentInfo.roomNumber) + "]"
                           : "");
            }
            function formatAssigneeLabel(personFullName, assigneeFullName, volunteerFullName, assistantFullName) {
                if (!assignmentInfo || !displayAssigneeLabel)
                    return "";
                return (assigneeFullName === personFullName || volunteerFullName === personFullName ? "\u2611" : (volunteerFullName !== "" ? volunteerFullName : assigneeFullName)) +
                        (assistantFullName !== "" ? (" | " + (assistantFullName === personFullName ? "\u2611" : "(" + assistantFullName + ")")) : "");
            }

            Rectangle {
                anchors.fill: parent
                color: itemDelegate.LookupControl.view.currentIndex === index ? myPalette.highlight : (index % 2 == 0 ? myPalette.base : myPalette.alternateBase)
            }

            ColumnLayout {
                id: gridLayout
                anchors.fill: parent
                Layout.leftMargin: 5
                Layout.rightMargin: 5

                RowLayout {
                    id: themeRow
                    Layout.fillWidth: true
                    Layout.leftMargin: 10
                    Layout.rightMargin: 10
                    height: childrenRect.height

                    Item {
                        height: 20
                        Layout.fillWidth: true
                        Layout.fillHeight: true
                        Layout.alignment: Qt.AlignLeft | Qt.AlignVCenter

                        Label {
                            width: parent.width
                            verticalAlignment: Text.AlignVCenter
                            anchors.centerIn: parent
                            text: formatAssignmentLabel(theme, assignmentInfo, congregationName, labelingMode, model.date)
                            elide: Text.ElideRight
                            font.strikeout: assigneeFullName === personFullName
                                            && volunteerFullName !== ""
                            visible: theme
                        }
                    }
                    Item {
                        height: 20
                        Layout.fillWidth: true
                        Layout.fillHeight: true
                        Layout.alignment: Qt.AlignLeft | Qt.AlignVCenter
                        visible: displayAssigneeLabel

                        Label {
                            width: parent.width
                            anchors.centerIn: parent
                            horizontalAlignment: Text.AlignRight
                            text: formatAssigneeLabel(personFullName, assigneeFullName, volunteerFullName, assistantFullName)
                            elide: Text.ElideMiddle
                            visible: !(assigneeFullName === personFullName
                                       && volunteerFullName !== "")
                        }
                    }
                    Item {
                        width: 90
                        implicitWidth: 90
                        height: 20
                        Layout.alignment: Qt.AlignRight | Qt.AlignVCenter

                        Label {
                            width: parent.width
                            horizontalAlignment: Text.AlignRight
                            anchors.centerIn: parent
                            text: date.toLocaleDateString(Qt.locale(), Locale.ShortFormat)
                            elide: Text.ElideMiddle
                            font.strikeout: assigneeFullName === personFullName
                                            && volunteerFullName !== ""
                        }
                    }
                }
                RowLayout {
                    Layout.fillWidth: true
                    Layout.leftMargin: 10
                    Layout.rightMargin: 10
                    height: childrenRect.height

                    Item {
                        height: 20
                        Layout.fillWidth: true
                        Layout.fillHeight: true
                        Layout.alignment: Qt.AlignLeft | Qt.AlignVCenter

                        Label {
                            width: parent.width
                            verticalAlignment: Text.AlignVCenter
                            anchors.centerIn: parent
                            text: assignmentInfo.assignmentTypeName
                            elide: Text.ElideRight
                            font.strikeout: assigneeFullName === personFullName
                                            && volunteerFullName !== ""
                            visible: theme
                        }
                    }
                    Item {
                        height: 20
                        Layout.fillWidth: true
                        Layout.fillHeight: true
                        Layout.alignment: Qt.AlignLeft | Qt.AlignVCenter

                        Label {
                            width: parent.width
                            verticalAlignment: Text.AlignVCenter
                            anchors.centerIn: parent
                            text: assignmentInfo.meetingPart === MeetingPart.PublicTalk ? congregationName : ""
                            elide: Text.ElideRight
                        }
                    }
                    Item {
                        height: 20
                        Layout.fillWidth: true
                        Layout.fillHeight: true
                        Layout.alignment: Qt.AlignRight | Qt.AlignVCenter

                        Label {
                            width: parent.width
                            verticalAlignment: Text.AlignVCenter
                            horizontalAlignment: Text.AlignRight
                            anchors.centerIn: parent
                            text: assignmentInfo.meetingSectionName
                            elide: Text.ElideRight
                            font.strikeout: assigneeFullName === personFullName
                                            && volunteerFullName !== ""
                            visible: theme
                        }
                    }
                }
                RowLayout {
                    id: noteRow
                    Layout.fillWidth: true
                    height: 0
                    visible: canViewStudentData
                    Label {
                        id: noteText
                        text: note
                        Layout.fillWidth: true
                        elide: Text.ElideRight
                        height: 0
                    }

                    Label {
                        id: timingText
                        width: 80
                        height: 0
                        text: timing
                    }
                }
            }
        }
    }
    PersonDetailFilterModel { id: historyDetailFilterModel }
    PersonDetailGroupByModel { id: historyDetailGroupByModel }
    Component {
        id: historySectionDelegate

        Pane {
            id: historySection
            width: LookupControl.view.width
            background: Rectangle {
                anchors.fill: parent
                color: TBStyle.primaryColor
            }
            SystemPalette { id: myPalette; colorGroup: SystemPalette.Active }
            Label {
                width: parent.width
                anchors.centerIn: parent
                text: switch(historySection.ListView.view.section.property) {
                      case "themeNumber": return SPScripts.formatNumberRange(section, 10)
                      default: return section;
                      }
                horizontalAlignment: Qt.AlignHCenter
                verticalAlignment: Qt.AlignVCenter
                font: TBStyle.bodyMediumFont
                color: TBStyle.onPrimaryColor
                elide: historySection.LookupControl.view.section.property === "theme" ? Text.ElideCenter : Text.ElideNone
            }
        }
    }
    LookupControlMoreMenu {
        id: historyLookupControlMoreMenu
        meetingType: MeetingType.MidweekMeeting
        isRowCountChangingAllowed: false
        isLabelChangingAllowed: false
        isIncludingPartsOfOtherMeetingAllowed: false
        isHideUnavailablesAllowed: false
        onResetDefaultSettings: resetDefaultHistoryLookupControlSettings()
    }

    MessageDialog {
        id: msgRemoveCongregation
        property Congregation congregation
        property int rowIndex
        buttons: MessageDialog.Yes | MessageDialog.No
        text: qsTr("Do you want to remove the congregation \"%1\"?").arg(congregation?.congregationName)
        onAccepted: {
            if (currentCongregation === congregation)
                currentCongregation = null;
            congregationTreeModel.removeCongregation(congregation.congregationId);
            if (congregationLookupControl.count > 0
                    && congregationLookupControl.currentIndex >= 0) {
                var congregationProxyIndex = congregationTreeProxyModel.index(congregationLookupControl.currentIndex, 0);
                var congregationIndex = congregationTreeProxyModel.mapToSource(congregationProxyIndex);
                currentCongregation = congregationTreeModel.data(congregationIndex, CongregationTreeModel.CongregationRole);
            }
        }
    }

    MessageDialog {
        id: msgRemovePerson
        property Publisher person
        property int rowIndex
        buttons: MessageDialog.Yes | MessageDialog.No
        text: qsTr("Do you want to remove this person?\n%1").arg(person?.fullName)
        onAccepted: {
            if (currentPerson === person)
                currentPerson = null;
            congregationTreeModel.removePerson(person.id);
            if (congregationLookupControl.count > 0
                    && congregationLookupControl.currentIndex >= 0) {
                var personProxyIndex = congregationTreeProxyModel.index(congregationLookupControl.currentIndex, 0, congregationDelegateModel.rootIndex);
                var personIndex = congregationTreeProxyModel.mapToSource(personProxyIndex);
                currentPerson = congregationTreeModel.data(personIndex, CongregationTreeModel.PersonRole);
            }
        }
    }

    MessageDialog {
        id: msgMovePerson
        property Publisher person
        property Congregation congregation
        property int rowIndex
        buttons: MessageDialog.Yes | MessageDialog.No

        text: qsTr("The speaker is scheduled for outgoing talks. These talks will be moved to the To Do List if you change the congregation.\nChange congregation to '%1'?").arg(congregation?.congregationName);

        onAccepted: {
            changePersonsCongregation(person.id, congregation.congregationId);
        }
    }

    // public talk
    PublicTalkModel { id: publicTalkModel }
    PublicTalkSFProxyModel {
        id: publicTalkProxyModel
        source: publicTalkModel
        isUnselectedDisplayed: publicTalksLookupControl.isEditing
        onGroupByChanged: {
            publicTalkProxyModel.sort(0, publicTalksLookupControl.sortOrder);
            publicTalksLookupControl.section.property = publicTalksGroupByModel.get(publicTalksLookupControl.groupByIndex).value;
        }
    }

    Component {
        id: publicTalksItemDelegate

        CheckDelegate {
            id: itemDelegate
            width: LookupControl.view.width

            checked: isSelected
            nextCheckState: function() {
                if (itemDelegate.LookupControl.view.isEditing) {
                    if (checkState === Qt.Checked)
                        return Qt.Unchecked;
                    else
                        return Qt.Checked;
                }
                else {
                    return checkState;
                }
            }

            onClicked: {
                var currProxyIndex = publicTalkProxyModel.index(index, 0);
                var currIndex = publicTalkProxyModel.mapToSource(currProxyIndex);
                publicTalkModel.setData(currIndex, checked, PublicTalkModel.IsSelectedRole);

                // restore binding, since the checked property was set to a
                // static value (true or false) when the CheckBox was clicked
                checked = Qt.binding(function() { return isSelected; });
            }

            contentItem: ColumnLayout {
                PublicTalkValidator {
                    id: publicTalkValidator
                    model: itemDelegate.LookupControl.view.model.source
                    role: PublicTalkModel.None
                    talkId: publicTalkId
                    onErrorChanged: error => publicTalkErrorLabel.text = error
                }
                RowLayout {
                    Layout.fillWidth: true

                    ValidationTextField {
                        implicitWidth: 40
                        text: " " // display alert icon only
                        validator: publicTalkValidator
                        readOnly: true
                        activeFocusOnPress: false
                        selectByMouse: false
                        background: null
                    }
                    RowLayout {
                        Layout.fillWidth: true
                        opacity: isNaN(discontinueDate) ? 1 : 0.5

                        Item {
                            implicitWidth: 80
                            Layout.fillHeight: true
                            Layout.alignment: Qt.AlignLeft | Qt.AlignVCenter
                            Label {
                                id: themeNumberText
                                text: themeNumber
                                anchors.centerIn: parent
                            }
                        }
                        Item {
                            Layout.fillWidth: true
                            Layout.alignment: Qt.AlignLeft | Qt.AlignVCenter
                            Label {
                                id: themeNameText
                                width: parent.width
                                anchors.centerIn: parent
                                text: themeName
                                elide: Text.ElideRight
                            }
                        }
                        Item {
                            implicitWidth: 80
                            Layout.fillHeight: true
                            Layout.alignment: Qt.AlignLeft | Qt.AlignVCenter
                            Label {
                                id: revisionText
                                anchors.centerIn: parent
                                text: revision
                                elide: Text.ElideRight
                            }
                        }
                    }
                }
                ColumnLayout {
                    Layout.fillWidth: true

                    Label {
                        id: publicTalkErrorLabel
                        font: TBStyle.bodySmallFont
                        color: TBStyle.alertColor
                        visible: text
                    }
                }
            }
            background: Rectangle {
                color: itemDelegate.down || itemDelegate.hovered
                       ? Qt.tint(index % 2 == 0
                                 ? myPalette.base
                                 : myPalette.alternateBase,
                                 Qt.rgba(myPalette.highlight.r,
                                         myPalette.highlight.g,
                                         myPalette.highlight.b,
                                         itemDelegate.down ? 1.0 : 0.2))
                       : index % 2 == 0
                         ? myPalette.base
                         : myPalette.alternateBase
            }
        }
    }
    ListModel {
        id: publicTalksGroupByModel
        ListElement {
            key: "Theme"
            value: "alphabet"
        }
        ListElement {
            key: "Number"
            value: "themeNumberRange"
        }
        ListElement {
            key: "Language"
            value: "language"
        }
    }
    Component {
        id: publicTalksSectionDelegate

        Pane {
            id: personDetailSection
            width: ListView.view.width
            background: Rectangle {
                anchors.fill: parent
                color: TBStyle.primaryColor
            }
            SystemPalette { id: myPalette; colorGroup: SystemPalette.Active }
            Label {
                width: parent.width
                anchors.centerIn: parent
                text: switch(personDetailSection.ListView.view.section.property) {
                      case "themeNumber": return SPScripts.formatNumberRange(section, 10)
                      default: return section;
                      }
                horizontalAlignment: Qt.AlignHCenter
                verticalAlignment: Qt.AlignVCenter
                font: TBStyle.bodyMediumFont
                color: TBStyle.onPrimaryColor
                elide: personDetailSection.ListView.view.section.property === "theme" ? Text.ElideCenter : Text.ElideNone
            }
        }
    }

    // STATES

    // TRANSITIONS

}
