import QtQuick
import QtQuick.Controls
import QtQuick.Layouts
import QtQuick.Dialogs
import QtQuick.Effects
import net.theocbase 1.0
import "controls"

Pane {
    id: dialog

    LayoutMirroring.enabled: Qt.application.layoutDirection === Qt.RightToLeft
    LayoutMirroring.childrenInherit: true

    property bool canDeleteCloudData: cloud.canResetCloudData()
    property bool isDarkMode: dialog.palette.windowText.hsvValue > dialog.palette.window.hsvValue

    property int preferredHeight: childrenRect.height

    ListModel { id: defaultList }

    function loadList(showAll) {
        if (showAll) {
            listView.model = cloud.storageService.searchFile("syncfile.json")
        } else {
            listView.model = defaultList
            listView.model.clear()
            listView.model.append({"path": cloud.storageService.account.syncFile, "checked": true})
        }
        dialog.enabled = true
    }

    Component.onCompleted: {
        if (cloud.isLoggedIn)
            loadList(false)
    }

    Connections {
        target: cloud
        function onIsLoggedInChanged() {
            if (cloud.isLoggedIn) {
                loadList();
            } else {
                listView.model = null;
            }
        }
        function onSyncStarted() { dialog.enabled = false }
        function onSyncFinished() { dialog.enabled = true }
        function onCloudResetStarted() { dialog.enabled = false }
        function onCloudResetFinished() { dialog.enabled = true }
        function onError() { dialog.enabled = true }
    }

    MessageDialog {
        id: msgbox
        text: qsTr("Are you sure you want to permanently delete your cloud data?")
        buttons: MessageDialog.Yes | MessageDialog.No
        //icon: StandardIcon.Question
        modality: Qt.ApplicationModal
        onAccepted: cloud.resetCloudData()
    }

    ColumnLayout {
        id: columnLayout1

        Frame {
            id: frame1
            Layout.fillWidth: true

            ColumnLayout {
                id: columnLayout
                anchors.fill: parent

                RowLayout {
                    id: rowLayout
                    Layout.fillWidth: true

                    Rectangle {
                        Layout.fillWidth: true
                        height: 40
                        width: 200
                        color: "transparent"
                        Image {
                            id: image
                            anchors.fill: parent
                            //Layout.fillWidth: true
                            sourceSize: Qt.size(width, height)
                            fillMode: Image.PreserveAspectFit
                            source: "qrc:/icons/" + cloud.storageService.providerName + ".svg"
                        }
                        MultiEffect {
                            anchors.fill: parent
                            visible: isDarkMode
                            source: image
                            brightness: 1
                            colorizationColor: "white"
                            colorization: 1
                        }
                    }

                    ToolButton {
                        id: buttonLogin
                        text: ""
                        onClicked: cloud.isLoggedIn ? cloud.logout(2) : cloud.login()
                        icon.source: "qrc:/icons/logout.svg"
                    }
                }
                Label {
                    id: labelName
                    text: cloud.storageService.account.name ?? ""
                    Layout.fillWidth: true
                }
                Label {
                    id: labelEmail
                    text: cloud.storageService.account.email ?? ""
                    Layout.fillWidth: true
                }
            }
        }

        Frame {
            id: frame2
            Layout.fillWidth: true
            enabled: cloud.isLoggedIn

            RowLayout {
                id: rowLayout1
                anchors.fill: parent
                ListView {
                    id: listView
                    Layout.fillHeight: true
                    Layout.fillWidth: true
                    implicitHeight: childrenRect.height
                    implicitWidth: contentItem.childrenRect.width

                    model: ListModel {}
                    delegate: RowLayout {
                        CheckBox {
                            id: listCheckBox
                            enabled: buttonEditSyncFile.listLoaded
                            checked: cloud.storageService.account.syncFile === model.path
                            text: model.path
                                  + (typeof (model.sharedby) == "undefined" || model.sharedby === ""
                                     ? ""
                                     : "\n("+ model.sharedby + ")")

//                            ToolTip.visible: hovered
//                            ToolTip.text: model.path

                            onCheckedChanged: {
                                if (checked)
                                    cloud.storageService.account.syncFile = model.path
                            }
                        }
                        MouseArea {
                            Layout.fillWidth: true
                            Layout.fillHeight: true
                            hoverEnabled: true
                            id: mousearea
                        }
                        ToolTip {
                            text: model.path
                            visible: mousearea.containsMouse
                        }
                    }
                }
                ToolButton {
                    id: buttonEditSyncFile
                    Layout.alignment: Qt.AlignLeft | Qt.AlignTop
                    icon.source: listLoaded ? "qrc:/icons/chevron_up.svg" : "qrc:/icons/chevron_down.svg"
                    property bool listLoaded: false
                    onClicked: {
                        dialog.enabled = false
                        loadList(!listLoaded)
                        listLoaded = !listLoaded
                    }
                }
            }
        }

        Frame {
            id: frame3
            Layout.fillWidth: true
            enabled: cloud.isLoggedIn

            GridLayout {
                id: columnLayout2
                columns: 2
                anchors.fill: parent
                Label {
                    id: label
                    text: qsTr("Last synchronized: %1").arg(cloud.lastSyncTime.toLocaleString(Qt.locale(), Locale.ShortFormat))
                    Layout.columnSpan: 2
                    Layout.fillWidth: true
                }
                Label {
                    text: qsTr("Synchronize")
                    Layout.fillWidth: true
                }
                ToolButton {
                    id: buttonSync
                    icon.source: "qrc:/icons/cloud_sync.svg"
                    onClicked: {
                        cloud.synchronize(false)
                        console.log("sync ready")
                    }
                }

                Label {
                    text: qsTr("Delete Cloud Data")
                    Layout.fillWidth: true
                    visible: canDeleteCloudData
                }
                ToolButton {
                    id: buttonReset
                    icon.source: "qrc:/icons/cloud_delete.svg"
                    visible: canDeleteCloudData
                    onClicked: msgbox.open()
                }
            }
        }
    }
    states: [
        State {
            name: "logout"
            when: !cloud.isLoggedIn

            PropertyChanges {
                target: buttonLogin
                icon.source: "qrc:/icons/login.svg"
            }

            PropertyChanges {
                target: labelEmail
                visible: false
            }

            PropertyChanges {
                target: labelName
                visible: false
            }

            PropertyChanges {
                target: frame2
                visible: false
            }

            PropertyChanges {
                target: frame3
                visible: false
            }
        }
    ]
}
