/**
 * This file is part of TheocBase.
 *
 * Copyright (C) 2011-2018, TheocBase Development Team, see AUTHORS.
 *
 * TheocBase is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * TheocBase is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with TheocBase.  If not, see <http://www.gnu.org/licenses/>.
 */
import QtQuick
import QtQuick.Controls
import QtQuick.Dialogs
import QtLocation
import QtPositioning
import net.theocbase 1.0

Item {
    property string address

    onAddressChanged: {
        _territories.geocodeAddress(address)
    }

    Component.onCompleted: {
        var parameters = new Array();
        for (var prop in geoServiceParameters) {
            var parameter = Qt.createQmlObject('import QtLocation; PluginParameter{ name: "'+ prop + '"; value: "' + geoServiceParameters[prop] + '"}', baseMap);
            parameters.push(parameter);
        }
        baseMap.pluginParameters = parameters;
        var provider = defaultGeoServiceProvider;
        var plugin;
        if (parameters && parameters.length > 0)
            plugin = Qt.createQmlObject ('import QtLocation; Plugin{ name:"' + provider + '"; parameters: baseMap.pluginParameters}', baseMap);
        else
            plugin = Qt.createQmlObject ('import QtLocation; Plugin{ name:"' + provider + '"}', baseMap);
        baseMap.plugin = plugin;
    }

    MessageDialog {
        id: messageDialog
        buttons: MessageDialog.Ok
        //icon: StandardIcon.Warning
        modality: Qt.ApplicationModal
    }

    Territories {
        id: _territories
        onGeocodingFinished: function(geocodeResults) {
            console.log( "geocode finished " + geocodeResults.length )
            if (geocodeResults.length > 0) {
                marker.location.location = geocodeResults[0];
                baseMap.zoomLevel = 14;
                baseMap.center = marker.location.coordinate;
            } else {
                // default
                baseMap.center = QtPositioning.coordinate(0, 0)
                baseMap.zoomLevel = 1
            }
        }
        onGeocodingError: {
            messageDialog.title = qsTr("Display congregation address", "Display marker at the location of the congregation on the map");
            messageDialog.text = message;
            messageDialog.visible = true;
        }
    }

    Map {
        id: baseMap

        property variant pluginParameters

        anchors.fill: parent
        //plugin: mapPlugin
        center: QtPositioning.coordinate(0, 0)
        zoomLevel: 1

        Slider {
            id: slider
            anchors.top: parent.top
            anchors.topMargin: 10
            anchors.bottom: parent.bottom
            anchors.bottomMargin: 10
            anchors.left: parent.left
            anchors.leftMargin: 10
            orientation: Qt.Vertical
            from: baseMap.minimumZoomLevel
            to: baseMap.maximumZoomLevel
            value: baseMap.zoomLevel
            onMoved: baseMap.zoomLevel = value
        }

        onErrorChanged: {
            console.log("Error: " + errorString)
        }

        MapSettings {
            id: mapSettings
        }

        SimpleMarker{
            id: marker
            color: '#000000'
            markerType: 1
            markerScale: mapSettings.markerScale
        }
    }
}
