#include "publictalk.h"

PublicTalk::PublicTalk()
{
}

PublicTalk::PublicTalk(const int id, const int themeNumber, const QString themeName, const QString revision,
                       const QDate releaseDate, const QDate discontinueDate, const int languageId,
                       const QString language, const bool isSelected, QObject *parent)
    : QObject(parent), m_publicTalkId(id), b_themeNumber(themeNumber), b_themeName(themeName), b_revision(revision), b_releaseDate(releaseDate), b_discontinueDate(discontinueDate), b_languageId(languageId), b_language(language), b_isSelected(isSelected)
{
}

int PublicTalk::publicTalkId() const
{
    return m_publicTalkId;
}

const QString PublicTalk::themeNumberRange(int step) const
{
    return tr("%1 to %2", "Number range, e.g. 1 to 10")
            .arg(qFloor((b_themeNumber - 1) / step) * step + 1)
            .arg(qFloor((b_themeNumber - 1) / step) * step + step);
}

PublicTalkModel::PublicTalkModel(QObject *parent)
    : QAbstractTableModel(parent)
{
}

PublicTalkModel::~PublicTalkModel()
{
    qDeleteAll(publicTalks);
    publicTalks.clear();
}

int PublicTalkModel::rowCount(const QModelIndex &parent) const
{
    Q_UNUSED(parent)
    return publicTalks.empty() ? 0 : publicTalks.count();
}

int PublicTalkModel::columnCount(const QModelIndex &parent) const
{
    Q_UNUSED(parent)
    return 6;
}

QHash<int, QByteArray> PublicTalkModel::roleNames() const
{
    QHash<int, QByteArray> items;
    items[PublicTalkIdRole] = "publicTalkId";
    items[PublicTalkRole] = "publicTalk";
    items[ThemeNumberRole] = "themeNumber";
    items[ThemeNumberRangeRole] = "themeNumberRange";
    items[ThemeNameRole] = "themeName";
    items[AlphabetRole] = "alphabet";
    items[RevisionRole] = "revision";
    items[ReleaseDateRole] = "releaseDate";
    items[DiscontinueDateRole] = "discontinueDate";
    items[LanguageIdRole] = "languageId";
    items[LanguageRole] = "language";
    items[IsSelectedRole] = "isSelected";
    return items;
}

QVariant PublicTalkModel::data(const QModelIndex &index, int role) const
{
    int row = index.row();
    if (row < 0 || row > publicTalks.count())
        return QVariant();

    switch (role) {
    case PublicTalkIdRole:
        return publicTalks[index.row()]->publicTalkId();
    case PublicTalkRole:
        return QVariant::fromValue(publicTalks[index.row()]);
    case ThemeNumberRole:
        return publicTalks[index.row()]->themeNumber();
    case ThemeNumberRangeRole:
        return publicTalks[index.row()]->themeNumberRange(10);
    case ThemeNameRole:
        return publicTalks[index.row()]->themeName();
    case AlphabetRole:
        return publicTalks[index.row()]->themeName().left(1);
    case RevisionRole:
        return publicTalks[index.row()]->revision();
    case ReleaseDateRole:
        return publicTalks[index.row()]->releaseDate();
    case DiscontinueDateRole:
        return publicTalks[index.row()]->discontinueDate();
    case LanguageIdRole:
        return publicTalks[index.row()]->languageId();
    case LanguageRole:
        return publicTalks[index.row()]->language();
    case IsSelectedRole:
        return publicTalks[index.row()]->isSelected();
    default:
        return QVariant();
    }
}

bool PublicTalkModel::setData(const QModelIndex &index, const QVariant &value, int role)
{
    PublicTalk *pt = publicTalks[index.row()];

    switch (role) {
    case IsSelectedRole:
        pt->setIsSelected(value.toBool());
        emit dataChanged(index, index, { role });
        break;
    default:
        break;
    }
    return true;
}

QVariantMap PublicTalkModel::get(int row) const
{
    QHash<int, QByteArray> names = roleNames();
    QHashIterator<int, QByteArray> i(names);
    QVariantMap res;
    QModelIndex idx = index(row, 0);
    while (i.hasNext()) {
        i.next();
        QVariant data = idx.data(i.key());
        res[i.value()] = data;
    }
    return res;
}

QModelIndex PublicTalkModel::getPublicTalkIndex(int publicTalkId) const
{
    for (int row = 0; row < this->rowCount(); ++row) {
        QModelIndex rowIndex = this->index(row, 0);
        if (rowIndex.data(PublicTalkIdRole) == publicTalkId)
            return rowIndex;
    }
    return QModelIndex();
}

int PublicTalkModel::addPublicTalk(PublicTalk *publicTalk)
{
    beginInsertRows(QModelIndex(), rowCount(), rowCount());
    publicTalks << publicTalk;
    endInsertRows();
    return publicTalk->publicTalkId();
}

// load lmm publicTalks history
// param personId: load publicTalks of the given person only
//                 0 = load all person's public talks (default)
void PublicTalkModel::loadPublicTalks(int personId)
{
    beginResetModel();
    qDeleteAll(publicTalks);
    publicTalks.clear();

    sql_class *sql = &Singleton<sql_class>::Instance();

    QString sqlQuery;
    if (personId != 0)
        sqlQuery = QString("SELECT p.*, l.desc, sp.speaker_id = %1 isSelected "
                           "FROM publictalks p "
                           "JOIN languages l ON p.lang_id = l.id "
                           "LEFT JOIN speaker_publictalks sp ON p.id = sp.theme_id AND sp.speaker_id = %1 AND sp.active "
                           "AND p.active "
                           "ORDER BY p.theme_number")
                           .arg(QVariant(personId).toString());
    else
        sqlQuery = QString("SELECT p.*, l.desc, 0 isSelected "
                           "FROM publictalks p "
                           "JOIN languages l ON p.lang_id = l.id AND p.active = 1 "
                           "ORDER BY p.theme_number");

    sql_items publicTalkRows = sql->selectSql(sqlQuery);

    if (!publicTalkRows.empty()) {
        for (unsigned int i = 0; i < publicTalkRows.size(); i++) {
            sql_item s = publicTalkRows[i];

            addPublicTalk(new PublicTalk(s.value("id").toInt(),
                                         s.value("theme_number").toInt(),
                                         s.value("theme_name").toString(),
                                         s.value("revision").toString(),
                                         s.value("release_date").toDate(),
                                         s.value("discontinue_date").toDate(),
                                         s.value("lang_id").toInt(),
                                         s.value("desc").toString(),
                                         s.value("isSelected").toBool(),
                                         this));
        }
    }
    endResetModel();
}

void PublicTalkModel::saveSpeakersTalks(int speakerId)
{
    AccessControl *ac = &Singleton<AccessControl>::Instance();
    if (!ac->user() || !ac->user()->hasPermission(PermissionRule::CanEditPublicSpeakers))
        return;

    sql_class *sql = &Singleton<sql_class>::Instance();
    sql->startTransaction();

    // save
    for (int publicTalkRow = 0; publicTalkRow < this->rowCount(); ++publicTalkRow) {
        QModelIndex publicTalkIndex = this->index(publicTalkRow, 0);
        PublicTalk *publicTalk = publicTalkIndex.data(PublicTalkRole).value<PublicTalk *>();
        if (publicTalk != nullptr) {
            sql_items pe = sql->selectSql(QString("SELECT * FROM speaker_publictalks WHERE speaker_id = %1 AND lang_id = %2 AND theme_id = %3 AND active")
                                                  .arg(speakerId)
                                                  .arg(publicTalk->languageId())
                                                  .arg(publicTalk->publicTalkId()));
            if (publicTalk->isSelected()) {
                // selected
                if (pe.empty()) {
                    // add a new speaker
                    sql_item s;
                    s.insert("speaker_id", speakerId);
                    s.insert("theme_id", publicTalk->publicTalkId());
                    s.insert("lang_id", publicTalk->languageId());
                    sql->insertSql("speaker_publictalks", &s, "id");
                }
            } else {
                // no selected
                if (!pe.empty()) {
                    // remove
                    sql_item s;
                    s.insert("speaker_id", speakerId);
                    s.insert("theme_id", publicTalk->publicTalkId());
                    s.insert("lang_id", publicTalk->languageId());
                    s.insert("time_stamp", 0);
                    sql->execSql("UPDATE speaker_publictalks SET active = 0, time_stamp = :time_stamp WHERE speaker_id = :speaker_id "
                                 "AND theme_id = :theme_id AND lang_id = :lang_id",
                                 &s, true);
                }
            }
        }
    }
    sql->commitTransaction();
}

PublicTalkSortFilterProxyModel::PublicTalkSortFilterProxyModel(QObject *parent)
    : QSortFilterProxyModel(parent)
{
}

QObject *PublicTalkSortFilterProxyModel::source() const
{
    return sourceModel();
}

void PublicTalkSortFilterProxyModel::setSource(QObject *source)
{
    setSourceModel(qobject_cast<QAbstractItemModel *>(source));
    emit sourceChanged();
}

QString PublicTalkSortFilterProxyModel::filterText() const
{
    return m_filterText;
}

void PublicTalkSortFilterProxyModel::setFilterText(QString newValue)
{
    beginFilterChange();
    m_filterText = newValue;
    endFilterChange(QSortFilterProxyModel::Direction::Rows);
    emit filterTextChanged();
}

QByteArray PublicTalkSortFilterProxyModel::sortRole() const
{
    return roleNames().value(QSortFilterProxyModel::sortRole());
}

void PublicTalkSortFilterProxyModel::setSortRole(const QByteArray &role)
{
    QSortFilterProxyModel::setSortRole(roleKey(role));
    emit sortChanged();
}

QByteArray PublicTalkSortFilterProxyModel::groupByRole() const
{
    return m_groupByRole;
}

void PublicTalkSortFilterProxyModel::setGroupByRole(const QByteArray &role)
{
    beginResetModel();
    m_groupByRole = role;
    endResetModel();
    emit groupByChanged();
}

bool PublicTalkSortFilterProxyModel::isUnselectedDisplayed() const
{
    return m_isUnselectedDisplayed;
}

void PublicTalkSortFilterProxyModel::setIsUnselectedDisplayed(bool newValue)
{
    beginFilterChange();
    m_isUnselectedDisplayed = newValue;
    endFilterChange(QSortFilterProxyModel::Direction::Rows);
    emit isUnselectedDisplayedChanged();
}

bool PublicTalkSortFilterProxyModel::filterAcceptsRow(int sourceRow,
                                                      const QModelIndex &sourceParent) const
{
    QModelIndex index = sourceModel()->index(sourceRow, 0, sourceParent);
    return ((sourceModel()->data(index, PublicTalkModel::Roles::ThemeNumberRole).toString().contains(filterText(), Qt::CaseInsensitive)
             || sourceModel()->data(index, PublicTalkModel::Roles::ThemeNameRole).toString().contains(filterText(), Qt::CaseInsensitive))
            && (isUnselectedDisplayed() || sourceModel()->data(index, PublicTalkModel::Roles::IsSelectedRole).toBool()));
}

bool PublicTalkSortFilterProxyModel::lessThan(const QModelIndex &left, const QModelIndex &right) const
{
    switch (roleKey(groupByRole())) {
    case PublicTalkModel::Roles::ThemeNameRole:
    case PublicTalkModel::Roles::AlphabetRole: {
        QVariant leftData = sourceModel()->data(left, PublicTalkModel::Roles::ThemeNameRole);
        QVariant rightData = sourceModel()->data(right, PublicTalkModel::Roles::ThemeNameRole);
        int compResult = QString::compare(leftData.toString(), rightData.toString(), Qt::CaseInsensitive);
        if (compResult != 0)
            return compResult < 0;
        break;
    }
    default:
        break;
    }

    QVariant leftData = sourceModel()->data(left, PublicTalkModel::Roles::ThemeNumberRole);
    QVariant rightData = sourceModel()->data(right, PublicTalkModel::Roles::ThemeNumberRole);
    return leftData.toInt() < rightData.toInt();
}

int PublicTalkSortFilterProxyModel::roleKey(const QByteArray &role) const
{
    QHash<int, QByteArray> roles = roleNames();
    QHashIterator<int, QByteArray> it(roles);
    while (it.hasNext()) {
        it.next();
        if (it.value() == role)
            return it.key();
    }
    return -1;
}

PublicTalkValidator::PublicTalkValidator(QObject *parent)
    : QValidator(parent), m_model(nullptr), m_talkId(-1), m_role(PublicTalkModel::Roles::None)
{
}

PublicTalkValidator::~PublicTalkValidator()
{
}

QValidator::State PublicTalkValidator::validate(QString &input, int &pos) const
{
    Q_UNUSED(pos)

    if (input.isEmpty()) {
        emit errorChanged("");
        return Acceptable;
    }
    if (!model()) {
        emit errorChanged("");
        return Acceptable;
    }
    // if (role() == PublicTalkModel::Roles::None) {
    //     emit errorChanged("");
    //     return Acceptable;
    // }
    QModelIndex modelIndex = model()->getPublicTalkIndex(talkId());
    if (!modelIndex.isValid()) {
        emit errorChanged("");
        return Acceptable;
    }
    PublicTalk *publicTalk = model()->data(modelIndex, PublicTalkModel::PublicTalkRole).value<PublicTalk *>();
    if (role() == PublicTalkModel::Roles::None
        || role() == PublicTalkModel::Roles::LanguageIdRole) {
        sql_class *sql = &Singleton<sql_class>::Instance();
        int defaultlang = sql->getLanguageDefaultId();
        if (publicTalk->languageId() != defaultlang) {
            emit errorChanged(tr("The language of this talk doesn't match the default language setting."));
            return Invalid;
        }
    }
    if (role() == PublicTalkModel::Roles::None
        || role() == PublicTalkModel::Roles::DiscontinueDateRole) {
        if (publicTalk->discontinueDate().isValid() && publicTalk->discontinueDate() < QDate::currentDate()) {
            emit errorChanged(tr("This talk has been discontinued."));
            return Invalid;
        }
    }

    emit errorChanged("");
    return Acceptable;
}

PublicTalkModel *PublicTalkValidator::model() const
{
    return m_model;
}

void PublicTalkValidator::setModel(PublicTalkModel *newModel)
{
    if (m_model == newModel)
        return;
    m_model = newModel;
    emit modelChanged();
}

int PublicTalkValidator::talkId() const
{
    return m_talkId;
}

void PublicTalkValidator::setTalkId(int newTalkId)
{
    if (m_talkId == newTalkId)
        return;
    m_talkId = newTalkId;
    emit talkIdChanged();
}

PublicTalkModel::Roles PublicTalkValidator::role() const
{
    return m_role;
}

void PublicTalkValidator::setRole(PublicTalkModel::Roles newRole)
{
    if (m_role == newRole)
        return;
    m_role = newRole;
    emit roleChanged();
}
