/**
 * This file is part of TheocBase.
 *
 * Copyright (C) 2011-2015, TheocBase Development Team, see AUTHORS.
 *
 * TheocBase is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * TheocBase is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with TheocBase.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "personsui.h"
#include "ui_personsui.h"
#include "lmm_meeting.h"
#include "historytable.h"
#include <ciso646>

// persons class creator
personsui::personsui(QWidget *parent)
    : QDialog(parent), m_ui(new Ui::personsui)
{
    sql = &Singleton<sql_class>::Instance();
    ac = &Singleton<AccessControl>::Instance();

    editmode = false;
    m_ui->setupUi(this);

    TBStyle *tbStyle = &TBStyle::Instance();
    connect(tbStyle, &TBStyle::stylesChanged, this, &personsui::updateStyles);
    m_ui->tabPersonalInfo_2->setCurrentIndex(0);
    updatePersonList();
    updateFamiliesList();

    m_ui->framePersonalInfo->setEnabled(false);

    m_ui->dateUnavailableStart->calendarWidget()->setFirstDayOfWeek(Qt::Monday);
    m_ui->dateUnavailableStop->calendarWidget()->setFirstDayOfWeek(Qt::Monday);
    m_ui->dateUnavailableStart->setDate(QDate::currentDate());
    m_ui->radBrother_2->setChecked(true);

    // Get default Watchtower Study Conductor
    defaultWtConductor = sql->getSetting("wtconductor", "-1").toInt();
    m_ui->buttonWtConductorDefault->setVisible(false);

    // Get own congregation
    myCongregationId = c.getMyCongregation().id;

    m_ui->tabGeneral_2->setEnabled(false);
    m_ui->chkLeadsMinistry_2->setVisible(false);

    m_studiesGridScrollPosition = 0;

    applyAuthorizationRules();
    m_ui->splitter->setStretchFactor(1, 1);
    updateStyles();
}

personsui::~personsui()
{
    qDeleteAll(m_families);
    delete m_ui;
}

void personsui::updateStyles()
{
    TBStyle *tbStyle = &TBStyle::Instance();
    general::changeButtonIconColor(findChildren<QAbstractButton *>(QString(), Qt::FindChildrenRecursively));
    general::changeLabelIconColor(findChildren<QLabel *>(QString(), Qt::FindChildrenRecursively));
    m_ui->frameMWFirstSection->setStyleSheet(QString("QFrame#frameMWFirstSection { border: 1px solid rgba(%1, %2, %3, 127); }")
                                                     .arg(tbStyle->lmmSection1TextColor().red())
                                                     .arg(tbStyle->lmmSection1TextColor().green())
                                                     .arg(tbStyle->lmmSection1TextColor().blue()));
    m_ui->frameMWSecondSection->setStyleSheet(QString("QFrame#frameMWSecondSection { border: 1px solid rgba(%1, %2, %3, 127); }")
                                                      .arg(tbStyle->lmmSection2TextColor().red())
                                                      .arg(tbStyle->lmmSection2TextColor().green())
                                                      .arg(tbStyle->lmmSection2TextColor().blue()));
    m_ui->line->setStyleSheet(QString("QFrame#line { border: 1px solid rgba(%1, %2, %3, 127); }")
                                      .arg(tbStyle->lmmSection2TextColor().red())
                                      .arg(tbStyle->lmmSection2TextColor().green())
                                      .arg(tbStyle->lmmSection2TextColor().blue()));
    m_ui->frameMWThirdSection->setStyleSheet(QString("QFrame#frameMWThirdSection { border: 1px solid rgba(%1, %2, %3, 127); }")
                                                     .arg(tbStyle->lmmSection3TextColor().red())
                                                     .arg(tbStyle->lmmSection3TextColor().green())
                                                     .arg(tbStyle->lmmSection3TextColor().blue()));
    m_ui->frame_7->setStyleSheet(QString("QFrame#frame_7 { border: 1px solid rgba(%1, %2, %3, 127); }")
                                         .arg(tbStyle->publicTalkTextColor().red())
                                         .arg(tbStyle->publicTalkTextColor().green())
                                         .arg(tbStyle->publicTalkTextColor().blue()));
    m_ui->frame_6->setStyleSheet(QString("QFrame#frame_6 { border: 1px solid rgba(%1, %2, %3, 127); }")
                                         .arg(tbStyle->watchtowerStudyTextColor().red())
                                         .arg(tbStyle->watchtowerStudyTextColor().green())
                                         .arg(tbStyle->watchtowerStudyTextColor().blue()));
}

std::unique_ptr<Person> personsui::currentPublisher() const
{
    auto item = m_ui->lstPublishers->currentItem();
    if (not item)
        return nullptr;

    return std::unique_ptr<Person> { cpersons::getPerson(item->data(Qt::UserRole).toInt()) };
}

void personsui::updatePersonList()
{
    // Get all publishers in own congregation
    cpersons *cp = new cpersons();
    auto users = cp->getAllPersons(0);

    // Add publishers to list
    for (Person *p : users) {
        auto item = new QListWidgetItem(m_ui->lstPublishers);
        item->setText(p->fullName());
        item->setData(Qt::UserRole, p->id());
    }

    // Show quantity of publishers
    m_ui->lblCount->setText(QVariant(users.size()).toString());

    qDeleteAll(users);
}

void personsui::changeEvent(QEvent *e)
{
    QWidget::changeEvent(e);
    switch (e->type()) {
    case QEvent::LanguageChange:
        m_ui->retranslateUi(this);
        break;
    default:
        break;
    }
}

void personsui::applyAuthorizationRules()
{
    // basic publisher infos and functions
    if (!ac->user()->hasPermission(Permission::Rule::CanEditPublishers)) {
        m_ui->buttonAddPerson->setEnabled(false);
        m_ui->buttonRemovePerson->setEnabled(false);
        m_ui->information1Widget->setEnabled(false);
        m_ui->information2Widget->setEnabled(false);
        m_ui->chkActive->setEnabled(false);
    }

    // privileges and student assignments
    if (!ac->user()->hasPermission(Permission::Rule::CanViewPrivileges) && !ac->user()->hasPermission(Permission::Rule::CanViewStudentData)) {
        m_ui->labelDetails->setHidden(true);
        m_ui->tabPersonalInfo_2->setHidden(true);
    }

    // MW student assignments
    if (!ac->user()->hasPermission(Permission::Rule::CanViewStudentData)) {
        m_ui->chkLMMTRBibleReading->setHidden(true);
        m_ui->frameMWSecondSectionStudenTalks->setHidden(true);
    } else if (!ac->user()->hasPermission(Permission::Rule::CanEditStudentData)) {
        m_ui->chkLMMTRBibleReading->setEnabled(false);
        m_ui->frameMWSecondSectionStudenTalks->setEnabled(false);
    }

    // MW and WE privileges
    if (!ac->user()->hasPermission(Permission::Rule::CanViewPrivileges)) {
        m_ui->chkNominated_2->setHidden(true);
        m_ui->chkLMMChairman->setHidden(true);
        m_ui->chkLMMTRTalk->setHidden(true);
        m_ui->chkLMMTRSpiritualGems->setHidden(true);
        m_ui->chkLMMFMDiscussion->setHidden(true);
        m_ui->frameMWThirdSection->setHidden(true);
        m_ui->frameWeekendMeeting->setHidden(true);
        m_ui->chkLeadsMinistry_2->setHidden(true);
        m_ui->chkPrayer->setHidden(true);
    } else if (!ac->user()->hasPermission(Permission::Rule::CanEditPrivileges)) {
        m_ui->chkNominated_2->setEnabled(false);
        m_ui->chkLMMChairman->setEnabled(false);
        m_ui->chkLMMTRTalk->setEnabled(false);
        m_ui->chkLMMTRSpiritualGems->setEnabled(false);
        m_ui->chkLMMFMDiscussion->setEnabled(false);
        m_ui->frameMWThirdSection->setEnabled(false);
        m_ui->frameWeekendMeeting->setEnabled(false);
        m_ui->chkLeadsMinistry_2->setEnabled(false);
        m_ui->chkPrayer->setEnabled(false);
    }

    // hospitality
    if (!ac->user()->hasPermission(Permission::Rule::CanScheduleHospitality)) {
        m_ui->chkHospitality->setHidden(true);
    }

    // availabilities
    if (!ac->user()->hasPermission(Permission::Rule::CanViewAvailabilities)) {
        m_ui->tabPersonalInfo_2->removeTab(3);
    } else if (!ac->user()->hasPermission(Permission::Rule::CanEditAvailabilities)) {
        m_ui->widgetEditUnavailable->setHidden(true);
    }

    // talk history
    if (!ac->user()->hasPermission(Permission::Rule::CanViewMidweekMeetingTalkHistory)) {
        m_ui->tabPersonalInfo_2->removeTab(1);
    }
}

// this is always called when a publisher item is clicked.
// Note that if the click causes the current item to change
// (invoking "on_lstPublishers_currentItemChanged"), this handler
// is always called _after_ "on_lstPublishers_currentItemChanged"
void personsui::on_lstPublishers_itemClicked(QListWidgetItem *item)
{
    // qDebug() << "on_lstPublishers_clicked:";
    // qDebug() << "   current:" << (item ? item->data(Qt::UserRole).toInt() : -1) << "," <<
    //             (item ? item->text() : "<empty>");

    editmode = false;
    saveChanges(item);
    editmode = true;
}

void personsui::on_lstPublishers_currentItemChanged(QListWidgetItem *current, QListWidgetItem *previous)
{
    // qDebug() << "lstPublisherItemChanged:";
    // qDebug() << "   current:" << (current ? current->data(Qt::UserRole).toInt() : -1) << "," <<
    //             (current ? current->text() : "<empty>");
    // qDebug() << "   previous:" << (previous ? previous->data(Qt::UserRole).toInt() : -1) << "," <<
    //             (previous ? previous->text() : "<empty>");

    editmode = false;

    // Try to save the changes before loading a selected publisher
    if (previous && !saveChanges(previous)) {
        qDebug("personsui: current publisher changed CANCELED");
        // Cannot save changes
        m_ui->lstPublishers->setCurrentItem(previous);
        editmode = true;
        return;
    }

    updateFamiliesList();

    if (current)
        current->setSelected(true);
    updateDetailsPaneWith(current);
}

void personsui::ClearDetailsPane()
{
    m_ui->framePersonalInfo->setEnabled(false);

    // Personal info...
    m_ui->tabGeneral_2->setEnabled(true);
    m_ui->txtFirstName->setText(QString());
    m_ui->txtLastName->setText(QString());
    m_ui->txtPhone->setText("");
    m_ui->txtMobile->setText("");
    m_ui->txtEmail->setText("");
    m_ui->radBrother_2->setChecked(true);
    m_ui->chkNominated_2->setChecked(false); // "servant"
    m_ui->chkPrayer->setChecked(false);
    m_ui->chkFamilyHead->setChecked(false);
    m_ui->chkActive->setChecked(true);

    // Details...
    m_ui->chkLMMChairman->setChecked(false);

    m_ui->chkLMMTRTalk->setChecked(false);
    m_ui->chkLMMTRSpiritualGems->setChecked(false);
    m_ui->chkLMMTRBibleReading->setChecked(false);

    m_ui->chkLMMFMStartingConversation->setChecked(false);
    m_ui->chkLMMFMFollowingUp->setChecked(false);
    m_ui->chkLMMFMMakingDisciples->setChecked(false);
    m_ui->chkLMMFMExplainingBeliefs->setChecked(false);
    m_ui->chkAssistant->setChecked(false);
    m_ui->chkLMMFMTalk->setChecked(false);
    m_ui->chkLMMFMDiscussion->setChecked(false);
    m_ui->radioAllClasses->setChecked(true);

    m_ui->chkLMMCLTalk->setChecked(false);
    m_ui->chkCBS->setChecked(false);
    m_ui->chkBibleStudyReader->setChecked(false);

    // Weekend meeting...
    m_ui->chkPuheenjohtaja_2->setChecked(false); // weekend chairman
    m_ui->checkBoxPublicTalk->setChecked(false);

    m_ui->chkWTConductor->setChecked(false);
    m_ui->buttonWtConductorDefault->setChecked(false);
    m_ui->chkWTReader_2->setChecked(false);

    m_ui->chkLeadsMinistry_2->setChecked(false);

    m_ui->tableWidgetSchoolTask->setRowCount(0);
}

//
// update details pane
//
void personsui::updateDetailsPaneWith(QListWidgetItem const *item)
{
    std::unique_ptr<Person> p { cpersons::getPerson(item ? item->data(Qt::UserRole).toInt() : 0) };
    if (!p) {
        ClearDetailsPane();
        editmode = true;
        return;
    }

    m_ui->framePersonalInfo->setEnabled(true);

    // qDebug() << "name" + p.name;
    m_ui->txtLastName->setText(p->lastName());
    m_ui->txtFirstName->setText(p->firstName());
    // qDebug() << "phone: " + p.phone;
    m_ui->txtPhone->setText(p->phone());
    m_ui->txtMobile->setText(p->mobile());
    m_ui->txtEmail->setText(p->email());

    m_ui->radBrother_2->setChecked(p->gender() == Person::Male);
    m_ui->radSister_2->setChecked(p->gender() == Person::Female);

    // is servant ?
    m_ui->chkNominated_2->setChecked(p->servant());
    on_chkNominated_2_toggled(m_ui->chkNominated_2->isChecked());

    m_ui->chkActive->setChecked(!(Person::IsBreak & p->usefor()));
    if (p->usefor() & Person::SchoolMain) {
        m_ui->radioMainClass->setChecked(true);
    } else if (p->usefor() & Person::SchoolAux) {
        m_ui->radioAuxClasses->setChecked(true);
    } else {
        m_ui->radioAllClasses->setChecked(true);
    }

    m_ui->chkLeadsMinistry_2->setChecked(Person::FieldMinistry & p->usefor());

    m_ui->chkPuheenjohtaja_2->setChecked(Person::Chairman & p->usefor());
    m_ui->chkWTConductor->setChecked(Person::WtCondoctor & p->usefor());

    m_ui->buttonWtConductorDefault->setChecked(p->id() == defaultWtConductor);
    m_ui->chkWTReader_2->setChecked(Person::WtReader & p->usefor());
    m_ui->chkAssistant->setChecked(Person::Assistant & p->usefor());
    m_ui->checkBoxPublicTalk->setChecked(Person::PublicTalk & p->usefor());
    m_ui->chkCBS->setChecked(Person::CBSConductor & p->usefor());
    m_ui->chkBibleStudyReader->setChecked(Person::CBSReader & p->usefor());
    m_ui->chkPrayer->setChecked(Person::Prayer & p->usefor());

    // hospitality
    m_ui->chkHospitality->setChecked(Person::Hospitality & p->usefor());

    // LMM
    m_ui->chkLMMChairman->setChecked(Person::LMM_Chairman & p->usefor());
    m_ui->chkLMMTRTalk->setChecked(Person::LMM_TR_Talk & p->usefor());
    m_ui->chkLMMTRSpiritualGems->setChecked(Person::LMM_TR_SpiritualGems & p->usefor());
    m_ui->chkLMMTRBibleReading->setChecked(Person::LMM_TR_BibleReading & p->usefor());
    m_ui->chkLMMFMDiscussion->setChecked(Person::LMM_FM_Discussion & p->usefor());
    m_ui->chkLMMFMStartingConversation->setChecked(Person::LMM_FM_StartingConversation & p->usefor());
    m_ui->chkLMMFMFollowingUp->setChecked(Person::LMM_FM_FollowingUp & p->usefor());
    m_ui->chkLMMFMMakingDisciples->setChecked(Person::LMM_FM_MakingDisciples & p->usefor());
    m_ui->chkLMMFMExplainingBeliefs->setChecked(Person::LMM_FM_ExplainingBeliefs & p->usefor());
    m_ui->chkLMMFMTalk->setChecked(Person::LMM_FM_Talk & p->usefor());
    m_ui->chkLMMCLTalk->setChecked(Person::LMM_CL_Talk & p->usefor());

    m_ui->tabGeneral_2->setEnabled(true);

    // family
    QSharedPointer<family> personsFamily(family::getPersonFamily(p->id()));
    // is family head ?
    m_ui->chkFamilyHead->setChecked(personsFamily ? personsFamily->getHeadId() == p->id() : false);
    // member linked to
    m_ui->comboFamilies->setCurrentText(personsFamily ? (personsFamily->getHeadId() < 1 ? "" : personsFamily->getName()) : "");

    // get all school assignments
    getSchoolHistory();

    // add unavailabilities
    m_ui->tableUnavailabilities->setRowCount(0);
    QList<QPair<QDate, QDate>> ulist = p->getUnavailabilities();
    if (!ulist.empty()) {
        m_ui->tableUnavailabilities->setRowCount(ulist.size());
        for (int i = 0; i < (ulist.size()); i++) {
            QPair<QDate, QDate> datepair = ulist[i];
            m_ui->tableUnavailabilities->setItem(i, 0, new QTableWidgetItem(datepair.first.toString(Qt::ISODate)));
            m_ui->tableUnavailabilities->setItem(i, 1, new QTableWidgetItem(datepair.second.toString(Qt::ISODate)));
        }
    }

    editmode = true;
}

bool personsui::saveChanges(QListWidgetItem *item)
{
    if (not item) {
        if (m_ui->txtFirstName->text() == "" && m_ui->txtLastName->text() == "") {
            // qDebug() << "saveChanges(nullptr) without name -> true";
            return true;
        }
        // qDebug() << "saveChanges(nullptr) with name -> false";
        return false;
    }

    auto id = item->data(Qt::UserRole).toInt();
    std::unique_ptr<Person> p { cpersons::getPerson(id) };
    if (not p) {
        // qDebug() << "saveChanges(item with ID:" << id << ") but CANNOT retreive in DB -> false";
        return false;
    }

    // name check
    // check duplicates
    Person *checkperson = cpersons::getPerson(m_ui->txtFirstName->text() + " " + m_ui->txtLastName->text());

    if (checkperson && checkperson->id() != p->id()) {
        if (QMessageBox::question(this, "",
                                  tr("A person with the same name already exists: '%1'. Do you want to change the name?").arg(m_ui->txtFirstName->text() + " " + m_ui->txtLastName->text()),
                                  QMessageBox::No, QMessageBox::Yes)
            == QMessageBox::Yes) {
            return false;
        }
    }

    // public talk check
    // check for scheduled talks
    bool removeSpeaker = false;
    int speakerId = p->id();
    if (!m_ui->checkBoxPublicTalk->isChecked() && (Person::PublicTalk & p->usefor())) {
        // check for scheduled talks
        QDate startDate = QDateTime::currentDateTime().date().addDays(-7);

        sql_items incomingTalks;
        incomingTalks = sql->selectSql("SELECT * from publicmeeting WHERE date > '" + startDate.toString(Qt::ISODate) + "' AND speaker_id = " + QVariant(speakerId).toString() + " AND active");

        sql_items outgoingTalks;
        outgoingTalks = sql->selectSql("SELECT * from outgoing WHERE date > '" + startDate.toString(Qt::ISODate) + "' AND speaker_id = " + QVariant(speakerId).toString() + " AND active");

        if (!incomingTalks.empty() || !outgoingTalks.empty()) {
            if (QMessageBox::question(this, "",
                                      tr("%1 is scheduled for public talks! These talks will\n"
                                         "be moved to the To Do List if you remove him as speaker.\nRemove him as speaker?")
                                              .arg(p->fullName()),
                                      QMessageBox::No, QMessageBox::Yes)
                == QMessageBox::Yes) {
                removeSpeaker = true;
            } else
                return false;
        }
    }

    // p->setIsDirty(false);

    p->setLastName(m_ui->txtLastName->text());
    p->setFirstName(m_ui->txtFirstName->text());
    p->setPhone(m_ui->txtPhone->text());
    p->setMobile(m_ui->txtMobile->text());
    p->setEmail(m_ui->txtEmail->text());
    p->setServant(m_ui->chkNominated_2->isChecked());
    p->setGender(m_ui->radBrother_2->isChecked() ? Person::Male : Person::Female);

    int usefor = 0;

    if (!m_ui->chkActive->isChecked())
        usefor += Person::IsBreak;
    if (m_ui->chkLeadsMinistry_2->isChecked())
        usefor += Person::FieldMinistry;
    if (m_ui->chkPuheenjohtaja_2->isChecked())
        usefor += Person::Chairman;
    if (m_ui->chkWTReader_2->isChecked())
        usefor += Person::WtReader;
    if (m_ui->chkWTConductor->isChecked())
        usefor += Person::WtCondoctor;
    if (m_ui->buttonWtConductorDefault->isChecked()) {
        sql->saveSetting("wtconductor", QVariant(p->id()).toString());
        defaultWtConductor = p->id();
    } else {
        if (defaultWtConductor == p->id())
            sql->saveSetting("wtconductor", "-1");
    }

    if (m_ui->chkAssistant->isChecked())
        usefor += Person::Assistant;
    if (m_ui->checkBoxPublicTalk->isChecked())
        usefor += Person::PublicTalk;
    if (m_ui->chkCBS->isChecked())
        usefor += Person::CBSConductor;
    if (m_ui->chkBibleStudyReader->isChecked())
        usefor += Person::CBSReader;
    if (m_ui->chkPrayer->isChecked())
        usefor += Person::Prayer;

    // hospitality
    if (m_ui->chkHospitality->isChecked())
        usefor += Person::Hospitality;

    // LMM
    if (m_ui->chkLMMChairman->isChecked())
        usefor += Person::LMM_Chairman;
    if (m_ui->chkLMMTRTalk->isChecked())
        usefor += Person::LMM_TR_Talk;
    if (m_ui->chkLMMTRSpiritualGems->isChecked())
        usefor += Person::LMM_TR_SpiritualGems;
    if (m_ui->chkLMMTRBibleReading->isChecked())
        usefor += Person::LMM_TR_BibleReading;
    if (m_ui->chkLMMFMTalk->isChecked())
        usefor += Person::LMM_FM_Talk;
    if (m_ui->chkLMMFMStartingConversation->isChecked())
        usefor += Person::LMM_FM_StartingConversation;
    if (m_ui->chkLMMFMFollowingUp->isChecked())
        usefor += Person::LMM_FM_FollowingUp;
    if (m_ui->chkLMMFMMakingDisciples->isChecked())
        usefor += Person::LMM_FM_MakingDisciples;
    if (m_ui->chkLMMFMExplainingBeliefs->isChecked())
        usefor += Person::LMM_FM_ExplainingBeliefs;
    if (m_ui->chkLMMCLTalk->isChecked())
        usefor += Person::LMM_CL_Talk;
    if (m_ui->chkLMMFMDiscussion->isChecked())
        usefor += Person::LMM_FM_Discussion;

    // use in these classes. all is default
    if (m_ui->radioMainClass->isChecked())
        usefor += Person::SchoolMain;
    if (m_ui->radioAuxClasses->isChecked())
        usefor += Person::SchoolAux;

    qDebug() << "save usefor:" << usefor;
    p->setUsefor(usefor);

    // congregation
    p->setCongregationId(myCongregationId);

    // family
    SaveFamilySetting(p->id());

    bool saved = p->save();

    if (saved) {
        item->setText(p->fullName());
        if (removeSpeaker) {
            // move scheduled talks to To Do List
            todo::addScheduledTalks(speakerId, true);
        }
    }

    return saved;
}

void personsui::SaveFamilySetting(int personId)
{
    QSharedPointer<family> personsFamily(family::getPersonFamily(personId));

    if (m_ui->chkFamilyHead->isChecked()) {
        if (!personsFamily || personsFamily->getHeadId() != personId) {
            family::getOrAddFamily(personId);
        }
    } else {
        if (personsFamily && personsFamily->getHeadId() == personId) {
            // remove family
            personsFamily->removeFamily();
        }

        // family member
        if (m_ui->comboFamilies->currentIndex() > 0) {
            // add member
            if (!personsFamily || (personsFamily->getHeadId() != m_families.at(m_ui->comboFamilies->currentIndex() - 1)->getHeadId())) {
                m_families.at(m_ui->comboFamilies->currentIndex() - 1)->addMember(personId);
            }
        } else {
            // remove member
            if (personsFamily) {
                personsFamily->removeMember(personId);
            }
        }
    }
}

//+ button
void personsui::on_buttonAddPerson_clicked()
{
    editmode = false;
    if (!saveChanges(m_ui->lstPublishers->currentItem())) {
        editmode = true;
        return;
    }

    std::unique_ptr<Person> newperson { new Person() };
    newperson->setGender(Person::Male);
    newperson->setServant(false);
    newperson->setLastName(tr("Last name"));
    newperson->setFirstName(tr("First name"));
    newperson->setCongregationId(myCongregationId);

    auto cp = std::make_shared<cpersons>();
    newperson->setId(cp->addPerson(newperson.get()));

    auto item = new QListWidgetItem(m_ui->lstPublishers);
    item->setText(newperson->fullName());
    item->setData(Qt::UserRole, newperson->id());
    m_ui->lstPublishers->setCurrentItem(item);

    // Show quantity of publishers
    m_ui->lblCount->setText(QVariant(m_ui->lstPublishers->count()).toString());

    // qDebug() << "created person id:" << newperson->id();
}

//- button
void personsui::on_buttonRemovePerson_clicked()
{
    auto item = m_ui->lstPublishers->currentItem();
    if (not item)
        return;

    std::unique_ptr<Person> p { cpersons::getPerson(item->data(Qt::UserRole).toInt()) };
    qDebug() << "Poistetaan";

    QString msg = tr("Remove student") + " " + p->fullName() + "?";
    bool talksScheduled = false;
    int speakerId = p->id();
    if (Person::PublicTalk & p->usefor()) {
        // check for scheduled talks
        QDate startDate = QDateTime::currentDateTime().date().addDays(-7);

        sql_items incomingTalks;
        incomingTalks = sql->selectSql("SELECT * from publicmeeting WHERE date > '" + startDate.toString(Qt::ISODate) + "' AND speaker_id = " + QVariant(speakerId).toString() + " AND active");

        sql_items outgoingTalks;
        outgoingTalks = sql->selectSql("SELECT * from outgoing WHERE date > '" + startDate.toString(Qt::ISODate) + "' AND speaker_id = " + QVariant(speakerId).toString() + " AND active");

        if (!incomingTalks.empty() || !outgoingTalks.empty()) {
            msg = tr("%1 is scheduled for public talks! These talks will\n"
                     "be moved to the To Do List if you remove the student.")
                            .arg(p->fullName())
                    + "\n" + msg;
            talksScheduled = true;
        }
    }

    // QMessageBox:: ( Icon icon, const QString & title, const QString & text, StandardButtons buttons = NoButton, QWidget * parent = 0, Qt::WindowFlags f = Qt::Dialog | Qt::MSWindowsFixedSizeDialogHint )
    if (QMessageBox::question(this, tr("Remove student?"),
                              msg,
                              QMessageBox::No, QMessageBox::Yes)
        == QMessageBox::Yes) {
        m_ui->lstPublishers->setCurrentItem(nullptr);

        cpersons::removePerson(p->id());

        delete item;

        // Show quantity of publishers
        m_ui->lblCount->setText(QVariant(m_ui->lstPublishers->count()).toString());

        // move scheduled talks to To Do List
        if (talksScheduled)
            todo::addScheduledTalks(speakerId, true);
    }
}

void personsui::on_radBrother_2_toggled(bool checked)
{
    //  m_ui->chkAssistant->setEnabled(!checked);
    m_ui->chkLeadsMinistry_2->setEnabled(checked);
    m_ui->chkPuheenjohtaja_2->setEnabled(checked);
    m_ui->chkWTReader_2->setEnabled(checked);
    m_ui->chkWTConductor->setEnabled(checked);
    m_ui->chkNominated_2->setEnabled(checked);
    m_ui->checkBoxPublicTalk->setEnabled(checked);
    m_ui->chkCBS->setEnabled(checked);
    m_ui->chkBibleStudyReader->setEnabled(checked);
    m_ui->chkPrayer->setEnabled(checked);
    m_ui->chkLMMChairman->setEnabled(checked);
    m_ui->chkLMMFMDiscussion->setEnabled(checked);
    m_ui->chkLMMTRTalk->setEnabled(checked);
    m_ui->chkLMMTRSpiritualGems->setEnabled(checked);
    m_ui->chkLMMTRBibleReading->setEnabled(checked);
    m_ui->chkLMMFMTalk->setEnabled(checked);
    m_ui->chkLMMCLTalk->setEnabled(checked);
    applyAuthorizationRules();
}

void personsui::on_chkNominated_2_toggled(bool checked)
{
    m_ui->checkBoxPublicTalk->setEnabled(checked);
    m_ui->chkWTConductor->setEnabled(checked);
    m_ui->chkCBS->setEnabled(checked);
    // m_ui->chkPrayer->setEnabled(checked);
    m_ui->chkLMMChairman->setEnabled(checked);
    m_ui->chkLMMTRTalk->setEnabled(checked);
    m_ui->chkLMMTRSpiritualGems->setEnabled(checked);
    m_ui->chkLMMFMDiscussion->setEnabled(checked);
    m_ui->chkLMMCLTalk->setEnabled(checked);
    applyAuthorizationRules();
}

void personsui::on_buttonAddUnavailable_clicked()
{
    // add new row
    auto item = m_ui->lstPublishers->currentItem();
    if (not item)
        return;

    std::unique_ptr<Person> p { cpersons::getPerson(item->data(Qt::UserRole).toInt()) };
    if (not p)
        return;

    p->setUnavailability(m_ui->dateUnavailableStart->date(), m_ui->dateUnavailableStop->date());
    m_ui->tableUnavailabilities->setRowCount(m_ui->tableUnavailabilities->rowCount() + 1);
    m_ui->tableUnavailabilities->setItem(m_ui->tableUnavailabilities->rowCount() - 1, 0, new QTableWidgetItem(m_ui->dateUnavailableStart->date().toString(Qt::ISODate)));
    m_ui->tableUnavailabilities->setItem(m_ui->tableUnavailabilities->rowCount() - 1, 1, new QTableWidgetItem(m_ui->dateUnavailableStop->date().toString(Qt::ISODate)));
}

void personsui::on_dateUnavailableStart_dateChanged(QDate date)
{
    m_ui->dateUnavailableStop->setDate(date);
}

void personsui::on_buttonRemoveUnavailable_clicked()
{
    if (not m_ui->lstPublishers->currentItem())
        return;

    if (m_ui->tableUnavailabilities->selectedItems().count() < 1)
        return;

    QTableWidgetItem *selecteditem = m_ui->tableUnavailabilities->selectedItems().first();
    qDebug() << QString::number(selecteditem->row());

    if (currentPublisher()->removeUnavailability(QDate::fromString(m_ui->tableUnavailabilities->item(selecteditem->row(), 0)->text(), Qt::ISODate),
                                                 QDate::fromString(m_ui->tableUnavailabilities->item(selecteditem->row(), 1)->text(), Qt::ISODate)))
        m_ui->tableUnavailabilities->removeRow(selecteditem->row());
}

// column name reminder: (0) date (1) num (2) source (3) note (4) time (5) assistant
void personsui::getSchoolHistory()
{
    auto cp = std::make_shared<cpersons>();
    m_ui->tableWidgetSchoolTask->setRowCount(0);
    m_ui->tableWidgetSchoolTask->setColumnWidth(1, 30);
    m_ui->tableWidgetSchoolTask->setColumnWidth(4, 60);

    auto assigneeID = m_ui->lstPublishers->currentItem()->data(Qt::UserRole).toInt();
    auto assigneeIDstr = QVariant(assigneeID).toString();

    auto history = ::getSchoolHistory(assigneeID,
                                      QTableWidgetItem().foreground().color(),
                                      QTableWidgetItem().background().color());

    m_ui->tableWidgetSchoolTask->setRowCount(history.rowCount());
    for (int iLine = 0; iLine < history.rowCount(); ++iLine) // auto const& assignment : history)
    {
        for (int iCol = 0; iCol <= 5; ++iCol) {
            auto const &formatedText = history.item(iLine, iCol);
            auto item = new QTableWidgetItem(formatedText.text());
            m_ui->tableWidgetSchoolTask->setItem(iLine, iCol, item);

            // coloring
            if (formatedText.customForeground()) {
                QBrush fgBrush = item->foreground();
                QColor fg = formatedText.foreground();
                fgBrush.setColor(fg);
                item->setForeground(fgBrush);
            }

            if (formatedText.customBackground()) {
                item->setBackground(formatedText.background());
            }

            if (formatedText.italic() || formatedText.bold() || formatedText.strikeOut()) {
                QFont f(item->font());
                f.setItalic(formatedText.italic());
                f.setBold(formatedText.bold());
                f.setStrikeOut(formatedText.strikeOut());

                item->setFont(f);
            }
            // icon
            if (!formatedText.iconName().isEmpty())
                item->setIcon(QIcon(formatedText.iconName()));
        }
    }

    m_ui->tableWidgetSchoolTask->sortItems(0, Qt::DescendingOrder);
    m_ui->tableWidgetSchoolTask->resizeColumnToContents(0); // date
    m_ui->tableWidgetSchoolTask->resizeColumnToContents(1); // what
    m_ui->tableWidgetSchoolTask->resizeColumnToContents(4); // time
    m_ui->tableWidgetSchoolTask->resizeColumnToContents(5); // assistant
}

/*void personsui::selectPerson(int personid)
{
    // choose a person from list
    foreach (Person *p,m_users){
        if(personid == p->id()){
            m_ui->lstPublishers->setCurrentIndex(
                    m_ui->lstPublishers->model()->index(m_users.indexOf(p),0));
            iPersonIndex = m_users.indexOf(p);
            break;
        }
    }
}*/

void personsui::updateFamiliesList()
{
    m_ui->comboFamilies->clear();
    m_ui->comboFamilies->addItem("");
    qDeleteAll(m_families);
    m_families.clear();
    m_families = family::getFamilies();
    foreach (family *f, m_families) {
        m_ui->comboFamilies->addItem(f->getName());
    }
}

void personsui::setDefaultPerson(QString name)
{
    m_defaultPersonName = name;
}

bool personsui::saveChanges()
{
    bool ok = saveChanges(m_ui->lstPublishers->currentItem());
    if (ok) {
        m_ui->lstPublishers->setCurrentItem(nullptr);
        ClearDetailsPane();
        m_ui->lstPublishers->clearFocus();
    }
    return ok;
}

void personsui::showEvent(QShowEvent *)
{
    if (m_defaultPersonName != "") {
        std::unique_ptr<Person> defaultperson { cpersons::getPerson(m_defaultPersonName) };
        if (!defaultperson)
            return;
        // select person from list
        for (auto i = 0, n = m_ui->lstPublishers->count(); i < n; ++i) {
            auto item = m_ui->lstPublishers->item(i);
            if (defaultperson->id() == item->data(Qt::UserRole).toInt()) {
                on_lstPublishers_currentItemChanged(item, m_ui->lstPublishers->currentItem());
                break;
            }
        }
    }
}

void personsui::closeEvent(QCloseEvent *e)
{
    // save changes in close event
    if (!saveChanges(m_ui->lstPublishers->currentItem())) {
        e->ignore();
    } else {
        e->accept();
    }
}

void personsui::on_buttonWtConductorDefault_toggled(bool checked)
{
    m_ui->buttonWtConductorDefault->setIcon(QIcon(checked ? ":/icons/star.svg" : ":/icons/star_filled.svg"));
}
