/**
 * This file is part of TheocBase.
 *
 * Copyright (C) 2011-2015, TheocBase Development Team, see AUTHORS.
 *
 * TheocBase is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * TheocBase is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with TheocBase.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef PERSON_H
#define PERSON_H

#include <QString>
#include <QStringList>
#include <QPair>
#include <QDate>
#include <QList>
#include <QAbstractItemModel>
#include <QSortFilterProxyModel>
#include <QObjectBindableProperty>
#include "sql_class.h"
#include "tbstyle.h"

/**
 * @brief The person class - This class represent a one person
 *                           Save changes using update() function.
 */
class Person : public QObject
{
    Q_OBJECT
    Q_PROPERTY(int id READ id WRITE setId NOTIFY idChanged)
    Q_PROPERTY(QString firstName READ firstName WRITE setFirstName BINDABLE bindableFirstName)
    Q_PROPERTY(QString lastName READ lastName WRITE setLastName BINDABLE bindableLastName)
    Q_PROPERTY(QString fullName READ fullName BINDABLE bindableFullName)
    Q_PROPERTY(QString info READ info WRITE setInfo BINDABLE bindableInfo)
    Q_PROPERTY(Gender gender READ gender WRITE setGender BINDABLE bindableGender)
    Q_PROPERTY(QString phone READ phone WRITE setPhone BINDABLE bindablePhone)
    Q_PROPERTY(QString mobile READ mobile WRITE setMobile BINDABLE bindableMobile)
    Q_PROPERTY(QString email READ email WRITE setEmail BINDABLE bindableEmail)
    Q_PROPERTY(bool servant READ servant WRITE setServant BINDABLE bindableServant)
    Q_PROPERTY(int usefor READ usefor WRITE setUsefor BINDABLE bindableUsefor)
    Q_PROPERTY(int congregationId READ congregationId BINDABLE bindableCongregationId)
    Q_PROPERTY(QString congregationName READ congregationName BINDABLE bindableCongregationName)
    Q_PROPERTY(bool isValid READ isValid BINDABLE bindableIsValid)

public:
    Person(QObject *parent = nullptr);
    Person(int id, QString uuid = "", QObject *parent = nullptr);
    virtual ~Person() = default;

    /**
     * @brief The Gender enum - Male or Female
     */
    enum Gender { Male,
                  Female };
    Q_ENUM(Gender)

    /**
     * @brief The UseFor enum
     */
    enum UseFor {
        None = 0,
        // use for bit values
        IsBreak = 1 << 4, // is break                                 16
        FieldMinistry = 1 << 5, // field ministry meeting conductor   32
        Chairman = 1 << 6, // public meeting chairman                 64
        WtReader = 1 << 7, // watchtower reader                       128
        Assistant = 1 << 8, // assistant                              256
        PublicTalk = 1 << 9, // public talks                          512
        CBSConductor = 1 << 10, // congregation bible study conductor 1024
        Prayer = 1 << 11, // prayer                                   2048
        CBSReader = 1 << 13, // congregation bible study reader       8192
        WtCondoctor = 1 << 14, // watchtower study conductor          16384
        SchoolMain = 1 << 15, // school - only for Main Class         32768
        SchoolAux = 1 << 16, // school - only for Auxiliary Classes   65536
        LMM_Chairman = 1 << 17, // midweek meeting - chairman         131072
        LMM_TR_Talk = 1 << 18, // talk in "Treasures From..."         262144
        LMM_TR_SpiritualGems = 1 << 19, // Spiritual Gems             524288
        LMM_TR_BibleReading = 1 << 20, // Bible Reading               1048576
        LMM_FM_ExplainingBeliefs = 1 << 21, // Explain. Your Beliefs  2097152
        LMM_FM_StartingConversation = 1 << 22, // Starting a Convers. 4194304
        LMM_FM_FollowingUp = 1 << 23, // Following Up                 8388608
        LMM_FM_MakingDisciples = 1 << 24, // Making Disciples         16777216
        LMM_CL_Talk = 1 << 25, // talk in "Living..."                 33554432
        Hospitality = 1 << 26, // Hospitality                         67108864
        LMM_FM_Talk = 1 << 27, // talk in "Apply..." section          134217728
        LMM_FM_Discussion = 1 << 28 // discussion in "Apply..."       268435456
    };
    Q_ENUM(UseFor)

    /**
     * @brief id - Get object id in database
     * @return - Id
     */
    int id();
    bool isNew();
    /**
     * @brief firstName - Get first name
     * @return - first name
     */
    QString firstName() { return b_firstName.value(); }
    /**
     * @brief setFirstname - Set first name
     * @param arg - Firstname
     */
    void setFirstName(QString newValue) { b_firstName = newValue; }
    QBindable<QString> bindableFirstName() { return &b_firstName; }

    /**
     * @brief lastName - Get last name
     * @return - last name
     */
    QString lastName() { return b_lastName.value(); }
    /**
     * @brief setLastname - Set last name
     * @param arg - Lastname
     */
    void setLastName(QString newValue) { b_lastName = newValue; }
    QBindable<QString> bindableLastName() { return &b_lastName; }

    /**
     * @brief fullName - Get full name according to the property fullNameFormat
     * @return - Fullname
     */
    QString fullName() { return b_fullName.value(); }
    QBindable<QString> bindableFullName() { return &b_fullName; }

    /**
     * @brief info - Get info text
     * @return - Info
     */
    QString info() { return b_info.value(); }
    /**
     * @brief setInfo - Set info value
     * @param value - Info
     */
    void setInfo(QString newValue) { b_info = newValue; }
    QBindable<QString> bindableInfo() { return &b_info; }

    /**
     * @brief gender - Get gender (Male or Female)
     * @return - gender
     */
    Gender gender() { return b_gender.value(); }
    /**
     * @brief setGender - Set gender
     * @param gender - Gender (Male or Female) e.g. setGender(Person::Male)
     */
    void setGender(Gender newValue) { b_gender = newValue; }
    QBindable<QString> bindableGender() { return &b_gender; }

    /**
     * @brief phone - Get phone number
     * @return - Phone number
     */
    QString phone() { return b_phone.value(); }
    /**
     * @brief setPhone - Set phone number
     * @param value - phone
     */
    void setPhone(QString newValue) { b_phone = newValue; }
    QBindable<QString> bindablePhone() { return &b_phone; }

    /**
     * @brief mobile - Get mobile number
     * @return - Mobile number
     */
    QString mobile() { return b_mobile.value(); }
    /**
     * @brief setMobile - Set mobile number
     * @param value - mobile
     */
    void setMobile(QString newValue) { b_mobile = newValue; }
    QBindable<QString> bindableMobile() { return &b_mobile; }

    /**
     * @brief email - Get email address
     * @return - email
     */
    QString email() { return b_email.value().simplified(); }
    /**
     * @brief setEmail - Set email address
     * @param value - email
     */
    void setEmail(QString newValue) { b_email = newValue; }
    QBindable<QString> bindableEmail() { return &b_email; }

    /**
     * @brief servant - Is a brother servant?
     * @return - true or false
     */
    bool servant() { return b_servant.value(); }
    /**
     * @brief setServant - Set servant value for brother
     * @param value - true if servant or false
     */
    void setServant(bool newValue) { b_servant = newValue; }
    QBindable<bool> bindableServant() { return &b_servant; }

    /**
     * @brief usefor - Get use for value. This value tells what assignments is allowed for this person.
     * @return - Binary value of assignments. Use bitwise operations to check allowed assignments.
     *           Example: Is allowed to use Wt reader? bool useWtReader = (Person::usefor() & Person::wt_reader);
     */
    int usefor() { return b_usefor.value(); }
    /**
     * @brief setUsefor - Set use for value. This value tells what assignments is allowed for this person.
     * @param value - Value is in a binary format and saved in a one column in the database.
     *                Example: Use for Bible reading assignment AND Wt reader in the Watchtower Study - setUsefor(Person::LMM_TR_BibleReading + Person::WtReader)
     */
    void setUsefor(int newValue) { b_usefor = newValue; }
    QBindable<int> bindableUsefor() { return &b_usefor; }

    /**
     * @brief congregationId - Get congregation id number
     * @return - id number
     */
    int congregationId() { return b_congregationId.value(); }
    /**
     * @brief setCongregationId - Set congregation id number
     * @param id - id number
     */
    void setCongregationId(int newValue) { b_congregationId = newValue; }
    QBindable<int> bindableCongregationId() { return &b_congregationId; }

    /**
     * @brief congregationName - Get congregation name
     * @return - Name of congregation
     */
    QString congregationName() { return b_congregationName.value(); }
    QBindable<QString> bindableCongregationName() { return &b_congregationName; }

    /**
     * @brief congregationInfo - Get congregation record
     * @return - the db record for the congregation
     */
    sql_items congregationInfo();

    bool isDirty() { return b_isDirty.value(); }
    void setIsDirty(bool newValue) { b_isDirty = newValue; }
    QBindable<bool> bindableIsDirty() { return &b_isDirty; }

    QString uuid() const;

    bool isValid() const { return b_isValid.value(); }
    QBindable<bool> bindableIsValid() { return &b_isValid; }

    QList<QPair<QDate, QDate>> getUnavailabilities();
    void setUnavailability(QDate start, QDate end);
    bool removeUnavailability(QDate start, QDate end);
    /**
     * @brief save - This function save changes to database
     * @return - success or failure
     */
    Q_INVOKABLE virtual bool save();

public slots:

    /**
     * @brief setId - Set database id for person object
     * @param id - id number
     */
    void setId(int id);

signals:
    void idChanged();
    void firstNameChanged();
    void lastNameChanged();
    void fullNameChanged();
    void infoChanged();
    void genderChanged();
    void phoneChanged();
    void mobileChanged();
    void emailChanged();
    void servantChanged();
    void useforChanged();
    void congregationIdChanged();

private:
    sql_class *sql;
    TBStyle *tbStyle;
    int m_id;
    Q_OBJECT_BINDABLE_PROPERTY(Person, QString, b_firstName, &Person::firstNameChanged);
    Q_OBJECT_BINDABLE_PROPERTY(Person, QString, b_lastName, &Person::lastNameChanged);
    QProperty<QString> b_fullName;
    Q_OBJECT_BINDABLE_PROPERTY(Person, QString, b_info, &Person::infoChanged);
    Q_OBJECT_BINDABLE_PROPERTY(Person, Gender, b_gender, &Person::genderChanged);
    Q_OBJECT_BINDABLE_PROPERTY(Person, QString, b_phone, &Person::phoneChanged);
    Q_OBJECT_BINDABLE_PROPERTY(Person, QString, b_mobile, &Person::mobileChanged);
    Q_OBJECT_BINDABLE_PROPERTY(Person, QString, b_email, &Person::emailChanged);
    Q_OBJECT_BINDABLE_PROPERTY(Person, bool, b_servant, &Person::servantChanged);
    Q_OBJECT_BINDABLE_PROPERTY(Person, int, b_usefor, &Person::useforChanged);
    Q_OBJECT_BINDABLE_PROPERTY(Person, int, b_congregationId, &Person::congregationIdChanged);
    QProperty<QString> b_congregationName;
    QProperty<QString> b_congregationInfo;
    QProperty<bool> b_isDirty { false };
    QString m_uuid;
    QProperty<bool> b_isValid { false };
};
#endif // PERSON_H
