/**
 * This file is part of TheocBase.
 *
 * Copyright (C) 2011-2015, TheocBase Development Team, see AUTHORS.
 *
 * TheocBase is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * TheocBase is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with TheocBase.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "person.h"
#include "accesscontrol.h"

Person::Person(QObject *parent)
    : Person(-1, "", parent)
{
}

Person::Person(int id, QString uuid, QObject *parent)
    : QObject(parent), m_id(id), b_gender(Male), b_servant(0), b_usefor(0), b_congregationId(0), m_uuid(uuid)
{
    sql = &Singleton<sql_class>::Instance();
    tbStyle = &TBStyle::Instance();

    QObject::connect(this, &Person::firstNameChanged, [&]() { b_isDirty = true; });
    QObject::connect(this, &Person::lastNameChanged, [&]() { b_isDirty = true; });
    QObject::connect(this, &Person::infoChanged, [&]() { b_isDirty = true; });
    QObject::connect(this, &Person::genderChanged, [&]() { b_isDirty = true; });
    QObject::connect(this, &Person::phoneChanged, [&]() { b_isDirty = true; });
    QObject::connect(this, &Person::mobileChanged, [&]() { b_isDirty = true; });
    QObject::connect(this, &Person::emailChanged, [&]() { b_isDirty = true; });
    QObject::connect(this, &Person::servantChanged, [&]() { b_isDirty = true; });
    QObject::connect(this, &Person::useforChanged, [&]() { b_isDirty = true; });
    QObject::connect(this, &Person::congregationIdChanged, [&]() { b_isDirty = true; });

    b_fullName.setBinding([&]() {
        return QString(tbStyle->fullNameFormat()).arg(firstName()).arg(lastName());
    });
    b_congregationName.setBinding([&]() {
        sql_items c = sql->selectSql("congregations", "id", QString::number(this->congregationId()), "");
        if (!c.empty()) {
            return c[0].value("name").toString();
        }
        return QString();
    });
    b_isValid.setBinding([&]() {
        // check validity of a person
        return congregationId() > 0 && !this->firstName().isEmpty() && !this->lastName().isEmpty();
    });
}

// id
int Person::id()
{
    return m_id;
}

void Person::setId(int id)
{
    if (m_id == id)
        return;
    m_id = id;
    emit idChanged();
}

bool Person::isNew()
{
    return m_id < 1;
}

QString Person::uuid() const
{
    return m_uuid;
}

sql_items Person::congregationInfo()
{
    return sql->selectSql("congregations", "id", QString::number(this->congregationId()), "");
}

QList<QPair<QDate, QDate>> Person::getUnavailabilities()
{
    // list person's all unavailabilities
    QList<QPair<QDate, QDate>> list;
    sql_item s;
    s.insert("person_id", this->id());
    sql_items unavailabilities = sql->selectSql("SELECT * FROM unavailables WHERE person_id=:person_id AND active ORDER BY start_date DESC", &s);
    if (!unavailabilities.empty()) {
        for (unsigned int i = 0; i < (unavailabilities.size()); i++) {
            sql_item u = unavailabilities[i];
            QDate start = u.value("start_date").toDate();
            QDate end = u.value("end_date").toDate();
            list.append(qMakePair(start, end));
        }
    }
    return list;
}

void Person::setUnavailability(QDate start, QDate end)
{
    // add new unvailability for person
    sql_item s;
    s.insert("person_id", this->id());
    s.insert("start_date", start);
    s.insert("end_date", end);
    sql->insertSql("unavailables", &s, "id");
}

bool Person::removeUnavailability(QDate start, QDate end)
{
    // remove unavailability
    sql_item s;
    s.insert("start_date", start);
    s.insert("end_date", end);
    s.insert("active", 0);
    return sql->execSql("UPDATE unavailables SET active = :active, time_stamp = strftime('%s','now') WHERE start_date = :start_date AND end_date = :end_date", &s, true);
}

bool Person::save()
{
    if (not this->isValid())
        return false;

    AccessControl *ac = &Singleton<AccessControl>::Instance();
    if (!ac->user()
        || !(ac->user()->hasPermission(PermissionRule::CanEditPublishers)
             || ac->user()->hasPermission(PermissionRule::CanEditPrivileges)
             || ac->user()->hasPermission(PermissionRule::CanEditPublicSpeakers)))
        return false;

    if (!isDirty())
        return true;

    // save person's information to database
    sql_item s;
    // now that sql_class::insertSql uses parameters, the apostrophes are not an issue
    s.insert("lastname", this->lastName());
    s.insert("firstname", this->firstName());
    s.insert("phone", this->phone());
    s.insert("mobile", this->mobile());
    s.insert("email", this->email());
    s.insert("servant", this->servant());
    s.insert("usefor", this->usefor());
    s.insert("gender", this->gender() == Male ? "B" : "S");
    s.insert("congregation_id", this->congregationId());
    s.insert("info", this->info());

    setIsDirty(false);
    if (this->isNew()) {
        int newID = sql->insertSql("persons", &s, "id");
        if (newID < 1)
            return false;
        setId(newID);
        setIsDirty(false);
        return true;
    }
    return sql->updateSql("persons", "id", QString::number(this->id()), &s);
}
