#include "shareutils.h"
#include <QJniObject>
#include <QDebug>
#include <jni.h>
#include <QDir>
#include <QFile>
#include <QUrl>
#include <QIODevice>

ShareUtils *ShareUtils::mInstance = nullptr;

ShareUtils::ShareUtils(QObject *parent)
    : QObject(parent), m_receivedUrl("")
{
    mInstance = this;
}

ShareUtils *ShareUtils::getInstance()
{
    if (!mInstance)
        mInstance = new ShareUtils();
    return mInstance;
}

void ShareUtils::setReceivedUrl(QString path)
{
    if (m_receivedUrl == path)
        return;
    m_receivedUrl = path;
    emit receivedUrlChanged();
}

QString ShareUtils::receivedUrl()
{
    if (m_receivedUrl.isEmpty())
        QJniObject::callStaticMethod<void>("net/theocbase/mobile/TBActivity",
                                           "checkReceivedUrl");
    return m_receivedUrl;
}

void ShareUtils::saveBackup()
{
    QString dbfilename = QStandardPaths::writableLocation(QStandardPaths::AppDataLocation) + "/theocbase.sqlite";
    if (!QFile::exists(dbfilename)) {
        qWarning() << "Database file not found" << dbfilename;
        return;
    }

    QString docLocation = QStandardPaths::standardLocations(QStandardPaths::AppDataLocation).value(0);
    docLocation.append("/share");
    if (!QDir(docLocation).exists()) {
        if (!QDir("").mkpath(docLocation)) {
            qWarning() << "Failed to create directory" << docLocation;
            return;
        }
    }

    QString tempFilePath = docLocation.append("/theocbase_backup.sqlite");
    if (QFile::exists(tempFilePath))
        QFile::remove(tempFilePath);
    if (!QFile::copy(dbfilename, tempFilePath)) {
        qWarning() << "Failed to copy temporary file. From: " << dbfilename << "To: " << tempFilePath;
        return;
    }
    QFile(tempFilePath).setPermissions(QFileDevice::ReadUser | QFileDevice::WriteUser);

    QJniObject javaFileName = QJniObject::fromString(tempFilePath);

    QJniEnvironment env;
    QJniObject::callStaticMethod<void>("net/theocbase/mobile/TBActivity",
                                       "saveFile",
                                       "(Ljava/lang/String;)V",
                                       javaFileName.object<jstring>());

    if (env->ExceptionCheck()) {
        qDebug() << " JNI exception ";
        env->ExceptionDescribe();
        env->ExceptionClear();
    } else {
        qDebug() << "OK";
    }
}

void ShareUtils::sendMail()
{
    QJniEnvironment env;
    QString dbfilename = QStandardPaths::writableLocation(QStandardPaths::AppDataLocation) + "/theocbase.sqlite";

    if (!QFile::exists(dbfilename)) {
        qWarning() << "Database file not found" << dbfilename;
        return;
    }

    QJniObject javaFileName = QJniObject::fromString(dbfilename);
    qDebug() << "db filename " << dbfilename;

    if (QJniObject::isClassAvailable("net/theocbase/mobile/TBActivity")) {
        qDebug() << "class available";
    } else {
        qDebug() << "class not found";
        return;
    }

    //    QJniObject data =  QJniObject::callStaticObjectMethod<jstring>
    //            ("net/theocbase/mobile/TBActivity",
    //             "myTest");
    //    qDebug() << "myTest" << data.toString();

    //    int intTest = QJniObject::callStaticMethod<jint>
    //            ("net/theocbase/mobile/TBActivity",
    //             "myTest2",
    //             "(I)I",
    //             23);
    //    qDebug() << "myTest2" << intTest;

    QJniObject::callStaticMethod<void>("net/theocbase/mobile/TBActivity",
                                       "sendMail",
                                       "(Ljava/lang/String;)V",
                                       javaFileName.object<jstring>());
    if (env->ExceptionCheck()) {
        // Handle exception here.
        qDebug() << " JNI exception ";
        env->ExceptionDescribe();
        env->ExceptionClear();
    } else {
        qDebug() << "java call OK";
    }
}

void ShareUtils::shareText(const QString text)
{
    QJniEnvironment env;
    QJniObject textString = QJniObject::fromString(text);
    QJniObject::callStaticMethod<void>("net/theocbase/mobile/TBActivity",
                                       "shareText",
                                       "(Ljava/lang/String;)V",
                                       textString.object<jstring>());

    if (env->ExceptionCheck()) {
        qDebug() << " JNI exception ";
        env->ExceptionDescribe();
        env->ExceptionClear();
    } else {
        qDebug() << "OK";
    }
}

void ShareUtils::handleUrlReceived(const QUrl &url)
{
    setReceivedUrl(url.toString());
}

QString ShareUtils::openFile(QString format)
{
    QJniObject javaUrlString = QJniObject::fromString(format);
    QJniObject::callStaticMethod<void>("net/theocbase/mobile/TBActivity",
                                       "openFileBrowser",
                                       "(Ljava/lang/String;)V",
                                       javaUrlString.object<jstring>());
    return format;
}

bool ShareUtils::copyFile(QString sourcePath, QString destPath)
{
    const QUrl sourceUrl(sourcePath);
    if (sourceUrl.isLocalFile())
        sourcePath = sourceUrl.toLocalFile();
    const QUrl destUrl(destPath);
    if (destUrl.isLocalFile())
        destPath = destUrl.toLocalFile();

    QFile sourceFile(sourcePath);
    QFile destFile(destPath);
    if (destFile.exists())
        destFile.remove();
    bool copied = sourceFile.copy(destPath);
    return copied;
}

bool ShareUtils::deleteFile(QString file)
{
    const QUrl fileUrl(file);
    if (fileUrl.isLocalFile())
        file = fileUrl.toLocalFile();
    return QFile::remove(file);
}

#ifdef __cplusplus
extern "C" {
#endif

JNIEXPORT void JNICALL
Java_net_theocbase_mobile_TBActivity_setUrlReceived(JNIEnv *env,
                                                    jobject obj,
                                                    jstring uri)
{
    Q_UNUSED(obj);
    jboolean isCopy = false;
    const char *utf = env->GetStringUTFChars(uri, &isCopy);
    qDebug() << "C++" << utf;

    QUrl url(QString { utf });
    env->ReleaseStringUTFChars(uri, utf);

    if (url.toString().startsWith("theocbase")) {
        // oauth2 redirect
    } else {
        ShareUtils::getInstance()->setReceivedUrl(url.toString());
    }
}
#ifdef __cplusplus
}
#endif
