/**
 * This file is part of TheocBase.
 *
 * Copyright (C) 2011-2020, TheocBase Development Team, see AUTHORS.
 *
 * TheocBase is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * TheocBase is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with TheocBase.  If not, see <http://www.gnu.org/licenses/>.
 */
#include <QApplication>
#include <QQmlApplicationEngine>
#include <QQuickView>
#include <QQmlContext>
#include <QQmlComponent>
#include <QQuickItem>
#include <QObject>
#include <QString>
#include <QDebug>
#include <QObject>
#include <QStandardPaths>
#include <QQmlPropertyMap>
#include <QScreen>
#include "src/mobileapplication.h"
#include "../../sql_class.h"
#include "publishers_model.h"
#include "../../cloud/cloud_controller.h"
#include "messagebox.h"
#include "qmltranslator.h"
#include <QScreen>
#include <QtMath>
#include "../../lmm_meeting.h"
#include "../../lmm_schedule.h"
#include "../../lmm_assignment.h"
#include "../../lmm_assignmentcontoller.h"
#include "../../publicmeeting_controller.h"
#include "../../csync.h"
#include "../../importlmmworkbook.h"
#include "../../wtimport.h"
#include "../../outgoingspeakersmodel.h"
#include "../../weekinfo.h"
#include "../../cterritories.h"
#include "printcontroller.h"
#include "src/territorymanager.h"

#if defined(Q_OS_IOS)
#include "iosutil.h"
#endif

#if defined(Q_OS_ANDROID) || defined(Q_OS_IOS)
#include "shareutils.h"
#endif

bool transactionStarted;

void initDatabase()
{
    // database from resource file
    QString databasepath = ":/database/theocbase.sqlite";
    QString localdatabasedir = QStandardPaths::writableLocation(QStandardPaths::AppDataLocation);
    QString localdatabasepath = localdatabasedir + "/theocbase.sqlite";

    QDir dir(localdatabasedir);
    if (!dir.exists()) {
        if (!dir.mkpath(localdatabasedir)) {
            qDebug() << "directory not created!";
            QMessageBox::warning(nullptr, "TheocBase", "Directory not found:\n" + localdatabasedir);
            return;
        }
        QFile oldfile(QStandardPaths::writableLocation(QStandardPaths::DocumentsLocation) + "/theocbase.sqlite");
        if (oldfile.exists())
            oldfile.copy(localdatabasepath);
    }

    if (!QFile::exists(localdatabasepath)) {
        // copy database to user's local
        QFile::copy(databasepath, localdatabasepath);
        QFile::setPermissions(localdatabasepath, QFile::ReadOwner | QFile::WriteOwner | QFile::ReadUser | QFile::WriteUser | QFile::ReadGroup | QFile::WriteOwner | QFile::ReadOther | QFile::WriteOther);
        qDebug() << "New database copied to " + localdatabasedir;
        // QMessageBox::information(0,"",QObject::tr("Database copied to ") + localdatabasedir);
    } else {
        qDebug() << "Use existing database!";
    }

    sql_class *sql = &Singleton<sql_class>::Instance();
    sql->databasepath = localdatabasepath;
    sql->createConnection();
    sql->updateDatabase(APP_VERSION);
}

int main(int argc, char *argv[])
{
    qputenv("QT_QUICK_CONTROLS_STYLE", "Material");
    qputenv("QT_QUICK_CONTROLS_CONF", ":/style/qtquickcontrols2.conf");
    MobileApplication app(argc, argv);
    app.setApplicationName("TheocBase");
    app.setOrganizationDomain("mobile.theocbase.net");

    qDebug() << app.applicationDirPath();

    initDatabase();

    qmlRegisterType<publishers_modelview>("net.theocbase.mobile", 1, 0, "Publishers");
    qmlRegisterType<Person>("net.theocbase.mobile", 1, 0, "Publisher");
    qRegisterMetaType<Person::UseFor>("UseFor");
    qmlRegisterSingletonType<cpersons>("net.theocbase.mobile", 1, 0, "CPersons", [](QQmlEngine *engine, QJSEngine *scriptEngine) -> QObject * {
        Q_UNUSED(engine);
        Q_UNUSED(scriptEngine);
        return new cpersons();
    });
    qmlRegisterType<PersonsModel>("net.theocbase.mobile", 1, 0, "PersonsModel");
    qmlRegisterType<PersonsSortFilterProxyModel>("net.theocbase.mobile", 1, 0, "PersonsSFProxyModel");
    qmlRegisterType<family>("net.theocbase.mobile", 1, 0, "Family");
    qmlRegisterType<MessageBox>("net.theocbase.mobile", 1, 0, "MsgBox");
    qmlRegisterType<cloud_controller>("net.theocbase.mobile", 1, 0, "Cloud");
    qmlRegisterType<StorageService>("net.theocbase.mobile", 1, 0, "Dropbox");
    qmlRegisterType<LMM_Schedule>("net.theocbase.mobile", 1, 0, "LMM_Schedule");
    qmlRegisterType<LMM_Meeting>("net.theocbase.mobile", 1, 0, "LMM_Meeting");
    qmlRegisterType<LMM_Assignment>("net.theocbase.mobile", 1, 0, "LMM_Assignment");
    qmlRegisterType<LMM_AssignmentContoller>("net.theocbase.mobile", 1, 0, "AssignmentController");
    qmlRegisterType<SortFilterProxyModel>("net.theocbase.mobile", 1, 0, "SortFilterProxyModel");
    qmlRegisterType<ccongregation>("net.theocbase.mobile", 1, 0, "CongregationCtrl");
    qmlRegisterType<publicmeeting_controller>("net.theocbase.mobile", 1, 0, "PublicMeetingController");
    qmlRegisterType<cptmeeting>("net.theocbase.mobile", 1, 0, "CPTMeeting");
    qmlRegisterType<csync>("net.theocbase.mobile", 1, 0, "CSync");
    qmlRegisterType<OutgoingSpeakersModel>("net.theocbase.mobile", 1, 0, "OutgoingSpeakersModel");
    qmlRegisterType<importlmmworkbook>("net.theocbase.mobile", 1, 0, "WorkbookImport");
    qmlRegisterType<wtimport>("net.theocbase.mobile", 1, 0, "WTImport");
    qmlRegisterType<AccessControl>("net.theocbase.mobile", 1, 0, "AccessControl");
    qRegisterMetaType<SpecialEvent>("SpecialEvent");
    qmlRegisterUncreatableType<SpecialEventClass>("net.theocbase.mobile", 1, 0, "SpecialEvent", "Not creatable as it is an enum type");
    qRegisterMetaType<SpecialEventExclusivity>("SpecialEventExclusivity");
    qmlRegisterUncreatableType<SpecialEventExclusivityClass>("net.theocbase.mobile", 1, 0, "SpecialEventExclusivity", "Not creatable as it is an enum type");
    qmlRegisterType<SpecialEventRule>("net.theocbase.mobile", 1, 0, "SpecialEventRule");
    qmlRegisterSingletonType<SpecialEvents>("net.theocbase.mobile", 1, 0, "SpecialEvents", &SpecialEvents::qmlInstance);
    qmlRegisterType<WeekInfo>("net.theocbase.mobile", 1, 0, "WeekInfo");
    qmlRegisterType<PrintController>("net.theocbase.mobile", 1, 0, "PrintController");
    qmlRegisterUncreatableType<TemplateData>("net.theocbase.mobile", 1, 0, "TemplateData", "Not creatable as it is an enum type.");
    qRegisterMetaType<MeetingType>("MeetingType");
    qmlRegisterUncreatableType<MeetingTypeClass>("net.theocbase.mobile", 1, 0, "MeetingType", "Not creatable as it is an enum type");
    qRegisterMetaType<MeetingSection>("MeetingSection");
    qmlRegisterUncreatableType<MeetingSectionClass>("net.theocbase.mobile", 1, 0, "MeetingSection", "Not creatable as it is an enum type");
    qRegisterMetaType<MeetingPart>("MeetingPart");
    qmlRegisterUncreatableType<MeetingPartClass>("net.theocbase.mobile", 1, 0, "MeetingPart", "Not creatable as it is an enum type");
    qRegisterMetaType<AssignmentType>("AssignmentType");
    qmlRegisterUncreatableType<AssignmentTypeClass>("net.theocbase.mobile", 1, 0, "AssignmentType", "Not creatable as it is an enum type");
    qmlRegisterType<AssignmentInfo>("net.theocbase.mobile", 1, 0, "AssignmentInfo");
    qmlRegisterSingletonType<AssignmentInfos>("net.theocbase.mobile", 1, 0, "AssignmentInfos", &AssignmentInfos::qmlInstance);
    qmlRegisterSingletonType<TBStyle>("net.theocbase.mobile", 1, 0, "TBStyle", &TBStyle::qmlInstance);
    qmlRegisterType<DataObjectListModel>("net.theocbase.mobile", 1, 0, "DataObjectListModel");
    qmlRegisterType<DataObjectListSortFilterProxyModel>("net.theocbase.mobile", 1, 0, "DataObjectListSFProxyModel");
    qmlRegisterType<Territory>("net.theocbase.mobile", 1, 0, "Territory");
    qmlRegisterType<TerritoryTreeItem>("net.theocbase.mobile", 1, 0, "TerritoryTreeItem");
    qmlRegisterType<TerritoryTreeModel>("net.theocbase.mobile", 1, 0, "TerritoryTreeModel");
    qmlRegisterType<TerritoryTreeSFProxyModel>("net.theocbase.mobile", 1, 0, "TerritoryTreeSFProxyModel");
    qmlRegisterType<cterritories>("net.theocbase.mobile", 1, 0, "Territories");
    qmlRegisterType<TerritoryAssignmentModel>("net.theocbase.mobile", 1, 0, "TerritoryAssignmentModel");
    qmlRegisterType<TerritoryAssignmentValidator>("net.theocbase.mobile", 1, 0, "TerritoryAssignmentValidator");
    qmlRegisterType<TerritoryStreet>("net.theocbase.mobile", 1, 0, "TerritoryStreet");
    qmlRegisterType<TerritoryStreetModel>("net.theocbase.mobile", 1, 0, "TerritoryStreetModel");
    qmlRegisterType<TerritoryStreetValidator>("net.theocbase.mobile", 1, 0, "TerritoryStreetValidator");
    qmlRegisterType<TerritoryAddress>("net.theocbase.mobile", 1, 0, "TerritoryAddress");
    qmlRegisterType<TerritoryAddressModel>("net.theocbase.mobile", 1, 0, "TerritoryAddressModel");
    qmlRegisterType<TerritoryAddressValidator>("net.theocbase.mobile", 1, 0, "TerritoryAddressValidator");
    qmlRegisterType<GeocodeResult>("net.theocbase.mobile", 1, 0, "GeocodeResult");
    qmlRegisterType<GeocodeResultModel>("net.theocbase.mobile", 1, 0, "GeocodeResultModel");
    qmlRegisterType<GeocodeResultSortFilterProxyModel>("net.theocbase.mobile", 1, 0, "GeocodeResultSFProxyModel");
    qmlRegisterType<StreetResult>("net.theocbase.mobile", 1, 0, "StreetResult");
    qmlRegisterType<StreetResultModel>("net.theocbase.mobile", 1, 0, "StreetResultModel");
    qmlRegisterType<StreetResultSortFilterProxyModel>("net.theocbase.mobile", 1, 0, "StreetResultSFProxyModel");

#if defined(Q_OS_ANDROID) || defined(Q_OS_IOS)
    qmlRegisterType<ShareUtils>("net.theocbase.mobile", 1, 0, "ShareUtils");
#endif
    // qmlRegisterSingletonType() works without instance...

    ShareUtils sUtils;

    QQmlApplicationEngine engine;
    QQmlContext *ctxt = engine.rootContext();
    ctxt->setContextProperty("shareUtils", &sUtils);
    QmlTranslator qmlTranslator;
    qmlTranslator.initTranslator();
    ctxt->setContextProperty("qmlTranslator", &qmlTranslator);

    // Access control
    AccessControl *ac = &Singleton<AccessControl>::Instance();
    ctxt->setContextProperty("accessControl", ac);

    QQmlPropertyMap appInfo;
    ctxt->setContextProperty("app_info", &appInfo);
    appInfo.insert("version", QString(APP_VERSION));
    // font sizes
    QFont defaultfont;
    TBStyle *tbStyle = &TBStyle::Instance();
    // set toolbar background color

    appInfo.insert("fontpixels", defaultfont.pixelSize());
    appInfo.insert("fontsizepoint", defaultfont.pointSize());
    appInfo.insert("fontsize", tbStyle->bodyMediumFont().pointSize());
    appInfo.insert("fontsizeSmall", tbStyle->bodySmallFont().pointSize());
    appInfo.insert("devicepixelratio", qApp->primaryScreen()->devicePixelRatio());
    appInfo.insert("uilanguages", QLocale::system().uiLanguages().join(","));

    // calculate screen size
    qreal w = qApp->primaryScreen()->physicalSize().width();
    qreal h = qApp->primaryScreen()->physicalSize().height();
    qreal inches = qSqrt(w * w + h * h) / 25.4;
    appInfo.insert("screensize", QString::number(inches, 'f', 2).toDouble());
    qDebug() << "Screen size: " << appInfo.value("screensize");

    // Territory Manager
    TerritoryManager *territoryManager = new TerritoryManager();
    TerritoryTreeModel *territoryTreeModel = new TerritoryTreeModel();
    // assignments
    TerritoryAssignmentModel *assignmentModel = new TerritoryAssignmentModel();
    TerritoryAssignmentSortFilterProxyModel *assignmentProxyModel = new TerritoryAssignmentSortFilterProxyModel();
    assignmentProxyModel->setSourceModel(assignmentModel);
    // streets
    TerritoryStreetModel *streetModel = new TerritoryStreetModel();
    TerritoryStreetSortFilterProxyModel *streetProxyModel = new TerritoryStreetSortFilterProxyModel();
    streetProxyModel->setSourceModel(streetModel);
    // addresses
    TerritoryAddressModel *addressModel = new TerritoryAddressModel();
    TerritoryAddressSortFilterProxyModel *addressProxyModel = new TerritoryAddressSortFilterProxyModel();
    addressProxyModel->setSourceModel(addressModel);
    ctxt->setContextProperty("territoryManager", territoryManager);
    ctxt->setContextProperty("territoryTreeModel", territoryTreeModel);
    ctxt->setContextProperty("typeListModel", territoryManager->typeListModel());
    ctxt->setContextProperty("cityListModel", territoryManager->cityListModel());
    ctxt->setContextProperty("assignmentModel", assignmentModel);
    ctxt->setContextProperty("assignmentProxyModel", assignmentProxyModel);
    ctxt->setContextProperty("streetModel", streetModel);
    ctxt->setContextProperty("streetProxyModel", streetProxyModel);
    ctxt->setContextProperty("addressModel", addressModel);
    ctxt->setContextProperty("addressProxyModel", addressProxyModel);
    ctxt->setContextProperty("defaultGeoServiceProvider", territoryManager->defaultGeoServiceProviderName());
    ctxt->setContextProperty("geoServiceParameters", QVariant::fromValue(territoryManager->geoServiceParameters()));
#if defined(Q_OS_ANDROID) || defined(Q_OS_IOS)
    ctxt->setContextProperty("isGdalAvailable", false);
#elif defined(Q_OS_LINUX) || defined(Q_OS_MAC) || defined(Q_OS_WIN)
    ctxt->setContextProperty("isGdalAvailable", true);
#endif

#if defined(Q_OS_IOS)
    iosutil ios_util;
    ctxt->setContextProperty("IOSUtil", &ios_util);
    // ios fix
    // QObject::connect(&app, &QApplication::paletteChanged, &ios_util, &iosutil::initiOS);
    ios_util.initiOS();
    appInfo.insert("devicename", iosutil::getDeviceName());
#endif

    engine.load(QUrl(QStringLiteral("qrc:///main.qml")));
#ifdef Q_OS_ANDROID
    QNativeInterface::QAndroidApplication::hideSplashScreen();
    // set status bar color
    QJniObject activity = QNativeInterface::QAndroidApplication::context();
    QJniObject window = activity.callObjectMethod("getWindow", "()Landroid/view/Window;");
    window.callMethod<void>("setStatusBarColor", "(I)V", tbStyle->primaryColor().rgba());
#endif

    return app.exec();
}
