/**
 * This file is part of TheocBase.
 *
 * Copyright (C) 2011-2019, TheocBase Development Team, see AUTHORS.
 *
 * TheocBase is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * TheocBase is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with TheocBase.  If not, see <http://www.gnu.org/licenses/>.
 */
import QtCore
import QtQuick
import QtQuick.Window
import QtQuick.Controls
import QtQuick.Controls.Material
import QtQuick.Layouts
import net.theocbase.mobile 1.0
import net.theocbase 1.0
import "js/moment.js" as Moment

ApplicationWindow {   
    id: appWindow

    property var appState: Qt.application.state
    property int activepage: 1

    property bool canViewMidweekMeetingSchedule: accessControl.user && accessControl.user.hasPermission(PermissionRule.CanViewMidweekMeetingSchedule)
    property bool canEditMidweekMeetingSchedule: accessControl.user && accessControl.user.hasPermission(PermissionRule.CanEditMidweekMeetingSchedule)
    property bool canViewWeekendMeetingSchedule: accessControl.user && accessControl.user.hasPermission(PermissionRule.CanViewWeekendMeetingSchedule)
    property bool canEditWeekendMeetingSchedule: accessControl.user && accessControl.user.hasPermission(PermissionRule.CanEditWeekendMeetingSchedule)
    property bool canViewPublishers: accessControl.user && (accessControl.user.hasPermission(PermissionRule.CanViewPublishers))
    property bool canEditPublishers: accessControl.user && (accessControl.user.hasPermission(PermissionRule.CanEditPublishers))
    property bool canViewMeetingNotes: accessControl.user && (accessControl.user.hasPermission(PermissionRule.CanViewMeetingNotes))
    property bool canEditMeetingNotes: accessControl.user && (accessControl.user.hasPermission(PermissionRule.CanEditMeetingNotes))
    property SpecialEventRule specialEventRule

    function reloadMainpage(){
        stackView.navigation.get(0).refreshList(0)
    }

    // activate home, publishers or settings page
    function openPage(page){
        switch (page) {
        case 1:
            if (stackView.navigation.depth === 0)
                stackView.navigation.push(Qt.resolvedUrl("MainPage.qml"), { objectName: "MainPage" }, StackView.Immediate);
            else if (stackView.navigation.currentItem.objectName === "MainPage")
               stackView.navigation.currentItem.refreshList();
            else
                stackView.navigation.pop(null, StackView.Immediate);
            stackView.navigation.get(0).activateTabByDay();
            break;
        case 2:
            stackView.navigation.pop(null);
            stackView.navigation.push(Qt.resolvedUrl("TerritoryManagerPage.qml"), { objectName : "TerritoryManagerPage" }, StackView.Immediate);
            break;
        case 3:
            stackView.navigation.pop(null);
            stackView.navigation.push(Qt.resolvedUrl("PublishersPage.qml"), { objectName : "PublishersPage" }, StackView.Immediate);
            break;
        case 4:
            stackView.navigation.pop(null);
            stackView.navigation.push(Qt.resolvedUrl("SettingsPage.qml"), StackView.Immediate);
            break;
        default: return;
        }
    }

    function readImportFile() {
        if (shareUtils.receivedUrl !== undefined && shareUtils.receivedUrl !== "") {
            var f = shareUtils.receivedUrl
            shareUtils.receivedUrl = ""

            if (!ccloud.isLoggedIn) {
                msg.show("Import File", "You are not logged in. Please log in and try again.")
                return
            }

            console.log(`Try to import "${f}"`)
            var filename = f.substring(f.lastIndexOf('/')+1)
            console.log(filename)
            var result = ""
            if (/^.*(.thb)$/.test(filename)) {
                // thb file
                var clickedFunc = function(ok) {
                    if (ok) {
                        var p = stackView.find(function(item,index){return item.objectName === "DataExchange"})
                        if (p === null) {
                            stackView.push(Qt.resolvedUrl("DataExchange.qml"),
                                           {objectName: "DataExchange", filePath: f}, StackView.Immediate)
                        }else if (p === stackView.currentItem) {
                            p.readCurrentFile(f)
                        }else{
                            stackView.pop(p,
                                          StackView.Immediate,
                                          {filePath: f})
                        }
                    }
                    msg.onButtonClicked.disconnect(clickedFunc)
                }
                msg.onButtonClicked.connect(clickedFunc)
                msg.showYesNo("Import File","Do you want to import .thb-file?",-1)
                return
            } else if (/^w_.*(.epub)$/.test(filename)) {
                result = wtImport.importFile(f)
            } else if (/^mwb_.*(.epub)$/.test(filename)) {
                result = mwbImport.importFile(f)
            } else {
                result = "Unknown file type"
            }
            msg.show("", result)
            reloadMainpage()
        }
    }

    width: 480
    height: 800
    title: "TheocBase"
    flags: Qt.Window | Qt.ExpandedClientAreaHint | Qt.NoTitleBarBackgroundHint
    visible: true
    visibility: settings_ui.fullScreen ? Window.FullScreen : Window.Maximized

    Material.background: palette.window
    Material.foreground: palette.windowText

    onClosing: (close) => {
        console.log("Closing signal received");
        if (Qt.platform.os === "android" || Qt.platform.os === "ios") {
            close.accepted = false;
            if (stackView.detail.depth > 0) {
                console.log("Close detail page");
                stackView.pop();
            }
            else if (stackView.navigation.depth > 1) {
                console.log("Return to main page");
                stackView.navigation.pop();
            }
            else
                Qt.quit();
        }
    }

    SystemPalette { id: palette }
    WorkbookImport { id: mwbImport }
    WTImport { id: wtImport }
    Connections {
        id: shareUtilsConnection
        target: shareUtils
        function onReceivedUrlChanged() {
            if (appState === Qt.ApplicationActive)
                readImportFile()
            else
                console.log("onReceivedUrlChanged !!! app is not in active state")
        }
    }

    onAppStateChanged: {
        if (appState === Qt.ApplicationActive){
            console.log("application state active")
            if(Qt.platform.os == "ios"){
                IOSUtil.orientationChanged(Screen.orientation)
            }

            cloudtimer.start()
            console.log("login requested ??????? " + settings_ui.loginRequested.toString())
            if (shareUtilsConnection.enabled)
                readImportFile()
        }else if (appState === Qt.ApplicationInactive){
            console.log("application state inactive")
        }else if (appState === Qt.ApplicationSuspended){
            console.log("application state suspended")
            //console.log("logged = " + ccloud.logged())
            //console.log("updates available = " + ccloud.checkCloudUpdates())
            //backgroundDebug
            //var d = new Date()
            //ccloud.backgroundDebug = "Background test " + Qt.formatDateTime(d, Qt.locale(), Locale.ShortFormat)
        }else if (appState === Qt.ApplicationHidden){
            console.log("application state hidden")
        }
    }

    SynchronizePage {
        id: syncpage
        z: 2
    }

    Timer {
        id: cloudtimer
        interval: 5000
        onTriggered: {
            var differenceMinutes = Math.floor((new Date().getTime() - settings_ui.lastCloudUpdates.getTime())/1000/60)
            if (differenceMinutes > 5){
                // more than 5 minutes from last check
                if (ccloud.isLoggedIn){
                    // logged - check updates
                    ccloud.showError = false
                    ccloud.checkCloudUpdates()
                    ccloud.showError = true
                    // save timestamp
                    settings_ui.lastCloudUpdates = new Date()
                }
            }
        }
    }

    Cloud {
        id: ccloud
        property bool autoSync: false
        property bool showError: true
        property bool running: false
        onStateChanged: function(state) {
            if (state === Cloud.Both){
                console.log("Sync state changed: both changes")
            }else if (state === Cloud.Upload){
                console.log("Sync state changed: upload")
            }else if (state === Cloud.Download){
                console.log("Sync state changed: download")
            }
        }
        onSyncConflict: function(values) {
            console.log("sync conflict");
            syncpage.visible = false
            msg.showYesNo("TheocBase Cloud",
                          qsTr("The same changes can be found both locally and in the cloud (%1 rows). Do you want keep the local changes?")
                          .arg(values.toString()),2);
        }
        onSyncStarted: running = true
        onSyncFinished: {
            running = false
            console.log("sync finished")
            console.log(stackView.navigation.depth)
            var mainPage = stackView.navigation.find(function(item, index) { return item.objectName === "MainPage" })
            mainPage.refreshList(0)
            //stackView.navigation.get(0).refreshList(0);
            switch(stackView.navigation.depth) {
            case 1:
                mainPage.activateTabByDay()
                //stackView.navigation.get(0).activateTabByDay()
                break
            case 2:
                //var p = stackView.find(function(item,index){return item.objectName === "DataExchange"})
                var item = stackView.navigation.find(function(item, index) { return item.objectName === "PublishersPage" }) //.get(1).loadList()
                if (item !== null)
                    item.loadList()
                break
            }
            territoryManager.initialize();
            assignmentModel.initialize();
            streetModel.initialize();
            addressModel.initialize();
            settings_ui.lastCloudUpdates = new Date();
            if (activepage === 2) {
                // reload territory page
                openPage(2);
            }
        }
        onError: function(message) {
            syncpage.visible = false;
            if (showError)
                msg.show("TheocBase Cloud", message);
        }
        onDifferentLastDbUser: {
            syncpage.visible = false;
            msg.show("TheocBase Cloud",
                     "The database has been synchronized last time by other cloud user.");
        }

        onCloudResetFound: {
            syncpage.visible = false;
            msg.showYesNo("TheocBase Cloud",
                          qsTr("The cloud data has been reset. Your local data will be replaced. Continue?"),3);
        }
        onAuthorizeWithBrowserStarted: {
            console.log("Browser opened to login");
            settings_ui.loginRequested = true;
        }
        onIsLoggedInChanged: {
            if (ccloud.isLoggedIn && !moment(ccloud.lastSyncTime).isValid())
                syncpage.runSync()            
        }
        Component.onCompleted: {
            ccloud.initAccessControl()
        }
    }

    MsgBox {
        id: msg        
        onButtonClicked: function(ok, id) {
            switch (id){
            case 1:
                // sync updates
                if (ok) syncpage.runSync();
                break
            case 2:
                // sync conflict
                syncpage.continueSync(ok);                
                break
            case 3:
                // cloud data reset
                if (ok) {
                    ccloud.clearDatabase();
                    syncpage.runSync();
                }
                break;
            default: break
            }
        }
    }
    Settings {
        id: settings_ui
        category: "ui"
        property bool showTime: true
        property bool showSongTitles: false
        property date lastCloudUpdates: new Date('2016-01-01')
        property bool loginRequested: false
        property bool fullScreen: false
    }

    header: ToolBar {
        background: Rectangle {
            color: TBStyle.primaryColor
        }
    }

    footer: ToolBar {
        id: navbar

        Material.background: palette.button
        Material.foreground: palette.buttonText
        Material.accent: TBStyle.primaryTextColor

        RowLayout {
            anchors.fill: parent
            spacing: 0
            ToolButton {
                id: todayButton
                Layout.fillWidth: true
                icon.source: checked ? "qrc:/icons/home_filled.svg" : "qrc:/icons/home.svg"
                checked: activepage === 1
                onClicked: {
                    openPage(1);
                }
            }
            ToolButton {
                id: territoryManagerButton
                Layout.fillWidth: true
                icon.source: checked ? "qrc:/icons/territories_filled.svg" : "qrc:/icons/territories.svg"
                checked: activepage === 2
                onClicked: {
                    openPage(2);
                }
            }
            ToolButton {
                id: publishersButton
                Layout.fillWidth: true
                icon.source: checked ? "qrc:/icons/manage_accounts_filled.svg" : "qrc:/icons/manage_accounts.svg"
                checked: activepage === 3
                visible: canViewPublishers
                onClicked: {
                    openPage(3);
                }
            }
            ToolButton {
                id: settingsButton
                Layout.fillWidth: true
                icon.source: checked ? "qrc:/icons/settings_filled.svg" : "qrc:/icons/settings.svg"
                checked: activepage === 4
                onClicked: {
                    openPage(4);
                }
            }
        }
    }

    NavigationStack {
        id: stackView
        anchors {
            fill: parent
            topMargin: parent.SafeArea.margins.top
            leftMargin: parent.SafeArea.margins.left
            rightMargin: parent.SafeArea.margins.right
        }
        Component.onCompleted: {
            openPage(1)
            if (!ccloud.isLoggedIn){
                if (!settings_ui.loginRequested)
                    ccloud.login()
            }
        }
    }
}

