/**
 * This file is part of TheocBase.
 *
 * Copyright (C) 2011-2024, TheocBase Development Team, see AUTHORS.
 *
 * TheocBase is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * TheocBase is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with TheocBase.  If not, see <http://www.gnu.org/licenses/>.
 */
import QtCore
import QtQuick
import QtQuick.Controls
import QtQuick.Controls.Material
import QtQuick.Layouts
import QtQuick.Dialogs
import QtWebView
import net.theocbase.mobile 1.0

Item {
    property date date
    property int meetingType
    property string currentUrl

    function importHtml(url, html) {
        header.enabled = false;
        var result = "";
        if (meetingType === MeetingType.MidweekMeeting) {
            result = mwbImport.importHtml(url, html);
        } else {
            result = wtImport.importHtml(url, html);
        }
        messagebox.text = result;
        messagebox.open();
        header.enabled = true;
    }

    function saveLastWtLocale(classList) {
        var regExp = /ml-(?<wtlocale>\w+)/g, match;
        var wtlocale;
        while (match = regExp.exec(classList))
        {
            wtlocale = match[1];
        }
        settings.lastWtLocale = wtlocale;
    }

    function resetDefaultSettings() {
        settings.url = "https://wol.jw.org/wol/finder?srctype=wol";
        settings.lastLanguage = "";
        settings.lastWtLocale = "";
        settings.sync();
        setUrl(settings.url);
    }

    function setUrl(baseUrl) {
        currentUrl = baseUrl;
        if (!isNaN(date)) {
            var urlstr = baseUrl;
            // try to set document id of the given date
            var documentId = meetingType === MeetingType.WeekendMeeting
                    ? wtImport.getDocumentId(date)
                    : mwbImport.getDocumentId(date);
            if (documentId > 0) {
                urlstr += "&docid=" + documentId;
            } else {
                // set 'meetings' and 'date' to jump to the week and let the user go to schedule
                urlstr += "&alias=meetings&date=" + date.toISOString().substring(0, 10);
            }
            // set locale (may be ignored)
            let lang = Qt.locale().name;
            // set wt locale
            var wtLocale = settings.lastWtLocale;
            if (wtLocale === "")
                wtLocale = mwbImport.getWtLocale(lang);
            if (wtLocale !== "") {
                urlstr += "&wtlocale=" + wtLocale;
            }
            if (settings.lastLanguage != "") {
                urlstr += "&locale=" + settings.lastLanguage;
            } else {
                let match = lang.match("_|-");
                if (match.length > 0) {
                    lang = lang.substring(0, lang.indexOf(match[0]));
                }
                urlstr += "&locale=" + lang;
            }
            console.log("URL: "+ urlstr);
            webView.url = urlstr;
        }
    }

    onCurrentUrlChanged: {
        setUrl(currentUrl);
    }

    Component.onCompleted: {
        setUrl(settings.url);
    }

    Component.onDestruction: {
        settings.url = currentUrl;
        settings.sync();
    }

    MessageDialog {
        id: messagebox
        // icon: StandardIcon.Information
        onButtonClicked: {
            reloadMainpage()
        }
    }

    Settings {
        id: settings
        category: "WebBrowser"
        property string lastLanguage: ""
        property string lastWtLocale: ""
        property string url: "https://wol.jw.org/wol/finder?srctype=wol"
    }
    WorkbookImport { id: mwbImport }
    WTImport { id: wtImport }

    ColumnLayout {
        anchors.fill: parent
        spacing: 0

        BaseToolbar {
            id: header

            Layout.fillWidth: true
            Layout.alignment: Qt.AlignTop
            title: webView.title ? webView.title : qsTr("Watchtower ONLINE LIBRARY")
            componentLeft: ToolButton  {
                icon.source: "qrc:/icons/back.svg"
                onClicked: stackView.pop()
            }
            componentLeftMiddleVisible: false
            componentRightMiddle: Item {
                BusyIndicator {
                    anchors.centerIn: parent
                    running: webView.loading
                    width: Material.buttonHeight
                    height: Material.buttonHeight
                    Material.accent: TBStyle.onPrimaryColor
                }
            }
            componentRightMiddleVisible: webView.loading
            componentRight: Row {
                spacing: 0
                visible: !webView.loading
                ToolButton {
                    visible: canEditMidweekMeetingSchedule
                    anchors.verticalCenter: parent.verticalCenter
                    icon.source: "qrc:/icons/import_contacts.svg"
                    onClicked: {
                        var js = "document.getElementsByTagName(\"article\")[0].outerHTML";
                        webView.runJavaScript(js, function(result) {
                            let html = "<html><body>" + result + "</body></html>";
                            importHtml(webView.url, html);
                        });
                    }
                }
                ToolButton {
                    anchors.verticalCenter: parent.verticalCenter
                    icon.source:  (currentUrl?.search("www.jw.org") ?? -1) > 0
                                  ? "qrc:/icons/wol.svg"
                                  : "qrc:/icons/jw.svg"
                    onClicked: {
                        if (currentUrl === "https://www.jw.org/finder?srctype=wol")
                            currentUrl = "https://wol.jw.org/wol/finder?srctype=wol"
                        else
                            currentUrl = "https://www.jw.org/finder?srctype=wol"
                    }
                }
            }
        }
        WebView {
            id: webView
            Layout.fillWidth: true
            Layout.fillHeight: true

            onLoadingChanged: function(loadRequest) {
                if (loadRequest.status === WebView.LoadSucceededStatus) {
                    console.log("Page loaded: " + loadRequest.url + " " + webView.loading);
                    // save last language
                    let urlparts = loadRequest.url.toString().split("/");
                    urlparts.forEach(function(item, i) {
                        if (item === "wol.jw.org") {
                            settings.lastLanguage = urlparts[i + 1];
                            settings.sync();
                        }
                    })
                    // save last wt locale
                    var js = "document.getElementsByTagName(\"article\")[0].classList.value";
                    webView.runJavaScript(js, function(result) {
                        saveLastWtLocale(result);
                    });
                }
            }
        }
    }
}
