/**
 * This file is part of TheocBase.
 *
 * Copyright (C) 2011-2023, TheocBase Development Team, see AUTHORS.
 *
 * TheocBase is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * TheocBase is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with TheocBase.  If not, see <http://www.gnu.org/licenses/>.
 */
import QtCore
import QtQuick
import QtQuick.Controls
import QtQuick.Controls.Material
import QtQuick.Dialogs
import QtQuick.Layouts
import QtQuick.Window
import QtPositioning
import net.theocbase.mobile 1.0
import "./"
import "sidePanelScripts.js" as SPScripts

StackView {
    id: territoryPanelStackView

    initialItem: Page {
        // JAVASCRIPT FUNCTIONS
        function resetDefaultLookupControlSettings() {
            settings.territoryAssignment_groupByIndex = 1;
            assignmentLookupControl.groupByIndex = settings.territoryAssignment_groupByIndex;
            settings.territoryAssignment_sortOrder = Qt.AscendingOrder;
            assignmentLookupControl.sortOrder = settings.territoryAssignment_sortOrder;
            settings.territoryStreet_groupByIndex = 1;
            streetLookupControl.groupByIndex = settings.territoryStreet_groupByIndex;
            settings.territoryStreet_sortOrder = Qt.AscendingOrder;
            streetLookupControl.sortOrder = settings.territoryStreet_sortOrder;
            settings.territoryAddress_groupByIndex = 1;
            addressLookupControl.groupByIndex = settings.territoryAddress_groupByIndex;
            settings.territoryAddress_sortOrder = Qt.AscendingOrder;
            addressLookupControl.sortOrder = settings.territoryAddress_sortOrder;
            settings.sync();
        }

        // OBJECT PROPERTIES
        header: BaseToolbar {
            Material.background: myPalette.button
            Material.foreground: myPalette.buttonText
            Material.accent: myPalette.highlight
            title: qsTr("Territory") + (currentTerritory
                                        ? " " + currentTerritory.territoryNumber
                                        : "")
            subtitle: currentTerritory && currentTerritory.locality
                      ? " " + currentTerritory.locality
                      : ""
            titleColor: myPalette.buttonText
            componentLeft: ToolButton {
                icon.source: "qrc:/icons/chevron_left.svg"
                onClicked: { switchSidePanel() }
            }
        }

        // CHILD OBJECTS
        Settings {
            id: settings
            category: "LookupControl"
            property int territoryAssignment_groupByIndex: 1
            property bool territoryAssignment_sortOrder: Qt.AscendingOrder
            property int territoryStreet_groupByIndex: 1
            property bool territoryStreet_sortOrder: Qt.AscendingOrder
            property bool territoryStreetResult_sortOrder: Qt.AscendingOrder
            property bool territoryStreetResult_hideAlreadyAdded: true
            property int territoryAddress_groupByIndex: 1
            property bool territoryAddress_sortOrder: Qt.AscendingOrder
            property int territoryGeocodeResult_groupByIndex: 1
            property bool territoryGeocodeResult_sortOrder: Qt.AscendingOrder
        }

        SystemPalette { id: myPalette; colorGroup: SystemPalette.Active }

        Connections {
            target: territoryManager
            function onTerritorySelectionChanged() {
                territoryTypeComboBox.currentIndex = currentTerritory ? territoryTypeComboBox.indexOfValue(currentTerritory.typeId) : -1;
                territoryCityComboBox.currentIndex = currentTerritory ? territoryCityComboBox.indexOfValue(currentTerritory.cityId) : -1;
                // reset lookup controls in case item editing was not collapsed
                assignmentLookupControl.isEditing = false;
                assignmentLookupControl.headerItem.enabled = true;
                assignmentLookupControl.footerItem.visible = true;
                addressLookupControl.isEditing = false;
                addressLookupControl.headerItem.enabled = true;
                addressLookupControl.footerItem.visible = true;
                streetLookupControl.isEditing = false;
                streetLookupControl.headerItem.enabled = true;
                streetLookupControl.footerItem.visible = true;
            }
        }

        Component.onCompleted: {
            assignmentLookupControl.groupByIndex = settings.territoryAssignment_groupByIndex;
            assignmentLookupControl.sortOrder = settings.territoryAssignment_sortOrder;
            streetLookupControl.groupByIndex = settings.territoryStreet_groupByIndex;
            streetLookupControl.sortOrder = settings.territoryStreet_sortOrder;
            addressLookupControl.groupByIndex = settings.territoryAddress_groupByIndex;
            addressLookupControl.sortOrder = settings.territoryAddress_sortOrder;

            if (!currentTerritory)
                tabBar.currentIndex = 4;
        }

        Component.onDestruction: {
            settings.territoryAssignment_groupByIndex = assignmentLookupControl.groupByIndex;
            settings.territoryAssignment_sortOrder = assignmentLookupControl.sortOrder;
            settings.territoryStreet_groupByIndex = streetLookupControl.groupByIndex;
            settings.territoryStreet_sortOrder = streetLookupControl.sortOrder;
            settings.territoryAddress_groupByIndex = addressLookupControl.groupByIndex;
            settings.territoryAddress_sortOrder = addressLookupControl.sortOrder;
            settings.sync();
        }

        Pane {
            anchors.fill: parent
            clip: true
            padding: 10

            ColumnLayout {
                anchors.fill: parent

                TabBar {
                    id: tabBar
                    currentIndex: 0
                    Layout.fillWidth: true

                    TabButton {
                        id: territoryTabButton
                        text: qsTr("Territory")
                        width: visible ? implicitWidth : 0
                        enabled: currentTerritory
                    }
                    TabButton {
                        id: assignmentsTabButton
                        text: qsTr("Assignments")
                        width: visible ? implicitWidth : 0
                        enabled: currentTerritory
                        visible: canViewTerritoryAssignments
                    }
                    TabButton {
                        id: streetsTabButton
                        text: qsTr("Streets")
                        width: visible ? implicitWidth : 0
                        enabled: currentTerritory
                        visible: canViewTerritoryAddresses
                                 || assignmentModel.currentAssigneeId === accessControl?.user?.personId
                    }
                    TabButton {
                        id: adressesTabButton
                        text: qsTr("Addresses")
                        width: visible ? implicitWidth : 0
                        enabled: currentTerritory
                        visible: canViewTerritoryAddresses
                                 || assignmentModel.currentAssigneeId === accessControl?.user?.personId
                    }
                    TabButton {
                        id: mapTabButton
                        text: qsTr("Map")
                        width: implicitWidth
                    }
                }

                StackLayout {
                    clip: true
                    Layout.fillWidth: true
                    Layout.fillHeight: true
                    currentIndex: tabBar.currentIndex

                    ScrollView {
                        id: territoryDetails
                        Layout.fillWidth: true
                        Layout.fillHeight: true
                        contentWidth: availableWidth

                        ColumnLayout {
                            width: parent.width

                            FilledGroupBox {
                                Layout.fillWidth: true

                                Column {
                                    width: parent.width

                                    RowLayout {
                                        width: parent.width
                                        InterimIcon {
                                            Layout.alignment: Qt.AlignLeft | Qt.AlignVCenter
                                            source: "qrc:/icons/123.svg"
                                            color: Material.foreground
                                            ToolTip.text: qsTr("No.:")
                                            ToolTip.visible: hovered
                                        }
                                        TextField {
                                            Layout.fillWidth: true
                                            Material.containerStyle: Material.Filled
                                            bottomPadding: 8
                                            selectByMouse: true
                                            wrapMode: Text.WordWrap
                                            text: currentTerritory ? currentTerritory.territoryNumber : ""
                                            placeholderText: qsTr("Number")
                                            readOnly: !canEditTerritories
                                            onEditingFinished: {
                                                currentTerritory.territoryNumber = text;
                                                currentTerritory.save();
                                            }
                                        }
                                    }
                                    MenuSeparator { width: parent.width; padding: 0 }
                                    RowLayout {
                                        width: parent.width
                                        InterimIcon {
                                            Layout.alignment: Qt.AlignLeft | Qt.AlignVCenter
                                            source: "qrc:/icons/location.svg"
                                            color: Material.foreground
                                            ToolTip.text: qsTr("Locality:")
                                            ToolTip.visible: hovered
                                        }
                                        TextField {
                                            Layout.fillWidth: true
                                            Material.containerStyle: Material.Filled
                                            bottomPadding: 8
                                            selectByMouse: true
                                            wrapMode: Text.WordWrap
                                            text: currentTerritory ? currentTerritory.locality : ""
                                            placeholderText: qsTr("Locality")
                                            readOnly: !canEditTerritories
                                            onEditingFinished: {
                                                currentTerritory.locality = text;
                                                currentTerritory.save();
                                            }
                                        }
                                    }
                                    MenuSeparator { width: parent.width; padding: 0 }
                                    RowLayout {
                                        width: parent.width
                                        InterimIcon {
                                            Layout.alignment: Qt.AlignLeft | Qt.AlignVCenter
                                            source: "qrc:/icons/category.svg"
                                            color: Material.foreground
                                            ToolTip.text: qsTr("Type:", "Territory-type that is used to group territories")
                                            ToolTip.visible: hovered
                                        }
                                        ComboBox{
                                            id: territoryTypeComboBox

                                            Layout.fillWidth: true

                                            enabled: canEditTerritories

                                            model: typeListModel

                                            textRole: "name"
                                            valueRole: "id"

                                            // When an item is selected, update the backend.
                                            onActivated: {
                                                currentTerritory.typeId = currentValue;
                                                currentTerritory.save();
                                            }
                                            // Set the initial currentIndex to the value stored in the backend.
                                            Component.onCompleted: {
                                                currentIndex = currentTerritory && currentTerritory.typeId ? indexOfValue(currentTerritory.typeId) : -1
                                            }
                                        }
                                    }
                                    MenuSeparator { width: parent.width; padding: 0 }
                                    RowLayout {
                                        width: parent.width
                                        InterimIcon {
                                            Layout.alignment: Qt.AlignLeft | Qt.AlignVCenter
                                            source: "qrc:/icons/home.svg"
                                            color: Material.foreground
                                            ToolTip.text: qsTr("City:", "Cityname that is used to group territories")
                                            ToolTip.visible: hovered
                                        }
                                        ComboBox{
                                            id: territoryCityComboBox

                                            Layout.fillWidth: true

                                            enabled: canEditTerritories

                                            model: cityListModel

                                            textRole: "name"
                                            valueRole: "id"

                                            // When an item is selected, update the backend.
                                            onActivated: {
                                                currentTerritory.cityId = currentValue;
                                                currentTerritory.save();
                                            }
                                            // Set the initial currentIndex to the value stored in the backend.
                                            Component.onCompleted: {
                                                currentIndex = currentTerritory && currentTerritory.typeId ? indexOfValue(currentTerritory.typeId) : -1
                                            }
                                        }
                                    }
                                }
                            }
                            FilledGroupBox {
                                Layout.fillWidth: true
                                Layout.minimumHeight: 100

                                visible: canViewTerritoryAssignments
                                         || assignmentModel.currentAssigneeId === accessControl?.user?.personId

                                RowLayout{
                                    anchors.fill: parent
                                    InterimIcon {
                                        Layout.alignment: Qt.AlignLeft | Qt.AlignTop
                                        source: "qrc:/icons/notes.svg"
                                        color: Material.foreground
                                        ToolTip.text: qsTr("Notes")
                                        ToolTip.visible: hovered
                                    }
                                    TextArea {
                                        id: remarkTextField
                                        Layout.fillWidth: true
                                        Layout.fillHeight: true
                                        Material.containerStyle: Material.Filled
                                        readOnly: !canEditTerritories
                                        text: currentTerritory ? currentTerritory.remark : ""
                                        placeholderText: qsTr("Notes")
                                        wrapMode: TextArea.WrapAnywhere
                                        selectByMouse: true
                                        onEditingFinished: {
                                            currentTerritory.remark = text;
                                            currentTerritory.save();
                                        }
                                    }
                                }
                            }
                        }
                    }

                    ColumnLayout {
                        id: assignmentsItem
                        Layout.fillWidth: true
                        Layout.fillHeight: true
                        implicitWidth: parent.width
                        implicitHeight: parent.height

                        Component {
                            id: assignmentDetailPage
                            TerritoryAssignmentPage {
                                onPageCanceled: {
                                    // drop changes and reload assignments
                                    assignmentModel.loadAssignments(territoryManager.currentTerritory.territoryId)
                                }
                            }
                        }

                        FilledGroupBox {
                            Layout.fillWidth: true
                            Layout.fillHeight: true

                            LookupControl {
                                id: assignmentLookupControl

                                anchors.fill: parent
                                clip: true
                                focus: true

                                showFilterControls: false
                                showGroupControls: true
                                showSearchField: false
                                groupByIndex: -1
                                showEditButton: true
                                isEditing: false

                                model: assignmentProxyModel
                                groupByModel: ListModel {
                                    ListElement {
                                        key: qsTr("Year", "Group territory assignments by year")
                                        value: "assignedYear"
                                    }
                                    ListElement {
                                        key: qsTr("Publisher", "Group territory assignments by publishers")
                                        value: "publisher"
                                    }
                                }
                                //moreMenu: assignmentLookupControlMoreMenu

                                delegate: TerritoryAssignmentDelegate {}

                                section.property: groupByModel.get(groupByIndex) ? groupByModel.get(groupByIndex).value : ""
                                section.criteria: ViewSection.FullString
                                section.delegate: Pane {
                                    id: assignmentSection
                                    width: ListView.view.width
                                    height: Material.menuItemHeight
                                    padding: 0
                                    background: Rectangle {
                                        anchors.fill: parent
                                        color: TBStyle.primaryColor
                                    }
                                    Label {
                                        width: parent.width
                                        anchors.centerIn: parent
                                        text: section
                                        horizontalAlignment: Qt.AlignHCenter
                                        verticalAlignment: Qt.AlignVCenter
                                        font: TBStyle.bodyMediumFont
                                        color: TBStyle.onPrimaryColor
                                    }
                                }

                                footer: Pane {
                                    width: ListView.view.width;
                                    z: 3

                                    RowLayout {
                                        anchors.right: parent.right
                                        Label {
                                            verticalAlignment: Text.AlignVCenter
                                            //: Number of rows in a list
                                            //~ Context the '%1'-part will be replaced by the actual number
                                            text: qsTr("Count=%1", "Lookup control").arg(assignmentLookupControl.count)
                                        }

                                        ToolButton {
                                            id: addAssignmentButton

                                            Layout.alignment: Qt.AlignRight | Qt.AlignVCenter
                                            visible: assignmentLookupControl.showEditButton
                                            icon.source: "qrc:/icons/add.svg"
                                            onClicked: {
                                                var newId = assignmentModel.addAssignment(territoryManager.currentTerritory.territoryId);
                                                var newIndex = assignmentModel.getAssignmentIndex(newId);
                                                var newProxyIndex = assignmentProxyModel.mapFromSource(newIndex);
                                                assignmentLookupControl.currentIndex = newProxyIndex.row;
                                                territoryPanelStackView.push(assignmentDetailPage, { assignment: assignmentModel.get(newIndex.row).assignment });
                                            }
                                        }
                                        ToolButton {
                                            id: assignmentMoreMenuButton
                                            Layout.alignment: Qt.AlignRight | Qt.AlignVCenter
                                            visible: assignmentLookupControl.moreMenu
                                            ToolTip.visible: hovered
                                            ToolTip.delay: Qt.styleHints.mousePressAndHoldInterval
                                            ToolTip.text: qsTr("Settings", "Lookup control")
                                            icon.source: "qrc:/icons/more.svg"

                                            onClicked: assignmentLookupControl.moreMenu.popup()
                                        }
                                    }
                                }

                                onSearchTextChanged: assignmentProxyModel.filterText = text
                                onGroupByIndexChanged: {
                                    if (groupByIndex >= 0) {
                                        settings.territoryAssignment_groupByIndex = groupByIndex;
                                        assignmentProxyModel.groupByRole = groupByModel.get(groupByIndex).value;
                                    }
                                }
                                onSortOrderChanged: {
                                    settings.territoryAssignment_sortOrder = sortOrder;
                                    assignmentProxyModel.sort(0, sortOrder);
                                }

                                ScrollBar.vertical: ScrollBar {}
                            }
                        }
                    }

                    ColumnLayout {
                        id: streetsTabPage

                        property var editTerritoryId

                        Layout.fillWidth: true
                        Layout.fillHeight: true
                        implicitWidth: parent.width
                        implicitHeight: parent.height

                        StackView {
                            id: streetsEditStack

                            Layout.fillWidth: true
                            Layout.fillHeight: true
                            implicitWidth: parent.width
                            implicitHeight: parent.height

                            Connections {
                                target: territoryManager
                                function onCurrentStreetChanged() {
                                    if (territoryManager.currentStreet) {
                                        var streetIndex = streetModel.getStreetIndex(territoryManager.currentStreet.streetId);
                                        var streetProxyIndex = streetProxyModel.mapFromSource(streetIndex);
                                        streetLookupControl.currentIndex = streetProxyIndex.row;
                                    } else {
                                        streetLookupControl.currentIndex = -1;
                                    }
                                }
                            }

                            initialItem: FilledGroupBox {
                                Layout.fillWidth: true
                                Layout.fillHeight: true

                                LookupControl {
                                    id: streetLookupControl

                                    anchors.fill: parent
                                    clip: true
                                    focus: true

                                    showFilterControls: false
                                    showGroupControls: true
                                    groupByIndex: -1
                                    showEditButton: true
                                    isEditing: false

                                    model: streetProxyModel
                                    groupByModel: ListModel {
                                        ListElement {
                                            key: qsTr("Street name", "Group streets by their names")
                                            value: "streetName"
                                        }
                                        ListElement {
                                            key: qsTr("Type",
                                                      "Group streets by their type")
                                            value: "streetTypeName"
                                        }
                                        ListElement {
                                            key: qsTr("Quantity",
                                                      "Group streets by the quantity of addresses")
                                            value: "quantity"
                                        }
                                    }
                                    //moreMenu: streetLookupControlMoreMenu

                                    delegate: TerritoryStreetDelegate {}

                                    onCurrentIndexChanged: {
                                        if (currentIndex < 0) {
                                            territoryManager.currentStreet = null;
                                            return;
                                        }
                                        var streetProxyIndex = streetProxyModel.index(currentIndex, 0);
                                        var streetIndex = streetProxyModel.mapToSource(streetProxyIndex);
                                        var streetId = streetModel.get(streetIndex.row).streetId;
                                        if (typeof(streetId) !== undefined && streetId > 0
                                                && territoryManager.currentStreet?.streetId !== streetId) {
                                            territoryManager.currentStreet = streetModel.findStreet(streetId);
                                        }
                                    }

                                    section.property: groupByModel.get(groupByIndex) ? groupByModel.get(groupByIndex).value : ""
                                    section.criteria: ViewSection.FullString
                                    section.delegate: Pane {
                                        id: streetSection
                                        width: ListView.view.width
                                        height: Material.menuItemHeight
                                        padding: 0
                                        background: Rectangle {
                                            anchors.fill: parent
                                            color: TBStyle.primaryColor
                                        }
                                        Label {
                                            width: parent.width
                                            anchors.centerIn: parent
                                            text: switch(streetSection.LookupControl.view.section.property) {
                                                  case "quantity": return SPScripts.formatNumberRange(section, 10)
                                                  default: return section;
                                                  }
                                            horizontalAlignment: Qt.AlignHCenter
                                            verticalAlignment: Qt.AlignVCenter
                                            font: TBStyle.bodyMediumFont
                                            color: TBStyle.onPrimaryColor
                                            elide: Text.ElideMiddle
                                        }
                                    }

                                    footer: Pane {
                                        width: ListView.view.width;
                                        z: 3

                                        RowLayout {
                                            anchors.right: parent.right
                                            Label {
                                                verticalAlignment: Text.AlignVCenter
                                                //: Number of rows in a list
                                                //~ Context the '%1'-part will be replaced by the actual number
                                                text: qsTr("Count=%1", "Lookup control").arg(streetLookupControl.count)
                                            }

                                            ToolButton {
                                                id: addStreetButton
                                                Layout.alignment: Qt.AlignRight | Qt.AlignVCenter
                                                visible: isGdalAvailable && streetLookupControl.showEditButton
                                                icon.source: "qrc:/icons/add.svg"
                                                onClicked: {
                                                    streetsTabPage.editTerritoryId = territoryManager.currentTerritory.territoryId;
                                                    streetsEditStack.push(addStreetPage);
                                                }
                                            }

                                            ToolButton {
                                                id: streetMoreMenuButton
                                                Layout.alignment: Qt.AlignRight | Qt.AlignVCenter
                                                visible: streetLookupControl.moreMenu
                                                ToolTip.visible: hovered
                                                ToolTip.delay: Qt.styleHints.mousePressAndHoldInterval
                                                ToolTip.text: qsTr("Settings", "Lookup control")
                                                icon.source: "qrc:/icons/more.svg"

                                                onClicked: streetLookupControl.moreMenu.popup()
                                            }
                                        }
                                    }

                                    onSearchTextChanged: streetProxyModel.filterText = text
                                    onGroupByIndexChanged: {
                                        if (groupByIndex >= 0) {
                                            settings.territoryStreet_groupByIndex = groupByIndex;
                                            streetProxyModel.groupByRole = groupByModel.get(groupByIndex).value;
                                        }
                                    }
                                    onSortOrderChanged: {
                                        settings.territoryStreet_sortOrder = sortOrder;
                                        streetProxyModel.sort(0, sortOrder);
                                    }
                                    ScrollBar.vertical: ScrollBar {}
                                }
                            }
                        }
                    }

                    ColumnLayout {
                        id: addressesTabPage

                        property var editTerritoryId
                        property bool isEditAddressMode: false

                        Layout.fillWidth: true
                        Layout.fillHeight: true
                        implicitWidth: parent.width
                        implicitHeight: parent.height

                        StackView {
                            id: addressesEditStack

                            Layout.fillWidth: true
                            Layout.fillHeight: true
                            implicitWidth: parent.width
                            implicitHeight: parent.height

                            initialItem: FilledGroupBox {
                                Layout.fillWidth: true
                                Layout.fillHeight: true

                                Connections {
                                    target: territoryManager
                                    function onCurrentAddressChanged() {
                                        if (territoryManager.currentAddress) {
                                            var addressIndex = addressModel.getAddressIndex(territoryManager.currentAddress.id);
                                            var addressProxyIndex = addressProxyModel.mapFromSource(addressIndex);
                                            addressLookupControl.currentIndex = addressProxyIndex.row;
                                        } else {
                                            addressLookupControl.currentIndex = -1;
                                        }
                                    }
                                }

                                LookupControl {
                                    id: addressLookupControl

                                    anchors.fill: parent
                                    clip: true
                                    focus: true

                                    showFilterControls: false
                                    showGroupControls: true
                                    groupByIndex: -1
                                    showEditButton: true
                                    isEditing: false

                                    model: addressProxyModel
                                    groupByModel: ListModel {
                                        ListElement {
                                            key: qsTr("Street name", "Group addresses by street name")
                                            value: "street"
                                        }
                                        ListElement {
                                            key: qsTr("Type",
                                                      "Group addresses by their type")
                                            value: "addressTypeName"
                                        }
                                    }
                                    //moreMenu: streetLookupControlMoreMenu

                                    delegate: TerritoryAddressDelegate {}

                                    onCurrentIndexChanged: {
                                        if (currentIndex < 0) {
                                            territoryManager.currentAddress = null;
                                            return;
                                        }
                                        var addressProxyIndex = addressProxyModel.index(currentIndex, 0);
                                        var addressIndex = addressProxyModel.mapToSource(addressProxyIndex);
                                        var addressId = addressModel.get(addressIndex.row).id;
                                        if (typeof(addressId) !== undefined && addressId > 0
                                                && territoryManager.currentAddress?.id !== addressId) {
                                            territoryManager.currentAddress = addressModel.findAddress(addressId);
                                        }
                                    }

                                    section.property: groupByModel.get(groupByIndex) ? groupByModel.get(groupByIndex).value : ""
                                    section.criteria: ViewSection.FullString
                                    section.delegate: Pane {
                                        id: addressSection
                                        width: ListView.view.width
                                        height: Material.menuItemHeight
                                        padding: 0
                                        background: Rectangle {
                                            anchors.fill: parent
                                            color: TBStyle.primaryColor
                                        }
                                        Label {
                                            width: parent.width
                                            anchors.centerIn: parent
                                            text: section
                                            horizontalAlignment: Qt.AlignHCenter
                                            verticalAlignment: Qt.AlignVCenter
                                            font: TBStyle.bodyMediumFont
                                            color: TBStyle.onPrimaryColor
                                        }
                                    }

                                    footer: Pane {
                                        width: ListView.view.width;
                                        z: 3

                                        RowLayout {
                                            anchors.right: parent.right
                                            Label {
                                                verticalAlignment: Text.AlignVCenter
                                                //: Number of rows in a list
                                                //~ Context the '%1'-part will be replaced by the actual number
                                                text: qsTr("Count=%1", "Lookup control").arg(addressLookupControl.count)
                                            }

                                            ToolButton {
                                                id: addAddressButton
                                                Layout.alignment: Qt.AlignRight | Qt.AlignVCenter
                                                visible: isGdalAvailable && addressLookupControl.showEditButton
                                                icon.source: "qrc:/icons/add.svg"
                                                onClicked: {
                                                    addressesTabPage.editTerritoryId = territoryManager.currentTerritory.territoryId;
                                                    addressesEditStack.push(geocodeAddressPage);
                                                }
                                            }

                                            ToolButton {
                                                id: addressMoreMenuButton
                                                Layout.alignment: Qt.AlignRight | Qt.AlignVCenter
                                                visible: addressLookupControl.moreMenu
                                                ToolTip.visible: hovered
                                                ToolTip.delay: Qt.styleHints.mousePressAndHoldInterval
                                                ToolTip.text: qsTr("Settings", "Lookup control")
                                                icon.source: "qrc:/icons/more.svg"

                                                onClicked: addressLookupControl.moreMenu.popup()
                                            }
                                        }
                                    }

                                    onSearchTextChanged: function(text) {
                                        addressProxyModel.filterText = text
                                    }
                                    onGroupByIndexChanged: {
                                        if (groupByIndex >= 0) {
                                            settings.territoryAddress_groupByIndex = groupByIndex;
                                            addressProxyModel.groupByRole = groupByModel.get(groupByIndex).value;
                                        }
                                    }
                                    onSortOrderChanged: {
                                        settings.territoryAddress_sortOrder = sortOrder;
                                        addressProxyModel.sort(0, sortOrder);
                                    }
                                    ScrollBar.vertical: ScrollBar {}
                                }
                            }
                        }
                    }

                    ScrollView {
                        id: mapTabPage
                        Layout.fillWidth: true
                        Layout.fillHeight: true
                        contentWidth: availableWidth

                        ColumnLayout {
                            width: parent.width

                            FilledGroupBox {
                                Layout.fillWidth: true
                                Layout.minimumHeight: 100

                                title: qsTr("Addresses") + qmlTranslator.changed

                                Column {
                                    width: parent.width

                                    RowLayout {
                                        width: parent.width
                                        InterimIcon {
                                            Layout.alignment: Qt.AlignLeft
                                            source: "qrc:/icons/opacity.svg"
                                            color: Material.foreground
                                            ToolTip.text: qsTr("Opacity")
                                            ToolTip.visible: hovered
                                        }
                                        Slider {
                                            Layout.fillWidth: true
                                            from: 0.0
                                            to: 1.0
                                            value: territoryManager.mapView.addressLayer.opacity
                                            onValueChanged: {
                                                territoryManager.mapView.addressLayer.opacity = value;
                                            }
                                        }
                                    }
                                }
                            }

                            FilledGroupBox {
                                Layout.fillWidth: true
                                Layout.minimumHeight: 100

                                title: qsTr("Sketch") + qmlTranslator.changed

                                Column {
                                    width: parent.width

                                    RowLayout {
                                        width: parent.width
                                        InterimIcon {
                                            Layout.alignment: Qt.AlignLeft
                                            source: "qrc:/icons/opacity.svg"
                                            color: Material.foreground
                                            ToolTip.text: qsTr("Opacity")
                                            ToolTip.visible: hovered
                                        }
                                        Slider {
                                            Layout.fillWidth: true
                                            from: 0.0
                                            to: 1.0
                                            value: territoryManager.mapView.sketchLayer.opacity
                                            onValueChanged: {
                                                territoryManager.mapView.sketchLayer.opacity = value;
                                            }
                                        }
                                        ToolButton {
                                            icon.source: "qrc:/icons/delete_sweep.svg"
                                            ToolTip.text: qsTr("Clear sketch",
                                                               "Territory map")
                                            ToolTip.visible: hovered
                                            onClicked: {
                                                territoryManager.mapView.sketchLayer.deleteAllSketchItems();
                                            }
                                        }
                                    }
                                }
                            }

                            FilledGroupBox {
                                Layout.fillWidth: true
                                Layout.minimumHeight: 100

                                title: qsTr("Boundaries") + qmlTranslator.changed

                                Column {
                                    width: parent.width

                                    RowLayout {
                                        width: parent.width
                                        InterimIcon {
                                            Layout.alignment: Qt.AlignLeft
                                            source: "qrc:/icons/opacity.svg"
                                            color: Material.foreground
                                            ToolTip.text: qsTr("Opacity")
                                            ToolTip.visible: hovered
                                        }
                                        Slider {
                                            Layout.fillWidth: true
                                            from: 0.0
                                            to: 1.0
                                            value: territoryManager.mapView.boundaryLayer.opacity
                                            onValueChanged: {
                                                territoryManager.mapView.boundaryLayer.opacity = value;
                                            }
                                        }
                                    }
                                }
                            }

                            FilledGroupBox {
                                Layout.fillWidth: true
                                Layout.minimumHeight: 100

                                title: qsTr("Streets") + qmlTranslator.changed

                                Column {
                                    width: parent.width

                                    RowLayout {
                                        width: parent.width
                                        InterimIcon {
                                            Layout.alignment: Qt.AlignLeft
                                            source: "qrc:/icons/opacity.svg"
                                            color: Material.foreground
                                            ToolTip.text: qsTr("Opacity")
                                            ToolTip.visible: hovered
                                        }
                                        Slider {
                                            Layout.fillWidth: true
                                            from: 0.0
                                            to: 1.0
                                            value: territoryManager.mapView.streetLayer.opacity
                                            onValueChanged: {
                                                territoryManager.mapView.streetLayer.opacity = value;
                                            }
                                        }
                                    }
                                }
                            }

                            FilledGroupBox {
                                Layout.fillWidth: true
                                Layout.minimumHeight: 100

                                title: qsTr("Basemap") + qmlTranslator.changed

                                Column {
                                    width: parent.width

                                    RowLayout {
                                        width: parent.width
                                        InterimIcon {
                                            Layout.alignment: Qt.AlignLeft | Qt.AlignVCenter
                                            source: "qrc:/icons/map.svg"
                                            color: Material.foreground
                                            ToolTip.text: qsTr("Basemap")
                                            ToolTip.visible: hovered
                                        }
                                        ComboBox {
                                            id: mapTypeCombobox

                                            Layout.fillWidth: true

                                            model: territoryManager.mapView.baseMap.supportedMapTypes

                                            textRole: "description"
                                            valueRole: "name"

                                            // When an item is selected, update the backend.
                                            onActivated: {
                                                territoryManager.mapView.baseMap.activeMapType = territoryManager.mapView.baseMap.supportedMapTypes[currentIndex]
                                            }
                                            // Set the initial currentIndex to the value stored in the backend.
                                            Component.onCompleted: {
                                                currentIndex = territoryManager.mapView.baseMap.activeMapType ? indexOfValue(territoryManager.mapView.baseMap.activeMapType.name) : -1
                                            }
                                        }
                                        ToolButton {
                                            icon.source: "qrc:/icons/delete_sweep.svg"
                                            ToolTip.text: qsTr("Clear cache")
                                            ToolTip.visible: hovered
                                            onClicked: territoryManager.mapView.baseMap.clearData()
                                        }
                                    }
                                    RowLayout {
                                        width: parent.width
                                        InterimIcon {
                                            Layout.alignment: Qt.AlignLeft
                                            source: "qrc:/icons/opacity.svg"
                                            color: Material.foreground
                                            ToolTip.text: qsTr("Opacity")
                                            ToolTip.visible: hovered
                                        }
                                        Slider {
                                            Layout.fillWidth: true
                                            from: 0.0
                                            to: 1.0
                                            value: territoryManager.mapView.baseMap.opacity
                                            onValueChanged: {
                                                territoryManager.mapView.baseMap.opacity = value;
                                            }
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
    }
}
