
/**
* This file is part of TheocBase.
*
* Copyright (C) 2011-2016, TheocBase Development Team, see AUTHORS.
*
* TheocBase is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* TheocBase is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with TheocBase.  If not, see <http://www.gnu.org/licenses/>.
*/
import QtCore
import QtQuick
import QtQuick.Window
import QtQuick.Controls
import QtQuick.Controls.Material
import QtQml.Models
import QtQuick.Layouts
import QtPositioning
import QtLocation
import net.theocbase.mobile 1.0
import "."

Item {
    id: control

    property alias baseMap: _baseMap
    property alias vectorLayer: vectorOverlay
    property alias addressLayer: addressItemView
    property alias streetLayer: streetItemView
    property alias boundaryLayer: boundaryItemView
    property alias sketchLayer: sketchOverlay
    property alias sketchPointHandler: _sketchPointHandler
    property alias sketchDrawingInkButton: _sketchDrawingInkButton
    property alias sketchEraserButton: _sketchEraserButton
    property alias locationPermission: _locationPermission
    property alias positionSource: _positionSource
    property bool isPositionSourceActive: true
    property alias addressTextField: addressTextField
    property bool showLayerButtons: false
    property alias showBoundariesButton: showBoundariesButton
    property alias showStreetsButton: showStreetsButton
    property alias showMarkersButton: showMarkersButton
    property alias showSketchButton: showSketchButton
    property alias createBoundaryButton: createBoundaryButton
    property alias splitTerritoryButton: splitTerritoryButton
    property alias removeBoundaryButton: removeBoundaryButton
    property alias zoomSlider: zoomSlider
    property alias positionSourceButton: _positionSourceButton
    property alias editModeButton: editModeButton
    property alias mapPinchHandler: _pinchHandler
    property alias mapDragHandler: _dragHandler
    property alias searchBar: searchBar
    property alias openSearchBarButton: _openSearchBarButton
    property alias clearSearchBarButton: _clearSearchBarButton

    LocationPermission {
        id: _locationPermission
        accuracy: LocationPermission.Precise
        availability: LocationPermission.WhenInUse
    }

    PositionSource {
        id: _positionSource
        updateInterval: 1000
        active: locationPermission.status === Qt.PermissionStatus.Granted
                && control.isPositionSourceActive
    }

    Map {
        id: _baseMap

        property variant pluginParameters
        property geoCoordinate startCentroid

        anchors.fill: parent
        zoomLevel: 15

        PinchHandler {
            id: _pinchHandler
            target: null
            // grabPermissions: PointerHandler.TakeOverForbidden
        }
        WheelHandler {
            id: wheelHandler
            // workaround for QTBUG-87646 / QTBUG-112394 / QTBUG-112432:
            // Magic Mouse pretends to be a trackpad but doesn't work with PinchHandler
            // and we don't yet distinguish mice and trackpads on Wayland either
            acceptedDevices: Qt.platform.pluginName === "cocoa"
                             || Qt.platform.pluginName
                             === "wayland" ? PointerDevice.Mouse
                                             | PointerDevice.TouchPad : PointerDevice.Mouse
            rotationScale: 1 / 120
            property: "zoomLevel"
        }
        DragHandler {
            id: _dragHandler
            target: null
            enabled: !(territoryManager.isDrawerOpen || territoryManager.isPrintingOpen)
        }
        Shortcut {
            enabled: baseMap.zoomLevel < baseMap.maximumZoomLevel
            sequence: StandardKey.ZoomIn
            onActivated: baseMap.zoomLevel = Math.round(baseMap.zoomLevel + 1)
        }
        Shortcut {
            enabled: baseMap.zoomLevel > baseMap.minimumZoomLevel
            sequence: StandardKey.ZoomOut
            onActivated: baseMap.zoomLevel = Math.round(baseMap.zoomLevel - 1)
        }
    }

    Map {
        id: streetOverlay

        anchors.fill: parent
        plugin: Plugin { name: "itemsoverlay" }
        //gesture.enabled: false
        center: baseMap.center
        color: 'transparent' // Necessary to make this map transparent
        minimumFieldOfView: baseMap.minimumFieldOfView
        maximumFieldOfView: baseMap.maximumFieldOfView
        minimumTilt: baseMap.minimumTilt
        maximumTilt: baseMap.maximumTilt
        minimumZoomLevel: baseMap.minimumZoomLevel
        maximumZoomLevel: baseMap.maximumZoomLevel
        zoomLevel: baseMap.zoomLevel
        tilt: baseMap.tilt;
        bearing: baseMap.bearing
        fieldOfView: baseMap.fieldOfView
        z: _baseMap.z + 1

        // The code below enables SSAA
        layer.enabled: true
        layer.smooth: true
        property int w : vectorOverlay.width
        property int h : vectorOverlay.height
        property int pr: Screen.devicePixelRatio
        layer.textureSize: Qt.size(w  * 2 * pr, h * 2 * pr)

        MapItemView {
            id: streetItemView

            property int streetDisplayOption: 2

            model: streetModel
            delegate: territoryStreetDelegate
            visible: baseMap.zoomLevel > (baseMap.maximumZoomLevel / 3 * 2)
        }
    }

    Map {
        id: boundaryOverlay

        anchors.fill: parent
        plugin: Plugin { name: "itemsoverlay" }
        //gesture.enabled: false
        center: baseMap.center
        color: 'transparent' // Necessary to make this map transparent
        minimumFieldOfView: baseMap.minimumFieldOfView
        maximumFieldOfView: baseMap.maximumFieldOfView
        minimumTilt: baseMap.minimumTilt
        maximumTilt: baseMap.maximumTilt
        minimumZoomLevel: baseMap.minimumZoomLevel
        maximumZoomLevel: baseMap.maximumZoomLevel
        zoomLevel: baseMap.zoomLevel
        tilt: baseMap.tilt;
        bearing: baseMap.bearing
        fieldOfView: baseMap.fieldOfView
        z: streetOverlay.z + 1

        // The code below enables SSAA
        layer.enabled: true
        layer.smooth: true
        property int w : vectorOverlay.width
        property int h : vectorOverlay.height
        property int pr: Screen.devicePixelRatio
        layer.textureSize: Qt.size(w  * 2 * pr, h * 2 * pr)

        MapItemView {
            id: boundaryItemView

            property int boundaryDisplayOption: 2

            model: territoryTreeModel
            delegate: territoryBoundaryDelegate
            visible: baseMap.zoomLevel > (baseMap.maximumZoomLevel / 3)
        }
    }

    Map {
        id: sketchOverlay

        property int displayMode: 0
        property int ink: 1
        property int activeDrawingInk: 1
        property variant unfinishedItem: undefined
        property bool autoFadeIn: false
        property variant referenceSurface: QtLocation.ReferenceSurface.Map

        property color defaultPenColor: "#000F55"
        property color defaultMarkerColor: "#f55656"
        property color defaultHighlighterColor: "#8cff32"

        anchors.fill: parent
        plugin: Plugin { name: "itemsoverlay" }
        //gesture.enabled: false
        center: baseMap.center
        color: 'transparent' // Necessary to make this map transparent
        minimumFieldOfView: baseMap.minimumFieldOfView
        maximumFieldOfView: baseMap.maximumFieldOfView
        minimumTilt: baseMap.minimumTilt
        maximumTilt: baseMap.maximumTilt
        minimumZoomLevel: baseMap.minimumZoomLevel
        maximumZoomLevel: baseMap.maximumZoomLevel
        zoomLevel: baseMap.zoomLevel
        tilt: baseMap.tilt;
        bearing: baseMap.bearing
        fieldOfView: baseMap.fieldOfView
        z: boundaryOverlay.z + 1
        visible: displayMode === 1

        // The code below enables SSAA
        layer.enabled: true
        layer.smooth: true
        property int w : vectorOverlay.width
        property int h : vectorOverlay.height
        property int pr: Screen.devicePixelRatio
        layer.textureSize: Qt.size(w  * 2 * pr, h * 2 * pr)

        signal showMainMenu(variant coordinate)
        signal deleteSketchItem(variant modelData)
        signal deleteAllSketchItems()

        MapItemView {
            model: penGeoDatabase.model

            delegate: GeoJsonDelegate {
                defaultColor: sketchOverlay.defaultPenColor
                defaultLineWidth: 2
                defaultOpacity: 0.8
            }
        }

        MapItemView {
            model: markerGeoDatabase.model

            delegate: GeoJsonDelegate {
                defaultColor: sketchOverlay.defaultMarkerColor
                defaultLineWidth: 5
                defaultOpacity: 0.7
            }
        }

        MapItemView {
            model: highlighterGeoDatabase.model

            delegate: GeoJsonDelegate {
                defaultColor: sketchOverlay.defaultHighlighterColor
                defaultLineWidth: 13
                defaultOpacity: 0.5
            }
        }

        PointHandler {
            id: _sketchPointHandler
            property variant currentCoordinate
            property variant lastCoordinate

            target: Rectangle {
                parent: sketchOverlay
                color: sketchOverlay.ink === 0 ? TBStyle.alertColor : TBStyle.mediumColor
                visible: sketchOverlay.ink === 0
                         && _sketchPointHandler.point.pressedButtons
                         && !mapPinchHandler.active
                x: _sketchPointHandler.point.position.x - width / 2
                y: _sketchPointHandler.point.position.y - height / 2
                width: 20; height: width; radius: width / 2
            }
            enabled: vectorLayer.editMode === 1
                     && sketchLayer.ink >= 0
                     && !mainButtonBarHoverHandler.hovered
                     && !searchBarHoverHandler.hovered
                     && !(zoomSlider.hovered || zoomSlider.pressed || positionSourceButton.hovered)
                     && !editButtonBarHoverHandler.hovered
        }
    }

    Map {
        id: addressOverlay

        anchors.fill: parent
        plugin: Plugin { name: "itemsoverlay" }
        //gesture.enabled: false
        center: baseMap.center
        color: 'transparent' // Necessary to make this map transparent
        minimumFieldOfView: baseMap.minimumFieldOfView
        maximumFieldOfView: baseMap.maximumFieldOfView
        minimumTilt: baseMap.minimumTilt
        maximumTilt: baseMap.maximumTilt
        minimumZoomLevel: baseMap.minimumZoomLevel
        maximumZoomLevel: baseMap.maximumZoomLevel
        zoomLevel: baseMap.zoomLevel
        tilt: baseMap.tilt;
        bearing: baseMap.bearing
        fieldOfView: baseMap.fieldOfView
        z: sketchOverlay.z + 1

        // The code below enables SSAA
        layer.enabled: true
        layer.smooth: true
        property int w : vectorOverlay.width
        property int h : vectorOverlay.height
        property int pr: Screen.devicePixelRatio
        layer.textureSize: Qt.size(w  * 2 * pr, h * 2 * pr)

        MapItemView {
            id: addressItemView

            property int markerDisplayOption: 2

            model: addressModel
            delegate: territoryMarkerDelegate
            visible: baseMap.zoomLevel > (baseMap.maximumZoomLevel / 2)
        }
    }

    Map {
        id: vectorOverlay

        property int editMode: 0
        property bool isDigitizing: false
        property TerritoryMapPolygon polygon
        property TerritoryMapMarker marker
        property TerritoryMapMarker addressSearchMarker
        property var addressSearchMarkers: []
        property var boundaries: []
        property double markerScale: 1.0

        // This property indicates the mode
        //   0 = polyline
        //   1 = polygon
        property int digitizeMode: 1
        property bool isLineComplete: false
        property SimpleMarker snapMarker
        property SimplePolygon newPolygon
        property SimplePolyline newPolyline
        property SimplePolyline rubberBand

        anchors.fill: parent
        plugin: Plugin { name: "itemsoverlay" }
        //gesture.enabled: false
        center: baseMap.center
        color: 'transparent' // Necessary to make this map transparent
        minimumFieldOfView: baseMap.minimumFieldOfView
        maximumFieldOfView: baseMap.maximumFieldOfView
        minimumTilt: baseMap.minimumTilt
        maximumTilt: baseMap.maximumTilt
        minimumZoomLevel: baseMap.minimumZoomLevel
        maximumZoomLevel: baseMap.maximumZoomLevel
        zoomLevel: baseMap.zoomLevel
        tilt: baseMap.tilt;
        bearing: baseMap.bearing
        fieldOfView: baseMap.fieldOfView
        z: addressOverlay.z + 1

        // The code below enables SSAA
        layer.enabled: true
        layer.smooth: true
        property int w : vectorOverlay.width
        property int h : vectorOverlay.height
        property int pr: Screen.devicePixelRatio
        layer.textureSize: Qt.size(w  * 2 * pr, h * 2 * pr)
    }

    GridLayout {
        columns: 2
        rows: 3
        rowSpacing: 20

        height: parent.height
        z: vectorLayer.z + 1

        Pane {
            id: searchBar

            property bool isSearchBarOpen: false

            Layout.row: 0
            Layout.column: 0
            Layout.columnSpan: 2
            Layout.topMargin: 30
            Layout.leftMargin: 10
            padding: 0
            topInset: 0
            bottomInset: 0
            leftInset: 0
            rightInset: 0

            Material.roundedScale: Material.FullScale
            Material.elevation: 2

            HoverHandler {
                id: searchBarHoverHandler
            }

            Row {
                spacing: 0

                RoundButton {
                    id: _openSearchBarButton
                    anchors.verticalCenter: parent.verticalCenter
                    Material.background: "transparent"
                    icon.source: "qrc:/icons/search.svg"
                    icon.color: Material.foreground
                    padding: 0
                    onClicked: searchBar.isSearchBarOpen = true
                }
                TextField {
                    id: addressTextField
                    anchors.verticalCenter: parent.verticalCenter
                    width: 200
                    Material.containerStyle: Material.Filled
                    // placeholderText: qsTr("search", "Search address in territory map")
                    selectByMouse: true
                    visible: searchBar.isSearchBarOpen
                    background: null
                }
                RoundButton {
                    id: _clearSearchBarButton
                    anchors.verticalCenter: parent.verticalCenter
                    Material.background: "transparent"
                    icon.source: "qrc:/icons/cancel.svg"
                    icon.color: Material.foreground
                    padding: 0
                    visible: searchBar.isSearchBarOpen
                }
            }
        }

        Slider {
            id: zoomSlider
            Layout.fillHeight: true
            Layout.row: 1
            Layout.column: 0
            Layout.alignment: Qt.AlignLeft
            Layout.leftMargin: 10
            orientation: Qt.Vertical
            from: baseMap.minimumZoomLevel
            to: baseMap.maximumZoomLevel
            value: baseMap.zoomLevel

            HoverHandler {
                id: zoomSliderHoverHandler
            }
        }

        RoundButton {
            id: _positionSourceButton
            Layout.fillHeight: false
            Layout.row: 2
            Layout.column: 0
            Layout.alignment: Qt.AlignLeft
            Layout.leftMargin: 10
            Layout.bottomMargin: 20

            Material.background: myPalette.window
            ToolTip.text: qsTr("Show your location",
                               "Territory map")
            icon.source: "qrc:/icons/location_disabled.svg"
            opacity: parent.enabled ? 1.0 : 0.5
            checkable: true
            visible: true
        }
    }

    ColumnLayout {
        anchors.bottom: parent.bottom
        width: parent.width
        z: vectorLayer.z + 1

        Row {
            Layout.fillHeight: false
            Layout.alignment: Qt.AlignHCenter
            // Layout.leftMargin: 100
            Layout.bottomMargin: 20

            HoverHandler {
                id: editButtonBarHoverHandler
            }

            RoundButton {
                id: _sketchDrawingInkButton

                Material.background: myPalette.window
                ToolTip.text: qsTr("Sketch tool",
                                   "Territory map")
                icon.source: "qrc:/icons/ink_pen.svg"
                icon.color: sketchOverlay.ink > 0 ? TBStyle.primaryTextColor : myPalette.buttonText
                visible: sketchLayer.displayMode > 0 && vectorLayer.editMode === 1
            }

            RoundButton {
                id: _sketchEraserButton

                Material.background: myPalette.window
                ToolTip.text: qsTr("Clear sketch",
                                   "Territory map")
                icon.source: sketchOverlay.ink === 0 ? "qrc:/icons/ink_eraser_filled.svg" : "qrc:/icons/ink_eraser.svg"
                icon.color: sketchOverlay.ink === 0 ? TBStyle.primaryTextColor : myPalette.buttonText
                visible: sketchLayer.displayMode > 0 && vectorLayer.editMode === 1
            }
        }
    }

    Column {
        topPadding: 30
        bottomPadding: 30
        rightPadding: 10
        //height: parent.height
        z: vectorLayer.z + 1
        spacing: 5
        anchors.right: parent.right

        HoverHandler {
            id: mainButtonBarHoverHandler
        }

        RoundButton {
            id: showMarkersButton
            Material.background: myPalette.window
            ToolTip.text: qsTr("Show/hide markers",
                               "Show/hide address markers of territories")
            icon.source: "qrc:/icons/location_all.svg"
            opacity: parent.enabled ? 1.0 : 0.5
            checkable: false
            visible: showLayerButtons
        }
        RoundButton {
            id: showStreetsButton
            Material.background: myPalette.window
            ToolTip.text: qsTr("Show/hide streets",
                               "Show/hide streets of territories")
            icon.source: "qrc:/icons/road_all.svg"
            opacity: parent.enabled ? 1.0 : 0.5
            checkable: false
            visible: showLayerButtons
        }
        RoundButton {
            id: showBoundariesButton
            Material.background: myPalette.window
            ToolTip.text: qsTr("Show/hide territories",
                               "Show/hide boundaries of territories")
            checkable: false
            icon.source: "qrc:/icons/boundary_all.svg"
            opacity: parent.enabled ? 1.0 : 0.5
            visible: showLayerButtons
        }
        RoundButton {
            id: showSketchButton
            Material.background: myPalette.window
            ToolTip.text: qsTr("Show/hide sketch",
                               "Show/hide sketch on territories")
            checkable: false
            icon.source: "qrc:/icons/sketch.svg"
            opacity: parent.enabled ? 1.0 : 0.5
            visible: showLayerButtons
        }
        RoundButton {
            id: editModeButton
            Material.background: myPalette.window
            ToolTip.text: qsTr("Switch edit mode",
                               "Switch edit mode on territory map")
            icon.source: "qrc:/icons/visibility.svg"
            opacity: parent.enabled ? 1.0 : 0.5
            checkable: false
            visible: showLayerButtons
                     && ((isGdalAvailable && canEditTerritories
                          && (addressLayer.markerDisplayOption > 0
                              || boundaryLayer.boundaryDisplayOption > 0))
                         || sketchLayer.displayMode > 0)
        }
        RoundButton {
            id: createBoundaryButton
            Material.background: myPalette.window
            visible: isGdalAvailable && vectorLayer.editMode === 3
            enabled: currentTerritory && isGdalAvailable
            ToolTip.text: qsTr("Create boundary",
                               "Create a new boundary for the territory")
            icon.source: "qrc:/icons/add_boundary.svg"
            opacity: parent.enabled ? 1.0 : 0.5
        }
        RoundButton {
            id: removeBoundaryButton
            Material.background: myPalette.window
            visible: isGdalAvailable && canEditTerritories && vectorLayer.editMode === 3
            enabled: currentTerritory
            ToolTip.text: qsTr("Remove boundary",
                               "Remove boundary or geometry of the territory")
            icon.source: "qrc:/icons/remove_boundary.svg"
            opacity: parent.enabled ? 1.0 : 0.5
        }
        RoundButton {
            id: splitTerritoryButton
            Material.background: myPalette.window
            visible: isGdalAvailable && vectorLayer.editMode === 3
            enabled: currentTerritory && isGdalAvailable
            ToolTip.text: qsTr("Split territory", "Cut territory in two parts")
            icon.source: "qrc:/icons/boundary_split.svg"
            opacity: parent.enabled ? 1.0 : 0.5
        }
    }
}
