import QtCore
import QtQuick
import QtQuick.Controls
import QtQuick.Dialogs
import QtQuick.Window
import QtPositioning
import QtLocation
import net.theocbase.mobile 1.0
import net.theocbase 1.0

TerritoryMapForm {
    id: territoryMapForm

    // PROPERTY DECLARATIONS
    property variant territoryBoundaries
    property int geocodeRetryCount: 0
    property string geocodeQuery: ""

    // SIGNAL DECLARATIONS
    signal refreshTerritoryMapPolygon(var showAllBoundaries)

    // JAVASCRIPT FUNCTIONS

    function createBoundary() {
        vectorLayer.digitizeMode = 1;
        vectorLayer.isLineComplete = false;
        vectorLayer.isDigitizing = true;
    }

    function splitTerritory() {
        vectorLayer.digitizeMode = 0;
        vectorLayer.isLineComplete = false;
        vectorLayer.isDigitizing = true;
    }

    function setBoundary(overlappedTerritoryIds, annexOverlappingAreas) {
       if (territories.setTerritoryBoundary(currentTerritory.territoryId, vectorLayer.newPolygon.path, true, overlappedTerritoryIds, annexOverlappingAreas)) {
           console.log("Territory boundaries changed")
       }
       // clean up map
       vectorLayer.removeMapItem(vectorLayer.newPolygon);
       vectorLayer.newPolygon = null;
    }

    function removeBoundary() {
        if (currentTerritory) {
            currentTerritory.wktGeometry = "";
            // TODO: Check if refreshing the map is possible without reloading all territories
            territoryManager.reloadTerritories();
        }
    }

    function zoomSelectedTerritory() {
        if (currentTerritory)
            baseMap.visibleRegion = currentTerritory.boundingGeoRectangle;
    }

    function zoomSelectedStreet() {
        if (currentStreet) {
            baseMap.visibleRegion = currentStreet.boundingGeoRectangle();
        }
    }

    function zoomSelectedAddress() {
        if (currentAddress) {
            var regExp = /((?:[-+]?\.\d+|[-+]?\d+(?:\.\d*)?))\s*((?:[-+]?\.\d+|[-+]?\d+(?:\.\d*)?))/g, match;
            var lat;
            var lon;
            while (match = regExp.exec(currentAddress.wktGeometry))
            {
                lat = match[2];
                lon = match[1];
            }
            zoomCoordinate(lat, lon);
        }
    }

    function zoomCurrentPosition() {
        positionSource.positionChanged.disconnect(zoomCurrentPosition)
        baseMap.center = positionSource.position.coordinate;
        if (baseMap.zoomLevel < 17)
            baseMap.zoomLevel = 17;
    }

    function zoomCoordinate(lat, lon) {
        baseMap.center = QtPositioning.coordinate(lat, lon);
        if (baseMap.zoomLevel < 17)
            baseMap.zoomLevel = 17;
    }

    function zoomFull() {
        if (addressModel.count === 0 && !territoryTreeModel.boundingGeoRectangle.isValid) {
            // zoom to congregation's address
            var congrAddress = territories.getCongregationAddress();
            if (congrAddress !== "")
                geocodeAddress(congrAddress);
        } else {
            baseMap.visibleRegion = territoryTreeModel.boundingGeoRectangle;
        }
    }

    function getRoot() {
        var par = parent;
        while ( par.parent !== null ) {
            par = par.parent;
        }
        return par;
    }

    function clearAddressSearchMarkers() {
        // remove previous markers
        var count = vectorLayer.addressSearchMarkers.length;
        for (var i = 0; i < count; i++) {
            vectorLayer.removeMapItem(vectorLayer.addressSearchMarkers[i]);
            vectorLayer.addressSearchMarkers[i].destroy();
        }
        vectorLayer.addressSearchMarkers = [];
    }

    function geocodeAddress(query) {
        // send the geocode request
        geocodeRetryCount = 0;
        geocodeQuery = query;
        territories.onGeocodingFinished.connect(geocodingFinished);
        territories.onGeocodingError.connect(geocodingFailed);
        territories.geocodeAddress(geocodeQuery);
    }

    function geocodingFinished(geocodeResults) {
        if (geocodeResults.length < 1 && geocodeRetryCount < 3 && geocodeQuery !== "") {
            // try again
            geocodeRetryCount++;
            territories.geocodeAddress(geocodeQuery);
            return;
        }

        territories.onGeocodingFinished.disconnect(geocodingFinished);
        territories.onGeocodingError.disconnect(geocodingFailed);

        displayGeocodeResults(geocodeResults);
    }

    function geoLocationToLocation(geoLocation) {
        var location = Object.create(geoLocationPrototype);
        location.location = geoLocation;
        return location;
    }

    function displayGeocodeResults(geocodeResults) {
        clearAddressSearchMarkers();

        var count = geocodeResults.length;
        if (count > 0) {
            var markerScale = addressModel.getMarkerScale();
            var minLat = Number.POSITIVE_INFINITY;
            var minLon = Number.POSITIVE_INFINITY;
            var maxLat = Number.NEGATIVE_INFINITY;
            var maxLon = Number.NEGATIVE_INFINITY;

            for (var j = 0; j < geocodeResults.length; j++) {
                var territoryId = 0;
                var addressTypeNumber = 0;
                var addressTypeIndex = 0;
                var color = "#000000";

                var location = geoLocationToLocation(geocodeResults[j]);
                var lat = location.coordinate.latitude;
                var lon = location.coordinate.longitude;
                minLat = Math.min(lat, minLat);
                minLon = Math.min(lon, minLon);
                maxLat = Math.max(lat, maxLat);
                maxLon = Math.max(lon, maxLon);

                var component = Qt.createComponent('TerritoryMapMarker.qml');
                if (component.status === Component.Ready) {
                    vectorLayer.addressSearchMarker = component.createObject(vectorLayer);
                    vectorLayer.addressSearchMarker.territoryId = 0;
                    vectorLayer.addressSearchMarker.isTerritorySelected = false;
                    vectorLayer.addressSearchMarker.addressId = 0;
                    vectorLayer.addressSearchMarker.coordinate = location.coordinate;
                    vectorLayer.addressSearchMarker.color = "#000000";
                    vectorLayer.addressSearchMarker.markerScale = markerScale < 1 ? 1 : markerScale;
                    vectorLayer.addressSearchMarker.location.location = geocodeResults[j];

                    vectorLayer.addressSearchMarkers.push(vectorLayer.addressSearchMarker);
                    vectorLayer.addMapItem(vectorLayer.addressSearchMarker);
                }
            }

            baseMap.visibleRegion = QtPositioning.rectangle(QtPositioning.coordinate(maxLat, minLon), QtPositioning.coordinate(minLat, maxLon));
            if (count > 1)
                baseMap.zoomLevel = baseMap.zoomLevel - 1;
        }
    }

    function geocodingFailed(errorMessage) {
        territories.onGeocodingFinished.disconnect(geocodingFinished);
        territories.onGeocodingError.disconnect(geocodingFailed);
        message.title = qsTr("Search address", "Add or edit territory address");
        message.text = errorMessage;
        message.visible = true;
        return;
    }

    function addAddress(location) {
        var newAddressId = addressModel.addAddress(currentTerritory.territoryId, location.address.country,
                                    location.address.state, location.address.county, location.address.city, location.address.district,
                                    location.address.street, location.address.streetNumber, location.address.postalCode,
                                    "POINT(" + location.coordinate.longitude + " " + location.coordinate.latitude + ")");
        if (newAddressId > 0) {
            clearAddressSearchMarkers();
            var addressIndex = addressModel.getAddressIndex(newAddressId);
            addressModel.setSelectedAddress(addressIndex);
        }
    }

    function reassignAddress(addressId) {
        if (addressId > 0 && currentTerritory !== null) {
            addressModel.updateAddress(addressId, currentTerritory.territoryId);
        }
    }

    function removeAddress(addressId) {
        if (addressId > 0) {
            addressModel.removeAddress(addressId);
        }
    }

    function createElements(point) {
        if (vectorLayer.snapMarker === null) {
            var componentMarker = Qt.createComponent("SimpleMarker.qml");
            if (componentMarker.status === Component.Ready) {
                vectorLayer.snapMarker = componentMarker.createObject(vectorLayer);
                vectorLayer.snapMarker.coordinate = point;
                vectorLayer.snapMarker.markerType = 3;
                vectorLayer.snapMarker.visible = false;
                vectorLayer.snapMarker.isClickable = false;
                vectorLayer.addMapItem(vectorLayer.snapMarker);
            } else {
                console.log("Marker not created");
            }
        }

        switch (vectorLayer.digitizeMode) {
        case 0:
            // Polyline mode
            if (vectorLayer.newPolyline === null) {
                createLine(point);
                createRubberBand(point);
            }
            vectorLayer.newPolyline.mainPolyline.insertCoordinate(vectorLayer.newPolyline.path.length - 1, point);
            break;
        default:
            // Polygon mode
            if (vectorLayer.newPolygon === null) {
                createPolygon(point);
                createRubberBand(point);
            }
            vectorLayer.newPolygon.mainPolygon.addCoordinate(point);
            break;
        }
    }

    function createLine(point) {
        var componentLine = Qt.createComponent("SimplePolyline.qml");
        if (componentLine.status === Component.Ready) {
            vectorLayer.newPolyline = componentLine.createObject(vectorLayer);
            vectorLayer.newPolyline.lineWidth = 2;
            vectorLayer.newPolyline.mainPolyline.addCoordinate(point);

            vectorLayer.addMapItem(vectorLayer.newPolyline);
        } else {
            console.log("Line not created");
        }
    }

    function createPolygon(point) {
        var componentPolygon = Qt.createComponent("SimplePolygon.qml");
        if (componentPolygon.status === Component.Ready) {
            vectorLayer.newPolygon = componentPolygon.createObject(vectorLayer);
            vectorLayer.newPolygon.mainPolygon.addCoordinate(point);

            vectorLayer.addMapItem(vectorLayer.newPolygon);
        } else {
            console.log("Polygon not created");
        }
    }

    function createRubberBand(point) {
        vectorLayer.removeMapItem(vectorLayer.rubberBand);
        var component = Qt.createComponent("SimplePolyline.qml");
        if (component.status === Component.Ready) {
            vectorLayer.rubberBand = component.createObject(vectorLayer);
            vectorLayer.rubberBand.mainPolyline.addCoordinate(point);
            vectorLayer.rubberBand.mainPolyline.addCoordinate(point);
            vectorLayer.rubberBand.mainPolyline.addCoordinate(point);
            vectorLayer.addMapItem(vectorLayer.rubberBand);
        } else {
            console.log("Rubber band not created");
        }
    }

    function addGeoItem(item)
    {
        var co = Qt.createComponent('mapitems/'+item+'.qml')
        if (co.status === Component.Ready) {
            sketchLayer.unfinishedItem = co.createObject(sketchLayer)
            sketchLayer.unfinishedItem.setGeometry(sketchPointHandler.lastCoordinate)
            sketchLayer.unfinishedItem.addGeometry(sketchPointHandler.currentCoordinate, false)
            sketchLayer.addMapItem(sketchLayer.unfinishedItem)
        } else {
            console.log(item + " is not supported right now, please call us later.")
        }
    }

    function finishGeoItem()
    {
        if (sketchLayer.unfinishedItem === undefined)
            return;

        sketchLayer.unfinishedItem.finishAddGeometry();

        switch(sketchLayer.ink) {
        case 1:
            penGeoDatabase.addItem(sketchLayer.unfinishedItem);
            break;
        case 2:
            markerGeoDatabase.addItem(sketchLayer.unfinishedItem);
            break;
        case 3:
            highlighterGeoDatabase.addItem(sketchLayer.unfinishedItem);
            break;
        }

        sketchLayer.removeMapItem(sketchLayer.unfinishedItem);
        sketchLayer.unfinishedItem = undefined;
    }

    sketchPointHandler.onGrabChanged: function(transition, eventPoint) {
        switch (eventPoint.state) {
            case EventPoint.Pressed:
                mapDragHandler.enabled = false;
                if (sketchLayer.ink > 0) {
                    sketchPointHandler.lastCoordinate = sketchLayer.toCoordinate(eventPoint.position);
                    sketchPointHandler.currentCoordinate = sketchLayer.toCoordinate(sketchPointHandler.point.position);
                    addGeoItem("PolylineItem");
                }
                break;
            case EventPoint.Updated:
                break;
            case EventPoint.Released:
                if (sketchLayer.ink > 0) {
                    finishGeoItem();
                    mapDragHandler.enabled = true;
                }
                break;
            default:
                break;
        }
    }

    sketchPointHandler.onPointChanged: {
        // pen or mouse button pressed or activated via finger touch
        if (sketchPointHandler.point.pressedButtons || sketchPointHandler.active) {
            if (sketchLayer.unfinishedItem !== undefined)
                sketchLayer.unfinishedItem.addGeometry(sketchLayer.toCoordinate(sketchPointHandler.point.position), false);
        }
    }

    sketchDrawingInkButton.states: [
        State {
            name: "Highlighter"; when: sketchLayer.activeDrawingInk === 3 && sketchLayer.ink !== 3
            PropertyChanges { target: sketchDrawingInkButton; icon.source: "qrc:/icons/ink_highlighter.svg"; }
        },
        State {
            name: "HighlighterActive"; when: sketchLayer.activeDrawingInk === 3 && sketchLayer.ink === 3
            PropertyChanges { target: sketchDrawingInkButton; icon.source: "qrc:/icons/ink_highlighter_filled.svg"; }
        },
        State {
            name: "Marker"; when: sketchLayer.activeDrawingInk === 2 && sketchLayer.ink !== 2
            PropertyChanges { target: sketchDrawingInkButton; icon.source: "qrc:/icons/ink_marker.svg"; }
        },
        State {
            name: "MarkerActive"; when: sketchLayer.activeDrawingInk === 2 && sketchLayer.ink === 2
            PropertyChanges { target: sketchDrawingInkButton; icon.source: "qrc:/icons/ink_marker_filled.svg"; }
        },
        State {
            name: "Pen"; when: sketchLayer.activeDrawingInk === 1 && sketchLayer.ink !== 1
            PropertyChanges { target: sketchDrawingInkButton; icon.source: "qrc:/icons/ink_pen.svg"; }
        },
        State {
            name: "PenActive"; when: sketchLayer.activeDrawingInk === 1 && sketchLayer.ink === 1
            PropertyChanges { target: sketchDrawingInkButton; icon.source: "qrc:/icons/ink_pen_filled.svg"; }
        }
    ]
    sketchDrawingInkButton.onClicked: {
        if (sketchLayer.ink > 0) {
            // select next drawing tool if the eraser was not used before
            sketchLayer.activeDrawingInk = sketchLayer.activeDrawingInk % 3 + 1;
        }
        sketchLayer.ink = sketchLayer.activeDrawingInk;
    }
    sketchEraserButton.onClicked: {
        sketchLayer.ink = 0;
    }

    // OBJECT PROPERTIES
    Keys.onPressed: (event)=> {
        if (event.key === Qt.Key_Plus) {
            baseMap.zoomLevel++;
        } else if (event.key === Qt.Key_Minus) {
            baseMap.zoomLevel--;
        } else if (event.key === Qt.Key_Left || event.key === Qt.Key_Right
                   || event.key === Qt.Key_Up   || event.key === Qt.Key_Down) {
            var dx = 0;
            var dy = 0;

            switch (event.key) {
            case Qt.Key_Left: dx = baseMap.width / 4; break;
            case Qt.Key_Right: dx = -baseMap.width / 4; break;
            case Qt.Key_Up: dy = baseMap.height / 4; break;
            case Qt.Key_Down: dy = -baseMap.height / 4; break;
            }

            var mapCenterPoint = Qt.point(baseMap.width / 2.0 - dx, baseMap.height / 2.0 - dy);
            baseMap.center = baseMap.toCoordinate(mapCenterPoint);
        } else if (event.key === Qt.Key_Escape) {
            if (vectorLayer.isDigitizing) {
                // cancel digitizing
                vectorLayer.isLineComplete = true;
                vectorLayer.isDigitizing = false;
                if (vectorLayer.rubberBand !== null) {
                    vectorLayer.removeMapItem(vectorLayer.rubberBand);
                    vectorLayer.rubberBand = null;
                }
                switch (vectorLayer.digitizeMode) {
                case 0: {
                    if (vectorLayer.newPolyline !== null) {
                        vectorLayer.removeMapItem(vectorLayer.newPolyline);
                        vectorLayer.newPolyline = null;
                    }
                    break;
                }
                default: {
                    if (vectorLayer.newPolygon !== null) {
                        vectorLayer.removeMapItem(vectorLayer.newPolyline);
                        vectorLayer.newPolyline = null;
                        vectorLayer.removeMapItem(vectorLayer.newPolygon);
                        vectorLayer.newPolygon = null;
                    }
                    break;
                }
                }
            }
        } else if (event.key === Qt.Key_Backspace) {
            if (vectorLayer.isDigitizing) {
                var point;
                switch (vectorLayer.digitizeMode) {
                case 0: {
                    // remove last but one point (last point = first point)
                    if (vectorLayer.newPolyline !== null && vectorLayer.newPolyline.mainPolyline.pathLength() > 2) {
                        vectorLayer.newPolyline.mainPolyline.removeCoordinate(vectorLayer.newPolyline.mainPolyline.pathLength() - 2);
                        point = vectorLayer.newPolyline.mainPolyline.path[vectorLayer.newPolyline.mainPolyline.pathLength() - 2];
                    }
                    break;
                }
                default: {
                    if (vectorLayer.newPolygon !== null && vectorLayer.newPolygon.mainPolygon.path.length > 2) {
                        point = vectorLayer.newPolygon.mainPolygon.path[vectorLayer.newPolygon.mainPolygon.path.length - 1];
                        vectorLayer.newPolygon.mainPolygon.removeCoordinate(point);
                        point = vectorLayer.newPolygon.mainPolygon.path[vectorLayer.newPolygon.mainPolygon.path.length - 1];
                    }
                    break;
                }
                }
                if (vectorLayer.rubberBand !== null) {
                    // update rubber band
                    vectorLayer.rubberBand.mainPolyline.replaceCoordinate(0, point);
                }
            }
        }
    }

    addressTextField.onAccepted: {
        geocodeAddress(addressTextField.text);
    }

    showBoundariesButton.states: [
        State {
            name: "ShowAllBoundaries"; when: boundaryLayer.boundaryDisplayOption === 2
            PropertyChanges { target: showBoundariesButton; icon.source: "qrc:/icons/boundary_all.svg"; }
        },
        State {
            name: "ShowSelectedBoundaries"; when: boundaryLayer.boundaryDisplayOption === 1
            PropertyChanges { target: showBoundariesButton; icon.source: "qrc:/icons/boundary.svg"; }
        },
        State {
            name: "HideBoundaries"; when: boundaryLayer.boundaryDisplayOption === 0
            PropertyChanges { target: showBoundariesButton; icon.source: "qrc:/icons/boundary_off.svg"; }
        }
    ]

    showBoundariesButton.onClicked: boundaryLayer.boundaryDisplayOption = (boundaryLayer.boundaryDisplayOption + 1) % 3

    showMarkersButton.states: [
        State {
            name: "ShowAllAddresses"; when: addressLayer.markerDisplayOption === 2
            PropertyChanges { target: showMarkersButton; icon.source: "qrc:/icons/location_all.svg"; }
        },
        State {
            name: "ShowTerritoryAddresses"; when: addressLayer.markerDisplayOption === 1
            PropertyChanges { target: showMarkersButton; icon.source: "qrc:/icons/location.svg"; }
        },
        State {
            name: "HideAddresses"; when: addressLayer.markerDisplayOption === 0
            PropertyChanges { target: showMarkersButton; icon.source: "qrc:/icons/location_off.svg"; }
        }
    ]
    showMarkersButton.onClicked: addressLayer.markerDisplayOption = (addressLayer.markerDisplayOption + 1) % 3

    showStreetsButton.states: [
        State {
            name: "ShowAllStreets"; when: streetLayer.streetDisplayOption === 2
            PropertyChanges { target: showStreetsButton; icon.source: "qrc:/icons/road_all.svg"; }
        },
        State {
            name: "ShowTerritoryStreets"; when: streetLayer.streetDisplayOption === 1
            PropertyChanges { target: showStreetsButton; icon.source: "qrc:/icons/road.svg"; }
        },
        State {
            name: "HideStreets"; when: streetLayer.streetDisplayOption === 0
            PropertyChanges { target: showStreetsButton; icon.source: "qrc:/icons/road_off.svg"; }
        }
    ]
    showStreetsButton.onClicked: streetLayer.streetDisplayOption = (streetLayer.streetDisplayOption + 1) % 3

    showSketchButton.states: [
        State {
            name: "ShowSketch"; when: sketchLayer.displayMode === 1
            PropertyChanges { target: showSketchButton; icon.source: "qrc:/icons/sketch.svg"; }
        },
        State {
            name: "HideSketch"; when: sketchLayer.displayMode === 0
            PropertyChanges { target: showSketchButton; icon.source: "qrc:/icons/sketch_off.svg"; }
        }
    ]
    showSketchButton.onClicked: sketchLayer.displayMode = (sketchLayer.displayMode + 1) % 2

    createBoundaryButton.onClicked: createBoundary()
    splitTerritoryButton.onClicked: splitTerritory()
    removeBoundaryButton.onClicked: removeBoundary()

    editModeButton.states: [
        State {
            name: "View"; when: vectorLayer.editMode === 0
            PropertyChanges { target: editModeButton; icon.source: "qrc:/icons/visibility.svg"; }
        },
        State {
            name: "DrawSketch"; when: vectorLayer.editMode === 1
            PropertyChanges { target: editModeButton; icon.source: "qrc:/icons/sketch_edit.svg"; }
        },
        State {
            name: "EditAddresses"; when: vectorLayer.editMode === 2
            PropertyChanges { target: editModeButton; icon.source: "qrc:/icons/edit_location.svg"; }
        },
        State {
            name: "EditBoundaries"; when: vectorLayer.editMode === 3
            PropertyChanges { target: editModeButton; icon.source: "qrc:/icons/edit_boundary.svg"; }
        }
    ]
    editModeButton.onClicked: isGdalAvailable && canEditTerritories
                              ? vectorLayer.editMode = (vectorLayer.editMode + 1) % 4
                              : vectorLayer.editMode = (vectorLayer.editMode + 1) % 2

    zoomSlider.onMoved: {
        baseMap.zoomLevel = zoomSlider.value;
    }

    positionSourceButton.states: [
        State {
            name: "LocationDisabled"; when: locationPermission.status === Qt.PermissionStatus.Denied
            PropertyChanges { target: positionSourceButton; icon.source: "qrc:/icons/location_disabled.svg"; }
        },
        State {
            name: "LocationOff"; when: locationPermission.status !== Qt.PermissionStatus.Granted || !isPositionSourceActive
            PropertyChanges { target: positionSourceButton; icon.source: "qrc:/icons/location_searching.svg"; }
        },
        State {
            name: "LocationOn"; when: locationPermission.status === Qt.PermissionStatus.Granted && isPositionSourceActive
            PropertyChanges { target: positionSourceButton; icon.source: "qrc:/icons/my_location.svg"; }
        }
    ]

    positionSourceButton.onClicked: {
        switch (locationPermission.status) {
           case Qt.PermissionStatus.Undetermined:
               locationPermission.request();
               return;
           case Qt.PermissionStatus.Denied:
               isPositionSourceActive = false;
               return;
           case Qt.PermissionStatus.Granted:
               isPositionSourceActive = !isPositionSourceActive;
               if (isPositionSourceActive)
                   positionSource.positionChanged.connect(zoomCurrentPosition);
        }
    }

    locationPermission.onStatusChanged: {
        if (locationPermission.status === Qt.PermissionStatus.Granted) {
            isPositionSourceActive = true;
            positionSource.positionChanged.connect(zoomCurrentPosition);
        }
    }

    // baseMap.plugin: osmPlugin

    openSearchBarButton.onClicked: {
        searchBar.isSearchBarOpen = true;
        addressTextField.focus = true;
    }

    clearSearchBarButton.onClicked: {
        if (addressTextField.text) {
            addressTextField.text = "";
            addressTextField.focus = true;
        }
        else
            searchBar.isSearchBarOpen = false;
    }

    mapPinchHandler.onActiveChanged: if (active) {
        baseMap.startCentroid = baseMap.toCoordinate(mapPinchHandler.centroid.position, false);
        if (sketchLayer.unfinishedItem !== undefined) {
            // remove undesired sketch
            sketchLayer.removeMapItem(sketchLayer.unfinishedItem);
            sketchLayer.unfinishedItem = undefined;
        }
    }
    mapPinchHandler.onScaleChanged: (delta) => {
        baseMap.zoomLevel += Math.log2(delta);
        baseMap.alignCoordinateToPoint(baseMap.startCentroid, mapPinchHandler.centroid.position);
    }
    mapDragHandler.onTranslationChanged: (delta) => baseMap.pan(-delta.x, -delta.y)

    baseMap.onSupportedMapTypesChanged: {
        if (baseMap.supportedMapTypes.length > 0) {
            var newActiveMapType = baseMap.supportedMapTypes[0];
            if (settings.osm_map_type !== "") {
                for (var mapType in baseMap.supportedMapTypes) {
                    if (baseMap.supportedMapTypes[mapType].name === settings.osm_map_type) {
                        newActiveMapType = baseMap.supportedMapTypes[mapType];
                        break;
                    }
                }
            }
            if (newActiveMapType !== undefined)
                baseMap.activeMapType = newActiveMapType;
        }
    }

    baseMap.onActiveMapTypeChanged: {
        settings.osm_map_type = baseMap.activeMapType.name;
    }

    // CHILD OBJECTS
    Settings {
        id: settings
        category: "map_view"
        property string osm_map_type: ""
        property double basemap_opacity: 1.0
        property int boundary_display_mode: 2
        property double boundary_layer_opacity: 1.0
        property int street_display_mode: 2
        property double street_layer_opacity: 1.0
        property int address_display_mode: 2
        property double address_layer_opacity: 1.0
        property int sketch_display_mode: 1
        property double sketch_layer_opacity: 1.0
        property int sketch_active_ink: 1
    }

    Connections {
        target: territoryManager
        function onZoomSelectedTerritory() { zoomSelectedTerritory(); }
        function onZoomSelectedAddress() { zoomSelectedAddress(); }
        function onZoomSelectedStreet() { zoomSelectedStreet(); }
        function onZoomCoordinate(lat, lon) { zoomCoordinate(lat, lon); }
        function onDisplayGeocodeResults(geocodeResults) { displayGeocodeResults(geocodeResults); }
        function onClearAddressSearchMarkers() { clearAddressSearchMarkers(); }
    }

    Connections {
        target: sketchLayer

        function onDeleteSketchItem(modelData) {
            Array.prototype.clone = function() {
                return this.map(e => Array.isArray(e) ? e.clone() : e);
            };

            // search and delete in pen db
            var penModel;
            for (var pRootIndex in penGeoDatabase.model) {
                if (JSON.stringify(penGeoDatabase.model[pRootIndex]) === JSON.stringify(modelData)) {
                    penModel = penGeoDatabase.model.clone();
                    penModel.splice(pRootIndex, 1);
                    break;
                } else {
                    for (var pItemIndex in penGeoDatabase.model[pRootIndex]["data"]) {
                        if (JSON.stringify(penGeoDatabase.model[pRootIndex]["data"][pItemIndex]) === JSON.stringify(modelData)) {
                            penModel = penGeoDatabase.model.clone();
                            penModel[pRootIndex]["data"].splice(pItemIndex, 1);
                            break;
                        }
                    }
                }
            }
            if (penModel !== undefined)
                penGeoDatabase.model = penModel;

            // search and delete in marker db
            var markerModel;
            for (var mRootIndex in markerGeoDatabase.model) {
                if (JSON.stringify(markerGeoDatabase.model[mRootIndex]) === JSON.stringify(modelData)) {
                    markerModel = markerGeoDatabase.model.clone();
                    markerModel.splice(mRootIndex, 1);
                    break;
                } else {
                    for (var mItemIndex in markerGeoDatabase.model[mRootIndex]["data"]) {
                        if (JSON.stringify(markerGeoDatabase.model[mRootIndex]["data"][mItemIndex]) === JSON.stringify(modelData)) {
                            markerModel = markerGeoDatabase.model.clone();
                            markerModel[mRootIndex]["data"].splice(mItemIndex, 1);
                            break;
                        }
                    }
                }
            }
            if (markerModel !== undefined)
                markerGeoDatabase.model = markerModel;

            // search and delete in highlighter db
            var highlighterModel;
            for (var hRootIndex in highlighterGeoDatabase.model) {
                if (JSON.stringify(highlighterGeoDatabase.model[hRootIndex]) === JSON.stringify(modelData)) {
                    highlighterModel = highlighterGeoDatabase.model.clone();
                    highlighterModel.splice(hRootIndex, 1);
                    break;
                } else {
                    for (var hItemIndex in highlighterGeoDatabase.model[hRootIndex]["data"]) {
                        if (JSON.stringify(highlighterGeoDatabase.model[hRootIndex]["data"][hItemIndex]) === JSON.stringify(modelData)) {
                            highlighterModel = highlighterGeoDatabase.model.clone();
                            highlighterModel[hRootIndex]["data"].splice(hItemIndex, 1);
                            break;
                        }
                    }
                }
            }
            if (highlighterModel !== undefined)
                highlighterGeoDatabase.model = highlighterModel;
        }

        function onDeleteAllSketchItems() {
            penGeoDatabase.clear();
            markerGeoDatabase.clear();
            highlighterGeoDatabase.clear();
        }
    }

    Location { id: geoLocationPrototype } // for conversion from QGeoLocation to QML Location

    GeoJsonData {
        id: penGeoDatabase
    }
    GeoJsonData {
        id: markerGeoDatabase
    }
    GeoJsonData {
        id: highlighterGeoDatabase
    }

    Component {
        id: territoryBoundaryDelegate

        // nested MapItemView for grouped territories in tree model
        MapItemView {
            id: groupDelegate
            required property int index
            required property int childCount

            model: childCount
            delegate: TerritoryMapPolygon {
                id: mapPolygon
                required property int index
                property var territory: territoryTreeModel.data(
                                                  territoryTreeModel.index(mapPolygon.index, 0, territoryTreeModel.index(groupDelegate.index, 0)),
                                                  TerritoryTreeModel.TerritoryRole)
                // model data is set via required properties
                isSelected: false
                territoryId: territory ? territory.territoryId : 0
                multiPolygon: territoryTreeModel.data(
                           territoryTreeModel.index(mapPolygon.index, 0, territoryTreeModel.index(groupDelegate.index, 0)),
                           TerritoryTreeModel.MultiPolygonRole)  // mapPolygon.multiPolygon
                model: multiPolygon
            }
        }
    }

    Component {
        id: territoryStreetDelegate

        TerritoryMapPolyline {
            // model data is set via required properties
            isTerritorySelected: currentTerritory?.territoryId === street.territoryId ? true : false
            isStreetSelected: currentStreet?.streetId === street.streetId ? true : false
            model: multiLine
        }
    }

    Component {
        id: territoryMarkerDelegate

        TerritoryMapMarker {
            territoryId: model.territoryId
            isTerritorySelected: currentTerritory?.territoryId === model.territoryId ? true : false
            isMarkerSelected: currentAddress?.id === model.id ? true : false
            addressId: model.id
            coordinate: model.coordinate
            color: model.addressTypeColor
            markerScale: vectorLayer.markerScale
        }
    }

    Plugin {
        id: osmPlugin
        name: "osm"
        // specify plugin parameters if necessary
        // PluginParameter {
        //     name:
        //     value:
        // }
        //PluginParameter{ name: "style"; value: "CycleMap"}
    }

    Menu {
        id: contextMenu
        property var marker;
        property int featureTerritoryId;
        property int markerAddressId;

        function show(sender, territoryId, addressId) {
            marker = sender;
            featureTerritoryId = territoryId;
            markerAddressId = addressId;
            open();
        }

        onAboutToShow: {
            addAddressMenuItem.enabled = currentTerritory;
            reassignAddressMenuItem.enabled = currentTerritory !== null
                    && currentTerritory.territoryId !== featureTerritoryId;
        }

        MenuItem {
            id: annexAreaMenuItem
            text: qsTr("Join to the selected territory", "Join two territories into one")
            icon.source: "qrc:/icons/boundary_join.svg"
            visible: canEditTerritories && contextMenu.featureTerritoryId !== 0 && contextMenu.markerAddressId === -1
                     && currentTerritory !== null && currentTerritory.territoryId !== contextMenu.featureTerritoryId
            // height: visible ? implicitHeight : 0
            onTriggered: {
                var territoryList = territories.getAllTerritories();
                var count = territoryList.length;
                for (var i = 0; i < count; i++) {
                    if (typeof territoryList[i] !== "undefined")
                        if (territoryList[i].territoryId === contextMenu.featureTerritoryId) {
                            var overlappedTerritoryNumbers = [];
                            overlappedTerritoryNumbers.push(contextMenu.featureTerritoryId);
                            if (territories.setTerritoryBoundary(currentTerritory.territoryId, territoryList[i].wktGeometry, true, overlappedTerritoryNumbers, true)) {
                                console.log("Territories joined");
                            }
                            break;
                        }
                }
            }
        }

        MenuItem {
            id: addAddressMenuItem
            text: qsTr("Add address to selected territory")
            icon.source: "qrc:/icons/add_location.svg"
            visible: canEditTerritories && contextMenu.featureTerritoryId === 0 && contextMenu.markerAddressId > -1
            // height: visible ? implicitHeight : 0
            onTriggered: addAddress(contextMenu.marker.location)
        }

        MenuItem {
            id: reassignAddressMenuItem
            text: qsTr("Assign to selected territory", "Reassign territory address")
            icon.source: "qrc:/icons/move_location.svg"
            visible: canEditTerritories && contextMenu.featureTerritoryId > 0 && contextMenu.markerAddressId > -1
            // height: visible ? implicitHeight : 0
            onTriggered: reassignAddress(contextMenu.markerAddressId)
        }

        MenuItem {
            id: removeAddressMenuItem
            text: qsTr("Remove address", "Delete territory address")
            icon.source: "qrc:/icons/remove_location.svg"
            visible: canEditTerritories && contextMenu.featureTerritoryId > 0 && contextMenu.markerAddressId > -1
            // height: visible ? implicitHeight : 0
            onTriggered: removeAddress(contextMenu.markerAddressId)
        }
    }

    Component.onCompleted: {
        // set map plugin
        var parameters = new Array();
        for (var prop in geoServiceParameters) {
            var parameter = Qt.createQmlObject('import QtLocation; PluginParameter{ name: "'+ prop + '"; value: "' + geoServiceParameters[prop] + '"}', baseMap);
            parameters.push(parameter);
        }
        baseMap.pluginParameters = parameters;
        var provider = defaultGeoServiceProvider;
        var plugin;
        if (parameters && parameters.length > 0)
            plugin = Qt.createQmlObject ('import QtLocation; Plugin{ name:"' + provider + '"; parameters: baseMap.pluginParameters}', baseMap);
        else
            plugin = Qt.createQmlObject ('import QtLocation; Plugin{ name:"' + provider + '"}', baseMap);
        baseMap.plugin = plugin;
        baseMap.opacity = settings.basemap_opacity;
        boundaryLayer.opacity = settings.boundary_layer_opacity;
        streetLayer.opacity = settings.street_layer_opacity;
        addressLayer.opacity = settings.address_layer_opacity;
        sketchLayer.opacity = settings.sketch_layer_opacity;

        vectorLayer.markerScale = addressModel.getMarkerScale();

        boundaryLayer.boundaryDisplayOption = settings.boundary_display_mode;
        streetLayer.streetDisplayOption = settings.street_display_mode;
        addressLayer.markerDisplayOption = settings.address_display_mode;
        sketchLayer.displayMode = settings.sketch_display_mode;
        sketchLayer.activeDrawingInk = settings.sketch_active_ink;

        // load territory sketch
        var penSketchPath = StandardPaths.locate(StandardPaths.AppDataLocation, "territorysketch_pen.json");
        if (penSketchPath.toString() !== "") {
            penGeoDatabase.openUrl(penSketchPath);
        } else {
            penGeoDatabase.model = {
                type : "FeatureCollection",
                data : []
            };
        }
        var markerSketchPath = StandardPaths.locate(StandardPaths.AppDataLocation, "territorysketch_marker.json");
        if (markerSketchPath.toString() !== "") {
            markerGeoDatabase.openUrl(markerSketchPath);
        } else {
            markerGeoDatabase.model = {
                type : "FeatureCollection",
                data : []
            };
        }
        var highlighterSketchPath = StandardPaths.locate(StandardPaths.AppDataLocation, "territorysketch_highlighter.json");
        if (highlighterSketchPath.toString() !== "") {
            highlighterGeoDatabase.openUrl(highlighterSketchPath);
        } else {
            highlighterGeoDatabase.model = {
                type : "FeatureCollection",
                data : []
            };
        }

        // geoDatabase.model = [{
        //     type : "FeatureCollection",
        //     data : []
        // }];

        zoomFull();
    }

    Component.onDestruction: {
        settings.osm_map_type = baseMap.activeMapType.name;
        settings.basemap_opacity = baseMap.opacity;
        settings.boundary_display_mode = boundaryLayer.boundaryDisplayOption;
        settings.boundary_layer_opacity = boundaryLayer.opacity;
        settings.street_display_mode = streetLayer.streetDisplayOption;
        settings.street_layer_opacity = streetLayer.opacity;
        settings.address_display_mode = addressLayer.markerDisplayOption;
        settings.address_layer_opacity = addressLayer.opacity;
        settings.sketch_display_mode = sketchLayer.displayMode;
        settings.sketch_layer_opacity = sketchLayer.opacity;
        settings.sketch_active_ink = sketchLayer.activeDrawingInk;
        settings.sync();

        // save territory sketch
        var appDataFolder = StandardPaths.writableLocation(StandardPaths.AppDataLocation)
        if (penGeoDatabase.model !== undefined) {
            penGeoDatabase.saveAs(appDataFolder + "/territorysketch_pen.json");
        }
        if (markerGeoDatabase.model !== undefined) {
            markerGeoDatabase.saveAs(appDataFolder + "/territorysketch_marker.json");
        }
        if (highlighterGeoDatabase.model !== undefined) {
            highlighterGeoDatabase.saveAs(appDataFolder + "/territorysketch_highlighter.json");
        }
    }
}
