/**
 * This file is part of TheocBase.
 *
 * Copyright (C) 2011-2024, TheocBase Development Team, see AUTHORS.
 *
 * TheocBase is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * TheocBase is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with TheocBase.  If not, see <http://www.gnu.org/licenses/>.
 */
import QtCore
import QtQuick
import QtQuick.Controls
import QtQuick.Controls.Material
import QtQuick.Controls.Material.impl
import QtQuick.Layouts
import QtQml.Models
import QtPositioning
import QtLocation
import net.theocbase.mobile 1.0
import net.theocbase 1.0

TerritoryManagerPageForm {
    id: territoryManager

    // PROPERTY DECLARATIONS
    property bool canEditTerritories: accessControl && accessControl.user && accessControl.user.hasPermission(PermissionRule.CanEditTerritories)
    property bool canViewTerritoryAssignments: accessControl && accessControl.user && accessControl.user.hasPermission(PermissionRule.CanViewTerritoryAssignments)
    property bool canViewTerritoryAddresses: accessControl && accessControl.user && accessControl.user.hasPermission(PermissionRule.CanViewTerritoryAddresses)
    property DataObjectListModel addressTypeListModel: territories.getAddressTypes()
    property DataObjectListModel streetTypeListModel: territories.getStreetTypes()
    property Territories territories: Territories {}
    property Territory currentTerritory
    property TerritoryStreet currentStreet
    property TerritoryAddress currentAddress
    property alias territoryTreeView: territoryTreeView
    property bool isDrawerOpen: drawer.position === 1.0
    property bool isPrintingOpen

    // SIGNAL DECLARATIONS
    signal territorySelectionChanged()
    signal zoomSelectedTerritory()
    signal zoomSelectedStreet()
    signal zoomSelectedAddress()
    signal zoomCoordinate(var lat, var lon)
    signal displayGeocodeResults(var geocodeResults)
    signal clearAddressSearchMarkers()

    // JAVASCRIPT FUNCTIONS
    function selectRootIndex(title) {
        // selected root index of the item in the territory tree model that matches the given title
        for (var i = 0; i < territoryTreeModel.rowCount(); i++) {
            var groupTitle = territoryTreeModel.get(i).title;
            if (groupTitle === title) {
                var modelIndex = territoryTreeModel.index(i, 0);
                var proxyIndex = territoriesProxyModel.mapFromSource(modelIndex)
                territoriesDetailModel.rootIndex = proxyIndex;
                break;
            }
        }
    }

    function resetDefaultTerritoryLookupTreeViewSettings() {
        settings.territories_groupByIndex = 1;
        territoryTreeView.groupByIndex = settings.territories_groupByIndex;
        settings.territories_groupBySortOrder = Qt.AscendingOrder;
        territoryTreeView.groupBySortOrder = settings.territories_groupBySortOrder;
        settings.territories_sortOrder = Qt.AscendingOrder;
        territoryTreeView.sortOrder = settings.territories_sortOrder;
        settings.territories_detailRowCount = 3;
        settings.territories_labelingMode = 1;
        territoryLookupTreeViewMoreMenu.detailRowCount = settings.territories_detailRowCount;
        territoryLookupTreeViewMoreMenu.labelingMode = settings.territories_labelingMode;
        settings.sync();
        reloadTerritories();
    }

    function reloadTerritories() {
        // save current selection
        var currentTerritoryId = currentTerritory?.territoryId ?? 0;
        if (canEditTerritories)
            territoryTreeModel.saveTerritories();
        territoryTreeModel.updateModel();
        // restore selection
        currentTerritory = territoryTreeModel.findTerritory(currentTerritoryId);
    }

    function refreshStreets(reload) {
        // save current selection
        var previousStreetId = currentStreet?.streetId ?? 0;
        if (canEditTerritories) {
            // save streets
            streetModel.saveStreets();
        }
        if (reload)
            streetModel.loadStreets();
        // restore selection
        var previousStreet = streetModel.findStreet(previousStreetId);
        if (previousStreet?.territoryId === currentTerritory.territoryId)
            currentStreet = previousStreet;
        else
            currentStreet = null;
    }

    function refreshAddresses(reload) {
        // save current selection
        var previousAddressId = currentAddress?.id ?? 0;
        if (canEditTerritories) {
            // save addresses
            addressModel.saveAddresses();
        }
        if (reload)
            addressModel.loadAddresses();
        // restore selection
        var previousAddress = addressModel.findAddress(previousAddressId);
        if (previousAddress?.territoryId === currentTerritory.territoryId)
            currentAddress = previousAddress;
        else
            currentAddress = null;
    }

    function save() {
        return true;
    }

    function switchSidePanel() {
        console.log("switchSidePanel")
        territoryManager.forceActiveFocus()

        if (!sidebar.isVisible)
            sidebar.loadPage("../TerritoryPanel.qml", {})
        sidebar.isVisible = !sidebar.isVisible;

        //if (!sidebar.isVisible)
            // stackView.clearAndPush(
            //     Qt.resolvedUrl("TerritoryPanel.qml"),
            //     {territoryManager: territoryManager,
            //     territories: territories,
            //     currentTerritory: currentTerritory,
            //     addressTypeListModel: addressTypeListModel,
            //     streetTypeListModel: streetTypeListModel})
        //sidebar.isVisible = !sidebar.isVisible;
    }


    // OBJECT PROPERTIES
    onWidthChanged: {
        if (width < 800) {
            splitView.visible = false
            itemView.visible = true
        } else {
            itemView.visible = false
            splitView.visible = true
        }
    }

    territoryTreeView.onSearchTextChanged: function(text) {
        territoriesProxyModel.filterText = text
    }

    territoryTreeView.onGroupByIndexChanged: {
        if (territoryTreeView.groupByIndex >= 0) {
            settings.territories_groupByIndex = territoryTreeView.groupByIndex;
            territoryTreeModel.groupByIndex = territoryTreeView.groupByIndex;
        }
    }

    // territoryTreeView.onGroupBySortOrderChanged: {
    //     settings.territories_groupBySortOrder = territoryTreeView.groupBySortOrder;
    //     territoryTreeModel.groupBySortOrder = territoryTreeView.groupBySortOrder ? Qt.DescendingOrder : Qt.AscendingOrder;
    // }

    territoryTreeView.onSortOrderChanged: {
        settings.territories_sortOrder = territoryTreeView.sortOrder;
        territoriesProxyModel.sort(0, territoryTreeView.sortOrder)
    }

    // territoryTreeView.onAddItem: {
    //     var newTerritoryId = territoryTreeModel.addNewTerritory();
    //     if (newTerritoryId >= 0) {
    //         currentTerritory = territoryTreeModel.findTerritory(newTerritoryId);
    //         if (!sidebar.isVisible)
    //             switchSidePanel();
    //     }
    // }

    showSidePanelButton.onClicked: {
        console.log("onShowSidePanelButtonChanged")
        switchSidePanel();
    }

    onCurrentTerritoryChanged: {
        if (canEditTerritories) {
            territoryTreeModel.saveTerritories();
            if (currentTerritory) {
                // reload territory, in case boundaries etc. were changed in the map
                // to avoid saving an older values when other properties are edited
                currentTerritory.reloadTerritory();
            }
        }

        if (canEditTerritories)
            assignmentModel.saveAssignments();
        assignmentModel.loadAssignments(currentTerritory ? currentTerritory.territoryId : 0);

        refreshStreets();

        refreshAddresses();

        // set selection in treeview
        var territoryIndex = territoryTreeModel.index(currentTerritory ? currentTerritory.territoryId : 0);
        //territoryTreeView.selectionModel.setCurrentIndex(territoriesProxyModel.mapFromSource(territoryIndex), ItemSelectionModel.ClearAndSelect);
        var parentIndex = territoryTreeModel.parent(territoryIndex);
        var parentProxyIndex = territoriesProxyModel.mapFromSource(parentIndex);
        territoriesDetailModel.rootIndex = parentProxyIndex;
        territoryTreeView.currentIndex = territoriesProxyModel.mapFromSource(territoryIndex).row;
        territorySelectionChanged();
    }

    onCurrentStreetChanged: {
        if (canEditTerritories)
            streetModel.saveStreets();
    }

    onCurrentAddressChanged: {
        if (canEditTerritories)
            addressModel.saveAddresses();
    }

    header: BaseToolbar {
        id: toolBar
        title: qsTr("Territory Manager", "application settings") + qmlTranslator?.changed
        componentLeft: ToolButton {
            action: navigateDrawerAction
            visible: true //appWindow.portraitMode
        }
        componentRightMiddle: Row {
            spacing: 0
            ToolButton {
                anchors.verticalCenter: parent.verticalCenter
                icon.source: "qrc:/icons/fullscreen.svg"
                onClicked: mapView.zoomFull()
            }
            ToolButton {
                anchors.verticalCenter: parent.verticalCenter
                icon.source: "qrc:/icons/layers.svg"
                onClicked: mapView.showLayerButtons = !mapView.showLayerButtons
            }
        }
        componentRight: Row {
            spacing: 0
            ToolButton {
                anchors.verticalCenter: parent.verticalCenter
                icon.source: ccloud.syncState === Cloud.Synchronized
                             ? "qrc:/icons/cloud_done.svg"
                             : ccloud.syncState === Cloud.Download
                               ? "qrc:/icons/cloud_download.svg"
                               : "qrc:/icons/cloud_upload.svg"
                onClicked: { syncpage.runSync() }
            }
            ToolButton {
                icon.source: "qrc:/icons/more.svg"
                onClicked: moreMenu.open()

                Menu {
                    id: moreMenu
                    MenuItem {
                        text: qsTr("Import data",
                                   "Import territory data from a file")
                        icon.source: "qrc:/icons/input.svg"
                        enabled: isGdalAvailable && canEditTerritories
                        visible: isGdalAvailable && canEditTerritories
                        height: enabled ? implicitHeight : 0
                        width: enabled ? implicitWidth : 0
                        onTriggered: importBoundaries()
                    }

                    MenuItem {
                        text: "Print"
                        icon.source: "qrc:/icons/print.svg"
                        onClicked: {
                            printLoader.source = ""
                            printLoader.source = "PrintOptions.qml"
                        }

                        Loader {
                            id: printLoader

                            Connections {
                                target: printLoader.item
                                function onClosed() {
                                    isPrintingOpen = false;
                                }
                            }

                            onLoaded: {
                                isPrintingOpen = true;
                                item.open()
                            }
                        }
                    }
                }
            }
        }
    }

    focus: true
    Keys.onReleased: (event)=> {
        if (event.key === Qt.Key_Back) {
            if (sidebar.isVisible) {
                var territoryPanel = sidebar.loaderItem;
                if (territoryPanel.depth > 1) {
                    territoryPanel.pop();
                } else {
                    // hide side panel
                    switchSidePanel();
                }
                event.accepted = true;
            }
        }
    }

    StackView.onActivated: {
        console.log("Territory page activated");
        appWindow.activepage = 2;
    }

    Component.onCompleted: {
        territoryTreeModel.updateModel();
        territoryTreeView.groupByIndex = settings.territories_groupByIndex;
        territoriesProxyModel.sortRole = "territoryNumber";
        territoryTreeView.sortOrder = settings.territories_sortOrder;
        mapView?.zoomFull();

        // set binding to only display data from the current territory
        addressProxyModel.filterTerritoryId = Qt.binding(function() { return currentTerritory?.territoryId ?? 0; });
        streetProxyModel.filterTerritoryId = Qt.binding(function() { return currentTerritory?.territoryId ?? 0; });

        streetModel.loadStreets();
        addressModel.loadAddresses();
    }

    Component.onDestruction: {
        // save current changes
        if (currentTerritory)
            currentTerritory.save();
        assignmentModel.saveAssignments();
        streetModel.saveStreets();
        addressModel.saveAddresses();
    }

    // CHILD OBJECTS
    SystemPalette { id: myPalette; colorGroup: SystemPalette.Active }

    Settings {
        id: settings
        category: "LookupControl"
        property int territories_groupByIndex: 1
        property bool territories_groupBySortOrder: Qt.AscendingOrder
        property bool territories_sortOrder: Qt.AscendingOrder
        property int territories_detailRowCount: 3
        property int territories_labelingMode: 1
    }

    TerritoryTreeSFProxyModel {
        id: territoriesProxyModel
        source: territoryTreeModel
        sortCaseSensitivity: Qt.CaseInsensitive

        //sortRole: territoryTreeView.he
    }

    Connections {
        target: territories
        function onBoundaryGeometryChanged() {
            reloadTerritories();
        }
        function onStreetGeometryChanged() {
            refreshStreets(true);
        }
        function onAddressGeometryChanged() {
            refreshAddresses(true);
        }
    }

    Action {
        id: navigateDrawerAction
        icon.source: "qrc:/icons/menu.svg"
        onTriggered: {
            drawer.open();
        }
    }

    DelegateModel {
        id: territoriesDetailModel
        model: territoriesProxyModel

        delegate: SwipeDelegate {
            id: swipeDelegate
            width: LookupControl.view.width
            highlighted: LookupControl.view.currentIndex === index

            swipe.enabled: canEditTerritories && territoryId > 0 && territoriesDetailModel.rootIndex.valid // column === 1
            swipe.right: Label {
                id: deleteLabel
                text: qsTr("Delete")
                color: "white"
                verticalAlignment: Label.AlignVCenter
                padding: 12
                height: parent.height
                anchors.right: parent.right

                SwipeDelegate.onClicked: {
                    if (territoryId >= 0)
                        territoryTreeModel.removeTerritory(territoryId);
                }

                background: Rectangle {
                    color: deleteLabel.SwipeDelegate.pressed ? Qt.darker("tomato", 1.1) : "tomato"
                }
            }

            contentItem: RowLayout {
                Label {
                    Layout.fillWidth: true
                    Layout.alignment: Qt.AlignVCenter | Qt.AlignLeft
                    text: (territoriesDetailModel && territoriesDetailModel.rootIndex.valid)
                          ? ((territory && territory.locality) ? territory.territoryNumber : "")
                          : (title ? title : "")
                }
                Label {
                    Layout.fillWidth: true
                    Layout.alignment: Qt.AlignVCenter | Qt.AlignRight
                    horizontalAlignment: Text.AlignRight
                    text: (territoriesDetailModel && territoriesDetailModel.rootIndex.valid)
                          ? ((territory && territory.locality) ? territory.locality : "")
                          : (summary ? summary : "")
                }
            }

            onClicked: {
                if (territoriesDetailModel.rootIndex.valid)
                {
                    if (territoryId > 0) {
                        if (currentTerritory !== territory)
                            currentTerritory = territory;
                        else
                            zoomSelectedTerritory();
                    }
                }
                else
                    territoriesDetailModel.rootIndex = territoriesProxyModel.index(index, 0)
            }
        }
    }

    Drawer {
        id: drawer

        width: Math.min(appWindow.width, appWindow.height) / 3 * 2
        height: appWindow.height
        // modal: appWindow.portraitMode
        // interactive: appWindow.portraitMode ? (stackView.depth === 1) : false
        position: 0
        visible: false //!appWindow.portraitMode

        Page {
            id: territoriesPage
            anchors.fill: parent

            header: BaseToolbar {
                Material.background: myPalette.window
                Material.foreground: myPalette.windowText
                title: qsTr("Territories")
                subtitle: (territoriesDetailModel.rootIndex.valid
                           ? territoryTreeModel.data(territoriesProxyModel.mapToSource(territoriesDetailModel.rootIndex), TerritoryTreeModel.TitleRole)
                           : territoryTreeView.groupByModel.get(territoryTreeView.groupByIndex).key)
                titleColor: myPalette.windowText
                componentLeft: ToolButton {
                    icon.source: "qrc:/icons/chevron_left.svg"
                    onClicked: {
                        if (territoriesDetailModel.rootIndex.valid)
                            territoriesDetailModel.rootIndex = undefined;
                        else
                            drawer.close();
                    }
                }
            }

            contentItem: LookupControl {
                id: territoryTreeView

                showFilterControls: false
                showGroupControls: true
                useMenuGroupByControl: true
                groupByIndex: -1
                //showAddButton: isGdalAvailable
                showEditButton: false
                isEditing: false

                model: territoriesDetailModel

                // selectionModel: ItemSelectionModel {}

                groupByModel: ListModel {
                    ListElement {
                        key: qsTr("City", "Group territories by city")
                        value: "city"
                    }
                    ListElement {
                        key: qsTr("Publisher",
                                  "Group territories by publisher")
                        value: "publisher"
                    }
                    ListElement {
                        key: qsTr("Type",
                                  "Group territories by type of the territory")
                        value: "type_id"
                    }
                    ListElement {
                        key: qsTr("Worked through",
                                  "Group territories by time frame they have been worked through")
                        value: "workedthrough"
                    }
                }

                sortMenu: Menu {
                    MenuItem {
                        text: qsTr("Number", "Sort territories by their number")
                              + (territoriesProxyModel.sortRole.toString() === "territoryNumber"
                                 ? territoriesProxyModel.sortOrder === Qt.AscendingOrder ? " ↑" : " ↓"
                                 : "")
                        onTriggered: {
                            // save currently selected root index
                            var proxyIndex = territoriesDetailModel.rootIndex;
                            var modelIndex = territoriesProxyModel.mapToSource(proxyIndex);
                            var selectedRootTitle = territoryTreeModel.data(modelIndex, TerritoryTreeModel.TitleRole);
                            // change sorting
                            territoriesProxyModel.sortRole = "territoryNumber";
                            territoriesProxyModel.sortOrder = territoriesProxyModel.sortOrder === Qt.AscendingOrder
                                    ? Qt.DescendingOrder
                                    : Qt.AscendingOrder;
                            // select previously selected root index
                            selectRootIndex(selectedRootTitle);
                        }
                    }
                    MenuItem {
                        text: qsTr("Locality",
                                   "Sort territories by locality")
                              + (territoriesProxyModel.sortRole.toString() === "locality"
                                 ? territoriesProxyModel.sortOrder === Qt.AscendingOrder ? " ↑" : " ↓"
                                 : "")
                        onTriggered: {
                            // save currently selected root index
                            var proxyIndex = territoriesDetailModel.rootIndex;
                            var modelIndex = territoriesProxyModel.mapToSource(proxyIndex);
                            var selectedRootTitle = territoryTreeModel.data(modelIndex, TerritoryTreeModel.TitleRole);
                            // change sorting
                            territoriesProxyModel.sortRole = "locality";
                            territoriesProxyModel.sortOrder = territoriesProxyModel.sortOrder === Qt.AscendingOrder
                                    ? Qt.DescendingOrder
                                    : Qt.AscendingOrder;
                            // select previously selected root index
                            selectRootIndex(selectedRootTitle);
                        }
                    }
                }

                onGroupByIndexChanged: territoriesDetailModel.rootIndex = undefined;

                ScrollBar.vertical: ScrollBar {
                    focus: true
                }
            }
        }
    }
}
