import QtCore
import QtQuick
import QtQuick.Window
import QtQuick.Controls
import QtQuick.Layouts
import net.theocbase 1.0
import net.theocbase.mobile 1.0
import "."
import "sidePanelScripts.js" as SPScripts

Page {
    id: assignmentPage

    // PROPERTY DECLARATIONS
    property var assignment
    property PersonsModel publisherListModel: assignmentModel.getPublisherList()
    property bool isPublisherLoading: true
    property bool isDateLoading: true

    signal pageCanceled()

    // OBJECT PROPERTIES
    header: BaseToolbar {
        Material.background: myPalette.button
        Material.foreground: myPalette.buttonText
        Material.accent: myPalette.highlight
        title: qsTr("Territory assignment")
        subtitle: currentTerritory && currentTerritory.locality
               ? (qsTr("Territory") + " " + currentTerritory.territoryNumber +
                  (currentTerritory.locality
                   ? ", " + currentTerritory.locality
                   : ""))
               : ""
        titleColor: myPalette.buttonText
        componentLeft: ToolButton {
            icon.source: "qrc:/icons/chevron_left.svg"
            onClicked: {
                assignmentModel.saveAssignments();
                territoryPanelStackView.pop();
            }
        }
        componentRight: ToolButton {
            icon.source: "qrc:/icons/cancel.svg"
            onClicked: {
                territoryPanelStackView.pop();
                pageCanceled();
            }
        }
    }

    // CHILD OBJECTS
    TerritoryAssignmentValidator {
        id: publisherValidator
        model: assignmentModel
        assignmentId: assignment.id
        role: TerritoryAssignmentModel.PersonFullNameRole
        onErrorChanged: function(error) { publisherErrorLabel.text = error }
        onAssignmentIdChanged: {
            isPublisherLoading = true;
            isPublisherLoading = false;
        }
    }
    TerritoryAssignmentValidator {
        id: assignedDateValidator
        model: assignmentModel
        assignmentId: assignment.id
        role: TerritoryAssignmentModel.AssignedDateRole
        onErrorChanged: function(error) { assignedDateErrorLabel.text = error }
        onAssignmentIdChanged: {
            isDateLoading = true;
            isDateLoading = false;
        }
    }
    TerritoryAssignmentValidator {
        id: completedDateValidator
        model: assignmentModel
        assignmentId: assignment.id
        role: TerritoryAssignmentModel.CompletedDateRole
        onErrorChanged: function(error) { completedDateErrorLabel.text = error }
        onAssignmentIdChanged: {
            isDateLoading = true;
            isDateLoading = false;
        }
    }

    PersonsSFProxyModel {
        id: publisherListProxyModel
        source: publisherListModel
    }

    Component {
        id: publishersPage

        Page {
            StackView.onActivated: {
                publisherListProxyModel.filterText = "";
            }

            header: BaseToolbar {
                Material.background: myPalette.button
                Material.foreground: myPalette.buttonText
                Material.accent: myPalette.highlight
                title: qsTr("Publisher")
                titleColor: myPalette.buttonText
                componentLeft: ToolButton {
                    icon.source: "qrc:/icons/chevron_left.svg"
                    onClicked: {
                        if (publisherLookupControl.currentIndex > -1) {
                            var modelIndex = publisherListProxyModel.mapToSource(publisherListProxyModel.index(publisherLookupControl.currentIndex, 0));
                            var selectedPersonId = publisherListModel.get(modelIndex.row).personId;
                            var assignmentIndex = assignmentModel.getAssignmentIndex(assignment.id);
                            assignmentModel.setData(assignmentIndex, selectedPersonId, TerritoryAssignmentModel.PersonIdRole);
                            isPublisherLoading = true;
                            isPublisherLoading = false;
                        }
                        territoryPanelStackView.pop();
                    }
                }
            }
            LookupControl {
                id: publisherLookupControl

                anchors.fill: parent

                showSearchField: true
                showFilterControls: false
                showGroupControls: false
                groupByIndex: -1
                showEditButton: false
                isEditing: false

                model: publisherListProxyModel

                section.property: "personFullName"
                section.criteria: ViewSection.FirstCharacter
                section.delegate: Pane {
                    id: publisherSection

                    width: LookupControl.view.width
                    background: Rectangle {
                        anchors.fill: parent
                        color: TBStyle.primaryColor
                    }

                    Label {
                        width: parent.width
                        anchors.centerIn: parent
                        text: section
                        horizontalAlignment: Qt.AlignHCenter
                        verticalAlignment: Qt.AlignVCenter
                        font: TBStyle.bodyMediumFont
                        color: TBStyle.onPrimaryColor
                        elide: Text.ElideMiddle
                    }
                }

                delegate: RadioDelegate {
                    width: LookupControl.view.width
                    text: personFullName
                    checked: LookupControl.view.currentIndex === index
                    onToggled: LookupControl.view.currentIndex = index
                }

                // Set the initial currentIndex to the value stored in the backend.
                Component.onCompleted: {
                    if (publisherListModel) {
                        var modelIndex = publisherListModel.getPersonIndex(assignment.personId);
                        var proxyIndex = publisherListProxyModel.mapFromSource(modelIndex);
                        currentIndex = proxyIndex.row;
                        positionViewAtIndex(currentIndex, ListView.Center)
                    }
                }

                onSearchTextChanged: publisherListProxyModel.filterText = text
            }
        }
    }

    Pane {
        anchors.fill: parent
        Column {
            anchors.fill: parent
            spacing: 16

            FilledGroupBox {
                width: parent.width
                title: qsTr("Publisher")

                Column {
                    width: parent.width

                    RowLayout {
                        width: parent.width
                        InterimIcon {
                            Layout.alignment: Qt.AlignLeft | Qt.AlignVCenter
                            source: "qrc:/icons/person.svg"
                            color: Material.foreground
                            ToolTip.text: qsTr("Publisher")
                            ToolTip.visible: hovered
                        }
                        ItemDelegate {
                            id: publisherControl
                            Layout.fillWidth: true
                            indicator: InterimIcon {
                                x: publisherControl.width - width - publisherControl.rightPadding
                                y: publisherControl.topPadding + publisherControl.availableHeight / 2 - height / 2
                                Layout.alignment: Qt.AlignLeft | Qt.AlignVCenter
                                source: "qrc:/icons/chevron_right.svg"
                                color: Material.foreground
                            }
                            text: assignment.personFullName

                            onClicked: {
                                territoryPanelStackView.push(publishersPage)
                            }
                        }
                    }
                    Label {
                        id: publisherErrorLabel
                        font: TBStyle.bodySmallFont
                        color: TBStyle.alertColor
                        visible: text
                    }
                }
            }

            FilledGroupBox {
                width: parent.width
                title: qsTr("Date")
                Column {
                    width: parent.width

                    RowLayout {
                        width: parent.width
                        InterimIcon {
                            Layout.alignment: Qt.AlignLeft | Qt.AlignVCenter
                            source: "qrc:/icons/event_upcoming.svg"
                            color: Material.foreground
                            ToolTip.text: qsTr("Assigned", "Territory assignment")
                            ToolTip.visible: hovered
                        }
                        LookupComboBox {
                            id: assignedDateEdit
                            Layout.fillWidth: true
                            Material.containerStyle: Material.Filled
                            displayText: assignment?.assignedDate.toLocaleDateString(Qt.locale(), Locale.ShortFormat)
                            validator: isDateLoading ? null : assignedDateValidator
                            //isButtonVisible: false

                            popup: CalendarPopup {
                                transformOrigin: Popup.Top
                                width: assignedDateEdit.width
                                // x: (Qt.platform.os == "ios" ? mainpage.leftPadding : 0)
                                // y: toolbarbase.height

                                onAboutToShow: {
                                    if (assignment && !isNaN(assignment.assignedDate.getTime()))
                                        selectedDate = assignment.assignedDate;
                                }

                                onActivated: {
                                    var modelIndex = assignmentModel.getAssignmentIndex(assignment.id);
                                    assignmentModel.setData(modelIndex, selectedDate, TerritoryAssignmentModel.AssignedDateRole);
                                    isDateLoading = true;
                                    isDateLoading = false;
                                }
                            }

                            onCleared: {
                                var modelIndex = assignmentModel.getAssignmentIndex(assignment.id);
                                assignmentModel.setData(modelIndex, undefined, TerritoryAssignmentModel.AssignedDateRole);
                                isDateLoading = true;
                                isDateLoading = false;
                            }
                        }
                    }
                    Label {
                        id: assignedDateErrorLabel
                        font: TBStyle.bodySmallFont
                        color: TBStyle.alertColor
                        visible: text
                    }
                    MenuSeparator { width: parent.width; padding: 0 }
                    RowLayout {
                        width: parent.width
                        InterimIcon {
                            Layout.alignment: Qt.AlignLeft | Qt.AlignVCenter
                            source: "qrc:/icons/event_available.svg"
                            color: Material.foreground
                            ToolTip.text: qsTr("Completed", "Territory assignment")
                            ToolTip.visible: hovered
                        }
                        LookupComboBox {
                            id: completedDateEdit
                            Layout.fillWidth: true
                            Material.containerStyle: Material.Filled
                            displayText: assignment?.completedDate.toLocaleDateString(Qt.locale(), Locale.ShortFormat)
                            validator: isDateLoading ? null : completedDateValidator

                            popup: CalendarPopup {
                                transformOrigin: Popup.Top
                                width: completedDateEdit.width

                                onAboutToShow: {
                                    if (assignment && !isNaN(assignment.completedDate.getTime()))
                                        selectedDate = assignment.completedDate;
                                }

                                onActivated: {
                                    var modelIndex = assignmentModel.getAssignmentIndex(assignment.id);
                                    assignmentModel.setData(modelIndex, selectedDate, TerritoryAssignmentModel.CompletedDateRole);
                                    isDateLoading = true;
                                    isDateLoading = false;
                                }
                            }

                            onCleared: {
                                var modelIndex = assignmentModel.getAssignmentIndex(assignment.id);
                                assignmentModel.setData(modelIndex, undefined, TerritoryAssignmentModel.CompletedDateRole);
                                isDateLoading = true;
                                isDateLoading = false;
                            }
                        }
                    }
                    Label {
                        id: completedDateErrorLabel
                        font: TBStyle.bodySmallFont
                        color: TBStyle.alertColor
                        visible: text
                    }
                }
            }
        }
    }
}
