import QtCore
import QtQuick
import QtQuick.Controls
import QtQuick.Dialogs
import QtQuick.Layouts
import QtQuick.Window
import QtPositioning
import QtLocation
import net.theocbase.mobile 1.0
import "."
import "sidePanelScripts.js" as SPScripts

Page {
    // PROPERTY DECLARATIONS
    property Address searchAddress

    // SIGNAL DECLARATIONS

    // JAVASCRIPT FUNCTIONS
    function geocodingFinished(geocodeResults) {
        territories.onGeocodingFinished.disconnect(geocodingFinished);
        territories.onGeocodingError.disconnect(geocodingFailed);

        geocodeResultModel.removeRows(0, geocodeResultModel.rowCount());
        busyIndicator.running = false;
        if (geocodeResults.length > 0) {
            for (var i=0; i<geocodeResults.length; i++)
            {
                geocodeResultModel.addGeocodeResult(geocodeResults[i]);
            }
            territoryManager.displayGeocodeResults(geocodeResults);
        }
        else
        {
            geocodeResultMessage.title = qsTr("Search address", "Add or edit territory address");
            geocodeResultMessage.text = qsTr("No address found.", "Add or edit territory address");
            geocodeResultMessage.visible = true;
            return;
        }
    }

    function geocodingFailed(errorMessage) {
        territories.onGeocodingFinished.disconnect(geocodingFinished);
        territories.onGeocodingError.disconnect(geocodingFailed);
        busyIndicator.running = false;
        geocodeResultMessage.title = qsTr("Search address", "Add or edit territory address");
        geocodeResultMessage.text = errorMessage;
        geocodeResultMessage.visible = true;
        return;
    }

    function saveAddress(address) {
        if (typeof address === "undefined")
        {
            geocodeResultMessage.title = qsTr("Add address", "Add address to territory");
            geocodeResultMessage.text = qsTr("Please select an address in the list of search results.", "Add address to territory");
            geocodeResultMessage.visible = true;
            return;
        }
        if (addressesTabPage.editTerritoryId > 0)
        {
            if (addressesTabPage.isEditAddressMode)
            {
                if (addressesTabPage.editAddressId > 0)
                    if (addressModel.updateAddress(addressesTabPage.editAddressId,
                                                   addressModel.get(addressModel.getAddressIndex(addressesTabPage.editAddressId).row).territoryId,
                                                   address.country, address.state, address.county,
                                                   address.city, address.district, address.street,
                                                   address.streetNumber, address.postalCode,
                                                   "POINT(" + address.longitude + " " + address.latitude + ")"));
            }
            else
            {
                var newAddressId = addressModel.addAddress(addressesTabPage.editTerritoryId, address.country,
                                            address.state, address.county, address.city, address.district,
                                            address.street, address.streetNumber, address.postalCode,
                                            "POINT(" + address.longitude + " " + address.latitude + ")");

                // select new address
                if (newAddressId > 0)
                {
                    var addressIndex = addressModel.getAddressIndex(newAddressId);
                    var addressProxyIndex = addressProxyModel.mapFromSource(addressIndex);
                    if (addressProxyIndex)
                    {
                        addressLookupControl.currentIndex = addressProxyIndex.row;
                    }
                }
            }
        }
    }

    function resetDefaultSettings() {
        settings.territoryGeocodeResult_groupByIndex = 1;
        geocodeResultLookupControl.groupByIndex = settings.territoryGeocodeResult_groupByIndex;
        settings.territoryGeocodeResult_sortOrder = Qt.AscendingOrder;
        geocodeResultLookupControl.sortOrder = settings.territoryGeocodeResult_sortOrder;
        settings.sync();
    }

    // OBJECT PROPERTIES
    StackView.onActivating: {
        busyIndicator.running = true;
        geocodeResultModel.removeRows(0, geocodeResultModel.rowCount());
        geocodeResultLookupControl.groupByIndex = settings.territoryGeocodeResult_groupByIndex;
        geocodeResultLookupControl.sortOrder = settings.territoryGeocodeResult_sortOrder;

        territories.onGeocodingFinished.connect(geocodingFinished);
        territories.onGeocodingError.connect(geocodingFailed);
        territories.geocodeAddress(searchAddress.address, territoryManager.currentTerritory.boundingGeoRectangle);
    }

    StackView.onDeactivating: {
        settings.territoryGeocodeResult_groupByIndex = geocodeResultLookupControl.groupByIndex;
        settings.territoryGeocodeResult_sortOrder = geocodeResultLookupControl.sortOrder;
        settings.sync();
        territoryManager.clearAddressSearchMarkers();
    }

    // CHILD OBJECTS
    GeocodeResultModel {
        id: geocodeResultModel
    }
    GeocodeResultSFProxyModel {
        id: geocodeResultSFProxyModel
        source: geocodeResultModel
        //filterText: geocodeResultLookupControl.searchText
    }

    Menu {
        id: geocodeResultLookupControlMoreMenu

        MenuItem { text: qsTr("Reset default settings", "Lookup control settings"); onTriggered: resetDefaultSettings() }
    }

    MessageDialog {
        id: geocodeResultMessage
        buttons: MessageDialog.Ok
        //icon: StandardIcon.Warning
        modality: Qt.ApplicationModal
    }

    header: RowLayout {
        ToolButton {
            Layout.alignment: Qt.AlignRight
            icon.source: "qrc:/icons/chevron_left.svg"
            onClicked: {
                addressesEditStack.pop();
            }
        }
        Label {
            Layout.fillWidth: true
            text: qsTr("Add address")
            font: TBStyle.titleSmallFont
        }
        ToolButton {
            Layout.alignment: Qt.AlignRight
            icon.source: "qrc:/icons/close.svg"
            onClicked: {
                addressesEditStack.popToIndex(0);
            }
        }
    }

    LookupControl {
        id: geocodeResultLookupControl

        anchors.fill: parent

        clip: true
        focus: true

        showFilterControls: false
        showGroupControls: true
        groupByIndex: -1
        showEditButton: false
        isEditing: false

        model: geocodeResultSFProxyModel
        groupByModel: ListModel {
            ListElement {
                key: qsTr("Street name", "Group addresses by street name")
                value: "street"
            }
            ListElement {
                key: qsTr("City", "Group addresses by city")
                value: "city"
            }
        }

        moreMenu: geocodeResultLookupControlMoreMenu

        delegate: ItemDelegate {
            id: itemDelegate

            clip: true

            width: LookupControl.view.width
            height: delegateContent.implicitHeight + topPadding + bottomPadding

            contentItem: Item {
                Column {
                    id: delegateContent

                    width: parent.width

                    HoverHandler {
                        id: hoverHandler
                        acceptedDevices: PointerDevice.Mouse
                        // cursorShape: Qt.PointingHandCursor
                    }
                    RowLayout {
                        width: parent.width
                        spacing: 6
                        Label {
                            id: streetNameTextField
                            Layout.fillWidth: true
                            Layout.alignment: Qt.AlignTop
                            Layout.margins: 0
                            horizontalAlignment: Text.AlignLeft
                            verticalAlignment: Text.AlignVCenter

                            text: model.text
                            wrapMode: TextInput.WordWrap
                            font: TBStyle.bodyLargeFont
                        }
                        Item {
                            id: editButtonRow
                            implicitWidth: Math.max(buttonSave.implicitBackgroundWidth + buttonSave.leftInset + buttonSave.rightInset,
                                                    buttonSave.implicitContentWidth + buttonSave.leftPadding + buttonSave.rightPadding) * 3
                            Layout.alignment: Qt.AlignVCenter | Qt.AlignRight
                            Layout.margins: 0
                            Layout.leftMargin: 5
                            Layout.fillHeight: true

                            RowLayout {
                                anchors.right: parent.right
                                anchors.verticalCenter: parent.verticalCenter
                                ToolButton {
                                    id: buttonZoom
                                    Layout.fillWidth: true
                                    Layout.alignment: Qt.AlignVCenter
                                    padding: 0
                                    spacing: 0
                                    topInset: 0
                                    bottomInset: 0
                                    z: 2
                                    icon.source: "qrc:/icons/fullscreen.svg"
                                    visible: hoverHandler.hovered
                                    onClicked: territoryManager.zoomCoordinate(latitude, longitude)
                                }
                                ToolButton {
                                    id: buttonSave
                                    Layout.fillWidth: true
                                    Layout.alignment: Qt.AlignVCenter
                                    padding: 0
                                    spacing: 0
                                    topInset: 0
                                    bottomInset: 0
                                    icon.source: "qrc:/icons/add_location.svg"
                                    z: 2
                                    visible: hoverHandler.hovered
                                    onClicked: {
                                        var currProxyIndex = geocodeResultSFProxyModel.index(index, 0);
                                        var currIndex = geocodeResultSFProxyModel.mapToSource(currProxyIndex);
                                        var address = geocodeResultModel.get(currIndex);
                                        saveAddress(address);
                                    }
                                }
                            }
                        }
                    }
                }
            }
            background: Rectangle {
                width: itemDelegate.LookupControl.view.width
                color: index % 2 == 0 ? myPalette.base : myPalette.alternateBase
            }
        }

        section.property: groupByModel.get(groupByIndex) ? groupByModel.get(groupByIndex).value : ""
        section.criteria: ViewSection.FullString
        section.delegate: Pane {
            id: geocodeResultSection

            width: LookupControl.view.width
            height: Material.menuItemHeight
            padding: 0
            background: Rectangle {
                anchors.fill: parent
                color: TBStyle.primaryColor
            }

            Label {
                width: parent.width
                anchors.centerIn: parent
                text: section
                horizontalAlignment: Qt.AlignHCenter
                verticalAlignment: Qt.AlignVCenter
                font: TBStyle.bodyMediumFont
                color: TBStyle.onPrimaryColor
                elide: Text.ElideMiddle
            }
        }

        footer: Pane {
            width: LookupControl.view.width;
            z: 3

            RowLayout {
                anchors.right: parent.right
                Label {
                    verticalAlignment: Text.AlignVCenter
                    //: Number of rows in a list
                    //~ Context the '%1'-part will be replaced by the actual number
                    text: qsTr("Count=%1", "Lookup control").arg(geocodeResultLookupControl.count)
                }

                ToolButton {
                    id: geocodeResultMoreMenuButton
                    Layout.alignment: Qt.AlignRight | Qt.AlignVCenter
                    visible: geocodeResultLookupControl.moreMenu
                    ToolTip.visible: hovered
                    ToolTip.delay: Qt.styleHints.mousePressAndHoldInterval
                    ToolTip.text: qsTr("Settings", "Lookup control")
                    icon.source: "qrc:/icons/more.svg"

                    onClicked: geocodeResultLookupControl.moreMenu.popup()
                }
            }
        }
        onSearchTextChanged: geocodeResultSFProxyModel.filterText = text
        onGroupByIndexChanged: {
            if (groupByIndex >= 0) {
                settings.territoryGeocodeResult_groupByIndex = groupByIndex;
                geocodeResultSFProxyModel.groupByRole = groupByModel.get(groupByIndex).value;
            }
        }
        onSortOrderChanged: {
            settings.territoryGeocodeResult_sortOrder = sortOrder;
            geocodeResultSFProxyModel.sort(0, sortOrder);
        }
        ScrollBar.vertical: ScrollBar {}
    }

    BusyIndicator {
        id: busyIndicator
        anchors.centerIn: parent
        running: true
    }

    // STATES

    // TRANSITIONS
}
