/**
 * This file is part of TheocBase.
 *
 * Copyright (C) 2011-2019, TheocBase Development Team, see AUTHORS.
 *
 * TheocBase is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * TheocBase is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with TheocBase.  If not, see <http://www.gnu.org/licenses/>.
 */
import QtCore
import QtQuick
import QtQuick.Controls
import QtQuick.Controls.Material
import QtQuick.Layouts
import net.theocbase.mobile 1.0
import net.theocbase 1.0

Page {
    id: pageStorageService

    property bool isStorageServiceChanging: false

    signal storageServiceChanged(string newStorageService, string baseUrl)

    function returnBackPage(){
        // hide keyboard
        if (Qt.inputMethod.visible) Qt.inputMethod.hide()
        // return to previous page
        stackView.pop()
    }

    function saveChanges(accept){
        isStorageServiceChanging = false;
        var newStorageService = checkDropbox.checked ? "Dropbox" : "Nextcloud";
        if (dataStorageSettings.online_storage_service.toLowerCase() !== newStorageService.toLowerCase()) {
            isStorageServiceChanging = true;
            dataStorageSettings.online_storage_service = newStorageService;
            dataStorageSettings.sync();
        }
        if (checkNextcloud.checked) {
            if (nextcloudSettings.server.toLowerCase() !== textNextcloudServer.text.toLowerCase()) {
                isStorageServiceChanging = true;
                nextcloudSettings.server = textNextcloudServer.text;
                nextcloudSettings.sync();
            }
        }
        returnBackPage();
    }

    header: BaseToolbar {
        title: qsTr("Storage service")
        componentLeft: ToolButton {
            icon.source: "qrc:/icons/cancel.svg"
            onClicked: { returnBackPage() }
        }
        componentRight: ToolButton {
            icon.source: "qrc:/icons/check.svg"
            icon.color: enabled ? TBStyle.onPrimaryColor : TBStyle.mediumColor
            onClicked: { saveChanges() }
        }
    }

    // CHILD OBJECTS
    Settings {
        id: dataStorageSettings
        category: "data_storage"
        property string online_storage_service: "Dropbox"
    }
    Settings {
        id: nextcloudSettings
        category: "nextcloud"
        property string server: ""
    }

    StackView.onActivating: {
        checkDropbox.checked = dataStorageSettings.online_storage_service === "Dropbox";
        checkNextcloud.checked = !checkDropbox.checked;
        textNextcloudServer.text = nextcloudSettings.server;
    }

    StackView.onDeactivating: {
        if (isStorageServiceChanging){
            storageServiceChanged(dataStorageSettings.online_storage_service, nextcloudSettings.server);
        }
    }

    ScrollView {
        id: scrollView
        anchors.fill: parent
        contentWidth: availableWidth

        Pane {
            width: parent.width
            Column {
                id: layout
                width: parent.width
                spacing: 16

                FilledGroupBox {
                    width: parent.width

                    Column {
                        width: parent.width
                        ButtonGroup { buttons: serviceProviderGrp.children }
                        Column {
                            id: serviceProviderGrp
                            width: parent.width
                            RadioDelegate {
                                id: checkDropbox
                                width: parent.width
                                text: qsTr("Dropbox")
                                checked: false
                            }
                            RadioDelegate {
                                id: checkNextcloud
                                width: parent.width
                                text: qsTr("Nextcloud")
                                checked: false
                            }
                        }
                        MenuSeparator { width: parent.width; padding: 0; visible: checkNextcloud.checked }
                        RowText {
                            id: textNextcloudServer
                            width: parent.width
                            title: qsTr("Server")
                            editable: true
                            visible: checkNextcloud.checked
                        }
                    }
                }
            } // layout
        }
    } // scrollView
}
