/**
 * This file is part of TheocBase.
 *
 * Copyright (C) 2011-2019, TheocBase Development Team, see AUTHORS.
 *
 * TheocBase is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * TheocBase is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with TheocBase.  If not, see <http://www.gnu.org/licenses/>.
 */
import QtQuick
import QtQuick.Controls
import QtQuick.Controls.Material
import QtQuick.Layouts
import QtQuick.Effects
import net.theocbase.mobile 1.0
import net.theocbase 1.0

Page {
    id: settingsPage

    property bool isDarkMode: systemPalette.windowText.hsvValue > systemPalette.window.hsvValue

    header: BaseToolbar {
        title: qsTr("Settings", "application settings") + qmlTranslator.changed
        componentRight: ToolButton {
            icon.source: ccloud.syncState === Cloud.Synchronized
                         ? "qrc:/icons/cloud_done.svg"
                         : ccloud.syncState === Cloud.Download
                           ? "qrc:/icons/cloud_download.svg"
                           : "qrc:/icons/cloud_upload.svg"
            onClicked: { syncpage.runSync() }
        }
    }

    StackView.onActivated: {
        console.log("Settings page activated");
        appWindow.activepage = 4;
    }

    SystemPalette { id: systemPalette }

    ScrollView {
        id: scrollView
        anchors.fill: parent
        contentWidth: availableWidth

        Pane {
            width: parent.width
            Column {
                id: layout
                width: parent.width
                spacing: 16

                FilledGroupBox {
                    width: parent.width

                    Column {
                        width: parent.width
                        ItemDelegate {
                            id: cloudProvider
                            width: parent.width
                            implicitHeight: Material.delegateHeight

                            indicator: InterimIcon {
                                x: cloudProvider.width - width - cloudProvider.rightPadding
                                y: cloudProvider.topPadding + cloudProvider.availableHeight / 2 - height / 2
                                Layout.alignment: Qt.AlignLeft | Qt.AlignVCenter
                                source: "qrc:/icons/chevron_right.svg"
                                color: ccloud.isLoggedIn ? TBStyle.mediumColor : Material.foreground
                                enabled: !ccloud.isLoggedIn
                            }
                            contentItem: Control {
                                height: Material.delegateHeight

                                Item {
                                    anchors.fill: parent
                                    anchors.leftMargin: !cloudProvider.mirrored ? 0 : cloudProvider.indicator.width + cloudProvider.spacing
                                    anchors.rightMargin: cloudProvider.mirrored ? 0 : cloudProvider.indicator.width + cloudProvider.spacing

                                    RowLayout {
                                        anchors.fill: parent
                                        Item {
                                            Layout.fillWidth: true
                                            Layout.fillHeight: true
                                            Column {
                                                anchors.verticalCenter: parent.verticalCenter
                                                Label {
                                                    text: qsTr("Cloud storage provider")
                                                    font: cloudProvider.font
                                                }
                                                Label {
                                                    text: qsTr("Last synchronized: %1").arg(ccloud.lastSyncTime.toLocaleString(Qt.locale(), Locale.ShortFormat)) +
                                                          " " + (ccloud.syncState === Cloud.Upload ? "↑" : ccloud.syncState === Cloud.Download ? "↓" : (ccloud.syncState === Cloud.Both ? "⇅" : "✓")) +
                                                          qmlTranslator.changed
                                                    font: TBStyle.bodySmallFont
                                                    color: TBStyle.mediumColor
                                                }
                                            }
                                        }

                                        Item {
                                            Layout.fillHeight: true
                                            Layout.fillWidth: true

                                            Image {
                                                id: dropBoxImage
                                                anchors.verticalCenter: parent.verticalCenter
                                                anchors.right: parent.right
                                                height: parent.height
                                                fillMode: Image.PreserveAspectFit
                                                source: "qrc:/icons/" + ccloud.storageService.providerName + ".svg"
                                            }
                                            MultiEffect {
                                                anchors.fill: dropBoxImage
                                                visible: isDarkMode
                                                source: dropBoxImage
                                                brightness: 1
                                                colorizationColor: "white"
                                                colorization: 1
                                            }
                                        }
                                    }
                                }
                            }
                            onClicked: {
                                if (!ccloud.isLoggedIn)
                                    stackView.clearAndPush(storageServicePage);
                            }
                        }
                        MenuSeparator { width: parent.width; padding: 0 }
                        RowText {
                            width: parent.width
                            title: ccloud.isLoggedIn ? qsTr("Logout") : qsTr("Login") + qmlTranslator.changed
                            arrow: true
                            onClicked: {
                                ccloud.logout();
                                ccloud.login();
                            }
                        }
                        MenuSeparator { width: parent.width; padding: 0 }
                        RowText {
                            id: textEmail
                            width: parent.width
                            title: qsTr("Email")
                            text: ccloud.storageService.account.email
                        }
                        MenuSeparator { width: parent.width; padding: 0 }
                        RowText {
                            id: textName
                            width: parent.width
                            title: qsTr("Name")
                            text: ccloud.storageService.account.name
                        }
                        MenuSeparator { width: parent.width; padding: 0 }
                        RowText {
                            id: textSyncFile
                            width: parent.width
                            title: "Syncfile"
                            text: ccloud.storageService.account.syncFile
                            arrow: true
                            onClicked: {
                                var page = stackView.clearAndPush(Qt.resolvedUrl("DropboxList.qml"));
                                page.onSyncFileChanged.connect(function(path){
                                    ccloud.clearDatabase();
                                    syncpage.runSync();
                                })
                            }
                        }
                    }
                }

                FilledGroupBox {
                    width: parent.width
                    title: qsTr("Info") + qmlTranslator.changed

                    Column {
                        width: parent.width
                        RowText {
                            id: textVersion
                            width: parent.width
                            title: qsTr("Version") + qmlTranslator.changed
                            text: app_info.version
                        }
                        MenuSeparator { width: parent.width; padding: 0 }
                        RowText {
                            id: textHelp
                            width: parent.width
                            title: qsTr("TheocBase Homepage") + qmlTranslator.changed
                            arrow: true
                            onClicked: {
                                Qt.openUrlExternally("http://theocbase.net")
                            }
                        }
                        MenuSeparator { width: parent.width; padding: 0 }
                        RowText {
                            width: parent.width
                            title: qsTr("Feedback") + qmlTranslator.changed
                            arrow: true
                            onClicked: {
                                Qt.openUrlExternally("http://www.theocbase.net/support-forum.html")
                            }
                        }
                        MenuSeparator { width: parent.width; padding: 0 }
                        RowText {
                            width: parent.width
                            title: "Debug"
                            arrow: true
                            visible: true
                            onClicked: {
                                stackView.clearAndPush(Qt.resolvedUrl("DebugPage.qml"));
                            }
                        }
                    }
                }

                FilledGroupBox {
                    width: parent.width
                    title: qsTr("Schedule") + qmlTranslator.changed

                    Column {
                        width: parent.width
                        ButtonGroup { buttons: groupTimeDuration.children }
                        Column {
                            id: groupTimeDuration
                            width: parent.width
                            RadioDelegate {
                                id: switchShowTime
                                width: parent.width
                                text: qsTr("Show Time") + qmlTranslator.changed
                                checked: settings_ui.showTime
                                onCheckedChanged: {
                                    settings_ui.showTime = checked
                                    reloadMainpage()
                                }
                            }
                            RadioDelegate {
                                width: parent.width
                                text: qsTr("Show Duration") + qmlTranslator.changed
                                checked: !switchShowTime.checked
                            }
                        }
                        MenuSeparator { width: parent.width; padding: 0 }
                        SwitchDelegate {
                            width: parent.width
                            text: qsTr("Show Song Titles") + qmlTranslator.changed
                            checked: settings_ui.showSongTitles
                            onCheckedChanged: {
                                settings_ui.showSongTitles = checked
                                reloadMainpage()
                            }
                        }
                    }
                }

                FilledGroupBox {
                    width: parent.width
                    title: qsTr("User Interface")

                    Column {
                        width: parent.width
                        RowText {
                            width: parent.width
                            title: qsTr("Language")
                            arrow: true
                            text: qmlTranslator.getLanguageName()
                            onClicked: {
                                if (Qt.inputMethod.visible) Qt.inputMethod.hide()
                                var languagePage = stackView.clearAndPush(Qt.resolvedUrl("SelectionListPage.qml"),
                                                                          {model: qmlTranslator.getLanguages(), pageTitle: title,
                                                                              selectedRow: qmlTranslator.currentIndex})
                                languagePage.onActiveRowChanged.connect(function(index){
                                    qmlTranslator.currentIndex = index
                                    text = qmlTranslator.getLanguageName()
                                    message.show("TheocBase", "You need to restart application for the changes to take effect")
                                })
                            }
                            MsgBox { id: message }
                        }
                        MenuSeparator { width: parent.width; padding: 0 }
                        SwitchDelegate {
                            width: parent.width
                            text: qsTr("Full screen") + qmlTranslator.changed
                            checked: settings_ui.fullScreen
                            onCheckedChanged: {
                                settings_ui.fullScreen = checked
                            }
                        }
                        MenuSeparator { width: parent.width; padding: 0 }
                        ButtonGroup { buttons: groupNameOrder.children }
                        Column {
                            id: groupNameOrder
                            width: parent.width
                            RadioDelegate {
                                id: switchLexicalNameOrder
                                width: parent.width
                                text: qsTr("Lexical name order (surname, forename)") + qmlTranslator.changed
                                checked: TBStyle.fullNameFormat === "%2, %1"
                                onCheckedChanged: {
                                    TBStyle.fullNameFormat = "%2, %1";
                                }
                            }
                            RadioDelegate {
                                id: switchWesternNameOrder
                                width: parent.width
                                text: qsTr("Western name order (forename surname)") + qmlTranslator.changed
                                checked: TBStyle.fullNameFormat === "%1 %2"
                                onCheckedChanged: {
                                    TBStyle.fullNameFormat = "%1 %2";
                                }
                            }
                            RadioDelegate {
                                id: switchEasternNameOrder
                                width: parent.width
                                text: qsTr("Eastern name order (surname forename)") + qmlTranslator.changed
                                checked: TBStyle.fullNameFormat === "%2 %1"
                                onCheckedChanged: {
                                    TBStyle.fullNameFormat = "%2 %1";
                                }
                            }
                        }
                    }
                }

                FilledGroupBox {
                    width: parent.width
                    title: qsTr("Printing")

                    RowText {
                        width: parent.width
                        title: qsTr("Custom Templates")
                        arrow: true
                        onClicked: {
                            var page = stackView.clearAndPush(Qt.resolvedUrl("PrintTemplates.qml"));
                        }
                    }
                }
            }
        }
    }

    Component {
        id: storageServicePage
        StorageServicePage{
            onStorageServiceChanged: function(newStorageService, baseUrl) {
                ccloud.setStorageService(newStorageService, baseUrl);
            }
        }
    }
}
