/**
 * This file is part of TheocBase.
 *
 * Copyright (C) 2011-2019, TheocBase Development Team, see AUTHORS.
 *
 * TheocBase is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * TheocBase is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with TheocBase.  If not, see <http://www.gnu.org/licenses/>.
 */
import QtQuick
import QtQuick.Controls
import QtQuick.Controls.Material
import QtQuick.Layouts
import net.theocbase.mobile 1.0

Page {
    id: publisherspage

    property string searchtext: ""

    function loadList(positionname){
        searchtext = ""
        publishersListView.model = _publishers.getAllPublishersList()
        if (typeof positionname !== "undefined") {
            positionTimer.currentName = positionname
            positionTimer.start()
        }
    }

    SystemPalette { id: palette }

    Publishers { id: _publishers }

    Timer {
        id: positionTimer
        interval: 500
        property string currentName
        onTriggered: {
            for(var i=0;i<publishersListView.model.length;i++){
                if (publishersListView.model[i].name === currentName){
                    publishersListView.positionViewAtIndex(i,ListView.Center)
                    break;
                }
            }
        }
    }

    StackView.onActivated: {
        console.log("Publishers page activated");
        appWindow.activepage = 3;
    }

    Component.onCompleted: {
        publishersListView.model = _publishers.getAllPublishersList()
    }

    header: ToolBar {
        Material.background: TBStyle.primaryColor
        Material.foreground: TBStyle.onPrimaryColor
        RowLayout {
            anchors.fill: parent
            ToolButton {
                icon.source: "qrc:/icons/search.svg"
                onClicked: textSearch.forceActiveFocus()
            }
            TextField {
                id: textSearch
                Layout.fillWidth: true
                onTextChanged: searchtext = text
                visible: activeFocus || text.length > 0
                opacity: visible ? 1 : 0
                Material.containerStyle: Material.Filled
                Material.background: palette.window
                Material.foreground: palette.windowText
                Behavior on opacity { NumberAnimation{ duration: 500 } }
            }
            Label {
                text: qsTr("Publishers")
                visible: !textSearch.visible
                Layout.fillWidth: true
                horizontalAlignment: Text.AlignHCenter
                verticalAlignment: Text.AlignVCenter
                font: TBStyle.titleMediumFont
            }
            ToolButton {
                icon.source: "qrc:/icons/person_add.svg"
                enabled: canEditPublishers
                onClicked: {
                    stackView.clearAndPush(detailPage, {p: _publishers.addNew()})
                }
            }
        }
    }

    ListView {
        id: publishersListView

        anchors.fill: parent

        delegate: listComponent

        clip: false
    }

    // list view component
    Component {
        id: listComponent
        SwipeDelegate {
            id: swipeDelegate

            text: model.modelData.name

            icon.source: "qrc:/icons/account_circle.svg"
            icon.color: Material.color(!model.modelData.active
                        ? Material.Grey
                        : (model.modelData.gender === Publisher.Male
                           ? Material.Blue
                           : Material.Pink))

            visible: searchtext === "" || model.modelData.name.toLowerCase().indexOf(searchtext.toLowerCase()) > -1
            height: visible ? implicitHeight : 0
            width: ListView.view.width

            ListView.onRemove: SequentialAnimation {
                PropertyAction {
                    target: swipeDelegate
                    property: "ListView.delayRemove"
                    value: true
                }
                NumberAnimation {
                    target: swipeDelegate
                    property: "height"
                    to: 0
                    easing.type: Easing.InOutQuad
                }
                PropertyAction {
                    target: swipeDelegate
                    property: "ListView.delayRemove"
                    value: false
                }
            }

            onClicked: {
                publishersListView.currentIndex = model.index
                stackView.clearAndPush(detailPage, {p: _publishers.getPublisher(model.modelData.id)})
            }

            swipe.enabled: canEditPublishers
            swipe.right: Label {
                id: deleteLabel
                text: qsTr("Delete")
                color: "white"
                verticalAlignment: Label.AlignVCenter
                padding: 12
                height: parent.height
                anchors.right: parent.right

                SwipeDelegate.onClicked: {
                    console.log("delete publisher");
                    _publishers.remove(model.modelData.id);
                    loadList();
                }

                background: Rectangle {
                    color: deleteLabel.SwipeDelegate.pressed ? Qt.darker("tomato", 1.1) : "tomato"
                }
            }
        }
    }

    Component {
        id: detailPage
        PublisherDetail{
            onPageClosed: loadList(p.lastName + ", " + p.firstName)
        }
    }
}
