/**
 * This file is part of TheocBase.
 *
 * Copyright (C) 2011-2019, TheocBase Development Team, see AUTHORS.
 *
 * TheocBase is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * TheocBase is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with TheocBase.  If not, see <http://www.gnu.org/licenses/>.
 */
import QtQuick
import QtQuick.Controls
import QtQuick.Controls.Material
import QtQuick.Layouts
import net.theocbase.mobile 1.0
import net.theocbase 1.0

Page {
    id: pagePublisherDetail

    property bool canEditPublishers: accessControl.user && accessControl.user.hasPermission(PermissionRule.CanEditPublishers)
    property bool canViewStudentData: accessControl.user && accessControl.user.hasPermission(PermissionRule.CanViewStudentData)
    property bool canEditStudentData: accessControl.user && accessControl.user.hasPermission(PermissionRule.CanEditStudentData)
    property bool canViewPrivileges: accessControl.user && accessControl.user.hasPermission(PermissionRule.CanViewPrivileges)
    property bool canEditPrivileges: accessControl.user && accessControl.user.hasPermission(PermissionRule.CanEditPrivileges)
    property bool isNew: p.id < 0
    property Publisher p
    property Family f

    signal pageClosed()

    function returnBackPage(){
        // hide keyboard
        if (Qt.inputMethod.visible) Qt.inputMethod.hide()
        // return to previous page
        pageClosed()
        stackView.pop()
    }
    function saveChanges(accept){
        if (p.id < 0 && !accept){
            if (textFirstname.text === "" && textLastname.text === ""){
                _msg.show("Save","Type first and last name!",1);
            }else{
                _msg.showYesNo("Save","Add a new publisher?",1);
            }
            return;
        }

        // Save use for
        var v = p.usefor;
        v = prepareUsefor(switchAuxClasses.checked,v,Publisher.SchoolAux)
        v = prepareUsefor(switchMainClass.checked,v,Publisher.SchoolMain)
        v = prepareUsefor(switchChairman.checked,v,Publisher.LMM_Chairman)
        v = prepareUsefor(switchLMMTRTalk.checked,v,Publisher.LMM_TR_Talk)
        v = prepareUsefor(switchLMMTRSpiritualGems.checked,v,Publisher.LMM_TR_SpiritualGems)
        v = prepareUsefor(switchLMMTRBibleReading.checked,v,Publisher.LMM_TR_BibleReading)
        v = prepareUsefor(switchLMMFMDiscussion.checked,v,Publisher.LMM_FM_Discussion)
        v = prepareUsefor(switchLMMFMStartingConversation.checked,v,Publisher.LMM_FM_StartingConversation)
        v = prepareUsefor(switchLMMFMFollowingUp.checked,v,Publisher.LMM_FM_FollowingUp)
        v = prepareUsefor(switchLMMFMMakingDisciples.checked,v,Publisher.LMM_FM_MakingDisciples)
        v = prepareUsefor(switchLMMFMExplainingBeliefs.checked,v,Publisher.LMM_FM_ExplainingBeliefs)
        v = prepareUsefor(switchLMMFMTalk.checked,v,Publisher.LMM_FM_Talk)
        v = prepareUsefor(switchLMMCLTalk.checked,v,Publisher.LMM_CL_Talk)
        v = prepareUsefor(switchCBS.checked,v,Publisher.CBSConductor)
        v = prepareUsefor(switchCBSReader.checked,v,Publisher.CBSReader)
        v = prepareUsefor(switchPrayer.checked, v, Publisher.Prayer)

        v = prepareUsefor(switchAssistant.checked,v,Publisher.Assistant)
        v = prepareUsefor(!switchActive.checked,v,Publisher.IsBreak)

        v = prepareUsefor(checkHospitality.checked,v,Publisher.Hospitality)

        v = prepareUsefor(checkWeekendChairman.checked,v,Publisher.Chairman)
        v = prepareUsefor(checkWtConductor.checked,v,Publisher.WtCondoctor)
        v = prepareUsefor(checkWtReader.checked,v,Publisher.WtReader)

        var gender = checkBrother.checked ? Publisher.Male : Publisher.Female
        if (textFirstname.text !== p.firstName ||
                textLastname.text !== p.lastName ||
                checkServant.checked !== p.servant ||
                gender !== p.gender ||
                v !== p.usefor ||
                textMobile.text !== p.mobile ||
                textPhone.text !== p.phone ||
                textEmail.text !== p.email){
            console.log("need to save changes");
            p.firstName = textFirstname.text
            p.lastName = textLastname.text
            p.servant = checkServant.checked
            p.gender = gender
            p.usefor = v;
            p.mobile = textMobile.text
            p.phone = textPhone.text
            p.email = textEmail.text
            _publishers.save(p)
        }

        // Save publisher's family
        // Get previous saved family
        // var prev_family = _family.getPersonFamily(p.id)
        if (checkFamilyHead.checked){
            if(!f || f.headId !== p.id){
                // add a new family
                _family.getOrAddFamily(p.id)
            }
        }else{
            if (f && f.headId === p.id){
                // remove family
                f.removeFamily()
            }
            // Family member
            if (familiesListPage.selectedRow > 0){
                // add member
                if (!f || (f.headId !== familiesListPage.model[familiesListPage.selectedRow].headId)){
                    familiesListPage.model[familiesListPage.selectedRow].addMember(p.id)
                }
            }else{
                // remove member
                if (f) f.removeMember(p.id)
            }
        }

        returnBackPage();
    }

    function prepareUsefor(checked,allvalue, value){
        //console.log("checked:" + checked.toString() + " value:" + value.toString())
        if (checked){
            //console.log("not exist " + (!(allvalue & value)).toString())
            if (!(allvalue & value)) allvalue += value;
        }else{
            //console.log("should be removed " + (allvalue & value).toString())
            if (allvalue & value) allvalue -= value;
        }
        return allvalue;
    }

    MsgBox {id: _msg}
    Connections {
        target: _msg
        function onButtonClicked(ok, id) {
            if (ok){
                switch (id){
                case 1:
                    // add a new publisher
                    console.log("add a new publisher")
                    saveChanges(true)
                    break;
                case 2:
                    // call phone number
                    Qt.openUrlExternally("tel:" + textPhone.text); break;
                default: break;
                }
            }
        }
    }

    // Family
    Family { id:_family }
    SelectionListPage {
        id: familiesListPage
        pageTitle: "Families"
        visible: false
        showName: true
        model: _family.getFamiliesVariantList()
        onActiveRowChanged: function(index) {
            console.log("families row changed - index:" + index)
            //f = familiesListPage.model[index]
            textLinkedFamily.text = familiesListPage.model[index].name
        }
        Component.onCompleted: {
            f = _family.getPersonFamily(p.id)
            if (f){
                for(var i=0;i<familiesListPage.model.length;i++){
                    if (familiesListPage.model[i].headId === f.headId){
                        familiesListPage.selectedRow = i
                        break;
                    }
                }
            }else{
                familiesListPage.selectedRow = 0
            }
        }
    }

    header: BaseToolbar {
        title: p.id < 0 ? qsTr("New publisher") : p.fullName;
        componentLeft: ToolButton {
            icon.source: "qrc:/icons/cancel.svg"
            onClicked: { returnBackPage() }
        }
        componentRight: ToolButton {
            icon.source: "qrc:/icons/check.svg"
            icon.color: enabled ? TBStyle.onPrimaryColor : TBStyle.mediumColor
            onClicked: { saveChanges() }
            enabled: textFirstname.text.length > 0 && textLastname.text.length > 0 &&
                     canEditPublishers
        }
    }

    ScrollView {
        id: scrollView
        anchors.fill: parent
        contentWidth: availableWidth

        Pane {
            width: parent.width
            Column {
                id: layout
                width: parent.width
                spacing: 16
                enabled: canEditPublishers

                FilledGroupBox {
                    width: parent.width
                    title: qsTr("Info")

                    Column {
                        width: parent.width
                        RowText {
                            id: textFirstname
                            width: parent.width
                            title: qsTr("First Name")
                            text: p.firstName
                            editable: true
                        }
                        MenuSeparator { width: parent.width; padding: 0 }
                        RowText {
                            id: textLastname
                            width: parent.width
                            title: qsTr("Last Name")
                            text: p.lastName
                            editable: true
                        }
                        MenuSeparator { width: parent.width; padding: 0 }
                        ButtonGroup { buttons: genderGrp.children }
                        Column {
                            id: genderGrp
                            width: parent.width
                            RadioDelegate {
                                id: checkBrother
                                width: parent.width
                                text: qsTr("Brother")
                                checked: false
                                Component.onCompleted: {
                                    checkBrother.checked = p.gender === Publisher.Male
                                    checkSister.checked = !checkBrother.checked
                                }
                            }
                            RadioDelegate {
                                id: checkSister
                                width: parent.width
                                text: qsTr("Sister")
                                checked: false
                            }
                        }
                        MenuSeparator { width: parent.width; padding: 0 }
                        SwitchDelegate {
                            id: switchActive
                            width: parent.width
                            text: qsTr("Active")
                            checked: (!(p.usefor & Publisher.IsBreak))
                        }
                        MenuSeparator { width: parent.width; padding: 0 }
                        SwitchDelegate {
                            id: checkHospitality
                            width: parent.width
                            text:  qsTr("Host for Public Speakers")
                            checked: p.usefor & Publisher.Hospitality
                        }
                        MenuSeparator { width: parent.width; padding: 0 }
                        SwitchDelegate {
                            id: checkServant
                            width: parent.width
                            text: qsTr("Servant")
                            checked: p.servant
                            visible: checkBrother.checked
                        }
                        MenuSeparator { width: parent.width; padding: 0; visible: checkBrother.checked }
                        SwitchDelegate {
                            id: switchPrayer
                            width: parent.width
                            text: qsTr("Prayer")
                            checked: p.usefor & Publisher.Prayer
                            enabled: p.gender === Publisher.Male
                        }
                    }
                }

                FilledGroupBox {
                    width: parent.width
                    title: qsTr("Family")

                    Column {
                        width: parent.width
                        SwitchDelegate {
                            id: checkFamilyHead
                            width: parent.width
                            text: qsTr("Family Head")
                            checked: (f && p.id == f.headId)
                        }
                        MenuSeparator { width: parent.width; padding: 0; visible: textLinkedFamily.visible }
                        RowText {
                            id: textLinkedFamily
                            width: parent.width
                            title: qsTr("Family member linked to")
                            text: f ? f.name : ""
                            visible: !checkFamilyHead.checked
                            arrow: true
                            onClicked: {
                                if (Qt.inputMethod.visible) Qt.inputMethod.hide()
                                stackView.push(familiesListPage)
                            }
                        }
                    }
                }

                FilledGroupBox {
                    width: parent.width
                    title: qsTr("Contact Information")

                    Column {
                        width: parent.width
                        RowText {
                            id: textMobile
                            width: parent.width
                            title: qsTr("Mobile")
                            text: p.mobile
                            editable: true
                            onPressAndHold: {
                                if (textMobile.text.length > 6 && app_info.screensize < 7) {
                                    var globalCoordinares = mapToItem(scrollView.contentItem, mouse.x, mouse.y)
                                    popupMobile.show(globalCoordinares.x,globalCoordinares.y)
                                }
                            }
                        }
                        MenuSeparator { width: parent.width; padding: 0; visible: textLinkedFamily.visible }
                        RowText {
                            id: textPhone
                            width: parent.width
                            title: qsTr("Phone")
                            text: p.phone
                            editable: true
                            onPressAndHold: {
                                if (textPhone.text.length > 6 && app_info.screensize < 7) {
                                    var globalCoordinares = mapToItem(scrollView.contentItem, mouse.x, mouse.y)
                                    popupPhone.show(globalCoordinares.x,globalCoordinares.y)
                                }
                            }
                        }
                        MenuSeparator { width: parent.width; padding: 0; visible: textLinkedFamily.visible }
                        RowText {
                            id: textEmail
                            width: parent.width
                            title: qsTr("Email")
                            text: p.email
                            editable: true
                        }
                    }
                }

                Column {
                    width: parent.width
                    spacing: 0

                    FilledGroupBox {
                        width: parent.width
                        title: qsTr("Midweek Meeting")
                        visible: canViewPrivileges || canViewStudentData
                        Material.background: TBStyle.lmmSection1Color

                        Column {
                            width: parent.width
                            SwitchDelegate {
                                id: switchChairman
                                width: parent.width
                                text: qsTr("Chairman")
                                checked: p.usefor & Publisher.LMM_Chairman
                                visible: canViewPrivileges
                                enabled: checkServant.checked &&
                                         canEditPrivileges
                            }
                            MenuSeparator { width: parent.width; padding: 0; visible: canViewPrivileges }
                            SwitchDelegate {
                                id: switchLMMTRTalk
                                width: parent.width
                                text: qsTr("Talk")
                                checked: p.usefor & Publisher.LMM_TR_Talk
                                visible: canViewPrivileges
                                enabled: checkServant.checked &&
                                         canEditPrivileges
                            }
                            MenuSeparator { width: parent.width; padding: 0; visible: canViewPrivileges }
                            SwitchDelegate {
                                id: switchLMMTRSpiritualGems
                                width: parent.width
                                text: qsTr("Spiritual Gems")
                                checked: p.usefor & Publisher.LMM_TR_SpiritualGems
                                visible: canViewPrivileges
                                enabled: checkServant.checked &&
                                         canEditPrivileges
                            }
                            MenuSeparator { width: parent.width; padding: 0; visible: canViewPrivileges && canViewStudentData }
                            SwitchDelegate {
                                id: switchLMMTRBibleReading
                                width: parent.width
                                text: qsTr("Bible Reading")
                                checked: p.usefor & Publisher.LMM_TR_BibleReading
                                visible: canViewStudentData
                                enabled: canEditStudentData
                            }
                        }
                    }

                    FilledGroupBox {
                        width: parent.width
                        visible: canViewStudentData || canViewPrivileges
                        Material.background: TBStyle.lmmSection2Color

                        Column {
                            width: parent.width

                            SwitchDelegate {
                                id: switchLMMFMStartingConversation
                                width: parent.width
                                text: qsTr("Starting a Conversation")
                                checked: p.usefor & Publisher.LMM_FM_StartingConversation
                                visible: canViewStudentData
                                enabled: canEditStudentData
                            }
                            MenuSeparator { width: parent.width; padding: 0; visible: canViewStudentData }
                            SwitchDelegate {
                                id: switchLMMFMFollowingUp
                                width: parent.width
                                text: qsTr("Following Up")
                                checked: p.usefor & Publisher.LMM_FM_FollowingUp
                                visible: canViewStudentData
                                enabled: canEditStudentData
                            }
                            MenuSeparator { width: parent.width; padding: 0; visible: canViewStudentData }
                            SwitchDelegate {
                                id: switchLMMFMMakingDisciples
                                width: parent.width
                                text: qsTr("Making Disciples")
                                checked: p.usefor & Publisher.LMM_FM_MakingDisciples
                                visible: canViewStudentData
                                enabled: canEditStudentData
                            }
                            MenuSeparator { width: parent.width; padding: 0; visible: canViewStudentData }
                            SwitchDelegate {
                                id: switchLMMFMExplainingBeliefs
                                width: parent.width
                                text: qsTr("Explaining Your Beliefs")
                                checked: p.usefor & Publisher.LMM_FM_ExplainingBeliefs
                                visible: canViewStudentData
                                enabled: canEditStudentData
                            }
                            MenuSeparator { width: parent.width; padding: 0; visible: canViewStudentData }
                            SwitchDelegate {
                                id: switchLMMFMTalk
                                width: parent.width
                                text: qsTr("Talk")
                                checked: p.usefor & Publisher.LMM_FM_Talk
                                visible: canViewStudentData
                                enabled: canEditStudentData
                            }
                            MenuSeparator { width: parent.width; padding: 0; visible: canViewStudentData }
                            SwitchDelegate {
                                id: switchAssistant
                                width: parent.width
                                text: qsTr("Assistant")
                                checked: p.usefor & Publisher.Assistant
                                visible: canViewStudentData
                                enabled: canEditStudentData
                            }
                            MenuSeparator { width: parent.width; padding: 0; visible: canViewStudentData && canViewPrivileges }
                            SwitchDelegate {
                                id: switchLMMFMDiscussion
                                width: parent.width
                                text: qsTr("Discussion")
                                checked: p.usefor & Publisher.LMM_FM_Discussion
                                visible: canViewPrivileges
                                enabled: checkServant.checked &&
                                         canEditPrivileges
                            }
                        }
                    }

                    FilledGroupBox {
                        width: parent.width
                        visible: canViewPrivileges
                        Material.background: TBStyle.lmmSection3Color

                        Column {
                            width: parent.width

                            SwitchDelegate {
                                id: switchLMMCLTalk
                                width: parent.width
                                text: qsTr("Talk or discussion")
                                checked: p.usefor & Publisher.LMM_CL_Talk
                                visible: canViewPrivileges
                                enabled: checkServant.checked &&
                                         canEditPrivileges
                            }
                            MenuSeparator { width: parent.width; padding: 0; visible: canViewPrivileges }
                            SwitchDelegate {
                                id: switchCBS
                                width: parent.width
                                text: qsTr("Congregation Bible Study")
                                checked: p.usefor & Publisher.CBSConductor
                                visible: canViewPrivileges
                                enabled: checkServant.checked &&
                                         canEditPrivileges
                            }
                            MenuSeparator { width: parent.width; padding: 0; visible: canViewPrivileges }
                            SwitchDelegate {
                                id: switchCBSReader
                                width: parent.width
                                text: qsTr("Cong. Bible Study reader")
                                checked: p.usefor & Publisher.CBSReader
                                visible: canViewPrivileges
                                enabled: p.gender === Publisher.Male &&
                                         canEditPrivileges
                            }
                        }
                    }

                    FilledGroupBox {
                        width: parent.width
                        visible: canViewStudentData

                        ButtonGroup { buttons: classesGrp.children }
                        Column {
                            id: classesGrp
                            width: parent.width
                            RadioDelegate {
                                id: switchAllClasses
                                width: parent.width
                                text: qsTr("All Classes")
                                checked: !(p.usefor & Publisher.SchoolMain || p.usefor & Publisher.SchoolAux)
                                visible: canViewStudentData
                                enabled: canEditStudentData
                            }
                            RadioDelegate {
                                id: switchMainClass
                                width: parent.width
                                text: qsTr("Only Main Class")
                                checked: p.usefor & Publisher.SchoolMain
                                visible: canViewStudentData
                                enabled: canEditStudentData
                            }
                            RadioDelegate {
                                id: switchAuxClasses
                                width: parent.width
                                text: qsTr("Only Auxiliary Classes")
                                checked: p.usefor & Publisher.SchoolAux
                                visible: canViewStudentData
                                enabled: canEditStudentData
                            }
                        }
                    }
                }

                FilledGroupBox {
                    width: parent.width
                    title: qsTr("Weekend Meeting")
                    visible: canViewPrivileges

                    Column {
                        width: parent.width

                        SwitchDelegate {
                            id: checkWeekendChairman
                            width: parent.width
                            text: qsTr("Chairman")
                            checked: p.usefor & Publisher.Chairman
                            visible: canViewPrivileges
                            enabled: checkServant.checked &&
                                     canEditPrivileges
                        }
                        MenuSeparator { width: parent.width; padding: 0; visible: canViewPrivileges }
                        SwitchDelegate {
                            id: checkWtConductor
                            width: parent.width
                            text: qsTr("Watchtower Study Conductor")
                            checked: p.usefor & Publisher.WtCondoctor
                            visible: canViewPrivileges
                            enabled: checkServant.checked &&
                                     canEditPrivileges
                        }
                        MenuSeparator { width: parent.width; padding: 0; visible: canViewPrivileges }
                        SwitchDelegate {
                            id: checkWtReader
                            width: parent.width
                            text: qsTr("Watchtower Study Reader")
                            checked: p.usefor & Publisher.WtReader
                            visible: canViewPrivileges
                            enabled: checkBrother.checked &&
                                     canEditPrivileges
                        }
                    }
                }
            } // layout
        }
    } // scrollView

    PopupMenu {
        id: popupPhone
        menuContent: RowLayout {
            anchors.fill: parent
            ToolButton {
                Layout.fillWidth: true
                Layout.fillHeight: true
                Layout.preferredWidth: height
                icon.source: "qrc:/icons/call.svg"
                icon.color: Material.foreground
                onClicked: {
                    // call phone number
                    Qt.openUrlExternally("tel:" + textPhone.text)
                    popupPhone.visible = false
                }
            }
            ToolButton {
                Layout.fillWidth: true
                Layout.fillHeight: true
                Layout.preferredWidth: height
                icon.source: "qrc:/icons/sms.svg"
                icon.color: Material.foreground
                onClicked: {
                    // send sms
                    Qt.openUrlExternally("sms:" + textPhone.text)
                    popupPhone.visible = false
                }
            }
        }
    }

    PopupMenu {
        id: popupMobile
        menuContent: RowLayout {
            anchors.fill: parent
            ToolButton {
                Layout.fillWidth: true
                Layout.fillHeight: true
                Layout.preferredWidth: height
                icon.source: "qrc:/icons/smartphone.svg"
                icon.color: Material.foreground
                onClicked: {
                    // call mobile number
                    Qt.openUrlExternally("tel:" + textMobile.text)
                    popupMobile.visible = false
                }
            }
            ToolButton {
                Layout.fillWidth: true
                Layout.fillHeight: true
                Layout.preferredWidth: height
                icon.source: "qrc:/icons/sms.svg"
                icon.color: Material.foreground
                onClicked: {
                    // send sms
                    Qt.openUrlExternally("sms:" + textMobile.text)
                    popupMobile.visible = false
                }
            }
        }
    }
}

