/**
 * This file is part of TheocBase.
 *
 * Copyright (C) 2011-2019, TheocBase Development Team, see AUTHORS.
 *
 * TheocBase is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * TheocBase is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with TheocBase.  If not, see <http://www.gnu.org/licenses/>.
 */
import QtQuick
import QtQuick.Controls
import QtQuick.Controls.Material
import QtQuick.Layouts
import net.theocbase.mobile 1.0

Pane {
    property bool showTime: false
    property bool editpossible: true
    property CPTMeeting cptMeeting

    function formatTime(time) {
        var newTime = Qt.formatTime(time, Qt.DefaultLocaleShortDate);
        var regex = /^(2[0-3]|[0-1]?[\d]):[0-5][\d]/;
        var found = newTime.match(regex);
        if (found !== null)
            newTime = found[0];
        return newTime;
    }

    function loadSchedule(date){
        showTime = settings_ui.showTime
        pmController.date = date

        cptMeeting = pmController.meeting

        var t = cptMeeting.startTime
        songStartRow.timeText = showTime ? formatTime(t) : "\u266B"
        t.setMinutes(t.getMinutes() + 5)
        ptThemeRow.timeText = showTime ? formatTime(t) : "30"
        t.setMinutes(t.getMinutes() + 30)
        wtSongRow.timeText = showTime ? formatTime(t) : "\u266B"
        t.setMinutes(t.getMinutes() + 5)
        wtStudyRow.timeText = showTime ? formatTime(t) : cptMeeting.wtTime
        t.setMinutes(t.getMinutes() + cptMeeting.wtTime)
        if (cptMeeting.finalTalk !== ""){
            finalTalkRow.timeText = showTime ? formatTime(t) : "30"
            t.setMinutes(t.getMinutes() + 30)
        }
        songEndRow.timeText = showTime ? formatTime(t) : "\u266B"
    }

    padding: 0

    PublicMeetingController { id: pmController }

    Column {
        id: columnLayout1
        width: parent.width
        spacing: 0

        RowLayout {
            width: parent.width
            Layout.preferredHeight: Material.delegateHeight
            Label {
                Layout.fillWidth: true
                Layout.leftMargin: 5
                Layout.alignment: Qt.AlignVCenter
                verticalAlignment: Text.AlignVCenter
                text: "○○● " + Qt.locale().dayName(wkDate.getDay(), Locale.LongFormat) +
                      " | " + qsTr("Weekend Meeting").toUpperCase()
                elide: Text.ElideRight
                font: TBStyle.titleMediumFont
                color: TBStyle.primaryTextColor
            }
            ToolButton {
                icon.source: "qrc:/icons/wol.svg"
                icon.color: TBStyle.primaryTextColor
                visible: canEditWeekendMeetingSchedule && editpossible
                onClicked: {
                    stackView.clearAndPush(Qt.resolvedUrl("WebBrowser.qml"), { "date" : cptMeeting.date, "meetingType" : MeetingType.WeekendMeeting });
                }
            }
            ToolButton {
                icon.source: cptMeeting && cptMeeting.notes ? "qrc:/icons/notes.svg" : "qrc:/icons/add_notes.svg"
                icon.color: TBStyle.primaryTextColor
                visible: canViewMeetingNotes && editpossible
                onClicked: {
                    stackView.clearAndPush(Qt.resolvedUrl("MeetingNotes.qml"),
                                           { "pageTitle": qsTr("Midweek Meeting"), "meeting": cptMeeting, "editable": canEditMeetingNotes })
                }
            }
        }
        ScheduleRowItem {
            id: songStartRow
            width: parent.width
            height: Material.delegateHeight
            themeText: !cptMeeting || cptMeeting.songTalk < 1 ? qsTr("Song and Prayer") : qsTr("Song %1 and Prayer").arg(cptMeeting.songTalk) +
                                                                     (settings_ui.showSongTitles ? "\n" + cptMeeting.songTalkTitle : "")
            themeColor: TBStyle.mediumColor
            timeColor: showTime ? Material.background : TBStyle.publicTalkTextColor
            timeBackgroundColor: showTime ? TBStyle.publicTalkTextColor : "transparent"
            nameText1.text: cptMeeting ? cptMeeting.chairman ? cptMeeting.chairman.fullName : "" : ""
            nameText1.color: TBStyle.mediumColor
            imageSource: clickable ? "qrc:/icons/chevron_right.svg" : ""
            clickable: canEditWeekendMeetingSchedule && editpossible
            onClicked: {
                var detailsPage = stackView.clearAndPush(Qt.resolvedUrl("WEMeetingChairmanPanel.qml"),
                                                         {"specialEventRule" : specialEventRule, meeting: cptMeeting})
            }
        }

        // Public talk
        Pane {
            width: parent.width
            Material.background: TBStyle.publicTalkColor
            padding: 0
            Column {
                width: parent.width
                spacing: 0
                RowLayout {
                    Rectangle {
                        color: TBStyle.publicTalkTextColor
                        height: Material.delegateHeight
                        width: Material.delegateHeight
                        InterimIcon {
                            id: ptIcon
                            anchors.fill: parent
                            anchors.margins: 4
                            source: "qrc:/icons/pt_48dp.svg"
                            color: Material.background
                        }
                    }
                    // header for meeting part
                    Label {
                        id: text1
                        width: parent.width
                        height: Material.delegateHeight
                        text: qsTr("Public Talk")
                        verticalAlignment: Text.AlignVCenter
                        horizontalAlignment: Text.AlignHCenter
                        font: TBStyle.titleLargeFont
                        color: TBStyle.publicTalkTextColor
                    }
                }
                ScheduleRowItem {
                    id: ptThemeRow
                    width: parent.width
                    height: Material.delegateHeight
                    themeText: cptMeeting ? cptMeeting.themeNumber > 0 ? cptMeeting.theme + " (" + cptMeeting.themeNumber + ")" : "" : ""
                    nameText1.text: cptMeeting ? cptMeeting.speaker ? cptMeeting.speaker.fullName : "" : ""
                    nameText2.text: cptMeeting ? cptMeeting.speaker ?  "(" + cptMeeting.speaker.congregationName + ")" : "" : ""
                    timeColor: Material.background
                    timeBackgroundColor: TBStyle.publicTalkTextColor
                    clickable: canEditWeekendMeetingSchedule && editpossible && (specialEventRule ? !specialEventRule.isWithoutPublicTalk : false)
                    imageSource: clickable ? "qrc:/icons/chevron_right.svg" : ""
                    onClicked: {
                        var detailsPage = stackView.clearAndPush(Qt.resolvedUrl("PublicTalkPanel.qml"),
                                                                 {meeting: cptMeeting})
                    }
                }
            }
        }

        // section separator
        Rectangle {
            width: parent.width
            height: Material.delegateHeight / 3
            color: "transparent"
        }

        // Watchtower study
        Pane {
            id: rectWtStudy
            width: parent.width
            Material.background: TBStyle.watchtowerStudyColor
            padding: 0

            Column {
                id: columnLayout3
                width: parent.width
                spacing: 0

                RowLayout {
                    Rectangle {
                        color: TBStyle.watchtowerStudyTextColor
                        height: Material.delegateHeight
                        width: Material.delegateHeight
                        InterimIcon {
                            id: wtIcon
                            anchors.fill: parent
                            anchors.margins: 4
                            source: "qrc:///icons/wt_48dp.svg"
                            color: Material.background
                        }
                    }
                    Label {
                        id: text5
                        width: parent.width
                        height: Material.delegateHeight
                        text: qsTr("Watchtower Study")
                        verticalAlignment: Text.AlignVCenter
                        horizontalAlignment: Text.AlignHCenter
                        font: TBStyle.titleLargeFont
                        color: TBStyle.watchtowerStudyTextColor
                    }
                }

                ScheduleRowItem {
                    id: wtSongRow
                    width: parent.width
                    height: Material.delegateHeight
                    themeText: qsTr("Song %1").arg(!cptMeeting || cptMeeting.songWtStart < 1
                                                        ? ""
                                                        : cptMeeting.songWtStart + (settings_ui.showSongTitles
                                                                                    ? "\n" + cptMeeting.songWtStartTitle
                                                                                    : ""))
                    themeColor: TBStyle.mediumColor
                    timeColor: showTime ? Material.background : TBStyle.watchtowerStudyTextColor
                    timeBackgroundColor: showTime ? TBStyle.watchtowerStudyTextColor : "transparent"
                    clickable: canEditWeekendMeetingSchedule && editpossible
                    imageSource: clickable ? "qrc:/icons/chevron_right.svg" : ""
                    onClicked: {                        
                        var detailsPage = stackView.clearAndPush(Qt.resolvedUrl("WatchtowerSongPanel.qml"),
                                                                 {meeting: cptMeeting, startSong: true})
                    }
                }

                ScheduleRowItem {
                    id: wtStudyRow
                    width: parent.width
                    height: Material.delegateHeight
                    themeText: cptMeeting ? cptMeeting.wtTheme : ""
                    nameText1.text: cptMeeting ? cptMeeting.wtConductor ? cptMeeting.wtConductor.fullName +  " (" + qsTr("Conductor") + ")" : "" : ""
                    nameText2.text: cptMeeting ? cptMeeting.wtReader ? cptMeeting.wtReader.fullName + " (" + qsTr("Reader") + ")" : "" : ""
                    timeColor: Material.background
                    timeBackgroundColor: TBStyle.watchtowerStudyTextColor
                    clickable: canEditWeekendMeetingSchedule && editpossible
                    imageSource: clickable ? "qrc:/icons/chevron_right.svg" : ""
                    onClicked: {
                        stackView.pop(null)
                        var detailsPage = stackView.clearAndPush(Qt.resolvedUrl("WatchtowerStudyPanel.qml"),
                                                                 {meeting: cptMeeting})
                    }
                    Label {
                        anchors.horizontalCenter: parent.horizontalCenter
                        anchors.verticalCenter: parent.verticalCenter
                        horizontalAlignment: Text.AlignHCenter
                        verticalAlignment: Text.AlignVCenter
                        text: "<html><a href='#'>" + qsTr("Import WT...") + "</a></html>"
                        visible: canEditWeekendMeetingSchedule && editpossible && parent.themeText == "" && cptMeeting &&
                                 cptMeeting.date > new Date("2019-03-03")
                        onLinkActivated: {
                            console.log("WT IMPORT")
                            shareUtils.openFile("epub")
                        }
                    }
                }
            }
        }

        ScheduleRowItem {
            id: finalTalkRow
            width: parent.width
            height: Material.delegateHeight
            timeText: "30"
            timeColor: Material.background
            timeBackgroundColor: Material.foreground
            visible: cptMeeting && cptMeeting.finalTalk !== ""
            themeText: cptMeeting ? cptMeeting.finalTalk : ""
            nameText1.text: cptMeeting ? cptMeeting.finalTalkSpeakerName : ""
        }

        ScheduleRowItem {
            id: songEndRow
            width: parent.width
            height: Material.delegateHeight
            timeText: ""
            timeColor: showTime ? Material.background : TBStyle.watchtowerStudyTextColor
            timeBackgroundColor: showTime ? TBStyle.watchtowerStudyTextColor : "transparent"
            themeText: !cptMeeting || cptMeeting.songWtEnd < 1 ? qsTr("Song and Prayer") : qsTr("Song %1 and Prayer").arg(cptMeeting.songWtEnd) +
                                                                      (settings_ui.showSongTitles ? "\n" + cptMeeting.songWtEndTitle : "")
            themeColor: TBStyle.mediumColor
            nameText1.text: cptMeeting ? cptMeeting.finalPrayer ? cptMeeting.finalPrayer.fullName : "" : ""
            clickable: canEditWeekendMeetingSchedule && editpossible
            imageSource: clickable ? "qrc:/icons/chevron_right.svg" : ""
            onClicked: {               
                var detailsPage = stackView.clearAndPush(Qt.resolvedUrl("WatchtowerSongPanel.qml"),
                                                         {meeting: cptMeeting, startSong: false})
            }
        }
    }
}
