/**
 * This file is part of TheocBase.
 *
 * Copyright (C) 2011-2020, TheocBase Development Team, see AUTHORS.
 *
 * TheocBase is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * TheocBase is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with TheocBase.  If not, see <http://www.gnu.org/licenses/>.
 */
import QtQuick
import QtQuick.Controls
import QtQuick.Controls.Material
import QtQuick.Controls.Material.impl
import QtQuick.Layouts
import net.theocbase.mobile 1.0
import net.theocbase 1.0

Popup {
    id: printPage
    modal: true
    parent: Overlay.overlay
    x: (parent.width - width) / 2
    y: (parent.height - height) / 2
    implicitWidth:  parent.width / 5 * 4
    implicitHeight: parent.height / 5 * 4
    closePolicy: Popup.NoAutoClose
    padding: 0

    property date weekDate
    property var modelTemplates: printController.templateList
    property bool showBusyIndicator: false

    property bool canPrintMidweekSchedule: accessControl.user.hasPermission(PermissionRule.CanPrintMidweekMeetingSchedule)
    property bool canPrintMidweekSlips: accessControl.user.hasPermission(PermissionRule.CanPrintMidweekMeetingAssignmentSlips)
    property bool canPrintMidweekWorksheet: accessControl.user.hasPermission(PermissionRule.CanPrintMidweekMeetingWorksheets)
    property bool canPrintWeekendSchedule: accessControl.user.hasPermission(PermissionRule.CanPrintWeekendMeetingSchedule)
    property bool canPrintWeekendWorksheet: accessControl.user.hasPermission(PermissionRule.CanPrintWeekendMeetingWorksheets)
    property bool canPrintOutgoingSchedule: accessControl.user.hasPermission(PermissionRule.CanPrintSpeakersSchedule)
    property bool canPrintOutgoingSlips: accessControl.user.hasPermission(PermissionRule.CanPrintSpeakersAssignments)
    property bool canPrintHospitality: accessControl.user.hasPermission(PermissionRule.CanPrintHospitality)
    property bool canPrintPublicTalkList: accessControl.user.hasPermission(PermissionRule.CanPrintPublicTalkList)
    property bool canPrintCombination: accessControl.user.hasPermission(PermissionRule.CanPrintMidweekMeetingSchedule) &&
                                       accessControl.user.hasPermission(PermissionRule.CanPrintWeekendMeetingSchedule) &&
                                       accessControl.user.hasPermission(PermissionRule.CanPrintSpeakersSchedule) &&
                                       accessControl.user.hasPermission(PermissionRule.CanPrintHospitality)
    property bool canPrintTerritoryRecord: accessControl.user.hasPermission(PermissionRule.CanPrintTerritoryRecord)
    property bool canPrintTerritoryMapCard: accessControl.user.hasPermission(PermissionRule.CanPrintTerritoryMapCard)
    property bool canPrintTerritoryMapAndAddressSheets: accessControl.user.hasPermission(PermissionRule.CanPrintTerritoryMapAndAddressSheets)

    Material.theme: TBStyle.colorScheme === Qt.Dark ? Material.Dark : Material.Light

    onOpened: {
        printController.initDate(weekDate)
        console.log(printController.fromDate)
        console.log(printController.toDate)
    }
    focus: true

    PrintController {
        id: printController
        fromDate: weekDate
        onTypeChanged: {
            modelTemplates = printController.templateList
            if (modelTemplates.length > 0) {
                printController.templateName = modelTemplates[0]
            }
        }
        onPreviewReady: {
            showBusyIndicator = false
        }
    }

    ToolBar {
        id: toolbar
        width: parent.width
        Material.foreground: TBStyle.onPrimaryColor
        RowLayout {
            anchors.fill: parent
            ToolButton {
                icon.source: stack.depth > 1 ? "qrc:/icons/back.svg" : "qrc:/icons/cancel.svg"
                icon.color: TBStyle.onPrimaryColor
                onClicked: {
                    if (stack.depth > 1)
                        stack.pop()
                    else
                        printPage.close()
                }
            }
            Label {
                Layout.fillWidth: true
                horizontalAlignment: Qt.AlignHCenter
                text: qsTr("Print Options")
                color: TBStyle.onPrimaryColor
            }
            Item { Layout.fillWidth: true }
        }
    }

    StackView {
        id: stack
        anchors.fill: parent
        anchors.topMargin: toolbar.height

        focus: true
        Keys.onReleased: (event)=> {
            if (event.key === Qt.Key_Back) {
                if (stack.depth > 1)
                    stack.pop()
                else
                    printPage.close()
            }
        }

        initialItem: optionsPage
    }

    Component {
        id: optionsPage
        Page {
            Material.background: palette.window
            Material.foreground: palette.windowText

            SystemPalette { id: palette }

            ColumnLayout {
                anchors.fill: parent

                ScrollView {
                    id: scrollView
                    Layout.fillWidth: true
                    Layout.fillHeight: true
                    contentWidth: availableWidth
                    //clip: true

                    Column {
                        width: scrollView.availableWidth

                        Pane {
                            width: parent.width
                            Column {
                                width: parent.width
                                ListView {
                                    id: listTypeGroup

                                    width: parent.width
                                    height: 120

                                    orientation: Qt.Horizontal
                                    currentIndex: 0

                                    onCurrentIndexChanged: {
                                        switch(currentIndex) {
                                        case 0:
                                            radioMWSchedule.checked = true
                                            break
                                        case 1:
                                            radioWESchedule.checked = true
                                            break
                                        case 2:
                                            printController.templateType = TemplateData.CombinationTemplate
                                            break
                                        case 3:
                                            radioTerritoryRecord.checked = true
                                        }
                                    }

                                    model: ListModel {
                                        ListElement {
                                            name: qsTr("Midweek Meeting")
                                        }
                                        ListElement {
                                            name: qsTr("Weekend Meeting")
                                        }
                                        ListElement {
                                            name: qsTr("Combination", "Print template")
                                        }
                                        ListElement {
                                            name: qsTr("Territories")
                                        }
                                    }

                                    delegate: AbstractButton {
                                        id: card
                                        width: 100
                                        height: 100
                                        leftInset: 5
                                        rightInset: 5

                                        display: AbstractButton.TextOnly

                                        background: Pane {
                                            padding: 0

                                            SystemPalette { id: panePalette }

                                            Material.background: listTypeGroup.currentIndex == index
                                                                 ? panePalette.highlight
                                                                 : panePalette.button
                                            Material.elevation: 6

                                            Ripple {
                                                width: parent.width
                                                height: parent.height

                                                clip: visible
                                                pressed: card.pressed
                                                anchor: card
                                                active: enabled && (card.down || card.visualFocus || card.hovered)
                                                color: card.Material.rippleColor
                                            }
                                        }

                                        Label {
                                            id: labelName
                                            anchors.centerIn: parent
                                            width: parent.width
                                            text: name
                                            horizontalAlignment: Text.AlignHCenter
                                            wrapMode: Text.WordWrap
                                        }

                                        onClicked: {
                                            listTypeGroup.currentIndex = index
                                        }
                                    }
                                }
                                Flow {
                                    width: parent.width
                                    RadioButton {
                                        id: radioMWSchedule
                                        text: qsTr("Schedule")
                                        visible: listTypeGroup.currentIndex == 0
                                        checked: true
                                        onCheckedChanged: if (checked)
                                                              printController.templateType = TemplateData.MidweekScheduleTemplate
                                    }
                                    RadioButton {
                                        id: radioMWSlip
                                        text: qsTr("Assignment Slips")
                                        visible: listTypeGroup.currentIndex == 0
                                        onCheckedChanged: if (checked)
                                                              printController.templateType = TemplateData.MidweekSlipTemplate
                                    }
                                    RadioButton {
                                        id: radioMWWorkSheet
                                        text: qsTr("Worksheets")
                                        visible: listTypeGroup.currentIndex == 0
                                        onCheckedChanged:  if (checked)
                                                               printController.templateType = TemplateData.MidweekWorksheetTemplate
                                    }
                                    RadioButton {
                                        id: radioWESchedule
                                        text: qsTr("Schedule")
                                        visible: listTypeGroup.currentIndex == 1
                                        onCheckedChanged: if (checked)
                                                              printController.templateType = TemplateData.PublicMeetingTemplate
                                    }
                                    RadioButton {
                                        id: radioWEWorkSheet
                                        text: qsTr("Worksheets")
                                        visible: listTypeGroup.currentIndex == 1
                                        onCheckedChanged: if (checked)
                                                              printController.templateType = TemplateData.PublicMeetingWorksheet
                                    }
                                    RadioButton {
                                        id: radioOutgoingSchedule
                                        text: qsTr("Outgoing Speakers Schedule")
                                        visible: listTypeGroup.currentIndex == 1
                                        onCheckedChanged: if (checked)
                                                              printController.templateType = TemplateData.OutgoingScheduleTemplate
                                    }
                                    RadioButton {
                                        id: radioOutgoingSlips
                                        text: qsTr("Outgoing Speakers Assignments")
                                        visible: listTypeGroup.currentIndex == 1
                                        onCheckedChanged: if (checked)
                                                              printController.templateType = TemplateData.OutgoingSlipTemplate
                                    }
                                    RadioButton {
                                        id: radioHospitality
                                        text: qsTr("Call List and Hospitality Schedule")
                                        visible: listTypeGroup.currentIndex == 1
                                        onCheckedChanged: if (checked)
                                                              printController.templateType = TemplateData.HospitalityTemplate
                                    }
                                    RadioButton {
                                        id: radioTalksOfSpeakers
                                        text: qsTr("Talks of Speakers")
                                        visible: listTypeGroup.currentIndex == 1
                                        onCheckedChanged: if (checked)
                                                              printController.templateType = TemplateData.TalksOfSpeakersTemplate
                                    }
                                    RadioButton {
                                        id: radioTerritoryRecord
                                        text: qsTr("Territory record")
                                        visible: listTypeGroup.currentIndex == 3
                                        onCheckedChanged: if (checked)
                                                              printController.templateType = TemplateData.TerritoryAssignmentRecordTemplate
                                    }
                                    RadioButton {
                                        id: radioTerritoryMapCard
                                        text: qsTr("Territory Map card")
                                        visible: listTypeGroup.currentIndex == 3
                                        onCheckedChanged: if (checked)
                                                              printController.templateType = TemplateData.TerritoryMapCardTemplate
                                    }
                                    RadioButton {
                                        id: radioMapAndAddressSheets
                                        text: qsTr("Map and address sheets")
                                        visible: listTypeGroup.currentIndex == 3
                                        onCheckedChanged: if (checked)
                                                              printController.templateType = TemplateData.TerritoryCardTemplate
                                    }
                                }
                            }
                        }

                        RowText {
                            width: parent.width
                            arrow: true
                            title: qsTr("Template", "Print template")
                            text: printController.templateName
                            onClicked: stack.push(templatePage)
                        }
                        RowText {
                            width: parent.width
                            title: qsTr("Weeks", "Print template")
                            visible: !(printController.templateType === TemplateData.TerritoryMapCardTemplate
                                       || printController.templateType === TemplateData.TerritoryCardTemplate)
                            text: printController.fromDate.toLocaleDateString(Qt.locale(), Locale.ShortFormat) + " - " +
                                  printController.toDate.toLocaleDateString(Qt.locale(), Locale.ShortFormat)
                        }
                        RowLayout {
                            width: parent.width
                            visible: printController.templateType === TemplateData.TerritoryMapCardTemplate
                                     || printController.templateType === TemplateData.TerritoryCardTemplate
                            Label {
                                Layout.leftMargin: 16
                                text: qsTr("Territory", "Print template")
                            }
                            SpinBox {
                                Layout.fillWidth: true
                                from: 1
                                value: printController.territoryNumber
                                editable: true
                                stepSize: 1
                                onValueModified: printController.territoryNumber = value
                                Layout.rightMargin: 16
                            }
                        }
                        SwitchDelegate {
                            id: checkBoxAssistantSlip
                            width: parent.width
                            text: qsTr("Assignment Slips for Assistant")
                            visible: radioMWSlip.checked
                            checked: printController.assistantSlip
                            onToggled: printController.assistantSlip = checked
                        }
                        SwitchDelegate {
                            id: checkBoxBlank
                            width: parent.width
                            text: qsTr("Print Assigned Only", "Assignment slip printing")
                            visible: radioMWSlip.checked
                            checked: printController.assignedOnly
                            onToggled: printController.assignedOnly = checked
                        }
                        RowLayout {
                            width: parent.width
                            Label {
                                Layout.leftMargin: 16
                                text: qsTr("Text size")
                            }
                            Slider {
                                Layout.fillWidth: true
                                from: -5
                                to: 5
                                value: printController.fontSize
                                stepSize: 1
                                onMoved: printController.fontSize = value
                                Layout.rightMargin: 16
                            }
                        }
                    }
                }

                Button {
                    id: printButton
                    Layout.preferredWidth: 200
                    Layout.alignment: Qt.AlignHCenter
                    Layout.margins: 10
                    text: qsTr("Print")
                    enabled: listTypeGroup.currentIndex == 0 && (
                                 radioMWSchedule.checked && canPrintMidweekSchedule ||
                                 radioMWWorkSheet.checked && canPrintMidweekWorksheet ||
                                 radioMWSlip.checked && canPrintMidweekSlips) ||
                             listTypeGroup.currentIndex == 1 && (
                                 radioWESchedule.checked && canPrintWeekendSchedule ||
                                 radioWEWorkSheet.checked && canPrintWeekendWorksheet ||
                                 radioOutgoingSchedule.checked && canPrintOutgoingSchedule ||
                                 radioOutgoingSlips.checked && canPrintOutgoingSlips ||
                                 radioHospitality.checked && canPrintHospitality ||
                                 radioTalksOfSpeakers.checked && canPrintPublicTalkList) ||
                             listTypeGroup.currentIndex === 2 && canPrintCombination ||
                             listTypeGroup.currentIndex == 3 && (
                                 radioTerritoryRecord.checked && canPrintTerritoryRecord ||
                                 radioTerritoryMapCard.checked && canPrintTerritoryMapCard ||
                                 radioMapAndAddressSheets.checked && canPrintTerritoryMapAndAddressSheets)
                    onClicked: {
                        console.log(printController.templateName)
                        printPage.showBusyIndicator = true
                        printController.print()
                    }
                }
            }
        }
    }

    Component {
        id: templatePage
        Page {
            ListView {
                id: listTemplates
                model: printPage.modelTemplates
                anchors.fill: parent
                currentIndex: 0
                delegate: RadioDelegate {
                    width: stack.width
                    text: modelData
                    checked: printController.templateName === text
                    onToggled: {
                        console.log(checked + " " + modelData)
                        if (checked) {
                            printController.templateName = modelData
                        }
                    }
                }
            }
        }
    }

    BusyIndicator {
        id: busy
        anchors.verticalCenter: parent.verticalCenter
        anchors.horizontalCenter: parent.horizontalCenter
        anchors.centerIn: parent.Center
        running: showBusyIndicator
    }
}
