/**
 * This file is part of TheocBase.
 *
 * Copyright (C) 2011-2019, TheocBase Development Team, see AUTHORS.
 *
 * TheocBase is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * TheocBase is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with TheocBase.  If not, see <http://www.gnu.org/licenses/>.
 */
import QtQuick
import QtQuick.Layouts
import QtQuick.Controls
import QtQuick.Controls.Material
import net.theocbase.mobile 1.0
import net.theocbase 1.0
import "js/moment.js" as Moment

Page {
    id: mainpage

    property date currentdate
    property int tabHeight: 45
    property int tabMargin: 0
    property string exceptionText: weekInfo.exceptionDisplayText
    property string weekStartingText: ""
    property date mwDate
    property date wkDate

    function activateTabByDay() {
        var d = new Date()
        swipe.currentIndex = (d.getDay() === 0 || d.getDay() === 6 ? 1 : 0)
    }

    function refreshList(addDays){
        stackView.pop(null)
        if (typeof addDays === 'undefined' || !moment(currentdate).isValid()){
            currentdate = moment().startOf('isoweek').toDate()
        }else{
            currentdate = moment(currentdate).add(addDays,'days').toDate()
        }

        console.log("refresh list " + currentdate.toString())
        lmMeeting.loadSchedule(currentdate)
        publicMeeting.loadSchedule(currentdate)
        outgoingSpeakers.loadList(currentdate)
        congCtrl.clearExceptionCache()
        weekInfo.load()
        specialEventRule = SpecialEvents.findSpecialEventRule(congCtrl.getSpecialEventId(currentdate))
        weekStartingText = qsTr("Week starting %1").arg(currentdate.toLocaleDateString(Qt.locale(), Locale.ShortFormat))
        // meeting days
        var mDay = congCtrl.getMeetingDay(MeetingType.MidweekMeeting, currentdate)
        var mDDay = congCtrl.getMeetingDay(MeetingType.MidweekMeeting)
        var wDay = congCtrl.getMeetingDay(MeetingType.WeekendMeeting, currentdate)
        var wDDay = congCtrl.getMeetingDay(MeetingType.WeekendMeeting)
        lmMeeting.editpossible = mDay > 0
        publicMeeting.editpossible = wDay > 0

        var mDate = new Date(currentdate)
        mDate.setDate(mDate.getDate() + (mDay === 0 ? mDDay : mDay) -1)
        mwDate = mDate
        var wDate = new Date(currentdate)
        wDate.setDate(wDate.getDate() + (wDay === 0 ? wDDay : wDay) -1)
        wkDate = wDate
    }

    // OBJECT PROPERTIES
    padding: 0
    header: ToolBar {
        id: toolbarbase
        RowLayout {
            id: topRow
            anchors.fill: parent

            ToolButton {
                Layout.alignment: Qt.AlignVCenter
                icon.source: "qrc:/icons/chevron_left.svg"
                icon.color: TBStyle.onPrimaryColor
                onClicked: { refreshList(-7) }
            }
            ToolButton {
                Layout.alignment: Qt.AlignVCenter
                icon.source: "qrc:/icons/chevron_right.svg"
                icon.color: TBStyle.onPrimaryColor
                onClicked: { refreshList(7) }
            }
            ComboBox {
                id: weekstartingButton
                Material.background: TBStyle.primaryColor
                Material.foreground: TBStyle.onPrimaryColor

                implicitHeight: parent.height
                flat: true
                background: null

                contentItem: ColumnLayout {
                    Label {
                        Layout.fillHeight: true
                        Layout.alignment: Qt.AlignVCenter
                        verticalAlignment: Text.AlignVCenter
                        text: weekStartingText
                    }
                    Label {
                        Layout.fillHeight: true
                        Layout.alignment: Qt.AlignVCenter
                        verticalAlignment: Text.AlignVCenter
                        text: lmMeeting.bibleReading
                    }
                }
                popup: calendar
            }
            Item {
                Layout.fillWidth: true
                Layout.minimumWidth: 10
            }
            ToolButton {
                icon.source: ccloud.syncState === Cloud.Synchronized ?
                                 "qrc:/icons/cloud_done.svg" : ccloud.syncState === Cloud.Download ?
                                     "qrc:/icons/cloud_download.svg" : "qrc:/icons/cloud_upload.svg"
                icon.color: TBStyle.onPrimaryColor
                onClicked: syncpage.runSync()
            }
            ToolButton {
                icon.source: "qrc:/icons/more.svg"
                icon.color: TBStyle.onPrimaryColor
                onClicked: moremenu.open()
                Menu {
                    id: moremenu
                    y: parent.height
                    MenuItem {
                        text: "Exceptions"
                        icon.source: "qrc:/icons/event_busy.svg"
                        enabled: accessControl.user && (accessControl.user.hasPermission(PermissionRule.CanViewSpecialEvents))
                        onClicked: {
                            if (exceptionPopup.active) {
                                exceptionPopup.item.weekDate = currentdate
                                exceptionPopup.item.weekInfo = weekInfo
                                exceptionPopup.item.open()
                            } else
                                exceptionPopup.active = true
                        }
                        Loader {
                            id: exceptionPopup
                            active: false
                            source: "Exceptions.qml"
                            onLoaded: {
                                item.weekDate = currentdate
                                item.weekInfo = weekInfo
                                item.open()
                            }
                        }
                    }
                    MenuItem {
                        text: "Print"
                        icon.source: "qrc:/icons/print.svg"
                        onClicked: {
                            printLoader.source = ""
                            printLoader.source = "PrintOptions.qml"
                        }
                        Loader {
                            id: printLoader
                            onLoaded: {
                                item.weekDate = currentdate
                                item.open()
                            }
                        }
                    }
                }
            }
        }
        CalendarPopup {
            id: calendar
            transformOrigin: Popup.Top
            x: (Qt.platform.os === "ios" ? mainpage.leftPadding : 0)
            y: toolbarbase.height
            onSelectedDateChanged: {
                currentdate = moment(selectedDate).startOf('isoWeek').toDate()
                refreshList(0)
            }
        }
    }

    StackView.onActivated: {
        console.log("Main page activated");
        appWindow.activepage = 1;
    }

    Component.onCompleted: {
        refreshList(0)
    }

    WeekInfo {
        id: weekInfo
        date: currentdate
    }

    CongregationCtrl { id: congCtrl }

    Page {
        anchors.fill: parent
        padding: 0
        header: Pane {
            visible: specialEventRule ? (specialEventRule.id !== SpecialEvent.None) : false
            Material.background: TBStyle.alertColor
            Label {
                anchors.fill: parent
                horizontalAlignment: Text.AlignHCenter
                verticalAlignment: Text.AlignVCenter
                text: exceptionText
                color: palette.window
                font: TBStyle.titleMediumFont
                visible: text !== ""
            }
        }

        SwipeView {
            id: swipe
            anchors.fill: parent
            spacing: 10
            Flickable {
                id: fl1
                contentHeight: lmMeeting.height
                onDragEnded: if (header1.refresh) syncpage.runSync()
                ListHeader {
                    id: header1
                    flickable: fl1
                    y: -height
                }

                LMMSchedule_Mobile {
                    id: lmMeeting
                    width: swipe.width
                    visible: currentdate.getFullYear() >= 2016 && canViewMidweekMeetingSchedule
                }
            }
            Flickable {
                id: fl2
                contentHeight: weLayout.height
                onDragEnded: if (header2.refresh) syncpage.runSync()
                ListHeader {
                    id: header2
                    flickable: fl2
                    y: -height
                }

                ColumnLayout {
                    id: weLayout
                    spacing: Material.delegateHeight / 2
                    width: swipe.width
                    PublicMeetingSchedule_Mobile {
                        id: publicMeeting
                        Layout.fillWidth: true
                        visible: canViewWeekendMeetingSchedule
                    }

                    OutgoingSpeakers {
                        id: outgoingSpeakers
                        Layout.fillWidth: true
                        visible: canViewWeekendMeetingSchedule
                        enabled: specialEventRule ? !specialEventRule.isWithoutPublicTalk : false
                    }
                    Item { Layout.fillHeight: true }
                }
            }
        }

        PageIndicator {
            count: swipe.count
            currentIndex: swipe.currentIndex

            anchors.bottom: swipe.bottom
            anchors.horizontalCenter: parent.horizontalCenter
        }
    }
}
