/**
 * This file is part of TheocBase.
 *
 * Copyright (C) 2011-2019, TheocBase Development Team, see AUTHORS.
 *
 * TheocBase is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * TheocBase is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with TheocBase.  If not, see <http://www.gnu.org/licenses/>.
 */
import QtQuick
import QtQuick.Controls
import QtQuick.Controls.Material
import QtQuick.Layouts
import QtQuick.Window
import net.theocbase.mobile 1.0
import net.theocbase 1.0
import "./"

Page {
    id: assignmentDialog
    property bool canSendMidweekMeetingReminders: accessControl.user && accessControl.user.hasPermission(PermissionRule.CanSendMidweekMeetingReminders)
    property string returnValue: ""
    property LMM_Assignment currentAssignment
    property int oldSpeakerId: -1
    property int oldAssistantId: -1
    property int oldVolunteerId: -1
    property int currentStudyId: 0
    property int nextStudyId: 0
    property string _current_timer_title: _timer_text
    property color _timer_color: palette.text
    property string _timer_text: qsTr("Stopwatch")
    property string _leave_study: qsTr("Leave on current study")
    property string _start_timer_text: qsTr("Start stopwatch")
    property string _stop_timer_text: qsTr("Stop stopwatch")
    property date _startTime
    property int currentindex: -1
    property int modelLength: 0
    signal gotoNext()
    signal gotoPrevious()

    AssignmentController { id: myController }
    onCurrentAssignmentChanged: {
        if (!currentAssignment) return
        myController.assignment = currentAssignment
        oldSpeakerId = currentAssignment.speaker ? currentAssignment.speaker.id : -1
        oldAssistantId = currentAssignment.assistant ? currentAssignment.assistant.id : -1
        oldVolunteerId = currentAssignment.volunteer ? currentAssignment.volunteer.id : -1
    }
    function saveChanges(){
        if (Qt.inputMethod.visible)
            Qt.inputMethod.hide()
        if (!currentAssignment)
            return
        if (_timer.running){
            _msg.showYesNo(_stop_timer_text,"",1)
            return false
        }

        var s_id = currentAssignment.speaker ? currentAssignment.speaker.id : -1
        var a_id = currentAssignment.assistant ? currentAssignment.assistant.id : -1
        var v_id = currentAssignment.volunteer ? currentAssignment.volunteer.id : -1

        if (currentAssignment.note !== textAreaNotes.text ||
                currentAssignment.completed !== checkCompleted.checked ||
                currentAssignment.timing !== textTiming.text ||
                s_id !== oldSpeakerId || a_id !== oldAssistantId || v_id !== oldVolunteerId){
            currentAssignment.note = textAreaNotes.text
            currentAssignment.completed = checkCompleted.checked
            currentAssignment.timing = textTiming.text
            currentAssignment.save()
            console.log("assignment saved")
            oldSpeakerId = s_id
            oldAssistantId = a_id
        }

        if (currentAssignment.speaker && currentAssignment.date.getFullYear() < 2019){
            if (currentAssignment.volunteer)
                myController.saveStudy(checkExerice.checked,
                                       false, currentStudyId)
            else
                myController.saveStudy(checkExerice.checked,
                                       checkCompleted.checked,
                                       nextStudyId)
        }
        return true
    }
    function timerTriggered(){
        if (_timer.running){
            _timer.stop()
            _timer_color = _timer_color
            //timertext.text = _start_timer_text
            _msg.showYesNo(qsTr("Add the timing?"), title.text,2)
        }else{
            _startTime = new Date()
            _timer.running = true
            _timer_color = palette.text
            //timertext.text = _stop_timer_text
        }
    }
    SystemPalette { id: palette }
    MsgBox { id: _msg }
    Connections {
        target: _msg
        function  onButtonClicked(ok, id) {
            switch (id){
            case 1:
                if (ok) timerTriggered()
                break
            case 2:
                if (ok) textTiming.text = _current_timer_title
                _current_timer_title = _timer_text
                saveChanges()
                break
            default: break
            }
        }
    }
    Timer {
        id: _timer
        interval: 500
        running: false
        repeat: true
        triggeredOnStart: true
        onTriggered: {
            var elapsedTime = Math.round((new Date().getTime() - _startTime.getTime()) / 1000)
            var minutes = Math.floor(elapsedTime / 60)
            var seconds = elapsedTime-minutes * 60
            _current_timer_title = (minutes < 10 ? "0" : "") + minutes.toString() +
                    ":" + (seconds < 10 ? "0" : "") + seconds.toString()
            _timer_color
        }
    }

    header: ToolBar {
        RowLayout {
            anchors.fill: parent
            ToolButton {
                icon.source: "qrc:/icons/back.svg"
                icon.color: TBStyle.onPrimaryColor
                onClicked: {
                    if (!saveChanges()) return
                    stackView.pop()
                }
            }
            Label {
                Layout.fillWidth: true
                text: qsTr("Details", "Page title")
                horizontalAlignment: Text.AlignHCenter
                font: TBStyle.titleMediumFont
                color: TBStyle.onPrimaryColor
            }
            ToolButton {
                icon.source: "qrc:/icons/share.svg"
                icon.color: TBStyle.onPrimaryColor
                visible: canSendMidweekMeetingReminders
                onClicked: {
                    shareUtils.setPoint(mapToGlobal(width/2, height))
                    shareUtils.shareText(currentAssignment.getReminderText())
                }
            }
            ToolButton {
                icon.source: "qrc:/icons/skip_previous.svg"
                icon.color: enabled ? TBStyle.onPrimaryColor : TBStyle.mediumColor
                enabled: currentindex > 0
                onClicked: {
                    if (!saveChanges()) return;
                    gotoPrevious()
                }
            }
            ToolButton {
                icon.source: "qrc:/icons/skip_next.svg"
                icon.color: enabled ? TBStyle.onPrimaryColor : TBStyle.mediumColor
                enabled: currentindex < (modelLength -1)
                onClicked: {
                    if (!saveChanges()) return;
                    gotoNext()
                }
            }
        }
    }

    ScrollView {
        id: flickable
        anchors.fill: parent
        contentWidth: availableWidth

        Pane {
            width: parent.width
            ColumnLayout {
                id: layout
                width: parent.width
                spacing: 16

                RowLayout {
                    Layout.fillWidth: true
                    visible: !checkCompleted.checked && currentAssignment && currentAssignment.speaker
                    ToolButton {
                        icon.source: "qrc:/icons/timer.svg"
                        Layout.fillHeight: true
                        Layout.preferredWidth: height
                        onClicked: timerTriggered()
                    }
                    Label {
                        id: timertext
                        Layout.fillWidth: true
                        text: _current_timer_title
                        color: _timer_color
                        horizontalAlignment: Text.AlignRight
                        verticalAlignment: Text.AlignVCenter
                    }
                }

                FilledGroupBox {
                    Layout.fillWidth: true

                    Column {
                        width: parent.width

                        Label {
                            width: parent.width
                            text: currentAssignment ? currentAssignment.theme : qsTr("Theme")
                            wrapMode: Text.WordWrap
                        }
                        MenuSeparator { width: parent.width; padding: 0 }
                        RowLayout {
                            width: parent.width
                            InterimIcon {
                                Layout.alignment: Qt.AlignLeft | Qt.AlignVCenter
                                source: "qrc:/icons/wt_source.svg"
                                color: Material.foreground
                                ToolTip.text: qsTr("Source")
                                ToolTip.visible: hovered
                            }
                            Label {
                                Layout.fillWidth: true
                                text: currentAssignment ? currentAssignment.source : qsTr("Source")
                                wrapMode: Text.WordWrap
                            }
                        }
                    }
                }

                FilledGroupBox {
                    id: studentGroupBox
                    Layout.fillWidth: true
                    title: qsTr("Student")

                    RowLayout {
                        width: parent.width
                        InterimIcon {
                            Layout.alignment: Qt.AlignLeft | Qt.AlignVCenter
                            source: "qrc:/icons/person.svg"
                            color: Material.foreground
                            ToolTip.text: qsTr("Student")
                            ToolTip.visible: hovered
                        }
                        ComboBoxTable {
                            Layout.fillWidth: true
                            currentText: currentAssignment && currentAssignment.speaker ? currentAssignment.speaker.fullName : ""
                            enabled: !checkCompleted.checked
                            col1Name: studentGroupBox.title
                            onBeforeMenuShown: {
                                model = currentAssignment.getSpeakerList()
                            }
                            onRowSelected: {
                                var student = CPersons.getPerson(id)
                                currentAssignment.speaker = student
                                saveChanges()
                            }
                            onCurrentTextChanged: function (text) {
                                if (text.length > 2) {
                                    var study = myController.currentStudy;
                                    if (study) {
                                        currentStudyId = study.number;
                                        textCurrentStudy.text = study.number + " " + study.name;
                                        checkExerice.checked = study.exercise;
                                        nextStudyId = myController.nextStudy;
                                        var nextnumber = currentAssignment.completed ? nextStudyId : study.number;
                                        textNextStudy.text = nextnumber === study.number ? _leave_study : myController.getStudies()[nextnumber];
                                    }
                                } else {
                                    textCurrentStudy.text = "";
                                }
                            }
                            onTooltipRequest: {
                                if (model.get(row).id !== "undefined") {
                                    var userid = model.get(row).id
                                    rowTooltip = myController.getHistoryTooltip(userid, TBStyle.bodySmallFont.pointSize)
                                }
                            }
                        }
                    }
                }

                FilledGroupBox {
                    id: assistantGroupBox
                    Layout.fillWidth: true
                    title: qsTr("Assistant", "Assistant to student")
                    visible: currentAssignment && currentAssignment.canHaveAssistant

                    RowLayout {
                        width: parent.width
                        InterimIcon {
                            Layout.alignment: Qt.AlignLeft | Qt.AlignVCenter
                            source: "qrc:/icons/assistant.svg"
                            color: Material.foreground
                            ToolTip.text: qsTr("Assistant","Assistant to student")
                            ToolTip.visible: hovered
                        }
                        ComboBoxTable {
                            Layout.fillWidth: true
                            currentText: currentAssignment && currentAssignment.assistant ? currentAssignment.assistant.fullName : ""
                            visible: currentAssignment && currentAssignment.canHaveAssistant
                            enabled: currentAssignment && currentAssignment.speaker  && !checkCompleted.checked
                            col1Name: assistantGroupBox.title
                            onBeforeMenuShown: {
                                model = currentAssignment.getAssistantList()
                            }
                            onRowSelected: {
                                var assistant = CPersons.getPerson(id)
                                currentAssignment.assistant = assistant
                                if (currentAssignment.assistant.gender !== currentAssignment.speaker.gender &&
                                        (currentAssignment.meetingPart === MeetingPart.LMM_FM_FollowingUp ||
                                         currentAssignment.meetingPart === MeetingPart.LMM_FM_MakingDisciples)){
                                    _message.show("TheocBase", qsTr('The assistant should not be someone of the opposite sex.'));
                                }
                                saveChanges()
                            }
                            MsgBox { id: _message }
                            onTooltipRequest: {
                                if (model.get(row).id !== "undefined") {
                                    var userid = model.get(row).id
                                    rowTooltip = myController.getHistoryTooltip(userid, TBStyle.bodySmallFont.pointSize)
                                }
                            }
                        }
                    }
                }

                FilledGroupBox {
                    Layout.fillWidth: true
                    title: qsTr("Study point")
                    visible: currentAssignment.date.getFullYear() >= 2019

                    Column {
                        width: parent.width
                        RowLayout {
                            width: parent.width
                            InterimIcon {
                                Layout.alignment: Qt.AlignLeft | Qt.AlignVCenter
                                source: "qrc:/icons/format_list_numbered.svg"
                                color: Material.foreground
                                ToolTip.text: qsTr("Study point")
                                ToolTip.visible: hovered
                            }
                            RowText {
                                id: textStudyPoint
                                Layout.fillWidth: true
                                text: currentAssignment.studyNumber + " " + currentAssignment.studyName
                                visible: currentAssignment.date.getFullYear() >= 2019
                                editable: false
                            }
                        }
                        MenuSeparator { width: parent.width; padding: 0 }
                        RowLayout {
                            width: parent.width
                            InterimIcon {
                                Layout.alignment: Qt.AlignLeft | Qt.AlignVCenter
                                source: "qrc:/icons/assignment_done.svg"
                                color: Material.foreground
                                ToolTip.text: qsTr("Result")
                                ToolTip.visible: hovered
                            }
                            SwitchDelegate {
                                id: checkCompleted
                                Layout.fillWidth: true
                                text: qsTr("Completed")
                                checked: currentAssignment && currentAssignment.completed
                                enabled: currentAssignment && currentAssignment.speaker
                                onClicked: saveChanges()
                            }
                        }
                    }
                }

                FilledGroupBox {
                    Layout.fillWidth: true
                    visible: checkCompleted.checked
                    title: qsTr("Volunteer")

                    RowLayout {
                        width: parent.width
                        InterimIcon {
                            Layout.alignment: Qt.AlignLeft | Qt.AlignVCenter
                            source: "qrc:/icons/person_substitute.svg"
                            color: Material.foreground
                            ToolTip.text: qsTr("Volunteer")
                            ToolTip.visible: hovered
                        }
                        RowText {
                            id: textVolunteer
                            Layout.fillWidth: true
                            title: qsTr("Volunteer")
                            text: currentAssignment && currentAssignment.volunteer ? currentAssignment.volunteer.fullName : ""
                            arrow: true
                            onClicked: {
                                var mylist = currentAssignment.getSpeakerList()
                                var rowindex = currentAssignment.volunteer ?
                                            mylist.find(currentAssignment.volunteer.id,1) : -1
                                var volunteersPage = stackView.push(Qt.resolvedUrl("SelectionListPage.qml"),
                                                                    {showName: true, model: mylist, selectedRow: rowindex, pageTitle: title})
                                volunteersPage.onActiveRowChanged.connect(function(index){
                                    var id = mylist.get(index).id
                                    if (typeof id === "undefined"){
                                        text = ""
                                        currentAssignment.volunteer = null
                                    }else{
                                        text = mylist.get(index).name
                                        currentAssignment.volunteer = myController.getPublisherById(id)
                                    }
                                    saveChanges()
                                })
                            }
                        }
                    }
                }

                FilledGroupBox {
                    Layout.fillWidth: true

                    RowLayout {
                        width: parent.width
                        InterimIcon {
                            Layout.alignment: Qt.AlignLeft | Qt.AlignVCenter
                            source: "qrc:/icons/study_timing.svg"
                            color: Material.foreground
                            ToolTip.text: qsTr("Timing")
                            ToolTip.visible: hovered
                        }
                        RowText {
                            id: textTiming
                            Layout.fillWidth: true
                            title: qsTr("Timing")
                            text: currentAssignment ? currentAssignment.timing : ""
                            editable: true
                            onEditingFinished: saveChanges()
                        }
                    }
                }

                FilledGroupBox {
                    Layout.fillWidth: true
                    Layout.minimumHeight: 100

                    RowLayout {
                        anchors.fill: parent
                        InterimIcon {
                            Layout.alignment: Qt.AlignLeft | Qt.AlignTop
                            source: "qrc:/icons/notes.svg"
                            color: Material.foreground
                            ToolTip.text: qsTr("Notes")
                            ToolTip.visible: hovered
                        }
                        TextArea {
                            id: textAreaNotes
                            Layout.fillWidth: true
                            Layout.fillHeight: true
                            Material.containerStyle: Material.Filled
                            text: currentAssignment ? currentAssignment.note : ""
                            placeholderText: qsTr("Notes")
                            wrapMode: Text.WordWrap
                            onEditingFinished: saveChanges()
                        }
                    }
                }
            }
        }
    }
}
