/**
 * This file is part of TheocBase.
 *
 * Copyright (C) 2011-2019, TheocBase Development Team, see AUTHORS.
 *
 * TheocBase is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * TheocBase is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with TheocBase.  If not, see <http://www.gnu.org/licenses/>.
 */
import QtQuick
import QtQuick.Controls
import QtQuick.Controls.Material
import QtQuick.Layouts
import QtQuick.Window
import net.theocbase.mobile 1.0
import net.theocbase 1.0
import "./"

Page {
    id: assignmentDialog

    property bool canSendMidweekMeetingReminders: accessControl.user && accessControl.user.hasPermission(PermissionRule.CanSendMidweekMeetingReminders)
    property string returnValue: ""
    property LMM_Assignment currentAssignment
    property int oldSpeakerId: -1
    property int oldAssistantId: -1
    property int currentindex: -1
    property int modelLength: 0
    property bool editableTheme: currentAssignment &&
                                 (currentAssignment.meetingPart === MeetingPart.Service_Talk ||
                                  currentAssignment.meetingPart === MeetingPart.LMM_CL_Talk ||
                                  currentAssignment.meetingPart === MeetingPart.LMM_CL_Discussion ||
                                  currentAssignment.meetingPart === MeetingPart.LMM_CL_Video)

    signal gotoNext()
    signal gotoPrevious()

    function saveChanges(){
        if (Qt.inputMethod.visible)
            Qt.inputMethod.hide()

        if (!currentAssignment)
            return

        var s_id = currentAssignment.speaker ? currentAssignment.speaker.id : -1
        var a_id = currentAssignment.assistant ? currentAssignment.assistant.id : -1

        if (currentAssignment.note !== textAreaNotes.text ||
                s_id !== oldSpeakerId || a_id !== oldAssistantId){
            currentAssignment.note = textAreaNotes.text
            currentAssignment.save()
            console.log("assignment saved")
            oldSpeakerId = s_id
            oldAssistantId = a_id
        }
        return true
    }

    onCurrentAssignmentChanged: {
        if (!currentAssignment) return
        oldSpeakerId = currentAssignment.speaker ? currentAssignment.speaker.id : -1
        oldAssistantId = currentAssignment.assistant ? currentAssignment.assistant.id : -1
    }

    header: ToolBar {
        RowLayout {
            anchors.fill: parent
            ToolButton {
                icon.source: "qrc:/icons/back.svg"
                icon.color: TBStyle.onPrimaryColor
                onClicked: {
                    if (!saveChanges())
                        return
                    stackView.pop()
                }
            }
            Label {
                Layout.fillWidth: true
                text: qsTr("Details", "Page title")
                horizontalAlignment: Text.AlignHCenter
                font: TBStyle.titleMediumFont
                color: TBStyle.onPrimaryColor
            }
            ToolButton {
                icon.source: "qrc:/icons/share.svg"
                icon.color: TBStyle.onPrimaryColor
                visible: canSendMidweekMeetingReminders
                onClicked: {
                    shareUtils.setPoint(mapToGlobal(width/2, height))
                    shareUtils.shareText(currentAssignment.getReminderText())
                }
            }
            ToolButton {
                icon.source: "qrc:/icons/skip_previous.svg"
                icon.color: enabled ? TBStyle.onPrimaryColor : TBStyle.mediumColor
                enabled: currentindex > 0
                onClicked: {
                    if (!saveChanges()) return;
                    gotoPrevious()
                }
            }
            ToolButton {
                icon.source: "qrc:/icons/skip_next.svg"
                icon.color: enabled ? TBStyle.onPrimaryColor : TBStyle.mediumColor
                enabled: currentindex < (modelLength -1)
                onClicked: {
                    if (!saveChanges()) return;
                    gotoNext()
                }
            }
        }
    }

    AssignmentController { id: myController }
    ScrollView {
        id: flickable
        anchors.fill: parent
        contentWidth: availableWidth

        Pane {
            width: parent.width
            ColumnLayout {
                id: layout
                width: parent.width
                spacing: 16

                FilledGroupBox {
                    Layout.fillWidth: true

                    Column {
                        width: parent.width

                        TextField {
                            width: parent.width
                            Material.containerStyle: Material.Filled
                            text: currentAssignment ? currentAssignment.theme : ""
                            placeholderText: qsTr("Theme")
                            wrapMode: Text.WordWrap
                            readOnly: !editableTheme
                            onEditingFinished: {
                                if (editableTheme && currentAssignment.theme !== text) {
                                    currentAssignment.theme = text
                                    currentAssignment.save()
                                }
                            }
                        }
                        RowLayout {
                            width: parent.width
                            InterimIcon {
                                Layout.alignment: Qt.AlignLeft | Qt.AlignVCenter
                                source: "qrc:/icons/wt_source.svg"
                                color: Material.foreground
                                ToolTip.text: qsTr("Source")
                                ToolTip.visible: hovered
                            }
                            TextArea {
                                Layout.fillWidth: true
                                Material.containerStyle: Material.Filled
                                text: currentAssignment ? currentAssignment.source : ""
                                placeholderText: qsTr("Source")
                                wrapMode: Text.WordWrap
                                readOnly: !editableTheme
                                onEditingFinished: {
                                    if (editableTheme && currentAssignment.source !== text) {
                                        currentAssignment.source = text
                                        currentAssignment.save()
                                    }
                                }
                            }
                        }
                    }
                }

                FilledGroupBox {
                    id: speakerGroupBox
                    Layout.fillWidth: true
                    title: currentAssignment && currentAssignment.meetingPart === MeetingPart.LMM_CBS
                           ? qsTr("Conductor")
                           : qsTr("Speaker")

                    RowLayout {
                        width: parent.width
                        InterimIcon {
                            Layout.alignment: Qt.AlignLeft | Qt.AlignVCenter
                            source: "qrc:/icons/speaker.svg"
                            color: Material.foreground
                        }
                        ComboBoxTable {
                            Layout.fillWidth: true
                            currentText: currentAssignment
                                         ? currentAssignment.speakerFullName
                                         : ""
                            currentId: currentAssignment.speaker ? currentAssignment.speaker.id : -1
                            col1Name: speakerGroupBox.title
                            enabled: currentAssignment &&
                                     currentAssignment.meetingPart !== MeetingPart.Service_Talk &&
                                     currentAssignment.meetingPart !== MeetingPart.LMM_CL_Video
                            onBeforeMenuShown: {
                                model = currentAssignment.getSpeakerList()
                            }
                            onRowSelected: {
                                var speaker = CPersons.getPerson(id)
                                currentAssignment.speaker = speaker
                                saveChanges()
                            }
                            onTooltipRequest: {
                                if (model.get(row).id !== "undefined") {
                                    var userid = model.get(row).id
                                    rowTooltip = myController.getHistoryTooltip(userid, TBStyle.bodySmallFont.pointSize)
                                }
                            }
                        }
                    }
                }

                FilledGroupBox {
                    id: readerGroupBox
                    Layout.fillWidth: true
                    title: qsTr("Reader")
                    visible: currentAssignment && currentAssignment.meetingPart === MeetingPart.LMM_CBS

                    RowLayout {
                        width: parent.width
                        InterimIcon {
                            Layout.alignment: Qt.AlignLeft | Qt.AlignVCenter
                            source: "qrc:/icons/reader.svg"
                            color: Material.foreground
                            ToolTip.text: qsTr("Reader")
                            ToolTip.visible: hovered
                        }
                        ComboBoxTable {
                            Layout.fillWidth: true
                            currentText: currentAssignment && currentAssignment.assistant ? currentAssignment.assistant.fullName : ""
                            currentId: currentAssignment.assistant ? currentAssignment.assistant.id : -1
                            visible: currentAssignment && currentAssignment.meetingPart === MeetingPart.LMM_CBS
                            col1Name: readerGroupBox.title
                            onBeforeMenuShown: {
                                model = currentAssignment.getAssistantList()
                            }
                            onRowSelected: {
                                var reader = CPersons.getPerson(id)
                                currentAssignment.assistant = reader
                                saveChanges()
                            }
                        }
                    }
                }

                FilledGroupBox {
                    Layout.fillWidth: true
                    Layout.minimumHeight: 100

                    RowLayout {
                        anchors.fill: parent
                        InterimIcon {
                            Layout.alignment: Qt.AlignLeft | Qt.AlignTop
                            source: "qrc:/icons/notes.svg"
                            color: Material.foreground
                            ToolTip.text: qsTr("Notes")
                            ToolTip.visible: hovered
                        }
                        TextArea {
                            id: textAreaNotes
                            Layout.fillWidth: true
                            Layout.fillHeight: true
                            Material.containerStyle: Material.Filled
                            text: currentAssignment ? currentAssignment.note : ""
                            placeholderText: qsTr("Notes")
                            wrapMode: Text.WordWrap
                            onEditingFinished: saveChanges()
                        }
                    }
                }
            }
        }
    }
}
