/**
 * This file is part of TheocBase.
 *
 * Copyright (C) 2011-2020, TheocBase Development Team, see AUTHORS.
 *
 * TheocBase is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * TheocBase is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with TheocBase.  If not, see <http://www.gnu.org/licenses/>.
 */
import QtQuick
import QtQuick.Controls
import QtQuick.Controls.Material
import QtQuick.Layouts
import net.theocbase.mobile 1.0
import net.theocbase 1.0
import "js/moment.js" as Moment

Popup {
    id: exceptionPopup

    property int midweek: 4
    property int weekend: 6
    property int defaultMidweek: 4
    property int defaultWeekend: 7
    property int exStart: 0
    property date exStartDate: moment(weekDate).add(exStart, 'days').toDate()
    property int exEnd: 0
    property date exEndDate: moment(weekDate).add(exEnd, 'days').toDate()
    property date weekDate
    property bool canEditExceptions: accessControl.user && (accessControl.user.hasPermission(PermissionRule.CanEditSpecialEvents))
    property bool canChangeMidweekMeeting: true
    property bool canChangeWeekendMeeting: true

    property var specialEventRules: SpecialEvents.specialEventRules
    property SpecialEventRule specialEventRule: SpecialEvents.findSpecialEventRule(SpecialEvent.None)
    property string exceptionText: ""
    property WeekInfo weekInfo

    modal: true
    parent: Overlay.overlay
    x: (parent.width - width) / 2
    y: (parent.height - height) / 2
    implicitWidth:  parent.width / 5 * 4
    implicitHeight: parent.height / 5 * 4
    closePolicy: Popup.NoAutoClose

    padding: 0

    onOpened: {
        defaultMidweek = congCtrl.getMeetingDay(MeetingType.MidweekMeeting)
        defaultWeekend = congCtrl.getMeetingDay(MeetingType.WeekendMeeting)

        specialEventRule = SpecialEvents.findSpecialEventRule(weekInfo.specialEvent)
        midweek = weekInfo.midweekDay
        weekend = weekInfo.weekendDay
        exStart = isNaN(weekInfo.exceptionStart) ? 0 : moment(weekInfo.exceptionStart).diff(moment(weekInfo.date), 'day')
        exEnd = isNaN(weekInfo.exceptionEnd) ? 0 : moment(weekInfo.exceptionEnd).diff(moment(weekInfo.date), 'day')
        exceptionText = weekInfo.exceptionText
    }
    onSpecialEventRuleChanged: {
        exStart = specialEventRule.startDay - 1;
        if (exStart < 0)
            exStart = 0;
        exEnd = specialEventRule.endDay - 1;
        midweek = defaultMidweek;
        weekend = defaultWeekend;

        if (specialEventRule.isCircuitOverseersVisit)
            midweek = 2;
        if (specialEventRule.isConvention) {
            exStart = specialEventRule.isSingleDay ? 6 : 4
            exEnd = 6;
        }
        if (specialEventRule.exclusivity === SpecialEventExclusivity.NoOtherMeetingInSameWeek) {
            midweek = 0;
            weekend = 0;
        }
        if (specialEventRule.exclusivity === SpecialEventExclusivity.NoOtherMeetingInSamePartOfTheWeek) {
            if (exStart < 5)
                midweek = 0;
            else
                weekend = 0;
        }
        exceptionText = "";
        canChangeMidweekMeeting = midweek !== 0 && canEditExceptions;
        canChangeWeekendMeeting = weekend !== 0 && canEditExceptions;
    }
    onExStartChanged: {
        if (specialEventRule.exclusivity === SpecialEventExclusivity.NoOtherMeetingInSamePartOfTheWeek) {
            exEnd = exStart
            if (exStart < 5) {
                midweek = 0
                weekend = defaultWeekend
            } else {
                midweek = defaultMidweek
                weekend = 0
            }
        }
    }
    focus: true

    function saveChanges() {
        weekInfo.specialEvent = specialEventRule.id
        if (specialEventRule.id !== SpecialEvent.None) {
            weekInfo.exceptionText = (specialEventRule && specialEventRule.canChangeDescription) ? exceptionText : ""
            weekInfo.exceptionStart = exStartDate
            weekInfo.exceptionEnd = exEndDate
            weekInfo.midweekDay = midweek
            weekInfo.weekendDay = weekend
        }
        weekInfo.saveChanges()
    }

    WeekInfo {
        date: weekDate
    }

    CongregationCtrl { id: congCtrl }

    ToolBar {
        id: toolbar
        width: parent.width

        Material.foreground: TBStyle.onPrimaryColor

        RowLayout {
            anchors.fill: parent
            ToolButton {
                icon.source: stack.depth > 1 ? "qrc:/icons/back.svg" : "qrc:/icons/cancel.svg"
                onClicked: {
                    // close but do not save
                    if (stack.depth > 1)
                        stack.pop()
                    else
                        exceptionPopup.close()
                }
            }
            Label {
                text: qsTr("Week starting %1").arg(weekDate.toLocaleDateString(Qt.locale(), Locale.ShortFormat))
                Layout.fillWidth: true
            }

            ToolButton {
                icon.source: "qrc:/icons/check.svg"
                visible: stack.depth === 1
                enabled: canEditExceptions
                onClicked: {
                    // close and save the changes
                    saveChanges()
                    exceptionPopup.close()
                    reloadMainpage()
                }
            }
        }
    }

    StackView {
        id: stack
        anchors.fill: parent
        anchors.topMargin: toolbar.height

        focus: true
        Keys.onReleased: (event)=> {
            if (event.key === Qt.Key_Back) {
                // close but do not save
                if (stack.depth > 1)
                    stack.pop()
                else
                    exceptionPopup.close()
            }
        }

        initialItem: Pane {
            id: exceptionPage
            Column {
                spacing: 0
                anchors.fill: parent
                RowText {
                    width: parent.width
                    arrow: true
                    title: specialEventRule ? specialEventRule.description : ""
                    text: (specialEventRule && specialEventRule.canChangeDescription) ? exceptionText : ""
                    enabled: canEditExceptions
                    onClicked: stack.push(exceptionTypesPage)
                }
                RowText {
                    id: textExceptionDates
                    width: parent.width
                    arrow: true
                    title: exStartDate.toLocaleDateString(Qt.locale(), Locale.ShortFormat) +
                           (exEnd > exStart ?
                                " - " + exEndDate.toLocaleDateString(Qt.locale(), Locale.ShortFormat)
                              : "")
                    visible: specialEventRule && (specialEventRule.id !== SpecialEvent.None)
                    enabled: canEditExceptions
                    onClicked: stack.push(exceptionDatesPage)
                }

                RowTitle {
                    width: parent.width
                    text: qsTr("Meeting Days")
                }

                RowText {
                    id: textMidweekMeeting
                    width: parent.width
                    arrow: true
                    title: qsTr("Midweek Meeting")
                    text: midweek > 0 ? Qt.locale().dayName(midweek) : "-"
                    enabled: canChangeMidweekMeeting
                    onClicked: stack.push(midweekPage)
                }
                RowText {
                    id: textWeekendMeeting
                    width: parent.width
                    arrow: true
                    title: qsTr("Weekend Meeting")
                    text: weekend  > 0 ? Qt.locale().dayName(weekend) : "-"
                    enabled: canChangeWeekendMeeting
                    onClicked: stack.push(weekendPage)                    
                }
            }
        }
    }

    Component {
        id: exceptionTypesPage
        Pane {
            ScrollView {
                width: parent.width
                height: parent.height
                contentWidth: availableWidth
                ColumnLayout {
                    width: stack.availableWidth
                    Repeater {
                        id: rpt
                        model: specialEventRules
                        RadioButton {
                            text: specialEventRules[index].description
                            Layout.fillWidth: true
                            checked: specialEventRule && (specialEventRule.id === specialEventRules[index].id)
                            onCheckedChanged: specialEventRule = specialEventRules[index]
                        }
                    }
                    TextField {
                        text: exceptionText
                        Layout.fillWidth: true
                        visible: specialEventRule && specialEventRule.canChangeDescription
                        onTextEdited: exceptionText = text
                    }
                }
            }
        }
    }

    Component {
        id: midweekPage
        Pane {
            ExceptionDaySelector {
                anchors.centerIn: parent
                selectedDay: midweek
                showWeekDays: true
                showWeekend: false
                onSelectedDayChanged: midweek = selectedDay
            }
        }
    }
    Component {
        id: weekendPage
        Pane {
            ExceptionDaySelector {
                anchors.centerIn: parent
                selectedDay: weekend
                showWeekDays: false
                showWeekend: true
                onSelectedDayChanged: weekend = selectedDay
            }
        }
    }
    Component {
        id: exceptionDatesPage
        Pane {
            ExceptionDateSelector {
                anchors.centerIn: parent
                weekDate: exceptionPopup.weekDate
                start: exStart
                end: exEnd
                onStartChanged: exStart = start
                onEndChanged: exEnd = end
            }
        }
    }
}
