/**
 * This file is part of TheocBase.
 *
 * Copyright (C) 2011-2019, TheocBase Development Team, see AUTHORS.
 *
 * TheocBase is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * TheocBase is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with TheocBase.  If not, see <http://www.gnu.org/licenses/>.
 */
import QtQuick
import QtQuick.Controls
import QtQuick.Controls.Material
import QtQuick.Controls.Material.impl
import QtQuick.Layouts
import QtQuick.Window
import net.theocbase.mobile 1.0

Item {
    id: comboBoxTable

    property string currentText
    property int currentId
    property var model
    property string col1Name: qsTr("Name", "Column name")
    property string col2Name: qsTr("Date", "Column name")
    property string col3Name: ""
    property string rowTooltip: ""

    signal beforeMenuShown
    signal rowSelected(var id)
    signal tooltipRequest(var row)

    implicitHeight: row.height

    RowText {
        id: row
        width: parent.width
        title: currentText
        arrow: true
        onClicked: {
            beforeMenuShown()
            console.log(currentText)
            console.log(model.rowCount())
            console.log(width)
            var index = -1
            if (currentText.length > 0) {
                for (var i = 0; i < model.rowCount(); i++) {
                    if (currentId > 0 && model.get(i).id === currentId)
                        index = i
                    if (model.get(i).name === currentText)
                        index = i
                }
            }
            stackView.push(listPage, {selectedRow: index})
        }
    }

    Component {
        id: listPage
        Page {
            property int selectedRow
            onSelectedRowChanged: {
                listView.currentIndex = selectedRow
            }

            header: BaseToolbar {
                title: title
                componentLeft: ToolButton {
                    icon.source: "qrc:/icons/cancel.svg"
                    onClicked: { stackView.pop() }
                }
                componentRightMiddle : ToolButton {
                    // Sort Menu
                    icon.source: "qrc:/icons/sort.svg"
                    onClicked: menu.open()
                    Menu {
                        id: menu
                        y: parent.height
                        MenuItem {
                            text: col1Name + (listView.model.sortColumn === 1 ?
                                                        listView.model.sortOrder === Qt.AscendingOrder ? " ↑" : " ↓" : "")
                            onTriggered: {
                                var order = Qt.AscendingOrder
                                if(listView.model.sortColumn === 1)
                                    order = listView.model.sortOrder === Qt.AscendingOrder ? Qt.DescendingOrder : Qt.AscendingOrder
                                comboBoxTable.model.sort(1, order)
                            }
                        }
                        MenuItem {
                            text: col2Name + (listView.model.sortColumn === 2 ?
                                                        listView.model.sortOrder === Qt.AscendingOrder ? " ↑" : " ↓" : "")
                            onTriggered: {
                                var order = Qt.AscendingOrder
                                if(listView.model.sortColumn === 2)
                                    order = listView.model.sortOrder === Qt.AscendingOrder ? Qt.DescendingOrder : Qt.AscendingOrder
                                comboBoxTable.model.sort(2, order)
                            }
                        }
                        MenuItem {
                            text: col3Name + (listView.model.sortColumn === 3 ?
                                                        listView.model.sortOrder === Qt.AscendingOrder ? " ↑" : " ↓" : "")
                            visible: col3Name.length > 0
                            height: visible ? implicitHeight : 0
                            onTriggered: {
                                var order = Qt.AscendingOrder
                                if(listView.model.sortColumn === 3)
                                    order = listView.model.sortOrder === Qt.AscendingOrder ? Qt.DescendingOrder : Qt.AscendingOrder
                                comboBoxTable.model.sort(3, order)
                            }
                        }
                    }
                }

                componentRight: ToolButton {
                    icon.source: "qrc:/icons/check.svg"
                    onClicked: {
                        var index = listView.currentIndex
                        if (typeof listView.model.get(index).id === "undefined") {
                            currentText = ""
                            currentId = -1
                            rowSelected(-1)
                        } else {
                            currentText = listView.model.get(index).name
                            currentId = listView.model.get(index).id
                            rowSelected(listView.model.get(index).id)
                        }
                        stackView.pop()
                    }
                }
            }

            ListView {
                id: listView
                anchors.fill: parent
                model: comboBoxTable.model

                ScrollIndicator.vertical: ScrollIndicator { }
                delegate:  RadioDelegate {
                    id: listViewDelegate
                    width: ListView.view.width

                    onWidthChanged: {
                        if (width < 500) {
                            layout1.visible = false;
                            layout2.visible = true;
                        } else {
                            layout1.visible = true;
                            layout2.visible = false;
                        }
                    }

                    onClicked: {
                        listView.currentIndex = index
                        console.log("Row " + index + " clicked")
                    }
                    onPressAndHold: {
                        console.log("press and hold " + index)
                        comboBoxTable.tooltipRequest(index)
                    }

                    ToolTip {
                        visible: parent.pressed && rowTooltip != ""
                        delay: Qt.styleHints.mousePressAndHoldInterval
                        contentItem: Text {
                            text: rowTooltip
                            textFormat: Text.RichText
                            clip: true
                        }
                        padding: 0
                    }

                    Control {
                        id: nameLabel
                        Layout.fillWidth: true
                        implicitHeight: childrenRect.height
                        Label {
                            anchors.verticalCenter: parent.verticalCenter
                            verticalAlignment: Text.AlignVCenter
                            text: typeof name === "undefined" ? "-" : name
                            elide: Text.ElideRight
                        }
                    }
                    Control {
                        id: dateLabel
                        Layout.fillWidth: true
                        implicitHeight: childrenRect.height
                        Label {
                            anchors.verticalCenter: parent.verticalCenter
                            verticalAlignment: Text.AlignVCenter
                            text: typeof date == "undefined" ? "" : "◆ " + date
                            elide: Text.ElideRight
                            color: TBStyle.mediumColor
                        }
                    }
                    Control {
                        id: date2Label
                        Layout.fillWidth: true
                        Label {
                            anchors.verticalCenter: parent.verticalCenter
                            verticalAlignment: Text.AlignVCenter
                            text: typeof date2 == "undefined" ? "" : "◇ " + date2
                            elide: Text.ElideRight
                            color: TBStyle.mediumColor
                            font.italic: true
                        }
                    }

                    indicator: RadioIndicator {
                        x: listViewDelegate.text
                           ? (listViewDelegate.mirrored
                              ? listViewDelegate.leftPadding
                              : listViewDelegate.width - width - listViewDelegate.rightPadding)
                           : listViewDelegate.leftPadding + (listViewDelegate.availableWidth - width)
                        y: listViewDelegate.topPadding + (listViewDelegate.availableHeight - height) / 2
                        control: listViewDelegate
                    }

                    contentItem: ColumnLayout {
                        implicitHeight: childrenRect.height
                        RowLayout {
                            id: layout1
                            Layout.fillWidth: true
                            Layout.alignment: Qt.AlignVCenter
                            visible: false
                            LayoutItemProxy { target: nameLabel }
                            LayoutItemProxy { target: dateLabel }
                            LayoutItemProxy { target: date2Label }
                        }
                        Column {
                            id: layout2
                            Layout.fillWidth: true
                            Layout.alignment: Qt.AlignVCenter
                            spacing: 0
                            visible: true
                            RowLayout {
                                width: parent.width
                                LayoutItemProxy { target: nameLabel }
                            }
                            RowLayout {
                                width: parent.width
                                LayoutItemProxy { target: dateLabel }
                                LayoutItemProxy { target: date2Label }
                            }
                        }
                    }
                }
            }
        }
    }
}
