#include "lmm_schedule.h"

LMM_Schedule::LMM_Schedule(QObject *parent)
    : QObject(parent), m_meetingPart(MeetingPart::None)
{
    sql = &Singleton<sql_class>::Instance();
    m_schedule_db_id = -1;
}

LMM_Schedule::LMM_Schedule(MeetingPart meetingPart, int sequence, int roworder, QDate date, QString theme, QString source, QString study, int time, int dbId, QObject *parent)
    : QObject(parent), m_meetingPart(meetingPart), m_sequence(sequence), m_schedule_db_id(dbId), m_date(date), m_theme(theme), m_source(source), m_study_number(study), m_roworder(roworder)
{
    sql = &Singleton<sql_class>::Instance();
    if (time > 0) {
        m_time = time;
    } else {
        switch (meetingPart) {
        case MeetingPart::LMM_TR_Talk:
            m_time = 10;
            break;
        case MeetingPart::LMM_TR_SpiritualGems:
            m_time = 10;
            break;
        case MeetingPart::LMM_TR_BibleReading:
            m_time = 4;
            break;
        case MeetingPart::LMM_FM_Discussion:
            m_time = 7;
            break;
        case MeetingPart::LMM_FM_StartingConversation:
            m_time = 2;
            break;
        case MeetingPart::LMM_FM_FollowingUp:
            m_time = 3;
            break;
        case MeetingPart::LMM_FM_MakingDisciples:
        case MeetingPart::LMM_FM_Talk:
        case MeetingPart::LMM_FM_BeliefsTalk:
        case MeetingPart::LMM_FM_BeliefsDemonstration:
            m_time = 6;
            break;
        case MeetingPart::LMM_CBS:
            m_time = 30;
            break;
        case MeetingPart::Service_Talk:
            m_time = 30;
            break;
        default:
            m_time = 10;
            break;
        }
    }
}

const MeetingPart &LMM_Schedule::meetingPart() const
{
    return m_meetingPart;
}

int LMM_Schedule::dbTalkId() const
{
    return MeetingPartClass::toDbTalkId(meetingPart()) + m_sequence;
}

int LMM_Schedule::sequence() const
{
    return m_sequence;
}

int LMM_Schedule::scheduleId() const
{
    return m_schedule_db_id;
}

int LMM_Schedule::dbTalkId(int talkId, int sequence)
{
    return talkId * 10 + sequence;
}

void LMM_Schedule::splitDbTalkId(int dbTalkId, MeetingPart &meetingPart, int &sequence)
{
    int talkId = dbTalkId / 10;
    meetingPart = MeetingPartClass::fromInt(talkId);
    sequence = dbTalkId % 10;
}

QString LMM_Schedule::getStringTalkType(int src)
{
    MeetingPart meetingPart(MeetingPartClass::fromInt(src));
    switch (meetingPart) {
    case MeetingPart::LMM_Chairman:
        return "CMN";
    case MeetingPart::LMM_TR_Talk:
        return "TRS";
    case MeetingPart::LMM_TR_SpiritualGems:
        return "DIG";
    case MeetingPart::LMM_TR_BibleReading:
        return "BiR";
    case MeetingPart::LMM_FM_Discussion:
        return "SCV";
    case MeetingPart::LMM_FM_StartingConversation:
        return "ICL";
    case MeetingPart::LMM_FM_FollowingUp:
        return "RV";
    case MeetingPart::LMM_FM_MakingDisciples:
        return "BiS";
    case MeetingPart::LMM_FM_Talk:
        return "ST";
    case MeetingPart::LMM_FM_BeliefsTalk:
        return "EB(t)";
    case MeetingPart::LMM_FM_BeliefsDemonstration:
        return "EB(d)";
    case MeetingPart::LMM_CL_Talk:
        return "LT";
    case MeetingPart::LMM_CL_Discussion:
        return "LD";
    case MeetingPart::LMM_CBS:
        return "CBS";
    case MeetingPart::Service_Talk:
        return "COT";
    default:
        return "";
    }
    return "";
}

int LMM_Schedule::getTalkTypeFromFullString(QString name)
{
    QMetaEnum e = QMetaEnum::fromType<MeetingPart>();
    static QHash<QString, MeetingPart> meetingPartList;
    if (meetingPartList.count() == 0) {
        for (int k = 0; k < e.keyCount(); k++) {
            MeetingPart meetingPart = (MeetingPart)e.value(k);
            meetingPartList.insert(MeetingPartClass::toString(meetingPart), meetingPart);
        }
    }
    if (meetingPartList.contains(name))
        return MeetingPartClass::toTalkId(meetingPartList[name]);
    else
        return -1;
}

QList<int> LMM_Schedule::getDefaultTalks()
{
    QList<int> defaultTasks;
    // TREASURES FROM GOD’S WORD
    defaultTasks.append(MeetingPartClass::toDbTalkId(MeetingPart::LMM_TR_Talk));
    defaultTasks.append(MeetingPartClass::toDbTalkId(MeetingPart::LMM_TR_SpiritualGems));
    defaultTasks.append(MeetingPartClass::toDbTalkId(MeetingPart::LMM_TR_BibleReading));
    // APPLY YOURSELF TO THE FIELD MINISTRY
    defaultTasks.append(MeetingPartClass::toDbTalkId(MeetingPart::LMM_FM_StartingConversation));
    defaultTasks.append(MeetingPartClass::toDbTalkId(MeetingPart::LMM_FM_FollowingUp));
    defaultTasks.append(MeetingPartClass::toDbTalkId(MeetingPart::LMM_FM_MakingDisciples));
    // LIVING AS CHRISTIANS
    defaultTasks.append(MeetingPartClass::toDbTalkId(MeetingPart::LMM_CL_Talk));
    defaultTasks.append(MeetingPartClass::toDbTalkId(MeetingPart::LMM_CBS));
    return defaultTasks;
}

QList<int> LMM_Schedule::getAllPossibleTalks()
{
    QList<int> possibleTasks;
    // TREASURES FROM GOD’S WORD
    possibleTasks.append(MeetingPartClass::toDbTalkId(MeetingPart::LMM_TR_Talk));
    possibleTasks.append(MeetingPartClass::toDbTalkId(MeetingPart::LMM_TR_SpiritualGems));
    possibleTasks.append(MeetingPartClass::toDbTalkId(MeetingPart::LMM_TR_BibleReading));
    // APPLY YOURSELF TO THE FIELD MINISTRY
    possibleTasks.append(MeetingPartClass::toDbTalkId(MeetingPart::LMM_FM_StartingConversation));
    possibleTasks.append(MeetingPartClass::toDbTalkId(MeetingPart::LMM_FM_FollowingUp));
    possibleTasks.append(MeetingPartClass::toDbTalkId(MeetingPart::LMM_FM_MakingDisciples));
    possibleTasks.append(MeetingPartClass::toDbTalkId(MeetingPart::LMM_FM_Discussion));
    possibleTasks.append(MeetingPartClass::toDbTalkId(MeetingPart::LMM_FM_Talk));
    possibleTasks.append(MeetingPartClass::toDbTalkId(MeetingPart::LMM_FM_BeliefsTalk));
    possibleTasks.append(MeetingPartClass::toDbTalkId(MeetingPart::LMM_FM_BeliefsDemonstration));
    // LIVING AS CHRISTIANS
    possibleTasks.append(MeetingPartClass::toDbTalkId(MeetingPart::LMM_CL_Talk));
    possibleTasks.append(MeetingPartClass::toDbTalkId(MeetingPart::LMM_CL_Discussion));
    possibleTasks.append(MeetingPartClass::toDbTalkId(MeetingPart::LMM_CL_Video));
    possibleTasks.append(MeetingPartClass::toDbTalkId(MeetingPart::LMM_CBS));
    return possibleTasks;
}

QDate LMM_Schedule::date() const
{
    return m_date;
}

void LMM_Schedule::setDate(QDate date)
{
    m_date = date;
    emit dateChanged(date);
}

int LMM_Schedule::time() const
{
    return m_time;
}

void LMM_Schedule::setTime(int time)
{
    m_time = time;
    emit timeChanged(time);
}

QString LMM_Schedule::theme() const
{
    return m_theme;
}

void LMM_Schedule::setTheme(QString theme)
{
    m_theme = theme;
    emit themeChanged(theme);
}

QString LMM_Schedule::source() const
{
    return m_source;
}

void LMM_Schedule::setSource(QString source)
{
    m_source = source;
    emit sourceChanged(source);
}

QString LMM_Schedule::study_number() const
{
    return m_study_number;
}

void LMM_Schedule::setStudy_number(QString study_number)
{
    m_study_number = study_number;
}

QString LMM_Schedule::study_name() const
{
    QString study = "1";
    QRegularExpression re(QRegularExpression::anchoredPattern("\\d*"));
    if (re.match(study_number()).hasMatch()) {
        int istudy = study_number().toInt();
        if (istudy > 0)
            return sql->selectScalar("select study_name from lmm_studies where study_number = ? and active", istudy, "").toString();
    }

    return "";
}

bool LMM_Schedule::canMultiSchool() const
{
    return LMM_Schedule::canMultiSchool(meetingPart());
}

bool LMM_Schedule::canMultiSchool(MeetingPart meetingPart)
{
    AssignmentInfos *assignmentInfos = &AssignmentInfos::Instance();
    AssignmentInfo *assignmentInfo = assignmentInfos->findAssignmentInfo(MeetingType::MidweekMeeting, meetingPart, false, 2);
    return assignmentInfo;
}

bool LMM_Schedule::canCounsel() const
{
    if (meetingPart() == MeetingPart::None)
        return false;
    try {
        AssignmentInfos *assignmentInfos = &AssignmentInfos::Instance();
        AssignmentInfo *assignmentInfo = assignmentInfos->findAssignmentInfo(MeetingType::MidweekMeeting, meetingPart());
        if (!assignmentInfo)
            throw UnknownAssignmentInfoIException(MeetingType::MidweekMeeting, 1, meetingPart(), false);
        return assignmentInfo->canCounsel();
    } catch (const UnknownAssignmentInfoIException &e) {
        QMessageBox::critical(nullptr, "TheocBase", e.what());
        return false;
    }
}

bool LMM_Schedule::canHaveAssistant() const
{
    AssignmentInfos *assignmentInfos = &AssignmentInfos::Instance();
    AssignmentInfo *assignmentInfo = assignmentInfos->findAssignmentInfo(MeetingType::MidweekMeeting, meetingPart());
    return assignmentInfo && assignmentInfo->canHaveAssistant();
}

QString LMM_Schedule::talkName() const
{
    return MeetingPartClass::toString(meetingPart());
}

int LMM_Schedule::roworder() const
{
    return m_roworder;
}

void LMM_Schedule::setRoworder(int roworder)
{
    m_roworder = roworder;
}

bool LMM_Schedule::save()
{
    if (!date().isValid())
        return false;
    if (meetingPart() == MeetingPart::None)
        return false;

    sql_item queryitems;
    queryitems.insert(":date", date());
    queryitems.insert(":talk_id", dbTalkId());
    m_schedule_db_id = sql->selectScalar("SELECT id FROM lmm_schedule WHERE date = :date AND talk_id = :talk_id and active", &queryitems, -1).toInt();

    sql_item insertitems;
    insertitems.insert("date", date());
    insertitems.insert("talk_id", dbTalkId());
    insertitems.insert("source", source());
    insertitems.insert("theme", theme());
    insertitems.insert("study_number", study_number());
    insertitems.insert("time", time());

    if (roworder() > -1) {
        insertitems.insert("roworder", roworder());
    }

    insertitems.insert("active", 1);

    bool returnVal(false);
    if (m_schedule_db_id > 0) {
        // update
        returnVal = sql->updateSql("lmm_schedule", "id", QString::number(m_schedule_db_id), &insertitems);
    } else {
        // insert new row
        m_schedule_db_id = sql->insertSql("lmm_schedule", &insertitems, "id");
        returnVal = m_schedule_db_id > 0;
    }
    return returnVal;
}

void LMM_Schedule::RemoveDuplicates()
{
    sql_class *s(&Singleton<sql_class>::Instance());
    s->execSql("UPDATE lmm_schedule SET active=1,time_stamp=strftime('%s','now') WHERE id IN (SELECT id FROM lmm_schedule s inner join (SELECT date, theme, source FROM lmm_schedule GROUP BY date, theme, source HAVING count(theme) > 1) dups on s.date = dups.date and s.theme = dups.theme)");
    s->execSql("UPDATE lmm_schedule SET active=0,time_stamp=strftime('%s','now') WHERE id IN (SELECT id FROM lmm_schedule s inner join (SELECT date, theme, source, max(talk_id) keeptalkid FROM lmm_schedule GROUP BY date, theme, source HAVING count(theme) > 1) dups on s.date = dups.date and s.theme = dups.theme and s.talk_id != keeptalkid)");
    s->execSql("update lmm_assignment set lmm_schedule_id = ifnull( (select newid.id from lmm_schedule inactive inner join lmm_schedule newid on inactive.date = newid.date and inactive.theme = newid.theme and newid.active where lmm_assignment.lmm_schedule_id = inactive.id and inactive.active = 0 and lmm_assignment.id not in (select a.id from lmm_assignment a inner join lmm_schedule activesch on a.lmm_schedule_id = activesch.id and activesch.active) ), lmm_schedule_id), time_stamp=strftime('%s','now') ");
    s->execSql("delete from lmm_assignment where lmm_schedule_id in (select s.id from lmm_meeting m inner join lmm_schedule s on m.date = s.date inner join lmm_assignment a on s.id = a.lmm_schedule_id where s.date >= '2018-01-01' and s.talk_id = 5 and a.assignee_id <> m.chairman and a.assignee_id <> m.counselor2 and a.assignee_id <> m.counselor3)");
    s->execSql("delete from lmm_assignment where id in (select a.id from (select date, lmm_schedule_id, classnumber, max(id) keepid from lmm_assignment group by date, lmm_schedule_id, classnumber having count(*) >1) dups inner join lmm_assignment a on dups.date = a.date and dups.lmm_schedule_id = a.lmm_schedule_id and dups.classnumber = a.classnumber and dups.keepid <> a.id)");
}
